/*
 * Copyright 2019 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, SidecarExpect, Selectors } from '@kui-shell/test';
import { waitForGreen, waitForRed, createNS as create, defaultModeForGet } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
const wdescribe = process.env.USE_WATCH_PANE ? describe : xdescribe;
/** name of the namespace */
const nsName = create();
var Status;
(function (Status) {
    Status["Offline"] = "red-background";
    Status["Online"] = "green-background";
})(Status || (Status = {}));
/** after a CLI.command (res), wait for a table row with the given status */
const waitForStatus = function (status, nsName, res) {
    return __awaiter(this, void 0, void 0, function* () {
        const selector = yield ReplExpect.okWithCustom({
            selector: Selectors.BY_NAME(nsName)
        })(res);
        if (status === Status.Offline) {
            return waitForRed(this.app, selector);
        }
        else {
            return waitForGreen(this.app, selector);
        }
    });
};
/** create namespace, and expect status eventually to be green */
const createNS = function (kubectl) {
    return __awaiter(this, void 0, void 0, function* () {
        it(`should create namespace from URL via ${kubectl}`, () => __awaiter(this, void 0, void 0, function* () {
            const waitForOnline = waitForStatus.bind(this, Status.Online, nsName);
            try {
                yield waitForOnline(yield CLI.command(`${kubectl} create ns ${nsName}`, this.app));
            }
            catch (err) {
                yield Common.oops(this, false)(err);
            }
        }));
    });
};
/** delete namespace, and expect status eventually to be red; */
const deleteNS = function (kubectl) {
    it(`should delete the namespace ${nsName} from URL via ${kubectl}`, () => __awaiter(this, void 0, void 0, function* () {
        try {
            const waitForOffline = waitForStatus.bind(this, Status.Offline, nsName);
            const res = yield CLI.command(`${kubectl} delete ns ${nsName}`, this.app);
            yield waitForOffline(res);
        }
        catch (err) {
            yield Common.oops(this, false)(err);
        }
    }));
};
/**
 * Drilldown to a given namespace from the watch table
 *
 */
const testDrilldown = (nsName, badgeSelector, app) => __awaiter(void 0, void 0, void 0, function* () {
    yield app.client.click(`${badgeSelector}`);
    yield SidecarExpect.open(app)
        .then(SidecarExpect.mode(defaultModeForGet))
        .then(SidecarExpect.showing(nsName));
    yield app.client.click(Selectors.SIDECAR_FULLY_CLOSE_BUTTON);
    yield SidecarExpect.fullyClosed(app);
});
/** k get ns -w */
const watchNS = function (kubectl) {
    const watchCmds = [
        //    `${kubectl} get ns -w`, <-- not guaranteed to work locally, due to table pagination
        `${kubectl} get ns ${nsName} -w`,
        `${kubectl} get -w=true --watch ns ${nsName} --watch=true -w`
    ];
    watchCmds.forEach((_watchCmd, idx) => {
        const nsNameForIter = `${nsName}-${idx}`;
        const watchCmd = _watchCmd.replace(nsName, nsNameForIter);
        it('should reload', () => Common.refresh(this));
        it(`should watch namespaces via ${watchCmd}`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                const waitForOnline = waitForStatus.bind(this, Status.Online, nsNameForIter);
                const waitForOffline = waitForStatus.bind(this, Status.Offline, nsNameForIter);
                console.error('wait for create');
                const createBadge = yield waitForOnline(yield CLI.command(`${kubectl} create ns ${nsNameForIter}`, this.app));
                console.error('start watching and split terminal');
                // execute the watch command, and expect ok in repl
                yield CLI.command(watchCmd, this.app);
                console.error('wait for watcher1');
                /**
                 * console.error('wait for watch title')
                 * // expect watcher1 has title pod
                 * await this.app.client.waitForExist(Selectors.WATCHER_N_TITLE(1)) // or watcher1, assert
                 * const title = await this.app.client.getText(Selectors.WATCHER_N_TITLE(1))
                 * assert.strictEqual(title, 'Namespace')
                 */
                const watchBadge = Selectors.CURRENT_GRID_ONLINE_FOR_SPLIT(2, nsNameForIter);
                const watchBadgeButOffline = Selectors.CURRENT_GRID_OFFLINE_FOR_SPLIT(2, nsNameForIter);
                console.error('wait for badge online');
                yield this.app.client.waitForExist(watchBadge);
                console.error('drilldown from badge');
                yield testDrilldown(nsNameForIter, watchBadge, this.app);
                console.error('wait for delete');
                const deleteBadge = yield waitForOffline(yield CLI.command(`${kubectl} delete ns ${nsNameForIter}`, this.app));
                // the create and delete badges had better still exist
                yield this.app.client.waitForExist(createBadge);
                yield this.app.client.waitForExist(deleteBadge);
                // the "online" badge from the watch had better *NOT* exist after the delete
                // (i.e. we had better actually be watching!)
                yield this.app.client.waitForExist(watchBadge, 20000, true);
                // and, conversely, that watch had better eventually show Offline
                yield this.app.client.waitForExist(watchBadgeButOffline);
                // create again
                yield waitForOnline(yield CLI.command(`${kubectl} create ns ${nsNameForIter}`, this.app));
                // the "online" badge from the watch had better now exist again after the create
                // (i.e. we had better actually be watching!)
                yield this.app.client.waitForExist(watchBadge);
                // and, conversely, that watch had better NOT show Offline
                yield this.app.client.waitForExist(watchBadgeButOffline, 20000, true);
                // delete again
                yield waitForOffline(yield CLI.command(`${kubectl} delete ns ${nsNameForIter}`, this.app));
                // the "online" badge from the watch had better *NOT* exist after the delete
                // (i.e. we had better actually be watching!)
                yield this.app.client.waitForExist(watchBadge, 20000, true);
                // and, conversely, that watch had better eventually show Offline
                yield this.app.client.waitForExist(watchBadgeButOffline);
            }
            catch (err) {
                yield Common.oops(this, false)(err);
            }
        }));
    });
};
const synonyms = ['kubectl'];
wdescribe(`kubectl watch namespace via watch pane ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    synonyms.forEach(kubectl => {
        const createIt = createNS.bind(this, kubectl);
        const deleteIt = deleteNS.bind(this, kubectl);
        const watchIt = watchNS.bind(this, kubectl);
        const checkResponseOutput = () => {
            it(`should emit a helpful message in response to ${kubectl} watch request`, () => CLI.command(`${kubectl} get pods -w`, this.app)
                .then(ReplExpect.okWithString('Output has been pinned to a watch pane'))
                .catch(Common.oops(this, true)));
        };
        //
        // here come the tests
        //
        checkResponseOutput();
        createIt();
        deleteIt();
        watchIt();
    });
});
//# sourceMappingURL=aaa-get-namespaces-with-watch.js.map