"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.plugin = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _core = require("@kui-shell/core");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2019 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

const strings = (0, _core.i18n)('plugin-core-support');
const debug = (0, _debug.default)('plugins/core-support/theme');
/**
 * The command usage model
 *
 */

const usage = {
  theme: {
    command: 'theme',
    strict: 'theme',
    docs: strings('theme.usageDocs')
  },
  themes: {
    command: 'themes',
    strict: 'themes',
    docs: strings('theme.usageDocs')
  },
  list: {
    command: 'list',
    strict: 'list',
    docs: strings('theme.usageDocs')
  },
  reset: {
    command: 'reset',
    strict: 'reset',
    docs: strings('theme.resetUsageDocs')
  },
  set: {
    command: 'set',
    strict: 'set',
    docs: 'Set the current theme',
    required: [{
      name: 'string',
      docs: strings('theme.setUsageRequiredDocs')
    }]
  }
};
/**
 * List themes
 *
 */

const list = ({
  REPL
}) => __awaiter(void 0, void 0, void 0, function* () {
  const header = {
    cells: [strings('Theme'), {
      value: strings('Style'),
      hints: "hide-with-narrow-window"
      /* HideWhenNarrow */

    }, {
      value: strings('Provider'),
      hints: "hide-with-sidecar"
      /* HideWithSidecar */

    }]
  }; // careful: the user's chosen theme might not be available in the
  // settings.themes model; e.g. they previously selected a theme that
  // has since been eliminated

  const currentTheme = () => __awaiter(void 0, void 0, void 0, function* () {
    const chosenTheme = (yield (0, _core.getPersistedThemeChoice)()) || (yield (0, _core.getDefaultTheme)());
    return (0, _core.findThemeByName)(chosenTheme) ? chosenTheme : (0, _core.getDefaultTheme)();
  });

  debug('currentTheme', yield currentTheme()); // debug('theme list', uiThemes())

  const body = (0, _core.flatten)((yield (0, _core.uiThemes)()).map(({
    plugin,
    themes
  }) => themes.map(theme => ({
    nameIdx: 0,
    cells: [theme.name, {
      value: strings(theme.style),
      hints: "hide-with-narrow-window"
      /* HideWhenNarrow */

    }, {
      value: plugin,
      hints: ["hide-with-sidecar"
      /* HideWithSidecar */
      , "sub-text"
      /* BeGrayish */
      ]
    }],
    onSelect: () => __awaiter(void 0, void 0, void 0, function* () {
      yield REPL.qexec(`theme set ${REPL.encodeComponent(theme.name)}`);
    })
  }))));

  const getSelectedIdx = () => __awaiter(void 0, void 0, void 0, function* () {
    const current = yield currentTheme();
    return body.findIndex(_ => _.cells[0] === current);
  });

  const defaultSelectedIdx = yield getSelectedIdx();
  const table = {
    apiVersion: 'kui-shell/v1',
    kind: 'RadioTable',
    title: strings('theme.Available Themes'),
    header,
    body,
    defaultSelectedIdx
  };
  return table;
});
/**
 * REPL command to switch themes
 *
 */


const set = ({
  argvNoOptions
}) => __awaiter(void 0, void 0, void 0, function* () {
  const theme = argvNoOptions[argvNoOptions.indexOf('set') + 1];
  debug('set', theme);
  yield (0, _core.switchToTheme)(theme);
  return true;
});
/**
 * The command handlers
 *
 */


const plugin = commandTree => {
  debug('plugin');
  commandTree.listen('/theme/list', list, {
    usage: usage.list
  });
  commandTree.listen('/themes', list, {
    usage: usage.themes
  });
  commandTree.listen('/theme', list, {
    usage: usage.theme
  });
  commandTree.listen('/theme/set', set, {
    usage: usage.set
  }); // returns the current persisted theme choice; helpful for debugging

  commandTree.listen('/theme/current', () => __awaiter(void 0, void 0, void 0, function* () {
    return (yield (0, _core.getPersistedThemeChoice)()) || strings('theme.currentTheme');
  }), {
    hidden: true
  }); // for debugging

  commandTree.listen('/theme/reset', _core.resetToDefaultTheme, {
    usage: usage.reset
  });
};

exports.plugin = plugin;