/*
 * Copyright 2019 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, SidecarExpect, Selectors } from '@kui-shell/test';
import { waitForGreen, waitForRed, defaultModeForGet, createNS as create, waitTillNone } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
import * as assert from 'assert';
const synonyms = ['kubectl'];
describe(`kubectl get all-namespaces ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    synonyms.forEach(kubectl => {
        /** create the given namespace */
        const createNs = (name) => {
            it(`should create namespace ${name} via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create namespace ${name}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        /** create pod in the given namespace */
        const createPod = (ns) => {
            it(`should create sample pod in namespace ${ns} from URL via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create -f https://raw.githubusercontent.com/kubernetes/examples/master/staging/pod -n ${ns}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        /** list pods across all namespaces, then find the pod corresponding to the given namespace `ns` */
        const listAndClickOn = (ns) => {
            const allNamespaces = ['--all-namespaces', '-A'];
            allNamespaces.forEach(allNamespace => {
                it(`should list pods ${allNamespace} expecting ns ${ns} via ${kubectl} then click`, () => __awaiter(this, void 0, void 0, function* () {
                    try {
                        const { app, count } = yield CLI.command(`${kubectl} get pods ${allNamespace}`, this.app);
                        yield this.app.client.waitForExist(Selectors.TABLE_TITLE(count));
                        const actualTitle = yield this.app.client.getText(Selectors.TABLE_TITLE(count));
                        assert.strictEqual(actualTitle, 'Pod');
                        const secondaryTitle = yield this.app.client.getText(Selectors.TABLE_TITLE_SECONDARY(count));
                        assert.strictEqual(secondaryTitle, 'all');
                        const selector = yield ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(ns) })({ app, count });
                        // wait for the badge to become green
                        yield waitForGreen(this.app, selector);
                        // make sure the NAME cell is clickable (as opposed to, say, the NAMESPACE cell)
                        yield this.app.client.waitForExist(`${selector} .clickable[data-key="NAME"]`);
                        // now click on that cell
                        yield this.app.client.click(`${selector} .clickable`);
                        yield SidecarExpect.open(this.app)
                            .then(SidecarExpect.mode(defaultModeForGet))
                            .then(SidecarExpect.showing('nginx', undefined, undefined, ns));
                    }
                    catch (err) {
                        return Common.oops(this, true)(err);
                    }
                }));
            });
        };
        /** delete the given namespace */
        const deleteNs = (name) => {
            it(`should delete the namespace ${name} via ${kubectl}`, () => {
                return CLI.command(`${kubectl} delete namespace ${name}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForRed(this.app, selector))
                    .then(() => waitTillNone('namespace', undefined, name))
                    .catch(Common.oops(this, true));
            });
        };
        /** delete the pod in the given namespace `ns` */
        const deletePod = (ns) => {
            it(`should delete the pod in ns ${ns} by name via ${kubectl}`, () => {
                return CLI.command(`${kubectl} delete pod nginx -n ${ns}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') }))
                    .then(selector => waitForRed(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        //
        // here start the tests
        //
        const ns1 = create();
        const ns2 = create();
        createNs(ns1);
        createNs(ns2);
        createPod(ns1);
        createPod(ns2);
        listAndClickOn(ns1);
        listAndClickOn(ns2);
        deletePod(ns1);
        deletePod(ns2);
        deleteNs(ns1);
        deleteNs(ns2);
    });
});
//# sourceMappingURL=get-all-namespaces.js.map