/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import * as React from 'react';
import { isMessageWithUsageModel, isMessageWithCode, isHTML, isMarkdownResponse, isReactResponse, isRadioTable, isTable, isMixedResponse, isUsageError } from '@kui-shell/core';
import HTMLDom from './HTMLDom';
import renderTable from '../Table';
import Markdown from '../Markdown';
import { KuiContext } from '../../../';
import RadioTableSpi from '../../spi/RadioTable';
import { isError } from '../../Views/Terminal/Block/BlockModel';
/**
 * Component that renders a "ScalarResponse", which is a command
 * response that doesn't require any particularly special
 * interpretation or visualization of the inner structure --- i.e. a
 * response that is suitable for rendering directly in the Terminal.
 *
 */
export default class Scalar extends React.PureComponent {
    constructor(props) {
        super(props);
        this.state = {
            catastrophicError: undefined
        };
    }
    static getDerivedStateFromError(error) {
        return { catastrophicError: error };
    }
    componentDidCatch(error, errorInfo) {
        console.error('catastrophic error in Scalar', error, errorInfo);
    }
    render() {
        if (this.state.catastrophicError) {
            return React.createElement("div", { className: "oops" }, this.state.catastrophicError.toString());
        }
        const { tab, response } = this.props;
        try {
            if (typeof response === 'number' || typeof response === 'string' || typeof response === 'boolean') {
                return React.createElement("pre", null, response);
            }
            else if (isRadioTable(response)) {
                return (React.createElement(KuiContext.Consumer, null, config => React.createElement(RadioTableSpi, { table: response, title: !config.disableTableTitle })));
            }
            else if (isTable(response)) {
                const renderBottomToolbar = !this.props.isPinned;
                const renderGrid = this.props.isPinned;
                return renderTable(tab, tab.REPL, response, undefined, renderBottomToolbar, renderGrid, this.props.onRender);
                // ^^^ Notes: typescript doesn't like this, and i don't know why:
                // "is not assignable to type IntrinsicAttributes..."
                // <PaginatedTable {...props} />
            }
            else if (isMixedResponse(response)) {
                return (React.createElement("div", { className: "result-vertical flex-layout", style: { flex: 1, alignItems: 'unset' } }, response.map((part, idx) => (React.createElement(Scalar, Object.assign({}, this.props, { key: idx, response: part }))))));
            }
            else if (isUsageError(response)) {
                // hopefully we can do away with this shortly
                if (typeof response.raw === 'string') {
                    return React.createElement("pre", null, response.raw);
                }
                else if (isMessageWithUsageModel(response.raw) || isMessageWithCode(response.raw)) {
                    return React.createElement("pre", null, response.raw.message);
                }
                else {
                    return React.createElement(HTMLDom, { content: response.raw });
                }
            }
            else if (isError(response)) {
                return React.createElement("div", { className: "oops" }, response.message);
            }
            else if (isReactResponse(response)) {
                return response.react;
            }
            else if (isHTML(response)) {
                // ^^^ intentionally using an "else" so that typescript double
                // checks that we've covered every case of ScalarResponse
                return React.createElement(HTMLDom, { content: response });
            }
            else if (isMarkdownResponse(response)) {
                return React.createElement(Markdown, { source: response.content });
            }
        }
        catch (err) {
            console.error('catastrophic error rendering Scalar', err);
            return React.createElement("pre", null, err.toString());
        }
        console.error('unexpected null return from Scalar:', response);
        return React.createElement("pre", { className: "oops" }, "Internal Error in command execution");
    }
}
//# sourceMappingURL=index.js.map