/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import * as React from 'react';
import { i18n, isCodedError, isHTML, isReactResponse, isMarkdownResponse, isTable, eventChannelUnsafe, isWatchable } from '@kui-shell/core';
import { hasUUID, isFinished, isProcessing, isOk, isCancelled, isEmpty, isOops } from './BlockModel';
import Scalar from '../../../Content/Scalar/';
const strings = i18n('plugin-client-common');
export default class Output extends React.PureComponent {
    constructor(props) {
        super(props);
        const streamingConsumer = this.streamingConsumer.bind(this);
        const tabUUID = props.uuid;
        if (isProcessing(props.model)) {
            eventChannelUnsafe.on(`/command/stdout/${tabUUID}/${props.model.execUUID}`, streamingConsumer);
        }
        this.state = {
            isResultRendered: false,
            streamingOutput: [],
            streamingConsumer
        };
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    streamingConsumer(part) {
        return __awaiter(this, void 0, void 0, function* () {
            if (hasUUID(this.props.model)) {
                this.setState(curState => ({
                    streamingOutput: curState.streamingOutput.concat([part])
                }));
                this.props.onRender();
                eventChannelUnsafe.emit(`/command/stdout/done/${this.props.uuid}/${this.props.model.execUUID}`);
            }
        });
    }
    static getDerivedStateFromProps(props, state) {
        if (isFinished(props.model) && !state.isResultRendered) {
            const tabUUID = props.uuid;
            if (!isEmpty(props.model)) {
                eventChannelUnsafe.off(`/command/stdout/${tabUUID}/${props.model.execUUID}`, state.streamingConsumer);
            }
            if (!isEmpty(props.model) && !isCancelled(props.model)) {
                props.onRender();
            }
            return {
                isResultRendered: true,
                streamingConsumer: undefined
            };
        }
        else {
            return state;
        }
    }
    onRender(assertHasContent) {
        if (this.props.onRender) {
            this.props.onRender();
        }
        this.setState({ assertHasContent });
    }
    stream() {
        if (this.state.streamingOutput.length > 0) {
            return (React.createElement("div", { className: "repl-result-like result-vertical", "data-stream": true }, this.state.streamingOutput.map((part, idx) => (React.createElement(Scalar, { key: idx, tab: this.props.tab, response: part, isPinned: this.props.isPinned, onRender: this.onRender.bind(this) })))));
        }
    }
    result() {
        if (isProcessing(this.props.model)) {
            return React.createElement("div", { className: "repl-result" });
        }
        else if (isEmpty(this.props.model)) {
            // no result to display for these cases
            return React.createElement(React.Fragment, null);
        }
        else {
            const statusCode = isOops(this.props.model)
                ? isCodedError(this.props.model.response)
                    ? this.props.model.response.code || this.props.model.response.statusCode
                    : 500
                : isFinished(this.props.model)
                    ? 0
                    : undefined;
            return (React.createElement("div", { className: 'repl-result' + (isOops(this.props.model) ? ' oops' : ''), "data-status-code": statusCode }, isCancelled(this.props.model) ? (React.createElement(React.Fragment, null)) : (React.createElement(Scalar, { tab: this.props.tab, response: this.props.model.response, isPinned: this.props.isPinned, onRender: this.onRender.bind(this) }))));
        }
    }
    cursor() {
        if (isProcessing(this.props.model)) {
            return (React.createElement("div", { className: "repl-result-spinner" },
                React.createElement("div", { className: "repl-result-spinner-inner" })));
        }
    }
    isShowingSomethingInTerminal(block) {
        if (isFinished(block) && !isCancelled(block) && !isEmpty(block)) {
            const { response } = block;
            return (isReactResponse(response) ||
                isHTML(response) ||
                isMarkdownResponse(response) ||
                (typeof response === 'string' && response.length > 0) ||
                (isTable(response) && response.body.length > 0) ||
                this.state.streamingOutput.length > 0);
        }
        else {
            return false;
        }
    }
    ok(hasContent) {
        if (isOk(this.props.model)) {
            if (hasContent) {
                return React.createElement("div", { className: "ok" });
            }
            else if (isWatchable(this.props.model.response)) {
                return this.props.isPinned ? (React.createElement("div", { className: "kui--hero-text" }, strings('No resources'))) : (React.createElement("div", { className: "ok" }, strings('No resources')));
            }
            else {
                return React.createElement("div", { className: "ok" }, strings('ok'));
            }
        }
    }
    render() {
        const hasContent = this.state.assertHasContent !== undefined
            ? this.state.assertHasContent
            : this.isShowingSomethingInTerminal(this.props.model);
        return (React.createElement("div", { className: 'repl-output result-vertical' + (hasContent ? ' repl-result-has-content' : '') },
            this.stream(),
            this.result(),
            this.cursor(),
            this.ok(hasContent)));
    }
}
//# sourceMappingURL=Output.js.map