import _extends from "@babel/runtime/helpers/extends";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
var _excluded = ["activation", "aria-label", "children", "className", "size"],
    _excluded2 = ["children"],
    _excluded3 = ["children"];

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { settings } from 'carbon-components';
import PropTypes from 'prop-types';
import React from 'react';
import cx from 'classnames';
import { match, matches, keys } from '../../../internal/keyboard';
import { useId } from '../../../internal/useId';
import { useControllableState } from '../../../internal/useControllableState';
var prefix = settings.prefix; // Used to manage the overall state of the ContentSwitcher

var ContentSwitcherContext = /*#__PURE__*/React.createContext(); // Used to keep track of position in a tablist

var ContentTabContext = /*#__PURE__*/React.createContext(); // Used to keep track of position in a list of tab panels

var ContentPanelContext = /*#__PURE__*/React.createContext();

function ContentSwitcher(_ref) {
  var children = _ref.children,
      _ref$defaultSelectedI = _ref.defaultSelectedIndex,
      defaultSelectedIndex = _ref$defaultSelectedI === void 0 ? 0 : _ref$defaultSelectedI,
      _onChange = _ref.onChange,
      controlledSelectedIndex = _ref.selectedIndex;
  var baseId = useId('ccs'); // The active index is used to track the element which has focus in our tablist

  var _React$useState = React.useState(defaultSelectedIndex),
      _React$useState2 = _slicedToArray(_React$useState, 2),
      activeIndex = _React$useState2[0],
      setActiveIndex = _React$useState2[1]; // The selected index is used for the tab/panel pairing which is "visible"


  var _useControllableState = useControllableState({
    value: controlledSelectedIndex,
    defaultValue: defaultSelectedIndex,
    onChange: function onChange(value) {
      if (_onChange) {
        _onChange({
          selectedIndex: value
        });
      }
    }
  }),
      _useControllableState2 = _slicedToArray(_useControllableState, 2),
      selectedIndex = _useControllableState2[0],
      setSelectedIndex = _useControllableState2[1];

  var value = {
    baseId: baseId,
    activeIndex: activeIndex,
    setActiveIndex: setActiveIndex,
    selectedIndex: selectedIndex,
    setSelectedIndex: setSelectedIndex
  };
  return /*#__PURE__*/React.createElement(ContentSwitcherContext.Provider, {
    value: value
  }, children);
}

ContentSwitcher.propTypes = {
  /**
   * Provide child elements to be rendered inside of the `ContentSwitcher`.
   * These elements should render either `ContentTabs` or `ContentPanels`
   */
  children: PropTypes.node,

  /**
   * Specify which content tab should be initially selected when the component
   * is first rendered
   */
  defaultSelectedIndex: PropTypes.number,

  /**
   * Provide an optional function which is called whenever the state of the
   * `ContentSwitcher` changes
   */
  onChange: PropTypes.func,

  /**
   * Control which content panel is currently selected. This puts the component
   * in a controlled mode and should be used along with `onChange`
   */
  selectedIndex: PropTypes.number
};
/**
 * A `ContentPanel` corresponds to a tablist in the Tabs pattern as written in
 * WAI-ARIA Authoring Practices.
 *
 * @see https://w3c.github.io/aria-practices/#tabpanel
 */

function ContentTabs(_ref2) {
  var _ref2$activation = _ref2.activation,
      activation = _ref2$activation === void 0 ? 'automatic' : _ref2$activation,
      label = _ref2['aria-label'],
      children = _ref2.children,
      customClassName = _ref2.className,
      _ref2$size = _ref2.size,
      size = _ref2$size === void 0 ? 'md' : _ref2$size,
      rest = _objectWithoutProperties(_ref2, _excluded);

  var _React$useContext = React.useContext(ContentSwitcherContext),
      activeIndex = _React$useContext.activeIndex,
      selectedIndex = _React$useContext.selectedIndex,
      setSelectedIndex = _React$useContext.setSelectedIndex,
      setActiveIndex = _React$useContext.setActiveIndex;

  var ref = React.useRef(null);
  var className = cx(customClassName, "".concat(prefix, "--content-switcher"), _defineProperty({}, "".concat(prefix, "--content-switcher--").concat(size), size));
  var count = React.Children.count(children);
  var tabs = [];

  function onKeyDown(event) {
    if (matches(event, [keys.ArrowRight, keys.ArrowLeft, keys.Home, keys.End])) {
      var nextIndex = getNextIndex(event, count, activation === 'automatic' ? selectedIndex : activeIndex);

      if (activation === 'automatic') {
        setSelectedIndex(nextIndex);
      } else if (activation === 'manual') {
        setActiveIndex(nextIndex);
      }

      tabs[nextIndex].current.focus();
    }
  }

  return (
    /*#__PURE__*/
    // eslint-disable-next-line jsx-a11y/interactive-supports-focus
    React.createElement("div", _extends({}, rest, {
      "aria-label": label,
      ref: ref,
      role: "tablist",
      className: className,
      onKeyDown: onKeyDown
    }), React.Children.map(children, function (child, index) {
      var ref = /*#__PURE__*/React.createRef();
      tabs.push(ref);
      return /*#__PURE__*/React.createElement(ContentTabContext.Provider, {
        value: index
      }, /*#__PURE__*/React.cloneElement(child, {
        ref: ref
      }));
    }))
  );
}

ContentTabs.propTypes = {
  /**
   * Specify whether the content tab should be activated automatically or
   * manually
   */
  activation: PropTypes.oneOf(['automatic', 'manual']),

  /**
   * Provide an accessible label to be read when a user interacts with this
   * component
   */
  'aria-label': PropTypes.string.isRequired,

  /**
   * Provide child elements to be rendered inside of `ContentTabs`.
   * These elements should render a `ContentTab`
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be added to the container node
   */
  className: PropTypes.string,

  /**
   * Specify the size of the Content Switcher. Currently supports either `sm`, 'md' (default) or 'lg` as an option.
   */
  size: PropTypes.oneOf(['sm', 'md', 'lg'])
};
/**
 * Get the next index for a givne keyboard event given a count of the total
 * items and the current index
 * @param {Event} event
 * @param {number} total
 * @param {number} index
 * @returns {number}
 */

function getNextIndex(event, total, index) {
  if (match(event, keys.ArrowRight)) {
    return (index + 1) % total;
  } else if (match(event, keys.ArrowLeft)) {
    return (total + index - 1) % total;
  } else if (match(event, keys.Home)) {
    return 0;
  } else if (match(event, keys.End)) {
    return total - 1;
  }
}

var ContentTab = /*#__PURE__*/React.forwardRef(function ContentTab(_ref3, ref) {
  var children = _ref3.children,
      rest = _objectWithoutProperties(_ref3, _excluded2);

  var _React$useContext2 = React.useContext(ContentSwitcherContext),
      selectedIndex = _React$useContext2.selectedIndex,
      setSelectedIndex = _React$useContext2.setSelectedIndex,
      baseId = _React$useContext2.baseId;

  var index = React.useContext(ContentTabContext);
  var id = "".concat(baseId, "-tab-").concat(index);
  var panelId = "".concat(baseId, "-tabpanel-").concat(index);
  var className = cx("".concat(prefix, "--content-switcher-btn"), _defineProperty({}, "".concat(prefix, "--content-switcher--selected"), selectedIndex === index));
  return /*#__PURE__*/React.createElement("button", _extends({}, rest, {
    "aria-controls": panelId,
    "aria-selected": selectedIndex === index,
    ref: ref,
    id: id,
    role: "tab",
    className: className,
    onClick: function onClick() {
      setSelectedIndex(index);
    },
    tabIndex: selectedIndex === index ? '0' : '-1',
    type: "button"
  }), children);
});
ContentTab.propTypes = {
  /**
   * Provide child elements to be rendered inside of `ContentTab`.
   * These elements must be noninteractive
   */
  children: PropTypes.node
};
/**
 * Used to display all of the tab panels inside of a Content Switcher. This
 * components keeps track of position in for each ContentPanel.
 *
 * Note: children should either be a `ContentPanel` or should render a
 * `ContentPanel`. Fragments are not currently supported.
 */

function ContentPanels(_ref4) {
  var children = _ref4.children;
  return React.Children.map(children, function (child, index) {
    return /*#__PURE__*/React.createElement(ContentPanelContext.Provider, {
      value: index
    }, child);
  });
}

ContentPanels.propTypes = {
  /**
   * Provide child elements to be rendered inside of `ContentPanels`.
   * These elements should render a `ContentPanel`
   */
  children: PropTypes.node
};
/**
 * A `ContentPanel` corresponds to a tabpanel in the Tabs pattern as written in
 * WAI-ARIA Authoring Practices. This component reads the selected
 * index and base id from context in order to determine the correct `id` and
 * display status of the component.
 *
 * @see https://w3c.github.io/aria-practices/#tabpanel
 */

var ContentPanel = /*#__PURE__*/React.forwardRef(function ContentPanel(props, ref) {
  var children = props.children,
      rest = _objectWithoutProperties(props, _excluded3);

  var _React$useContext3 = React.useContext(ContentSwitcherContext),
      selectedIndex = _React$useContext3.selectedIndex,
      baseId = _React$useContext3.baseId;

  var index = React.useContext(ContentPanelContext);
  var id = "".concat(baseId, "-tabpanel-").concat(index);
  var tabId = "".concat(baseId, "-tab-").concat(index); // TODO: tabindex should only be 0 if no interactive content in children

  return /*#__PURE__*/React.createElement("div", _extends({}, rest, {
    "aria-labelledby": tabId,
    id: id,
    ref: ref,
    role: "tabpanel",
    tabIndex: "0",
    hidden: selectedIndex !== index
  }), children);
});
ContentPanel.propTypes = {
  /**
   * Provide child elements to be rendered inside of `ContentPanel`.
   */
  children: PropTypes.node
};
export { ContentSwitcher, ContentTabs, ContentTab, ContentPanels, ContentPanel };