import _extends from "@babel/runtime/helpers/extends";
import _slicedToArray from "@babel/runtime/helpers/slicedToArray";
import _objectWithoutProperties from "@babel/runtime/helpers/objectWithoutProperties";
import _toConsumableArray from "@babel/runtime/helpers/toConsumableArray";
import _defineProperty from "@babel/runtime/helpers/defineProperty";
var _excluded = ["className", "onChange", "totalItems", "itemsShown", "page", "loop", "translateWithId"];

/**
 * Copyright IBM Corp. 2020
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import PropTypes from 'prop-types';
import React, { useState, useEffect, useRef } from 'react';
import classnames from 'classnames';
import { CaretRight16, CaretLeft16, OverflowMenuHorizontal16 } from '@carbon/icons-react';
import { settings } from 'carbon-components';
import Button from '../Button';
var prefix = settings.prefix;
var translationIds = {
  'carbon.pagination-nav.next': 'Next',
  'carbon.pagination-nav.previous': 'Previous',
  'carbon.pagination-nav.item': 'Page',
  'carbon.pagination-nav.active': 'Active',
  'carbon.pagination-nav.of': 'of'
};

function translateWithId(messageId) {
  return translationIds[messageId];
} // https://reactjs.org/docs/hooks-faq.html#how-to-get-the-previous-props-or-state


function usePrevious(value) {
  var ref = useRef();
  useEffect(function () {
    ref.current = value;
  });
  return ref.current;
}

function getCuts(page, totalItems, itemsThatFit) {
  var splitPoint = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : null;

  if (itemsThatFit >= totalItems) {
    return {
      front: 0,
      back: 0
    };
  }

  var split = splitPoint || Math.ceil(itemsThatFit / 2) - 1;
  var frontHidden = page + 1 - split;
  var backHidden = totalItems - page - (itemsThatFit - split) + 1;

  if (frontHidden <= 1) {
    backHidden -= frontHidden <= 0 ? Math.abs(frontHidden) + 1 : 0;
    frontHidden = 0;
  }

  if (backHidden <= 1) {
    frontHidden -= backHidden <= 0 ? Math.abs(backHidden) + 1 : 0;
    backHidden = 0;
  }

  return {
    front: frontHidden,
    back: backHidden
  };
}

function DirectionButton(_ref) {
  var direction = _ref.direction,
      label = _ref.label,
      disabled = _ref.disabled,
      onClick = _ref.onClick;
  var icon = direction === 'forward' ? CaretRight16 : CaretLeft16;
  return /*#__PURE__*/React.createElement("li", {
    className: "".concat(prefix, "--pagination-nav__list-item")
  }, /*#__PURE__*/React.createElement(Button, {
    disabled: disabled,
    renderIcon: icon,
    kind: "ghost",
    hasIconOnly: true,
    iconDescription: label,
    tooltipAlignment: "center",
    tooltipPosition: "bottom",
    onClick: onClick
  }));
}

function PaginationItem(_ref2) {
  var page = _ref2.page,
      isActive = _ref2.isActive,
      onClick = _ref2.onClick,
      _ref2$translateWithId = _ref2.translateWithId,
      t = _ref2$translateWithId === void 0 ? translateWithId : _ref2$translateWithId;
  var itemLabel = t('carbon.pagination-nav.item');
  return /*#__PURE__*/React.createElement("li", {
    className: "".concat(prefix, "--pagination-nav__list-item")
  }, /*#__PURE__*/React.createElement("button", {
    type: "button",
    className: classnames("".concat(prefix, "--pagination-nav__page"), _defineProperty({}, "".concat(prefix, "--pagination-nav__page--active"), isActive)),
    onClick: onClick,
    "data-page": page,
    "aria-current": isActive ? 'page' : null
  }, /*#__PURE__*/React.createElement("span", {
    className: "".concat(prefix, "--pagination-nav__accessibility-label")
  }, isActive ? "".concat(t('carbon.pagination-nav.active'), ", ").concat(itemLabel) : itemLabel), page));
}

function PaginationOverflow(_ref3) {
  var fromIndex = _ref3.fromIndex,
      count = _ref3.count,
      onSelect = _ref3.onSelect,
      _ref3$translateWithId = _ref3.translateWithId,
      t = _ref3$translateWithId === void 0 ? translateWithId : _ref3$translateWithId;

  if (count > 1) {
    return /*#__PURE__*/React.createElement("li", {
      className: "".concat(prefix, "--pagination-nav__list-item")
    }, /*#__PURE__*/React.createElement("div", {
      className: "".concat(prefix, "--pagination-nav__select")
    }, /*#__PURE__*/React.createElement("select", {
      className: "".concat(prefix, "--pagination-nav__page ").concat(prefix, "--pagination-nav__page--select"),
      "aria-label": "Select ".concat(t('carbon.pagination-nav.item'), " number"),
      onChange: function onChange(e) {
        var index = Number(e.target.value);
        onSelect(index);
      }
    }, /*#__PURE__*/React.createElement("option", {
      value: "",
      hidden: true
    }), _toConsumableArray(Array(count)).map(function (e, i) {
      return /*#__PURE__*/React.createElement("option", {
        value: (fromIndex + i).toString(),
        "data-page": fromIndex + i + 1,
        key: "overflow-".concat(fromIndex + i)
      }, fromIndex + i + 1);
    })), /*#__PURE__*/React.createElement("div", {
      className: "".concat(prefix, "--pagination-nav__select-icon-wrapper")
    }, /*#__PURE__*/React.createElement(OverflowMenuHorizontal16, {
      className: "".concat(prefix, "--pagination-nav__select-icon")
    }))));
  }

  if (count === 1) {
    return /*#__PURE__*/React.createElement(PaginationItem, {
      page: fromIndex + 1,
      translateWithId: t,
      onClick: function onClick() {
        onSelect(fromIndex);
      }
    });
  }

  return null;
}

var PaginationNav = /*#__PURE__*/React.forwardRef(function PaginationNav(_ref4, ref) {
  var className = _ref4.className,
      _ref4$onChange = _ref4.onChange,
      onChange = _ref4$onChange === void 0 ? function () {} : _ref4$onChange,
      totalItems = _ref4.totalItems,
      _ref4$itemsShown = _ref4.itemsShown,
      itemsShown = _ref4$itemsShown === void 0 ? 10 : _ref4$itemsShown,
      _ref4$page = _ref4.page,
      page = _ref4$page === void 0 ? 0 : _ref4$page,
      _ref4$loop = _ref4.loop,
      loop = _ref4$loop === void 0 ? false : _ref4$loop,
      _ref4$translateWithId = _ref4.translateWithId,
      t = _ref4$translateWithId === void 0 ? translateWithId : _ref4$translateWithId,
      rest = _objectWithoutProperties(_ref4, _excluded);

  var _useState = useState(page),
      _useState2 = _slicedToArray(_useState, 2),
      currentPage = _useState2[0],
      setCurrentPage = _useState2[1];

  var _useState3 = useState(itemsShown >= 4 ? itemsShown : 4),
      _useState4 = _slicedToArray(_useState3, 2),
      itemsThatFit = _useState4[0],
      setItemsThatFit = _useState4[1];

  var _useState5 = useState(getCuts(currentPage, totalItems, itemsThatFit)),
      _useState6 = _slicedToArray(_useState5, 2),
      cuts = _useState6[0],
      setCuts = _useState6[1];

  var prevPage = usePrevious(currentPage);

  function jumpToItem(index) {
    if (index >= 0 && index < totalItems) {
      setCurrentPage(index);
      onChange(index);
    }
  }

  function jumpToNext() {
    var nextIndex = currentPage + 1;

    if (nextIndex >= totalItems) {
      if (loop) {
        jumpToItem(0);
      }
    } else {
      jumpToItem(nextIndex);
    }
  }

  function jumpToPrevious() {
    var previousIndex = currentPage - 1;

    if (previousIndex < 0) {
      if (loop) {
        jumpToItem(totalItems - 1);
      }
    } else {
      jumpToItem(previousIndex);
    }
  }

  function pageWouldBeHidden(page) {
    var startOffset = itemsThatFit <= 4 && page > 1 ? 0 : 1;
    var wouldBeHiddenInFront = page >= startOffset && page <= cuts.front;
    var wouldBeHiddenInBack = page >= totalItems - cuts.back - 1 && page <= totalItems - 2;
    return wouldBeHiddenInFront || wouldBeHiddenInBack;
  } // jump to new page if props.page is updated


  useEffect(function () {
    setCurrentPage(page);
  }, [page]); // re-calculate cuts if props.totalItems or props.itemsShown change

  useEffect(function () {
    setItemsThatFit(itemsShown >= 4 ? itemsShown : 4);
    setCuts(getCuts(currentPage, totalItems, itemsShown));
  }, [totalItems, itemsShown]); // eslint-disable-line react-hooks/exhaustive-deps
  // update cuts if necessary whenever currentPage changes

  useEffect(function () {
    if (pageWouldBeHidden(currentPage)) {
      var delta = currentPage - prevPage || 0;

      if (delta > 0) {
        var splitPoint = itemsThatFit - 3;
        setCuts(getCuts(currentPage, totalItems, itemsThatFit, splitPoint));
      } else {
        var _splitPoint = itemsThatFit > 4 ? 2 : 1;

        setCuts(getCuts(currentPage, totalItems, itemsThatFit, _splitPoint));
      }
    }
  }, [currentPage]); // eslint-disable-line react-hooks/exhaustive-deps

  var classNames = classnames("".concat(prefix, "--pagination-nav"), className);
  var backwardButtonDisabled = !loop && currentPage === 0;
  var forwardButtonDisabled = !loop && currentPage === totalItems - 1;
  var startOffset = itemsThatFit <= 4 && currentPage > 1 ? 0 : 1;
  return /*#__PURE__*/React.createElement("nav", _extends({
    className: classNames,
    ref: ref
  }, rest, {
    "aria-label": "pagination"
  }), /*#__PURE__*/React.createElement("ul", {
    className: "".concat(prefix, "--pagination-nav__list")
  }, /*#__PURE__*/React.createElement(DirectionButton, {
    direction: "backward",
    label: t('carbon.pagination-nav.previous'),
    disabled: backwardButtonDisabled,
    onClick: jumpToPrevious
  }), // render first item if at least 5 items can be displayed or
  // 4 items can be displayed and the current page is either 0 or 1
  (itemsThatFit >= 5 || itemsThatFit <= 4 && currentPage <= 1) && /*#__PURE__*/React.createElement(PaginationItem, {
    page: 1,
    translateWithId: t,
    isActive: currentPage === 0,
    onClick: function onClick() {
      jumpToItem(0);
    }
  }), /*#__PURE__*/React.createElement(PaginationOverflow, {
    fromIndex: startOffset,
    count: cuts.front,
    onSelect: jumpToItem
  }), // render items between overflows
  _toConsumableArray(Array(totalItems)).map(function (e, i) {
    return i;
  }).slice(startOffset + cuts.front, (1 + cuts.back) * -1).map(function (item) {
    return /*#__PURE__*/React.createElement(PaginationItem, {
      key: "item-".concat(item),
      page: item + 1,
      translateWithId: t,
      isActive: currentPage === item,
      onClick: function onClick() {
        jumpToItem(item);
      }
    });
  }), /*#__PURE__*/React.createElement(PaginationOverflow, {
    fromIndex: totalItems - cuts.back - 1,
    count: cuts.back,
    onSelect: jumpToItem
  }), // render last item unless there is only one in total
  totalItems > 1 && /*#__PURE__*/React.createElement(PaginationItem, {
    page: totalItems,
    translateWithId: t,
    isActive: currentPage === totalItems - 1,
    onClick: function onClick() {
      jumpToItem(totalItems - 1);
    }
  }), /*#__PURE__*/React.createElement(DirectionButton, {
    direction: "forward",
    label: t('carbon.pagination-nav.next'),
    disabled: forwardButtonDisabled,
    onClick: jumpToNext
  })), /*#__PURE__*/React.createElement("div", {
    "aria-live": "polite",
    "aria-atomic": "true",
    className: "".concat(prefix, "--pagination-nav__accessibility-label")
  }, "".concat(t('carbon.pagination-nav.item'), " ").concat(currentPage + 1, " ").concat(t('carbon.pagination-nav.of'), " ").concat(totalItems)));
});
DirectionButton.propTypes = {
  /**
   * The direction this button represents ("forward" or "backward").
   */
  direction: PropTypes.oneOf(['forward', 'backward']),

  /**
   * Whether or not the button should be disabled.
   */
  disabled: PropTypes.bool,

  /**
   * The label shown in the button's tooltip.
   */
  label: PropTypes.string,

  /**
   * The callback function called when the button is clicked.
   */
  onClick: PropTypes.func
};
PaginationItem.propTypes = {
  /**
   * Whether or not this is the currently active page.
   */
  isActive: PropTypes.bool,

  /**
   * The callback function called when the item is clicked.
   */
  onClick: PropTypes.func,

  /**
   * The page number this item represents.
   */
  page: PropTypes.number,

  /**
   * Specify a custom translation function that takes in a message identifier
   * and returns the localized string for the message
   */
  translateWithId: PropTypes.func
};
PaginationOverflow.propTypes = {
  /**
   * How many items to display in this overflow.
   */
  count: PropTypes.number,

  /**
   * From which index on this overflow should start displaying pages.
   */
  fromIndex: PropTypes.number,

  /**
   * The callback function called when the user selects a page from the overflow.
   */
  onSelect: PropTypes.func,

  /**
   * Specify a custom translation function that takes in a message identifier
   * and returns the localized string for the message
   */
  translateWithId: PropTypes.func
};
PaginationNav.displayName = 'PaginationNav';
PaginationNav.propTypes = {
  /**
   * Additional CSS class names.
   */
  className: PropTypes.string,

  /**
   * The number of items to be shown.
   */
  itemsShown: PropTypes.number,

  /**
   * Whether user should be able to loop through the items when reaching first / last.
   */
  loop: PropTypes.bool,

  /**
   * The callback function called when the current page changes.
   */
  onChange: PropTypes.func,

  /**
   * The current page.
   */
  page: PropTypes.number,

  /**
   * The total number of items.
   */
  totalItems: PropTypes.number,

  /**
   * Specify a custom translation function that takes in a message identifier
   * and returns the localized string for the message
   */
  translateWithId: PropTypes.func
};
export default PaginationNav;