/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import { Search16, Close16 } from '@carbon/icons-react';
import Search from './Search';
import { mount, shallow } from 'enzyme';
import { settings } from 'carbon-components';
var prefix = settings.prefix;
describe('Search', function () {
  describe('renders as expected', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Search, {
      id: "test",
      className: "extra-class",
      label: "Search Field",
      labelText: "testlabel"
    }));
    var label = wrapper.find('label');
    var textInput = wrapper.find('input');
    var container = wrapper.find(".".concat(prefix, "--search"));
    describe('container', function () {
      it('should add extra classes that are passed via className', function () {
        expect(container.hasClass('extra-class')).toEqual(true);
      });
    });
    describe('input', function () {
      it('renders as expected', function () {
        expect(textInput.length).toBe(1);
      });
      it('has the expected classes', function () {
        expect(textInput.hasClass("".concat(prefix, "--search-input"))).toEqual(true);
      });
      it('should set type as expected', function () {
        expect(textInput.props().type).toEqual('text');
        wrapper.setProps({
          type: 'email'
        });
        expect(wrapper.find('input').props().type).toEqual('email');
      });
      it('should set value as expected', function () {
        expect(textInput.props().defaultValue).toEqual(undefined);
        wrapper.setProps({
          defaultValue: 'test'
        });
        expect(wrapper.find('input').props().defaultValue).toEqual('test');
        expect(wrapper.find('input').props().value).toEqual(undefined);
      });
      it('should set placeholder as expected', function () {
        expect(textInput.props().placeholder).toEqual('');
        wrapper.setProps({
          placeholder: 'Enter text'
        });
        expect(wrapper.find('input').props().placeholder).toEqual('Enter text');
      });
    });
    describe('label', function () {
      it('renders a label', function () {
        expect(label.length).toBe(1);
      });
      it('has the expected classes', function () {
        expect(label.hasClass("".concat(prefix, "--label"))).toEqual(true);
      });
      it('should set label as expected', function () {
        expect(wrapper.props().label).toEqual('Search Field');
        wrapper.setProps({
          label: 'Email Input'
        });
        expect(wrapper.props().label).toEqual('Email Input');
      });
    });
    describe('Large Search', function () {
      var large = mount( /*#__PURE__*/React.createElement(Search, {
        id: "test",
        size: "lg",
        className: "extra-class",
        label: "Search Field",
        labelText: "testlabel"
      }));
      var largeContainer = large.find(".".concat(prefix, "--search"));
      it('renders correct search icon', function () {
        var icons = large.find(Search16);
        expect(icons.length).toBe(1);
      });
      it('should have the expected large class', function () {
        expect(largeContainer.hasClass("".concat(prefix, "--search--lg"))).toEqual(true);
      });
      it('should only have 1 button (clear)', function () {
        var btn = large.find('button');
        expect(btn.length).toEqual(1);
      });
      it('renders two Icons', function () {
        var iconTypes = [Search16, Close16];
        var icons = large.findWhere(function (n) {
          return iconTypes.includes(n.type());
        });
        expect(icons.length).toEqual(2);
      });
      describe('buttons', function () {
        var btns = wrapper.find('button');
        it('should be one button', function () {
          expect(btns.length).toBe(1);
        });
        it('should have type="button"', function () {
          var type1 = btns.first().instance().getAttribute('type');
          var type2 = btns.last().instance().getAttribute('type');
          expect(type1).toEqual('button');
          expect(type2).toEqual('button');
        });
      });
      describe('icons', function () {
        it('renders "search" icon', function () {
          var icons = wrapper.find(Search16);
          expect(icons.length).toBe(1);
        });
        it('renders two Icons', function () {
          wrapper.setProps({
            size: undefined
          });
          var iconTypes = [Search16, Close16];
          var icons = wrapper.findWhere(function (n) {
            return iconTypes.includes(n.type());
          });
          expect(icons.length).toEqual(2);
        });
      });
    });
    describe('Small Search', function () {
      var small = mount( /*#__PURE__*/React.createElement(Search, {
        id: "test",
        size: "sm",
        className: "extra-class",
        label: "Search Field",
        labelText: "testlabel"
      }));
      var smallContainer = small.find(".".concat(prefix, "--search"));
      it('renders correct search icon', function () {
        var icons = small.find(Search16);
        expect(icons.length).toBe(1);
      });
      it('should have the expected small class', function () {
        expect(smallContainer.hasClass("".concat(prefix, "--search--sm"))).toEqual(true);
      });
      it('should only have 1 button (clear)', function () {
        var btn = small.find('button');
        expect(btn.length).toEqual(1);
      });
      it('renders two Icons', function () {
        var iconTypes = [Search16, Close16];
        var icons = wrapper.findWhere(function (n) {
          return iconTypes.includes(n.type());
        });
        expect(icons.length).toEqual(2);
      });
    });
  });
  describe('events', function () {
    describe('enabled textinput', function () {
      var onClick = jest.fn();
      var onChange = jest.fn();
      var onClear = jest.fn();
      var wrapper = shallow( /*#__PURE__*/React.createElement(Search, {
        id: "test",
        labelText: "testlabel",
        onClick: onClick,
        onChange: onChange,
        onClear: onClear
      }));
      var input = wrapper.find('input');
      var eventObject = {
        target: {
          defaultValue: 'test'
        }
      };
      it('should invoke onClick when input is clicked', function () {
        input.simulate('click');
        expect(onClick).toHaveBeenCalled();
      });
      it('should invoke onChange when input value is changed', function () {
        input.simulate('change', eventObject);
        expect(onChange).toHaveBeenCalledWith(eventObject);
      });
      it('should call onClear when input value is cleared', function () {
        var node = document.createElement('div');
        document.body.appendChild(node);
        var wrapper = mount( /*#__PURE__*/React.createElement(Search, {
          id: "test",
          labelText: "testlabel",
          onClear: onClear,
          value: "test"
        }), {
          attachTo: node
        });
        wrapper.find('button').simulate('click', {
          target: {
            value: 'test'
          }
        });
        expect(onClear).toHaveBeenCalled();
        expect(wrapper.find('input').getDOMNode()).toHaveFocus();
        document.body.removeChild(node);
      });
    });
  });
});
/**
 * Find the <input> element.
 * @param {Enzymecontainer} wrapper
 * @returns {Enzymecontainer}
 */

var getInput = function getInput(wrapper) {
  return wrapper.find(".".concat(prefix, "--search-input"));
};
/**
 * Find the value of the <input> element
 * @param {EnzymeWrapper} wrapper
 * @returns {number}
 */


var getInputValue = function getInputValue(wrapper) {
  return getInput(wrapper).prop('value');
};

describe('Detecting change in value from props', function () {
  it('should have empty value', function () {
    var search = shallow( /*#__PURE__*/React.createElement(Search, {
      id: "test",
      className: "extra-class",
      labelText: "testlabel"
    }));
    expect(getInputValue(search)).toBe(undefined);
  });
  it('should set value if value prop is added', function () {
    var search = shallow( /*#__PURE__*/React.createElement(Search, {
      id: "test",
      className: "extra-class",
      labelText: "testlabel",
      value: "foo"
    }));
    expect(getInputValue(search)).toBe('foo');
  });
});