import _typeof from "@babel/runtime/helpers/typeof";

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import { shallow, mount } from 'enzyme';
import Tabs from '../Tabs';
import Tab from '../Tab';
import TabsSkeleton from '../Tabs/Tabs.Skeleton';
var prefix = 'bx';
Element.prototype.scrollIntoView = jest.fn();
describe('Tabs', function () {
  describe('renders as expected', function () {
    describe('navigation (<div>)', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Tabs, {
        className: "extra-class"
      }, /*#__PURE__*/React.createElement(Tab, {
        label: "firstTab"
      }, "content1"), /*#__PURE__*/React.createElement(Tab, {
        label: "lastTab"
      }, "content2")));
      it('renders [role="tablist"] props on <ul> by default', function () {
        expect(wrapper.find('ul').props().role).toEqual('tablist');
      });
      it('renders extra classes on wrapping <div> via className prop', function () {
        expect(wrapper // TODO: uncomment and replace in next major version
        // .find(`.${prefix}--tabs`).hasClass('extra-class')
        .find(".".concat(prefix, "--tabs--scrollable")).hasClass('extra-class')).toBe(true);
      });
      it('renders expected classes on wrapping <div> by default', function () {
        expect(wrapper.find('div').first().hasClass("".concat(prefix, "--tabs--scrollable"))).toBe(true);
      });
      it('supports container variant', function () {
        expect(mount( /*#__PURE__*/React.createElement(Tabs, {
          className: "extra-class",
          type: "container"
        }, /*#__PURE__*/React.createElement(Tab, {
          label: "firstTab"
        }, "content1"), /*#__PURE__*/React.createElement(Tab, {
          label: "lastTab"
        }, "content2"))).find('div').first().hasClass("".concat(prefix, "--tabs--scrollable--container"))).toBe(true);
      });
      it('has no selectionMode prop', function () {
        expect('selectionMode' in wrapper // TODO: uncomment in next major version
        // .find(`.${prefix}--tabs`)
        .find(".".concat(prefix, "--tabs--scrollable")).props()).toBe(false);
      });
    });
    describe('Children (<Tab>)', function () {
      var wrapper = shallow( /*#__PURE__*/React.createElement(Tabs, null, /*#__PURE__*/React.createElement(Tab, {
        label: "firstTab"
      }, "content1"), /*#__PURE__*/React.createElement(Tab, {
        label: "lastTab"
      }, "content2")));
      var firstTab = wrapper.find('[label="firstTab"]');
      var lastTab = wrapper.find('[label="lastTab"]');
      it('renders children as expected', function () {
        expect(wrapper.props().children.length).toEqual(2);
      });
      it('renders index prop', function () {
        expect(firstTab.props().index).toEqual(0);
        expect(lastTab.props().index).toEqual(1);
      });
      it('renders selected prop (where firstTab is selected by default)', function () {
        expect(firstTab.props().selected).toEqual(true);
        expect(lastTab.props().selected).toEqual(false);
      });
    });
  });
  describe('Children (<TabContent>)', function () {
    var wrapper = shallow( /*#__PURE__*/React.createElement(Tabs, null, /*#__PURE__*/React.createElement(Tab, {
      label: "firstTab"
    }, "content1"), /*#__PURE__*/React.createElement(Tab, {
      label: "lastTab"
    }, "content2")));
    it('renders content children as expected', function () {
      expect(wrapper.find('TabContent').length).toEqual(2);
    });
    it('allows for custom classNames on <TabContent>', function () {
      var customTabContentClassWrapper = shallow( /*#__PURE__*/React.createElement(Tabs, {
        tabContentClassName: "tab-content"
      }, /*#__PURE__*/React.createElement(Tab, {
        label: "firstTab"
      }, "content1"), /*#__PURE__*/React.createElement(Tab, {
        label: "lastTab"
      }, "content2")));
      expect(customTabContentClassWrapper.find('.tab-content').length).toEqual(2);
    });
    it('renders hidden props with boolean value', function () {
      var hiddenProp = wrapper.find('TabContent').first().props().hidden;
      expect(_typeof(hiddenProp)).toBe('boolean');
    });
    it('renders selected props with boolean value', function () {
      var selectedProp = wrapper.find('TabContent').first().props().hidden;
      expect(_typeof(selectedProp)).toBe('boolean');
    });
  });
  describe('events', function () {
    describe('keydown', function () {
      var leftKey = 37;
      var rightKey = 39;
      var spaceKey = 32;
      var enterKey = 13;
      var homeKey = 36;
      var endKey = 35;
      var wrapper;
      var firstTab;
      var lastTab;
      var buttonInFirstTab;
      var buttonInLastTab;
      var spyFocusButtonInFirstTab;
      var spyFocusButtonInLastTab;
      describe('state: selected', function () {
        beforeEach(function () {
          wrapper = mount( /*#__PURE__*/React.createElement(Tabs, {
            selected: 0
          }, /*#__PURE__*/React.createElement(Tab, {
            label: "firstTab",
            className: "firstTab"
          }, "content"), /*#__PURE__*/React.createElement(Tab, {
            label: "lastTab",
            className: "lastTab"
          }, "content")));
          firstTab = wrapper.find('.firstTab').last();
          lastTab = wrapper.find('.lastTab').last();
          buttonInFirstTab = firstTab.find('button').getDOMNode();
          buttonInLastTab = lastTab.find('button').getDOMNode();
        });
        it('updates selected state when pressing arrow keys', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: rightKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
          lastTab.simulate('keydown', {
            which: leftKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('updates selected state when pressing Home and End keys', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: endKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
          lastTab.simulate('keydown', {
            which: homeKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('loops focus and selected state from lastTab to firstTab', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          lastTab.simulate('keydown', {
            which: rightKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('loops focus and selected state from firstTab to lastTab', function () {
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: leftKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
        });
        it('updates selected state when pressing space or enter key', function () {
          firstTab.simulate('keydown', {
            which: spaceKey
          });
          expect(wrapper.state().selected).toEqual(0);
          lastTab.simulate('keydown', {
            which: enterKey
          });
          expect(wrapper.state().selected).toEqual(1);
        });
      });
      describe('ignore disabled child tab', function () {
        beforeEach(function () {
          wrapper = mount( /*#__PURE__*/React.createElement(Tabs, null, /*#__PURE__*/React.createElement(Tab, {
            label: "firstTab",
            className: "firstTab"
          }, "content1"), /*#__PURE__*/React.createElement(Tab, {
            label: "middleTab",
            className: "middleTab",
            disabled: true
          }, "content2"), /*#__PURE__*/React.createElement(Tab, {
            label: "lastTab",
            className: "lastTab"
          }, "content3")));
          firstTab = wrapper.find('.firstTab').last();
          lastTab = wrapper.find('.lastTab').last();
          buttonInFirstTab = firstTab.find('button').getDOMNode();
          buttonInLastTab = lastTab.find('button').getDOMNode();
        });
        it('updates selected state when pressing arrow keys', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: rightKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
          lastTab.simulate('keydown', {
            which: leftKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('updates selected state when pressing Home and End keys', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: endKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
          lastTab.simulate('keydown', {
            which: homeKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('loops focus and selected state from lastTab to firstTab', function () {
          spyFocusButtonInFirstTab = jest.spyOn(buttonInFirstTab, 'focus');
          wrapper.setState({
            selected: 2
          });
          lastTab.simulate('keydown', {
            which: rightKey
          });
          expect(spyFocusButtonInFirstTab).toHaveBeenCalled();
        });
        it('loops focus and selected state from firstTab to lastTab', function () {
          spyFocusButtonInLastTab = jest.spyOn(buttonInLastTab, 'focus');
          firstTab.simulate('keydown', {
            which: leftKey
          });
          expect(spyFocusButtonInLastTab).toHaveBeenCalled();
        });
        it('updates selected state when pressing space or enter key', function () {
          firstTab.simulate('keydown', {
            which: spaceKey
          });
          expect(wrapper.state().selected).toEqual(0);
          lastTab.simulate('keydown', {
            which: enterKey
          });
          expect(wrapper.state().selected).toEqual(2);
        });
      });
      afterEach(function () {
        if (spyFocusButtonInLastTab) {
          spyFocusButtonInLastTab.mockRestore();
          spyFocusButtonInLastTab = null;
        }

        if (spyFocusButtonInFirstTab) {
          spyFocusButtonInFirstTab.mockRestore();
          spyFocusButtonInFirstTab = null;
        }
      });
    });
  });
  describe('default state', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Tabs, null, /*#__PURE__*/React.createElement(Tab, {
      label: "firstTab",
      className: "firstTab"
    }, "content"), /*#__PURE__*/React.createElement(Tab, {
      label: "lastTab",
      className: "lastTab"
    }, "content")));
    describe('selected', function () {
      it('should be 0', function () {
        expect(wrapper.state().selected).toEqual(0);
      });
    });
  });
  describe('Allow initial state to draw from props', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(Tabs, {
      selected: 1
    }, /*#__PURE__*/React.createElement(Tab, {
      label: "firstTab",
      className: "firstTab"
    }, "content"), /*#__PURE__*/React.createElement(Tab, {
      label: "lastTab",
      className: "lastTab"
    }, "content")));
    var children = wrapper.find(Tab);
    it('Should apply the selected property on the selected tab', function () {
      expect(children.first().props().selected).toEqual(false);
      expect(children.last().props().selected).toEqual(true);
    });
  });
});
describe('props update', function () {
  var wrapper = mount( /*#__PURE__*/React.createElement(Tabs, {
    selected: 0
  }, /*#__PURE__*/React.createElement(Tab, {
    label: "firstTab",
    className: "firstTab"
  }, "content"), /*#__PURE__*/React.createElement(Tab, {
    label: "lastTab",
    className: "lastTab"
  }, "content")));
  it('updates selected state when selected prop changes', function () {
    wrapper.setProps({
      selected: 1
    });
    expect(wrapper.state().selected).toEqual(1);
    wrapper.setProps({
      selected: 0
    });
    expect(wrapper.state().selected).toEqual(0);
  });
  it('avoids updating state upon setting props, unless there the value actually changes', function () {
    wrapper.setProps({
      selected: 1
    });
    wrapper.setState({
      selected: 2
    });
    wrapper.setProps({
      selected: 1
    });
    expect(wrapper.state().selected).toEqual(2);
  });
});
describe('selection change', function () {
  var wrapper = mount( /*#__PURE__*/React.createElement(Tabs, {
    selected: 0,
    onSelectionChange: jest.fn()
  }, /*#__PURE__*/React.createElement(Tab, {
    label: "firstTab"
  }, "content"), /*#__PURE__*/React.createElement(Tab, {
    label: "lastTab",
    className: "secondTab"
  }, "content")));
  it('updates selected state when selected prop changes', function () {
    wrapper.find('.secondTab').last().simulate('click');
    expect(wrapper.props().onSelectionChange).toHaveBeenCalledWith(1);
  });
});
describe('TabsSkeleton', function () {
  describe('Renders as expected', function () {
    var wrapper = shallow( /*#__PURE__*/React.createElement(TabsSkeleton, null));
    it('Has the expected classes', function () {
      expect(wrapper.hasClass("".concat(prefix, "--skeleton"))).toEqual(true);
      expect(wrapper.hasClass("".concat(prefix, "--tabs"))).toEqual(true);
    });
  });
});