import _slicedToArray from "@babel/runtime/helpers/slicedToArray";

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import { Tile, ClickableTile, SelectableTile, ExpandableTile, TileAboveTheFoldContent, TileBelowTheFoldContent } from './Tile';
import Link from '../../Link';
import { render, cleanup, screen } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import '@testing-library/jest-dom';
import { mount } from 'enzyme';
var prefix = 'bx';
describe('Default', function () {
  afterEach(cleanup);
  it('adds extra classes that are passed via className', function () {
    render( /*#__PURE__*/React.createElement(Tile, {
      className: "\uD83D\uDE80"
    }, "Default tile", /*#__PURE__*/React.createElement("br", null), /*#__PURE__*/React.createElement("br", null), /*#__PURE__*/React.createElement(Link, {
      href: "https://www.carbondesignsystem.com"
    }, "Link")));
    expect(screen.getByText('Default tile').classList.contains('🚀')).toBe(true);
  });
});
describe('ClickableTile', function () {
  afterEach(cleanup);
  it('renders with a link', function () {
    render( /*#__PURE__*/React.createElement(ClickableTile, {
      href: "https://www.carbondesignsystem.com"
    }, "Clickable Tile"));
    expect(screen.getByRole('link')).toBeInTheDocument();
  });
});
describe('Multi Select', function () {
  afterEach(cleanup);
  it('does not invoke the click handler if SelectableTile is disabled', function () {
    var onClick = jest.fn();
    render( /*#__PURE__*/React.createElement("div", {
      role: "group",
      "aria-label": "selectable tiles"
    }, /*#__PURE__*/React.createElement(SelectableTile, {
      id: "tile-1",
      name: "tiles",
      value: "value",
      onClick: onClick,
      disabled: true
    }, /*#__PURE__*/React.createElement("span", {
      role: "img",
      "aria-label": "vertical traffic light"
    }, "\uD83D\uDEA6"))));
    var tile = screen.getByText('🚦');
    userEvent.click(tile);
    expect(onClick).not.toHaveBeenCalled();
  });
  it('should cycle elements in document tab order', function () {
    render( /*#__PURE__*/React.createElement("div", {
      role: "group",
      "aria-label": "selectable tiles"
    }, /*#__PURE__*/React.createElement(SelectableTile, {
      "data-testid": "element",
      id: "tile-1",
      name: "tiles",
      value: "value"
    }, "tile 1"), /*#__PURE__*/React.createElement(SelectableTile, {
      "data-testid": "element",
      id: "tile-2",
      name: "tiles",
      value: "value"
    }, "tile 2"), /*#__PURE__*/React.createElement(SelectableTile, {
      "data-testid": "element",
      id: "tile-3",
      name: "tiles",
      value: "value"
    }, "tile 3")));

    var _screen$getAllByTestI = screen.getAllByTestId('element'),
        _screen$getAllByTestI2 = _slicedToArray(_screen$getAllByTestI, 3),
        id1 = _screen$getAllByTestI2[0],
        id2 = _screen$getAllByTestI2[1],
        id3 = _screen$getAllByTestI2[2];

    expect(document.body).toHaveFocus();
    userEvent.tab();
    expect(id1).toHaveFocus();
    userEvent.tab();
    expect(id2).toHaveFocus();
    userEvent.tab();
    expect(id3).toHaveFocus();
    userEvent.tab(); // cycle goes back to the body element

    expect(document.body).toHaveFocus();
    userEvent.tab();
    expect(id1).toHaveFocus();
  });
});
describe('ExpandableTile', function () {
  var wrapper = mount( /*#__PURE__*/React.createElement(ExpandableTile, {
    className: "extra-class"
  }, /*#__PURE__*/React.createElement(TileAboveTheFoldContent, {
    className: "child"
  }, /*#__PURE__*/React.createElement("div", {
    style: {
      height: '200px'
    }
  }, "Test")), /*#__PURE__*/React.createElement(TileBelowTheFoldContent, {
    className: "child"
  }, /*#__PURE__*/React.createElement("div", {
    style: {
      height: '500px'
    }
  }, "Test"), /*#__PURE__*/React.createElement("a", {
    id: "test-link",
    href: "/"
  }, "Test Link"))));
  it('renders children as expected', function () {
    expect(wrapper.props().children.length).toBe(2);
  });
  it('has the expected classes', function () {
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--expandable"))).toEqual(true);
  });
  it('renders extra classes passed in via className', function () {
    expect(wrapper.hasClass('extra-class')).toEqual(true);
  });
  it('toggles the expandable class on click', function () {
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(false);
    wrapper.simulate('click');
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
  });
  it('allows click events to be ignored using onBeforeClick', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(ExpandableTile, {
      className: "extra-class"
    }, /*#__PURE__*/React.createElement(TileAboveTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '200px'
      }
    }, "Test")), /*#__PURE__*/React.createElement(TileBelowTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '500px'
      }
    }, "Test"), /*#__PURE__*/React.createElement("a", {
      id: "test-link",
      href: "/"
    }, "Test Link"))));
    wrapper.setProps({
      expanded: false
    });
    wrapper.setProps({
      onBeforeClick: function onBeforeClick(evt) {
        return evt.target.tagName.toLowerCase() !== 'a';
      } // ignore link clicks

    });
    wrapper.simulate('click');
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
    wrapper.find('#test-link').simulate('click');
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
    wrapper.simulate('click');
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(false);
  });
  it('displays the default tooltip for the button', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(ExpandableTile, {
      className: "extra-class"
    }, /*#__PURE__*/React.createElement(TileAboveTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '200px'
      }
    }, "Test")), /*#__PURE__*/React.createElement(TileBelowTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '500px'
      }
    }, "Test"), /*#__PURE__*/React.createElement("a", {
      id: "test-link",
      href: "/"
    }, "Test Link"))));
    var defaultExpandedIconText = 'Interact to collapse Tile';
    var defaultCollapsedIconText = 'Interact to expand Tile'; // Force the expanded tile to be collapsed.

    wrapper.setProps({
      expanded: false
    });
    var collapsedDescription = wrapper.find('button').prop('title');
    expect(collapsedDescription).toEqual(defaultCollapsedIconText); // click on the item to expand it.

    wrapper.simulate('click'); // Validate the description change

    var expandedDescription = wrapper.find('button').prop('title');
    expect(expandedDescription).toEqual(defaultExpandedIconText);
  });
  it('displays the custom tooltips for the button depending on state', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(ExpandableTile, {
      className: "extra-class"
    }, /*#__PURE__*/React.createElement(TileAboveTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '200px'
      }
    }, "Test")), /*#__PURE__*/React.createElement(TileBelowTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '500px'
      }
    }, "Test"), /*#__PURE__*/React.createElement("a", {
      id: "test-link",
      href: "/"
    }, "Test Link"))));
    var tileExpandedIconText = 'Click To Collapse';
    var tileCollapsedIconText = 'Click To Expand'; // Force the custom icon text and the expanded tile to be collapsed.

    wrapper.setProps({
      tileExpandedIconText: tileExpandedIconText,
      tileCollapsedIconText: tileCollapsedIconText,
      expanded: false
    });
    var collapsedDescription = wrapper.find('button').prop('title');
    expect(collapsedDescription).toEqual(tileCollapsedIconText); // click on the item to expand it.

    wrapper.simulate('click'); // Validate the description change

    var expandedDescription = wrapper.find('button').prop('title');
    expect(expandedDescription).toEqual(tileExpandedIconText);
  });
  it('supports setting initial expanded state from props', function () {
    var wrapper = mount( /*#__PURE__*/React.createElement(ExpandableTile, {
      expanded: true
    }, /*#__PURE__*/React.createElement(TileAboveTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '200px'
      }
    }, "Test")), /*#__PURE__*/React.createElement(TileBelowTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/React.createElement("div", {
      style: {
        height: '500px'
      }
    }, "Test"))));
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
  });
  it('supports setting expanded state from props', function () {
    wrapper.setProps({
      expanded: true
    });
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
    wrapper.setProps({
      expanded: false
    });
    expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(false);
  });
}); // Todo: Testing for a disabled ClickableTile
// Todo: Testing for ExpandableTile
// Todo: Testing for RadioTile