import _extends from "@babel/runtime/helpers/extends";

/**
 * Copyright IBM Corp. 2021
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import Toggle from './Toggle';
import { render } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { settings } from 'carbon-components';
var prefix = settings.prefix;
describe('Toggle', function () {
  var props = {
    id: 'toggle-id',
    labelA: 'labelA-unchecked',
    labelB: 'labelB-checked',
    labelText: 'Toggle label',
    toggled: false,
    onToggle: function onToggle() {}
  };
  var wrapper;
  beforeEach(function () {
    wrapper = render( /*#__PURE__*/React.createElement(Toggle, props));
  });
  describe('renders as expected', function () {
    it('button and label ids should match', function () {
      var button = wrapper.getByRole('switch');
      var label = wrapper.container.querySelector('label');
      expect(button.id).toBe(label.htmlFor);
    });
    it('renders labelA when unchecked', function () {
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        toggled: false
      })));
      expect(wrapper.queryByText(props.labelA)).toBeTruthy();
      expect(wrapper.queryByText(props.labelB)).toBeNull();
    });
    it('renders labelB when checked', function () {
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        toggled: true
      })));
      expect(wrapper.queryByText(props.labelA)).toBeNull();
      expect(wrapper.queryByText(props.labelB)).toBeTruthy();
    });
    it('supports additional css class names', function () {
      var className = 'some-additional-class';
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        className: className
      })));
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle")).classList.contains(className)).toBe(true);
    });
    it('supports sm size', function () {
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__appearance")).classList.contains("".concat(prefix, "--toggle__appearance--sm"))).toBe(false);
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__check"))).toBeNull();
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        size: "sm"
      })));
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__appearance")).classList.contains("".concat(prefix, "--toggle__appearance--sm"))).toBe(true);
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__check"))).toBeTruthy();
    });
    it('supports to use top label as side label', function () {
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        hideLabel: true
      })));
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__label-text")).classList.contains("".concat(prefix, "--visually-hidden"))).toBe(true);
      expect(wrapper.container.querySelector(".".concat(prefix, "--toggle__label-text")).textContent).toBe(props.labelText);
    });
  });
  describe('behaves as expected', function () {
    it('supports to be disabled', function () {
      expect(wrapper.getByRole('switch').disabled).toBe(false);
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        disabled: true
      })));
      expect(wrapper.getByRole('switch').disabled).toBe(true);
    });
    it('can be controlled with props.toggled', function () {
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        toggled: false
      })));
      expect(wrapper.getByRole('switch').getAttribute('aria-checked')).toBe('false');
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        toggled: true
      })));
      expect(wrapper.getByRole('switch').getAttribute('aria-checked')).toBe('true');
    });
  });
  describe('emits events as expected', function () {
    it('passes along props.onClick to button', function () {
      var onClick = jest.fn();
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        onClick: onClick
      })));
      expect(onClick).not.toHaveBeenCalled();
      userEvent.click(wrapper.getByRole('switch'));
      expect(onClick).toHaveBeenCalledTimes(1);
    });
    it('emits props.onToggle when toggled and passes current state', function () {
      var onToggle = jest.fn();
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        onToggle: onToggle,
        toggled: false
      })));
      expect(onToggle).not.toHaveBeenCalled();
      userEvent.click(wrapper.getByRole('switch'));
      expect(onToggle).toHaveBeenCalledTimes(1);
      expect(onToggle.mock.calls[0][0]).toBe(true);
      wrapper.rerender( /*#__PURE__*/React.createElement(Toggle, _extends({}, props, {
        onToggle: onToggle,
        toggled: true
      })));
      userEvent.click(wrapper.getByRole('switch'));
      expect(onToggle).toHaveBeenCalledTimes(2);
      expect(onToggle.mock.calls[1][0]).toBe(false);
    });
  });
});