/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { ExecType, isHeadless, inBrowser, hasProxy } from '@kui-shell/core';
const debug = Debug('plugins/bash-like/cmds/catchall');
/**
 * Command handler that dispatches to an outer shell
 *
 */
export const dispatchToShell = ({ tab, command, argv, argvNoOptions, execOptions, parsedOptions, createOutputStream }) => __awaiter(void 0, void 0, void 0, function* () {
    /** trim the first part of "/bin/sh: someNonExistentCommand: command not found" */
    const cleanUpError = (err) => {
        if (err.message && typeof err.message === 'string') {
            throw new Error(err.message.replace(/[a-zA-Z0-9/]+:\s*/, '').trim());
        }
        else {
            throw err;
        }
    };
    const useRaw = (execOptions.raw || execOptions.type === ExecType.Nested) &&
        execOptions.quiet === undefined &&
        execOptions.echo === undefined &&
        execOptions.replSilence === undefined;
    const eOptions = useRaw || execOptions.isProxied
        ? execOptions
        : Object.assign({}, { stdout: yield createOutputStream() }, execOptions);
    const actualCommand = command.replace(/^(!|sendtopty)\s+/, '');
    if (isHeadless() || (!inBrowser() && useRaw)) {
        const { doExec } = yield import('./bash-like');
        const actualArgv = argv[0] === 'sendtopty' ? argv.slice(1) : argv;
        const response = yield doExec(actualCommand, actualArgv, eOptions).catch(cleanUpError);
        if (useRaw && (typeof response === 'number' || typeof response === 'boolean')) {
            return response;
        }
        else if (useRaw && typeof response === 'string') {
            try {
                return !isNaN(parseInt(response)) ? response : JSON.parse(response); // prevents numeric conversion e.g. JSON.parse('1') = 1
            }
            catch (err) {
                // debug('response maybe is not JSON', response)
            }
        }
        return response;
    }
    else {
        const { doExec } = yield import(/* webpackMode: "lazy" */ '../../pty/client');
        const exec = () => doExec(tab, actualCommand, argvNoOptions, parsedOptions, eOptions);
        if (useRaw) {
            eOptions.quiet = true;
            eOptions.echo = false;
            eOptions.replSilence = true;
            let response = '';
            eOptions.onInit = () => (_) => {
                if (typeof _ === 'string') {
                    response += _;
                }
            };
            return exec()
                .then(() => response)
                .catch(cleanUpError);
        }
        return exec().catch(cleanUpError);
    }
});
/**
 * Execute the given command using a pty, but return a string
 *
 */
export function doExecWithStdoutViaPty(args) {
    return __awaiter(this, void 0, void 0, function* () {
        // eslint-disable-next-line no-async-promise-executor
        return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
            let stdout = '';
            // a bit of plumbing: tell the PTY that we will be handling everything
            const myExecOptions = Object.assign({}, args.execOptions, {
                rethrowErrors: true,
                quiet: true,
                replSilence: true,
                echo: false,
                // the PTY will call this when the PTY process is ready; in
                // return, we send it back a consumer of streaming output
                onInit: () => {
                    return (chunk) => {
                        if (typeof chunk === 'string') {
                            stdout += chunk;
                        }
                    };
                }
            });
            const myArgs = Object.assign({}, args, { execOptions: myExecOptions });
            yield dispatchToShell(myArgs).catch(err => {
                // if the PTY emitted anything on stdout, use this as the message
                const message = stdout || err.message;
                if (stdout && isHeadless()) {
                    // avoid stack traces to our own code? see
                    // https://github.com/kubernetes-sigs/kui/issues/7334
                    console.error(message);
                }
                else {
                    console.error(err);
                }
                err.message = message;
                reject(err);
            });
            resolve(stdout);
        }));
    });
}
/**
 * On preload, register the catchall handler
 *
 */
export const preload = (commandTree) => {
    if (inBrowser() && !hasProxy()) {
        debug('skipping catchall registration: in browser and no remote proxy to support it');
        return;
    }
    //
    // if we aren't running in a browser, then pass any command not
    // found exceptions to the outer shell
    //
    return commandTree.catchall(() => true, // we will accept anything
    dispatchToShell, // command handler dispatches to outer shell
    0 // priority
    );
};
//# sourceMappingURL=catchall.js.map