/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import React from 'react';
import { eventChannelUnsafe, eventBus, initializeSession } from '@kui-shell/core';
import KuiContext from './context';
const Confirm = React.lazy(() => import('../Views/Confirm'));
import getSize from '../Views/Terminal/getSize';
import ScrollableTerminal from '../Views/Terminal/ScrollableTerminal';
/**
 *
 * TabContent
 * ----------------  <Tab/> from here down
 * | ST  |        |
 * |     |        |
 * |     |        |
 * |     |        |
 * |     |        |
 * |     |        |
 * ----------------
 *  ST: <ScrollableTerminal/>
 *
 */
export default class TabContent extends React.PureComponent {
    constructor(props) {
        super(props);
        this.cleaners = [];
        /** switching back or away from this tab */
        this.activateHandlers = [];
        this.state = {
            tab: undefined,
            sessionInit: 'NotYet',
            showSessionInitDone: true,
            _terminal: undefined
        };
    }
    componentDidMount() {
        const onTabNew = () => {
            this.setState({ sessionInit: 'Done' });
            try {
                if (this.props.onTabReady) {
                    this.props.onTabReady(this.state.tab);
                }
            }
            catch (err) {
                console.error('Error in onTabReady', err);
            }
        };
        eventChannelUnsafe.once(`/tab/new/${this.props.uuid}`, onTabNew);
        this.cleaners.push(() => eventChannelUnsafe.off(`/tab/new/${this.props.uuid}`, onTabNew));
        const onError = (sessionInitError) => {
            this.setState({ sessionInit: 'Error', sessionInitError });
        };
        eventChannelUnsafe.on(`/tab/new/error/${this.props.uuid}`, onError);
        this.cleaners.push(() => eventChannelUnsafe.off(`/tab/new/error/${this.props.uuid}`, onError));
        const onOffline = this.onOffline.bind(this);
        eventBus.onWithTabId('/tab/offline', this.props.uuid, onOffline);
        this.cleaners.push(() => eventBus.offWithTabId('/tab/offline', this.props.uuid, onOffline));
    }
    onOffline() {
        return __awaiter(this, void 0, void 0, function* () {
            this.setState({
                sessionInit: 'Reinit'
            });
            initializeSession(this.state.tab)
                .then(() => {
                this.setState({
                    sessionInit: 'Done'
                });
            })
                .catch(TabContent.onSessionInitError.bind(undefined, this.props.uuid));
        });
    }
    static onSessionInitError(uuid, sessionInitError) {
        eventChannelUnsafe.emit(`/tab/new/error/${uuid}`, sessionInitError);
    }
    /** emit /tab/new event, if we have now a tab, but have not yet
     * emitted the event */
    static getDerivedStateFromProps(props, state) {
        if (state.tab && state.sessionInit === 'NotYet') {
            try {
                state.tab.state = props.state;
                // session init hook goes here
                initializeSession(state.tab)
                    .then(() => {
                    eventBus.emit('/tab/new', state.tab);
                    eventChannelUnsafe.emit(`/tab/new/${props.uuid}`, state.tab);
                })
                    .catch(TabContent.onSessionInitError.bind(undefined, props.uuid));
                // TabContent.hackResizer(state)
                return {
                    sessionInit: 'InProgress'
                };
            }
            catch (err) {
                console.error(err);
            }
        }
        else {
            if (props.active && state._terminal) {
                state._terminal.doFocusIfNeeded();
            }
            return state;
        }
    }
    componentWillUnmount() {
        eventBus.emit('/tab/close', this.state.tab);
        this.cleaners.forEach(cleaner => cleaner());
    }
    defaultLoading() {
        // this is a failsafe
        return 'Please wait while we connect to your cluster';
    }
    terminal() {
        if (this.state.sessionInit !== 'Done') {
            return (React.createElement(KuiContext.Consumer, null, config => {
                if (this.state.sessionInit === 'Error' && config.loadingError) {
                    return config.loadingError(this.state.sessionInitError);
                }
                else if (this.state.sessionInit === 'Reinit' && config.reinit) {
                    return config.reinit;
                }
                return config.loading || this.defaultLoading();
            }));
        }
        else {
            return (React.createElement(React.Fragment, null,
                React.createElement(KuiContext.Consumer, null, config => (React.createElement(ScrollableTerminal, Object.assign({}, this.props, { tab: this.state.tab, config: config, onClear: () => {
                        this.setState({ showSessionInitDone: false });
                        // reset the status stripe on clearing of the terminal
                        // eslint false positive:
                        // eslint-disable-next-line react/no-direct-mutation-state
                        this.state.tab.state.desiredStatusStripeDecoration = { type: 'default' };
                    }, ref: _terminal => {
                        // so that we can refocus/blur
                        this.setState({ _terminal });
                    } }), this.children())))));
        }
    }
    graft(node, key) {
        if (React.isValidElement(node)) {
            // ^^^ this check avoids tsc errors
            return React.cloneElement(node, {
                key,
                uuid: this.props.uuid,
                active: this.props.active
            });
        }
        else {
            return node;
        }
    }
    /** Graft on the REPL focus management */
    children() {
        if (Array.isArray(this.props.children)) {
            return this.props.children.map((child, idx) => this.graft(child, idx));
        }
        else {
            return this.graft(this.props.children);
        }
    }
    /** Graft on the tab uuid */
    bottom() {
        if (React.isValidElement(this.props.bottom)) {
            // ^^^ this check avoids tsc errors
            return React.cloneElement(this.props.bottom, {
                uuid: this.props.uuid,
                tab: this.state.tab
            });
        }
        else {
            return this.props.bottom;
        }
    }
    /** Construct the `className` property of the tab element */
    tabClassName() {
        return ('kui--tab-content' +
            (this.props.active ? ' visible' : '') +
            (!this.state.tabClassList ? '' : ' ' + Object.keys(this.state.tabClassList).join(' ')));
    }
    render() {
        this.activateHandlers.forEach(handler => handler(this.props.active));
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { ref: c => {
                    const tab = c;
                    this.setState({ tab });
                    if (tab) {
                        tab.uuid = this.props.uuid;
                        tab.getSize = getSize.bind(c);
                        tab.scrollToBottom = () => {
                            c.scrollTop = c.scrollHeight;
                        };
                        tab.onActivate = (handler) => {
                            this.activateHandlers.push(handler);
                        };
                        tab.offActivate = (handler) => {
                            const idx = this.activateHandlers.findIndex(_ => _ === handler);
                            if (idx >= 0) {
                                this.activateHandlers.splice(idx, 1);
                            }
                        };
                        tab.addClass = (cls) => {
                            this.setState(curState => {
                                if (!curState.tabClassList || !curState.tabClassList[cls]) {
                                    return {
                                        tabClassList: Object.assign({}, curState.tabClassList, { [cls]: true })
                                    };
                                }
                            });
                        };
                        tab.removeClass = (cls) => {
                            this.setState(curState => {
                                if (curState.tabClassList && curState.tabClassList[cls]) {
                                    const update = Object.assign({}, curState.tabClassList);
                                    delete update[cls];
                                    return {
                                        tabClassList: update
                                    };
                                }
                            });
                        };
                    }
                }, className: this.tabClassName(), "data-tab-id": this.props.uuid },
                React.createElement("div", { className: "kui--rows" },
                    React.createElement("div", { className: "kui--columns", style: { position: 'relative' } }, this.terminal()),
                    this.bottom()),
                this.state.tab && (React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
                    React.createElement(Confirm, { tab: this.state.tab, uuid: this.props.uuid }))))));
    }
}
//# sourceMappingURL=TabContent.js.map