"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.cloneObject = exports.processHTTPRequest = exports.runHttpQuery = exports.throwHttpGraphQLError = exports.HttpQueryError = void 0;
const apollo_server_env_1 = require("apollo-server-env");
const graphqlOptions_1 = require("./graphqlOptions");
const apollo_server_errors_1 = require("apollo-server-errors");
const requestPipeline_1 = require("./requestPipeline");
class HttpQueryError extends Error {
    constructor(statusCode, message, isGraphQLError = false, headers) {
        super(message);
        this.name = 'HttpQueryError';
        this.statusCode = statusCode;
        this.isGraphQLError = isGraphQLError;
        this.headers = headers;
    }
}
exports.HttpQueryError = HttpQueryError;
function throwHttpGraphQLError(statusCode, errors, options, extensions) {
    const defaultHeaders = { 'Content-Type': 'application/json' };
    const headers = apollo_server_errors_1.hasPersistedQueryError(errors)
        ? Object.assign(Object.assign({}, defaultHeaders), { 'Cache-Control': 'private, no-cache, must-revalidate' }) : defaultHeaders;
    const result = {
        errors: options
            ? apollo_server_errors_1.formatApolloErrors(errors, {
                debug: options.debug,
                formatter: options.formatError,
            })
            : errors,
    };
    if (extensions) {
        result.extensions = extensions;
    }
    throw new HttpQueryError(statusCode, prettyJSONStringify(result), true, headers);
}
exports.throwHttpGraphQLError = throwHttpGraphQLError;
function runHttpQuery(handlerArguments, request) {
    return __awaiter(this, void 0, void 0, function* () {
        let options;
        const debugDefault = process.env.NODE_ENV !== 'production' && process.env.NODE_ENV !== 'test';
        try {
            options = yield graphqlOptions_1.resolveGraphqlOptions(request.options, ...handlerArguments);
        }
        catch (e) {
            return throwHttpGraphQLError(500, [e], { debug: debugDefault });
        }
        if (options.debug === undefined) {
            options.debug = debugDefault;
        }
        if (typeof options.context === 'function') {
            try {
                options.context();
            }
            catch (e) {
                e.message = `Context creation failed: ${e.message}`;
                if (e.extensions &&
                    e.extensions.code &&
                    e.extensions.code !== 'INTERNAL_SERVER_ERROR') {
                    return throwHttpGraphQLError(400, [e], options);
                }
                else {
                    return throwHttpGraphQLError(500, [e], options);
                }
            }
        }
        const config = {
            schema: options.schema,
            schemaHash: options.schemaHash,
            logger: options.logger,
            rootValue: options.rootValue,
            context: options.context || {},
            validationRules: options.validationRules,
            executor: options.executor,
            fieldResolver: options.fieldResolver,
            cache: options.cache,
            dataSources: options.dataSources,
            documentStore: options.documentStore,
            extensions: options.extensions,
            persistedQueries: options.persistedQueries,
            tracing: options.tracing,
            formatError: options.formatError,
            formatResponse: options.formatResponse,
            debug: options.debug,
            plugins: options.plugins || [],
        };
        return processHTTPRequest(config, request);
    });
}
exports.runHttpQuery = runHttpQuery;
function processHTTPRequest(options, httpRequest) {
    return __awaiter(this, void 0, void 0, function* () {
        let requestPayload;
        switch (httpRequest.method) {
            case 'POST':
                if (!httpRequest.query || Object.keys(httpRequest.query).length === 0) {
                    throw new HttpQueryError(500, 'POST body missing. Did you forget use body-parser middleware?');
                }
                requestPayload = httpRequest.query;
                break;
            case 'GET':
                if (!httpRequest.query || Object.keys(httpRequest.query).length === 0) {
                    throw new HttpQueryError(400, 'GET query missing.');
                }
                requestPayload = httpRequest.query;
                break;
            default:
                throw new HttpQueryError(405, 'Apollo Server supports only GET/POST requests.', false, {
                    Allow: 'GET, POST',
                });
        }
        options = Object.assign(Object.assign({}, options), { plugins: [checkOperationPlugin, ...options.plugins] });
        function buildRequestContext(request) {
            const context = cloneObject(options.context);
            return {
                logger: options.logger || console,
                schema: options.schema,
                schemaHash: options.schemaHash,
                request,
                response: {
                    http: {
                        headers: new apollo_server_env_1.Headers(),
                    },
                },
                context,
                cache: options.cache,
                debug: options.debug,
                metrics: {},
            };
        }
        const responseInit = {
            headers: {
                'Content-Type': 'application/json',
            },
        };
        let body;
        try {
            if (Array.isArray(requestPayload)) {
                const requests = requestPayload.map(requestParams => parseGraphQLRequest(httpRequest.request, requestParams));
                const responses = yield Promise.all(requests.map((request) => __awaiter(this, void 0, void 0, function* () {
                    try {
                        const requestContext = buildRequestContext(request);
                        return yield requestPipeline_1.processGraphQLRequest(options, requestContext);
                    }
                    catch (error) {
                        return {
                            errors: apollo_server_errors_1.formatApolloErrors([error], options),
                        };
                    }
                })));
                body = prettyJSONStringify(responses.map(serializeGraphQLResponse));
            }
            else {
                const request = parseGraphQLRequest(httpRequest.request, requestPayload);
                try {
                    const requestContext = buildRequestContext(request);
                    const response = yield requestPipeline_1.processGraphQLRequest(options, requestContext);
                    if (response.errors && typeof response.data === 'undefined') {
                        return throwHttpGraphQLError((response.http && response.http.status) || 400, response.errors, undefined, response.extensions);
                    }
                    if (response.http) {
                        for (const [name, value] of response.http.headers) {
                            responseInit.headers[name] = value;
                        }
                    }
                    body = prettyJSONStringify(serializeGraphQLResponse(response));
                }
                catch (error) {
                    if (error instanceof requestPipeline_1.InvalidGraphQLRequestError) {
                        throw new HttpQueryError(400, error.message);
                    }
                    else if (error instanceof apollo_server_errors_1.PersistedQueryNotSupportedError ||
                        error instanceof apollo_server_errors_1.PersistedQueryNotFoundError) {
                        return throwHttpGraphQLError(200, [error], options);
                    }
                    else {
                        throw error;
                    }
                }
            }
        }
        catch (error) {
            if (error instanceof HttpQueryError) {
                throw error;
            }
            return throwHttpGraphQLError(500, [error], options);
        }
        responseInit.headers['Content-Length'] = Buffer.byteLength(body, 'utf8').toString();
        return {
            graphqlResponse: body,
            responseInit,
        };
    });
}
exports.processHTTPRequest = processHTTPRequest;
function parseGraphQLRequest(httpRequest, requestParams) {
    let queryString = requestParams.query;
    let extensions = requestParams.extensions;
    if (typeof extensions === 'string' && extensions !== '') {
        try {
            extensions = JSON.parse(extensions);
        }
        catch (error) {
            throw new HttpQueryError(400, 'Extensions are invalid JSON.');
        }
    }
    if (queryString && typeof queryString !== 'string') {
        if (queryString.kind === 'Document') {
            throw new HttpQueryError(400, "GraphQL queries must be strings. It looks like you're sending the " +
                'internal graphql-js representation of a parsed query in your ' +
                'request instead of a request in the GraphQL query language. You ' +
                'can convert an AST to a string using the `print` function from ' +
                '`graphql`, or use a client like `apollo-client` which converts ' +
                'the internal representation to a string for you.');
        }
        else {
            throw new HttpQueryError(400, 'GraphQL queries must be strings.');
        }
    }
    const operationName = requestParams.operationName;
    let variables = requestParams.variables;
    if (typeof variables === 'string' && variables !== '') {
        try {
            variables = JSON.parse(variables);
        }
        catch (error) {
            throw new HttpQueryError(400, 'Variables are invalid JSON.');
        }
    }
    return {
        query: queryString,
        operationName,
        variables,
        extensions,
        http: httpRequest,
    };
}
const checkOperationPlugin = {
    requestDidStart() {
        return {
            didResolveOperation({ request, operation }) {
                if (!request.http)
                    return;
                if (request.http.method === 'GET' && operation.operation !== 'query') {
                    throw new HttpQueryError(405, `GET supports only query operation`, false, {
                        Allow: 'POST',
                    });
                }
            },
        };
    },
};
function serializeGraphQLResponse(response) {
    return {
        errors: response.errors,
        data: response.data,
        extensions: response.extensions,
    };
}
function prettyJSONStringify(value) {
    return JSON.stringify(value) + '\n';
}
function cloneObject(object) {
    return Object.assign(Object.create(Object.getPrototypeOf(object)), object);
}
exports.cloneObject = cloneObject;
//# sourceMappingURL=runHttpQuery.js.map