'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopNamespace(e) {
    if (e && e.__esModule) { return e; } else {
        var n = {};
        if (e) {
            Object.keys(e).forEach(function (k) {
                var d = Object.getOwnPropertyDescriptor(e, k);
                Object.defineProperty(n, k, d.get ? d : {
                    enumerable: true,
                    get: function () {
                        return e[k];
                    }
                });
            });
        }
        n['default'] = e;
        return n;
    }
}

const graphql = require('graphql');
const utils = require('@graphql-tools/utils');
const validUrl = require('valid-url');
const crossFetch = require('cross-fetch');
const wrap = require('@graphql-tools/wrap');
const subscriptionsTransportWs = require('subscriptions-transport-ws');
const websocket = require('websocket');

/* eslint-disable no-case-declarations */
/**
 * This loader loads a schema from a URL. The loaded schema is a fully-executable,
 * remote schema since it's created using [@graphql-tools/wrap](/docs/remote-schemas).
 *
 * ```
 * const schema = await loadSchema('http://localhost:3000/graphql', {
 *   loaders: [
 *     new UrlLoader(),
 *   ]
 * });
 * ```
 */
class UrlLoader {
    loaderId() {
        return 'url';
    }
    async canLoad(pointer, options) {
        return this.canLoadSync(pointer, options);
    }
    canLoadSync(pointer, _options) {
        return !!validUrl.isWebUri(pointer);
    }
    buildAsyncExecutor({ pointer, fetch, extraHeaders, defaultMethod, useGETForQueries, }) {
        const HTTP_URL = pointer.replace('ws:', 'http:').replace('wss:', 'https:');
        return async ({ document, variables }) => {
            let method = defaultMethod;
            if (useGETForQueries) {
                method = 'GET';
                for (const definition of document.definitions) {
                    if (definition.kind === graphql.Kind.OPERATION_DEFINITION) {
                        if (definition.operation !== 'query') {
                            method = defaultMethod;
                        }
                    }
                }
            }
            let fetchResult;
            const query = graphql.print(document);
            switch (method) {
                case 'GET':
                    const urlObj = new URL(HTTP_URL);
                    urlObj.searchParams.set('query', query);
                    urlObj.searchParams.set(variables, JSON.stringify(variables));
                    const finalUrl = urlObj.toString();
                    fetchResult = await fetch(finalUrl, {
                        method: 'GET',
                        headers: extraHeaders,
                    });
                    break;
                case 'POST':
                    fetchResult = await fetch(HTTP_URL, {
                        method: 'POST',
                        body: JSON.stringify({
                            query,
                            variables,
                        }),
                        headers: extraHeaders,
                    });
                    break;
            }
            return fetchResult.json();
        };
    }
    buildSubscriber(pointer, webSocketImpl) {
        const WS_URL = pointer.replace('http:', 'ws:').replace('https:', 'wss:');
        const subscriptionClient = new subscriptionsTransportWs.SubscriptionClient(WS_URL, {}, webSocketImpl);
        return async ({ document, variables }) => {
            return utils.observableToAsyncIterable(subscriptionClient.request({
                query: document,
                variables,
            }));
        };
    }
    async getExecutorAndSubscriber(pointer, options) {
        let headers = {};
        let fetch = crossFetch.fetch;
        let defaultMethod = 'POST';
        let webSocketImpl = websocket.w3cwebsocket;
        if (options) {
            if (Array.isArray(options.headers)) {
                headers = options.headers.reduce((prev, v) => ({ ...prev, ...v }), {});
            }
            else if (typeof options.headers === 'object') {
                headers = options.headers;
            }
            if (options.customFetch) {
                if (typeof options.customFetch === 'string') {
                    const [moduleName, fetchFnName] = options.customFetch.split('#');
                    fetch = await new Promise(function (resolve) { resolve(_interopNamespace(require(moduleName))); }).then(module => (fetchFnName ? module[fetchFnName] : module));
                }
                else {
                    fetch = options.customFetch;
                }
            }
            if (options.webSocketImpl) {
                if (typeof options.webSocketImpl === 'string') {
                    const [moduleName, webSocketImplName] = options.webSocketImpl.split('#');
                    webSocketImpl = await new Promise(function (resolve) { resolve(_interopNamespace(require(moduleName))); }).then(module => webSocketImplName ? module[webSocketImplName] : module);
                }
                else {
                    webSocketImpl = options.webSocketImpl;
                }
            }
            if (options.method) {
                defaultMethod = options.method;
            }
        }
        const extraHeaders = {
            Accept: 'application/json',
            'Content-Type': 'application/json',
            ...headers,
        };
        const executor = this.buildAsyncExecutor({
            pointer,
            fetch,
            extraHeaders,
            defaultMethod,
            useGETForQueries: options.useGETForQueries,
        });
        let subscriber;
        if (options.enableSubscriptions) {
            subscriber = this.buildSubscriber(pointer, webSocketImpl);
        }
        return {
            executor,
            subscriber,
        };
    }
    async getSubschemaConfig(pointer, options) {
        const { executor, subscriber } = await this.getExecutorAndSubscriber(pointer, options);
        return {
            schema: await wrap.introspectSchema(executor, undefined, options),
            executor,
            subscriber,
        };
    }
    async load(pointer, options) {
        const subschemaConfig = await this.getSubschemaConfig(pointer, options);
        const remoteExecutableSchema = wrap.wrapSchema(subschemaConfig);
        return {
            location: pointer,
            schema: remoteExecutableSchema,
        };
    }
    loadSync() {
        throw new Error('Loader Url has no sync mode');
    }
}

exports.UrlLoader = UrlLoader;
//# sourceMappingURL=index.cjs.js.map
