/** *****************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 ****************************************************************************** */
/* Copyright (c) 2020 Red Hat, Inc. */
import _ from 'lodash';

async function getPolicyElements(clusterModel, complianceModel, model) {
  const links = [];
  const nodes = [];
  const clusters = await clusterModel.getClusters();

  // create policy node
  if (model.length > 0) {
    const policy = model[0];
    let { name } = policy;
    const { namespace, status } = policy;
    const policyId = `policy--${name}`;
    nodes.push({
      name,
      namespace,
      type: 'policy',
      id: policyId,
      uid: policyId,
      specs: { isDesign: true, policy },
    });

    // create placement node and links back to policy
    const placements = await complianceModel.getPlacementRules(model[0]);
    name = _.get(status, 'placement[0].placementRule');
    const placementId = `placement--${name}`;
    nodes.push({
      name,
      shortName: 'placement',
      type: 'placement',
      id: placementId,
      uid: placementId,
      specs: {
        isDesign: true,
        placements: placements.map(({ clusterLabels }) => ({ clusterLabels })),
      },
    });
    links.push({
      from: { uid: policyId },
      to: { uid: placementId },
      type: '',
      specs: { isDesign: true },
    });

    // create cluster nodes and links back to placement
    const clusterMap = {};
    clusters.forEach((cluster) => {
      const {
        k8sVersion, clusterip,
        metadata: {
          name: cname, namespace: cnamespace, labels, creationTimestamp,
        },
      } = cluster;
      clusterMap[cname] = {
        name: cname,
        namespace: cnamespace,
        labels,
        creationTimestamp,
        consoleURL: clusterip,
        version: k8sVersion,
        violations: [],
        cluster,
      };
    });
    const stats = _.get(status, 'status');
    if (stats) {
      stats.forEach((entry) => {
        const { compliance, clustername } = entry;
        const clusterId = `cluster--${clustername}`;
        nodes.push({
          name: clustername,
          type: 'cluster',
          id: clusterId,
          uid: clusterId,
          specs: {
            isLive: true,
            compliance,
            violations: clusterMap[clustername] && clusterMap[clustername].violations,
            cluster: clusterMap[clustername] && clusterMap[clustername].cluster,
            title: clusterMap[clustername] && clusterMap[clustername].labels
              && clusterMap[clustername].labels.environment,
          },
        });
        links.push({
          from: { uid: placementId },
          to: { uid: clusterId },
          type: 'binds',
          specs: { isDesign: true },
        });
      });
    }

    // add all compliance violations to shown clusters
    const compliances = await complianceModel.getCompliances();
    compliances.forEach(({ name: n, status: sts }) => {
      const stss = _.get(sts, 'status');
      if (stss) {
        stss.forEach((entry) => {
          const { compliance, compliant, clustername } = entry;
          const complianceStatus = compliance || compliant;
          if (clusterMap[clustername]
            && (!complianceStatus || complianceStatus.toLowerCase().trim() === 'noncompliant')) {
            clusterMap[clustername].violations.push(n);
          }
        });
      }
    });
  }

  return { resources: nodes, relationships: links };
}

export { getPolicyElements as default };
