package manifests

import (
	"fmt"
	"path/filepath"

	"github.com/ghodss/yaml"
	"github.com/pkg/errors"

	configv1 "github.com/openshift/api/config/v1"
	operatorv1 "github.com/openshift/api/operator/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"

	"github.com/openshift/installer/pkg/asset"
	"github.com/openshift/installer/pkg/asset/installconfig"
	"github.com/openshift/installer/pkg/types"
)

var (
	clusterIngressConfigFile     = filepath.Join(manifestDir, "cluster-ingress-02-config.yml")
	defaultIngressControllerFile = filepath.Join(manifestDir, "cluster-ingress-default-ingresscontroller.yaml")
)

// Ingress generates the cluster-ingress-*.yml files.
type Ingress struct {
	FileList []*asset.File
}

var _ asset.WritableAsset = (*Ingress)(nil)

// Name returns a human friendly name for the asset.
func (*Ingress) Name() string {
	return "Ingress Config"
}

// Dependencies returns all of the dependencies directly needed to generate
// the asset.
func (*Ingress) Dependencies() []asset.Asset {
	return []asset.Asset{
		&installconfig.InstallConfig{},
	}
}

// Generate generates the ingress cluster config and default ingresscontroller.
//
// A cluster ingress config is always created.
//
// A default ingresscontroller is only created if the cluster is using an internal
// publishing strategy. In this case, the default ingresscontroller is also set
// to use the internal publishing strategy.
func (ing *Ingress) Generate(dependencies asset.Parents) error {
	installConfig := &installconfig.InstallConfig{}
	dependencies.Get(installConfig)

	ing.FileList = []*asset.File{}

	clusterConfig, err := ing.generateClusterConfig(installConfig.Config)
	if err != nil {
		return errors.Wrap(err, "failed to create cluster config")
	}
	ing.FileList = append(ing.FileList, &asset.File{
		Filename: clusterIngressConfigFile,
		Data:     clusterConfig,
	})

	defaultIngressController, err := ing.generateDefaultIngressController(installConfig.Config)
	if err != nil {
		return errors.Wrap(err, "failed to create default ingresscontroller")
	}
	if len(defaultIngressController) > 0 {
		ing.FileList = append(ing.FileList, &asset.File{
			Filename: defaultIngressControllerFile,
			Data:     defaultIngressController,
		})
	}

	return nil
}

func (ing *Ingress) generateClusterConfig(config *types.InstallConfig) ([]byte, error) {
	obj := &configv1.Ingress{
		TypeMeta: metav1.TypeMeta{
			APIVersion: configv1.GroupVersion.String(),
			Kind:       "Ingress",
		},
		ObjectMeta: metav1.ObjectMeta{
			Name: "cluster",
			// not namespaced
		},
		Spec: configv1.IngressSpec{
			Domain: fmt.Sprintf("apps.%s", config.ClusterDomain()),
		},
	}
	return yaml.Marshal(obj)
}

func (ing *Ingress) generateDefaultIngressController(config *types.InstallConfig) ([]byte, error) {
	switch config.Publish {
	case types.InternalPublishingStrategy:
		obj := &operatorv1.IngressController{
			TypeMeta: metav1.TypeMeta{
				APIVersion: operatorv1.GroupVersion.String(),
				Kind:       "IngressController",
			},
			ObjectMeta: metav1.ObjectMeta{
				Namespace: "openshift-ingress-operator",
				Name:      "default",
			},
			Spec: operatorv1.IngressControllerSpec{
				EndpointPublishingStrategy: &operatorv1.EndpointPublishingStrategy{
					Type: operatorv1.LoadBalancerServiceStrategyType,
					LoadBalancer: &operatorv1.LoadBalancerStrategy{
						Scope: operatorv1.InternalLoadBalancer,
					},
				},
			},
		}
		return yaml.Marshal(obj)
	default:
		return nil, nil
	}
}

// Files returns the files generated by the asset.
func (ing *Ingress) Files() []*asset.File {
	return ing.FileList
}

// Load returns false since this asset is not written to disk by the installer.
func (ing *Ingress) Load(asset.FileFetcher) (bool, error) {
	return false, nil
}
