"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const git_response_error_1 = require("../errors/git-response-error");
const BranchSummary_1 = require("../responses/BranchSummary");
const BranchDeleteSummary_1 = require("../responses/BranchDeleteSummary");
function containsDeleteBranchCommand(commands) {
    const deleteCommands = ['-d', '-D', '--delete'];
    return commands.some(command => deleteCommands.includes(command));
}
exports.containsDeleteBranchCommand = containsDeleteBranchCommand;
function branchTask(customArgs) {
    const isDelete = containsDeleteBranchCommand(customArgs);
    const commands = ['branch', ...customArgs];
    if (commands.length === 1) {
        commands.push('-a');
    }
    if (!commands.includes('-v')) {
        commands.splice(1, 0, '-v');
    }
    return {
        format: 'utf-8',
        commands,
        parser(text) {
            return isDelete ? BranchDeleteSummary_1.parseBranchDeletions(text).all[0] : BranchSummary_1.parseBranchSummary(text);
        },
    };
}
exports.branchTask = branchTask;
function branchLocalTask() {
    return {
        format: 'utf-8',
        commands: ['branch', '-v'],
        parser(text) {
            return BranchSummary_1.parseBranchSummary(text);
        },
    };
}
exports.branchLocalTask = branchLocalTask;
function deleteBranchesTask(branches, forceDelete = false) {
    return {
        format: 'utf-8',
        commands: ['branch', '-v', forceDelete ? '-D' : '-d', ...branches],
        parser(text) {
            return BranchDeleteSummary_1.parseBranchDeletions(text);
        },
        onError(exitCode, error, done, fail) {
            if (!BranchDeleteSummary_1.hasBranchDeletionError(error, exitCode)) {
                return fail(error);
            }
            done(error);
        },
        concatStdErr: true,
    };
}
exports.deleteBranchesTask = deleteBranchesTask;
function deleteBranchTask(branch, forceDelete = false) {
    const parser = (text) => BranchDeleteSummary_1.parseBranchDeletions(text).branches[branch];
    return {
        format: 'utf-8',
        commands: ['branch', '-v', forceDelete ? '-D' : '-d', branch],
        parser,
        onError(exitCode, error, _, fail) {
            if (!BranchDeleteSummary_1.hasBranchDeletionError(error, exitCode)) {
                return fail(error);
            }
            throw new git_response_error_1.GitResponseError(parser(error), error);
        },
        concatStdErr: true,
    };
}
exports.deleteBranchTask = deleteBranchTask;
//# sourceMappingURL=branch.js.map