/** *****************************************************************************
 * Licensed Materials - Property of IBM
 * (c) Copyright IBM Corporation 2019. All Rights Reserved.
 *
 * Note to U.S. Government Users Restricted Rights:
 * Use, duplication or disclosure restricted by GSA ADP Schedule
 * Contract with IBM Corp.
 *
 * Copyright (c) 2020 Red Hat, Inc.
 ****************************************************************************** */
/* eslint-disable no-underscore-dangle */
/* eslint-disable max-len */
import _ from 'lodash';
import fs from 'fs';
import redis from 'redis';
import dns from 'dns';
import { Graph } from 'redisgraph.js';
import moment from 'moment';
import config from '../../../config';
import logger from '../lib/logger';
import { isRequired } from '../lib/utils';
import pollRbacCache, { getUserRbacFilter } from '../lib/rbacCaching';

// Is there a more efficient way?
function formatResult(results, removePrefix = true) {
  const startTime = Date.now();
  const resultList = [];
  while (results.hasNext()) {
    let resultItem = {};
    const record = results.next();
    record.keys().forEach((key) => {
      if (record.get(key) !== null) {
        if (removePrefix) {
          if (record.get(key).properties !== null && record.get(key).properties !== undefined) {
            resultItem = record.get(key).properties;
          } else {
            resultItem[key.substring(key.indexOf('.') + 1)] = record.get(key);
          }
        } else {
          resultItem[key] = record.get(key);
        }
      }
    });
    resultList.push(resultItem);
  }
  logger.perfLog(startTime, 100, 'formatResult()', `Result set size: ${resultList.length}`);
  return resultList;
}

const isNumber = (value) => !Number.isNaN(value * 1);
// TODO: Zack L - Need to come back to this once number values with units are normalized
// const isNumWithChars = (value) => {
//   if (!isNumber(value) && !Number.isNaN(parseInt(value, 10))) {
// eslint-disable-next-line
//     return ['Ei', 'Pi', 'Ti', 'Gi', 'Mi', 'Ki'].findIndex(unit => unit === value.substring(value.length - 2, value.length)) > -1;
//   }
//   return false;
// };
const isDate = (value) => !isNumber(value) && moment(value, 'YYYY-MM-DDTHH:mm:ssZ', true).isValid();
const isDateFilter = (value) => ['hour', 'day', 'week', 'month', 'year'].indexOf(value) > -1;
// const isVersion = property.toLowerCase().includes('version');

export function getOperator(value) {
  const match = value.match(/^<=|^>=|^!=|^!|^<|^>|^=]/);
  let operator = (match && match[0]) || '=';
  if (operator === '!' || operator === '!=') {
    operator = '<>';
  }
  return operator;
}

export function getDateFilter(value) {
  const currentTime = Date.now();
  switch (true) {
    case value === 'hour':
      return `> '${new Date(currentTime - 3600000).toISOString()}'`;
    case value === 'day':
      return `> '${new Date(currentTime - 86400000).toISOString()}'`;
    case value === 'week':
      return `> '${new Date(currentTime - 604800000).toISOString()}'`;
    case value === 'month':
      return `> '${new Date(currentTime - 2629743000).toISOString()}'`;
    case value === 'year':
      return `> '${new Date(currentTime - 31556926000).toISOString()}'`;
    default:
      // default to month
      return `> '${new Date(currentTime - 2629743000).toISOString()}'`;
  }
}

export function getFilterString(filters) {
  const filterStrings = [];
  filters.forEach((filter) => {
    // Use OR for filters with multiple values.
    filterStrings.push(`(${filter.values.map((value) => {
      const operatorRemoved = value.replace(/^<=|^>=|^!=|^!|^<|^>|^=/, '');
      if (isNumber(operatorRemoved)) { //  || isNumWithChars(operatorRemoved)
        return `n.${filter.property} ${getOperator(value)} ${operatorRemoved}`;
      } if (isDateFilter(value)) {
        return `n.${filter.property} ${getDateFilter(value)}`;
      }
      return `n.${filter.property} ${getOperator(value)} '${operatorRemoved}'`;
    }).join(' OR ')})`);
  });
  const resultString = filterStrings.join(' AND ');
  return resultString;
}

function getIPvFamily(redisHost) {
  return new Promise((resolve) => {
    dns.lookup(redisHost, (err, address, family) => {
      logger.info('RedisGraph address: %j family: IPv%s', address, family);
      if (family === 6) {
        resolve('IPv6');
      }
      resolve('IPv4');
    });
  });
}

let redisClient;
function getRedisClient() {
  // eslint-disable-next-line no-async-promise-executor
  return new Promise(async (resolve) => {
    if (redisClient) {
      resolve(redisClient);
      return;
    }

    logger.info('Initializing new Redis client.');

    if (config.get('redisPassword') === '') {
      logger.warn('Starting redis client without authentication. redisPassword was not provided in config.');
      redisClient = redis.createClient(config.get('redisEndpoint'));
    } else if (config.get('redisSSLEndpoint') === '') {
      logger.info('Starting Redis client using endpoint: ', config.get('redisEndpoint'));
      redisClient = redis.createClient(config.get('redisEndpoint'), { password: config.get('redisPassword') });
    } else {
      logger.info('Starting Redis client using SSL endpoint: ', config.get('redisSSLEndpoint'));
      const redisUrl = config.get('redisSSLEndpoint');
      const redisInfo = redisUrl.split(':');
      const redisHost = redisInfo[0];
      const redisPort = redisInfo[1];
      const redisCert = fs.readFileSync(process.env.redisCert || './rediscert/redis.crt', 'utf8');
      const ipFamily = await getIPvFamily(redisHost);
      redisClient = redis.createClient(
        redisPort,
        redisHost,
        {
          auth_pass: config.get('redisPassword'),
          tls: { servername: redisHost, ca: [redisCert] },
          family: ipFamily,
        },
      );
    }

    redisClient.ping((error, result) => {
      if (error) logger.error('Error with Redis SSL connection: ', error);
      else {
        logger.info('Redis SSL connection respone : ', result);
        if (result === 'PONG') {
          resolve(redisClient);
        }
      }
    });

    // Wait until the client connects and is ready to resolve with the connecction.
    redisClient.on('connect', () => {
      logger.info('Redis Client connected.');
    });
    redisClient.on('ready', () => {
      logger.info('Redis Client ready.');
    });

    // Log redis connection events.
    redisClient.on('error', (error) => {
      logger.info('Error with Redis connection: ', error);
    });
    redisClient.on('end', (msg) => {
      logger.info('The Redis connection has ended.', msg);
    });
  });
}

// Skip while running tests until we can mock Redis.
if (process.env.NODE_ENV !== 'test') {
  // Initializes the Redis client on startup.
  getRedisClient();
  // Check if user access has changed for any logged in user - if so remove them from the cache so the rbac string is regenerated
  pollRbacCache();
}

// Applications queries are only interested in resources on the local cluster, in the appropriate API group
const APPLICATION_MATCH = "(app:Application {cluster: 'local-cluster', apigroup: 'app.k8s.io'})";
const SUBSCRIPTION_MATCH = "(sub:Subscription {cluster: 'local-cluster', apigroup: 'apps.open-cluster-management.io'})";
const PLACEMENTRULE_MATCH = "(pr:PlacementRule {cluster: 'local-cluster', apigroup: 'apps.open-cluster-management.io'})";
const CHANNEL_MATCH = "(ch:Channel {cluster: 'local-cluster', apigroup: 'apps.open-cluster-management.io'})";

export default class RedisGraphConnector {
  constructor({
    rbac = isRequired('rbac'),
    req = isRequired('req'),
  } = {}) {
    this.rbac = rbac;
    this.req = req;
  }

  async isServiceAvailable() {
    await getRedisClient();
    if (this.g === undefined && redisClient) {
      this.g = new Graph('search-db', redisClient);
    }
    return redisClient.connected && redisClient.ready;
  }

  async getRbacValues() {
    const startTime = Date.now();
    const { allowedResources, allowedNS } = await getUserRbacFilter(this.req);
    logger.perfLog(startTime, 1000, 'getRbacValues()');
    return { allowedResources, allowedNS };
  }

  async createWhereClause(filters, aliases) {
    let whereClause = '';
    const { allowedResources, allowedNS } = await this.getRbacValues();
    let withClause = '';
    if (allowedNS.length > 0) {
      withClause = `WITH [${allowedResources}] AS allowedResources, [${allowedNS}] AS allowedNS`;
    } else {
      withClause = `WITH [${allowedResources}] AS allowedResources`;
    }
    const whereClauseRbac = aliases.map((alias) => {
      if (allowedNS.length > 0) {
        // When user is allowed to see all resources in a given namespace:
        //  - For hub resources, we use the `namespace` field.
        //  - For managed cluster resources, use the `_clusterNamespace`, which is the namespace where the cluster is mapped on the hub.
        return `${alias}._rbac IN allowedResources OR ((exists(${alias}._hubClusterResource) AND (${alias}.namespace IN allowedNS)) OR (exists(${alias}._clusterNamespace) AND ${alias}._clusterNamespace IN allowedNS))`;
      }
      return `${alias}._rbac IN allowedResources`;
    }).join(' AND ');
    const filterString = getFilterString(filters);
    if (filterString !== '') {
      whereClause = `WHERE ${filterString} AND (${whereClauseRbac})`;
    } else {
      whereClause = `WHERE (${whereClauseRbac})`;
    }
    return { withClause, whereClause };
  }

  /*
   * Execute a redis query and format the result as an array of Object.
   */
  async executeQuery({ query, removePrefix = true, queryName }) {
    await this.isServiceAvailable();
    const startTime = Date.now();
    const result = await this.g.query(query);
    logger.perfLog(startTime, 200, queryName);
    return formatResult(result, removePrefix);
  }

  /*
   * Get Applications.
   */
  async runApplicationsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app']);
    const matchClause = `MATCH ${APPLICATION_MATCH} ${whereClause}`;
    const returnClause = 'RETURN DISTINCT app._uid, app.name, app.namespace, app.created, app.dashboard, app.selfLink, app.label ORDER BY app.name, app.namespace ASC';
    const query = `${withClause} ${matchClause} ${returnClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runApplicationsQuery' });
  }

  /*
   * Get a list of applications that have related clusters.
   * NOTE: If an app doesn't have resources in any cluster it won't be in the result.
   *
   * Sample result:
   * [
   *    { 'app._uid': 'local-cluster/12345-67890', local: false, clusterCount: 1 },
   *    { 'app._uid': 'local-cluster/12345-67890', local: true, clusterCount: 1 }
   * ]
   */
  async runAppClustersQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'cluster']);
    const returnClause = "RETURN DISTINCT app._uid, cluster.name='local-cluster' as local, count(DISTINCT cluster._uid) as clusterCount";
    const query = `
      ${withClause} MATCH ${APPLICATION_MATCH}-->(:Subscription)<--(:Subscription)--(cluster:Cluster) ${whereClause} ${returnClause}
      UNION ${withClause} MATCH ${APPLICATION_MATCH}-->(:Subscription {cluster: 'local-cluster', localPlacement: 'true', apigroup: 'apps.open-cluster-management.io'})--(cluster:Cluster) ${whereClause} ${returnClause}
    `;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runAppClustersQuery' });
  }

  /*
   * Get Applications with their related Hub Subscriptions.
   */
  async runAppHubSubscriptionsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'sub']);
    const matchClause = `MATCH ${APPLICATION_MATCH}-->${SUBSCRIPTION_MATCH}`;
    const where = whereClause === '' ? 'WHERE' : `${whereClause} AND`;
    const additionalWhere = 'NOT exists(sub._hostingSubscription)';
    const returnClause = 'RETURN app._uid, sub._uid, sub.timeWindow, sub.localPlacement, sub.status, sub.channel, sub.name';
    const query = `${withClause} ${matchClause} ${where} ${additionalWhere} ${returnClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runAppHubSubscriptionsQuery' });
  }

  /*
   * Get Applications with their related Hub Channels.
   */
  async runAppHubChannelsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'sub', 'ch']);
    const matchClause = `${withClause} MATCH ${APPLICATION_MATCH}-[*1]->${SUBSCRIPTION_MATCH}-[*1]->(ch:Channel)`;
    const where = whereClause === '' ? 'WHERE' : `${whereClause} AND NOT exists(sub._hostingSubscription)`;
    const returnClause = 'RETURN app._uid, sub._uid, sub._gitbranch, sub._gitpath, sub._gitcommit, ch._uid, ch.type, ch.pathname';
    const query = `${matchClause} ${where} ${returnClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runAppHubChannelsQuery' });
  }

  /*
   * Get Applications with the pods counter
   * return the number of pods for this app as a string, grouped by their status
  */
  async runAppPodsCountQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'pod']);
    const query = `${withClause} MATCH ${APPLICATION_MATCH}-->(:Subscription)<--(:Subscription)--(pod:Pod) ${whereClause} RETURN app._uid, pod._uid, pod.status`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runAppPodsCountQuery' });
  }

  /*
   * Get Applications with their related remote subscriptions.
   # Remote subscriptions are those with the '_hostingSubscription' property.
   */
  async runAppRemoteSubscriptionsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'sub']);
    const matchClause = `MATCH ${APPLICATION_MATCH}-->(:Subscription)<--(sub:Subscription)`;
    const where = whereClause === '' ? 'WHERE' : `${whereClause} AND`;
    const additionalWhere = 'exists(sub._hostingSubscription)=true';
    const query = `${withClause} ${matchClause} ${where} ${additionalWhere} RETURN app._uid, sub._uid, sub.status`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runAppRemoteSubscriptionsQuery' });
  }

  /*
   * Get clusters related to any application.
   */
  async runGlobalAppClusterCountQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'cluster']);
    const query = `${withClause} MATCH ${APPLICATION_MATCH}-->(:Subscription)<--(:Subscription)-->(cluster:Cluster) ${whereClause} RETURN DISTINCT cluster._uid`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runGlobalAppClusterCountQuery' });
  }

  /*
   * Get channels related to any application.
   */
  async runGlobalAppChannelsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'ch']);
    const query = `${withClause} MATCH ${APPLICATION_MATCH}<-[]-(ch:Channel) ${whereClause} RETURN DISTINCT ch`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runGlobalAppChannelsQuery' });
  }

  /*
   * Get hub subscriptions related to any application.
   */
  async runGlobalAppHubSubscriptionsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'sub']);
    const matchClause = `MATCH ${APPLICATION_MATCH}-[]->${SUBSCRIPTION_MATCH}`;
    const where = whereClause === '' ? 'WHERE' : `${whereClause} AND`;
    const additionalWhere = 'NOT exists(sub._hostingSubscription)';
    const query = `${withClause} ${matchClause} ${where} ${additionalWhere} RETURN DISTINCT sub`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runGlobalAppHubSubscriptionsQuery' });
  }

  /*
   * Get remote subscriptions related to any application.
   */
  async runGlobalAppRemoteSubscriptionsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['app', 'sub']);
    const matchClause = `MATCH ${APPLICATION_MATCH}-->(:Subscription)<--(sub:Subscription)`;
    const where = whereClause === '' ? 'WHERE' : `${whereClause} AND`;
    const additionalWhere = 'exists(sub._hostingSubscription)=true';
    const query = `${withClause} ${matchClause} ${where} ${additionalWhere} RETURN DISTINCT sub._uid, sub.status`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runGlobalAppRemoteSubscriptionsQuery' });
  }

  /*
   * Get Subscriptions.
   */
  async runSubscriptionsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['sub']);
    const matchClause = `MATCH ${SUBSCRIPTION_MATCH} ${whereClause} AND NOT exists(sub._hostingSubscription)`;
    const returnClause = 'RETURN DISTINCT sub._uid, sub.name, sub.namespace, sub.created, sub.selfLink, sub.status, sub.channel, sub.timeWindow, sub.localPlacement';
    const orderClause = 'ORDER BY sub.name, sub.namespace ASC';
    const query = `${withClause} ${matchClause} ${returnClause} ${orderClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runSubscriptionsQuery' });
  }

  /*
   * Get a list of subscriptions that have related clusters.
   * NOTE: If a subsciption doesn't have resources in any cluster it won't be in the result.
   *
   * Sample result:
   * [
   *    { 'sub._uid': 'local-cluster/12345-67890', local: false, clusterCount: 1 },
   *    { 'sub._uid': 'local-cluster/12345-67890', local: true, clusterCount: 1 }
   * ]
   */
  async runSubClustersQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['sub', 'cluster']);
    const returnClause = "RETURN DISTINCT sub._uid, cluster.name='local-cluster' as local, count(DISTINCT cluster._uid) as clusterCount";
    const query = `
      ${withClause} MATCH ${SUBSCRIPTION_MATCH}<--(:Subscription)--(cluster:Cluster) ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
      UNION ${withClause} MATCH (sub:Subscription {cluster: 'local-cluster', localPlacement: 'true', apigroup: 'apps.open-cluster-management.io'})--(cluster:Cluster) ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
    `;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runSubClustersQuery' });
  }

  /*
   * Get a list of subscriptions that have related applications.
   * NOTE: If a subsciption doesn't have any related applications it won't be in the result.
   *
   * Sample result:
   * [
   *   {sub._uid: 'sub1', count: 3 },
   *   {sub._uid: 'sub2', count: 1 },
   * ]
   */
  async runSubAppsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['sub', 'app']);
    const matchClause = `MATCH ${SUBSCRIPTION_MATCH}<-[*1]-${APPLICATION_MATCH} ${whereClause} AND NOT exists(sub._hostingSubscription)`;
    const returnClause = 'RETURN DISTINCT sub._uid, count(DISTINCT app._uid) as count';
    const query = `${withClause} ${matchClause} ${returnClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runSubAppsQuery' });
  }

  /*
   * Get PlacementRules.
   */
  async runPlacementRulesQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['pr']);
    const matchClause = `MATCH ${PLACEMENTRULE_MATCH} ${whereClause}`;
    const returnClause = 'RETURN DISTINCT pr._uid, pr.name, pr.namespace, pr.created, pr.selfLink, pr.replicas';
    const orderClause = 'ORDER BY pr.name, pr.namespace ASC';
    const query = `${withClause} ${matchClause} ${returnClause} ${orderClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runPlacementRulesQuery' });
  }

  /*
   * Get a list of placement rules that have related clusters.
   * NOTE: If a placement rule doesn't have any related clusters it won't be in the result.
   *
   * Sample result:
   * [
   *    { 'pr._uid': 'local-cluster/12345-67890', local: false, clusterCount: 1 },
   *    { 'pr._uid': 'local-cluster/12345-67890', local: true, clusterCount: 1 }
   * ]
   */
  async runPRClustersQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['pr', 'sub', 'cluster']);
    const returnClause = "RETURN DISTINCT pr._uid, cluster.name='local-cluster' as local, count(DISTINCT cluster._uid) as clusterCount";
    const query = `
      ${withClause}
      MATCH ${PLACEMENTRULE_MATCH}<-[*1]-${SUBSCRIPTION_MATCH}<--(:Subscription)--(cluster:Cluster)
      ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
      UNION ${withClause}
      MATCH ${PLACEMENTRULE_MATCH}<-[*1]-(sub:Subscription {cluster: 'local-cluster', localPlacement: 'true', apigroup: 'apps.open-cluster-management.io'})--(cluster:Cluster)
      ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
    `;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runPRClustersQuery' });
  }

  /*
   * Get Channels.
   */
  async runChannelsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['ch']);
    const matchClause = `MATCH ${CHANNEL_MATCH} ${whereClause}`;
    const returnClause = 'RETURN DISTINCT ch._uid, ch.name, ch.namespace, ch.created, ch.selfLink, ch.type, ch.pathname';
    const orderClause = 'ORDER BY ch.name, ch.namespace ASC';
    const query = `${withClause} ${matchClause} ${returnClause} ${orderClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runChannelsQuery' });
  }

  /*
   * Get a list of channels that have related subscriptions.
   * NOTE: If a channel doesn't have any related subscriptions it won't be in the result.
   *
   * Sample result:
   * [
   *   {ch._uid: 'ch1', localPlacement: [], count: 2 },
   *   {ch._uid: 'ch2', localPlacement: [true, false], count: 1 },
   * ]
   */
  async runChannelSubsQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['ch', 'sub']);
    const matchClause = `MATCH ${CHANNEL_MATCH}<-[*1]-${SUBSCRIPTION_MATCH}`;
    const returnClause = 'RETURN DISTINCT ch._uid, collect(DISTINCT sub.localPlacement) as localPlacement, count(DISTINCT sub._uid) as count';
    const query = `${withClause} ${matchClause} ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}`;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runChannelSubsQuery' });
  }

  /*
   * Get a list of channels that have related clusters.
   * NOTE: If a channel doesn't have any related subscriptions and clusters, it won't be in the result.
   *
   * Sample result:
   * [
   *    { 'ch._uid': 'local-cluster/12345-67890', local: false, clusterCount: 1 },
   *    { 'ch._uid': 'local-cluster/12345-67890', local: true, clusterCount: 1 }
   * ]
   */
  async runChannelClustersQuery() {
    const { withClause, whereClause } = await this.createWhereClause([], ['ch', 'sub', 'cluster']);
    const returnClause = "RETURN DISTINCT ch._uid, cluster.name='local-cluster' as local, count(DISTINCT cluster._uid) as clusterCount";
    const query = `
      ${withClause}
      MATCH ${CHANNEL_MATCH}<-[*1]-${SUBSCRIPTION_MATCH}<--(:Subscription)--(cluster:Cluster)
      ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
      UNION ${withClause}
      MATCH ${CHANNEL_MATCH}<-[*1]-(sub:Subscription {cluster: 'local-cluster', localPlacement: 'true', apigroup: 'apps.open-cluster-management.io'})--(cluster:Cluster)
      ${whereClause} AND NOT exists(sub._hostingSubscription) ${returnClause}
    `;
    return this.executeQuery({ query, removePrefix: false, queryName: 'runChannelClustersQuery' });
  }

  /**
   * TODO: For users less than clusterAdmin we we do not currently handle non-namespaced resources
   * For users with access to >0 namespaces we create an RBAC string for resources user has access
   * For users with access to 0 namespaces we return an empty object
   */

  async runSearchQuery(filters, limit = config.get('defaultQueryLimit'), querySkipIdx = 0) {
    // logger.info('runSearchQuery()', filters);
    const startTime = Date.now();
    if (this.rbac.length > 0) {
      // RedisGraph 1.0.15 doesn't support an array as value. To work around this limitation we
      // encode labels in a single string. As a result we can't use an openCypher query to search
      // for labels so we need to filter here, which btw is inefficient.
      const specialFilters = filters.filter((f) => (f.property === 'label' || f.property === 'role'));
      if (specialFilters) {
        const { withClause, whereClause } = await this.createWhereClause(filters.filter((f) => f.property !== 'role' && f.property !== 'label'), ['n']);
        const q = `${withClause} MATCH (n) ${whereClause} RETURN n`;
        const res = await this.g.query(q);
        logger.perfLog(startTime, 150, 'SpecialFilterSearchQuery');
        return formatResult(res).filter((item) => {
          if (item.role && specialFilters.findIndex((filter) => filter.property === 'role') >= 0) {
            const roleIdx = specialFilters.findIndex((filter) => filter.property === 'role');
            return specialFilters[roleIdx].values.find((value) => {
              const values = value.replace(' ', '').split(',');
              return values.every((v) => item.role.includes(v));
            });
          }
          if (item.label && specialFilters.findIndex((filter) => filter.property === 'label') >= 0) {
            const labelIdx = specialFilters.findIndex((filter) => filter.property === 'label');
            return item.label && specialFilters[labelIdx].values.find((value) => item.label.indexOf(value) > -1);
          }
          return item;
        });
      }

      let limitClause = '';
      if (limit > 0) {
        limitClause = querySkipIdx > -1
          ? `SKIP ${querySkipIdx * config.get('defaultQueryLoopLimit')} LIMIT ${config.get('defaultQueryLoopLimit')}`
          : `LIMIT ${limit}`;
      }
      const { withClause, whereClause } = await this.createWhereClause(filters, ['n']);
      const query = `${withClause} MATCH (n) ${whereClause} RETURN n ${limitClause}`;
      const result = await this.g.query(query);
      logger.perfLog(startTime, 150, 'SearchQuery');
      return formatResult(result);
    }
    return [];
  }

  async runSearchQueryCountOnly(filters) {
    // logger.info('runSearchQueryCountOnly()', filters);

    if (this.rbac.length > 0) {
      // RedisGraph 1.0.15 doesn't support an array as value. To work around this limitation we
      // encode labels in a single string. As a result we can't use an openCypher query to search
      // for labels so we need to filter here, which btw is inefficient.
      const labelFilter = filters.find((f) => (f.property === 'label' || f.property === 'role'));
      if (labelFilter) {
        return this.runSearchQuery(filters, -1, -1).then((r) => r.length);
      }
      const { withClause, whereClause } = await this.createWhereClause(filters, ['n']);
      const startTime = Date.now();
      const result = await this.g.query(`${withClause} MATCH (n) ${whereClause} RETURN count(n)`);
      logger.perfLog(startTime, 150, 'runSearchQueryCountOnly()');
      if (result.hasNext() === true) {
        return result.next().get('count(n)');
      }
    }
    return 0;
  }

  async getAllProperties() {
    // logger.info('Getting all properties');

    // Adding these first to rank them higher when showin in UI.
    const values = ['cluster', 'kind', 'label', 'name', 'namespace', 'status'];

    if (this.rbac.length > 0) {
      const startTime = Date.now();
      const result = await this.g.query('CALL db.propertyKeys()');
      logger.perfLog(startTime, 150, 'getAllProperties()');

      result._results.forEach((record) => {
        if (record.values()[0].charAt(0) !== '_' && values.indexOf(record.values()[0]) < 0) {
          values.push(record.values()[0]);
        }
      });
    }
    return values;
  }

  async getAllValues(property, filters = [], limit = config.get('defaultQueryLimit')) {
    // logger.info('Getting all values for property:', property, filters);

    if (property === '') {
      logger.warn('getAllValues() called with empty value. Most likely this was an unecessary API call.');
      return Promise.resolve([]);
    }

    let valuesList = [];
    if (this.rbac.length > 0) {
      const startTime = Date.now();
      const limitClause = limit <= 0 || property === 'label'
        ? ''
        : `LIMIT ${limit}`;
      let f = filters.length > 0 ? filters : [];
      // Workaround for node resource queries - this will need to be removed when redis 2.0 features are used
      f = filters.filter((filter) => filter.property !== 'role' && filter.property !== 'label');
      const { withClause, whereClause } = await this.createWhereClause(f, ['n']);
      const result = await this.g.query(`${withClause} MATCH (n) ${whereClause} RETURN DISTINCT n.${property} ORDER BY n.${property} ASC ${limitClause}`);
      logger.perfLog(startTime, 500, 'getAllValues()');
      result._results.forEach((record) => {
        if (record.values()[0] !== 'NULL' && record.values()[0] !== null) {
          valuesList.push(record.values()[0]);
        }
      });

      // RedisGraph 1.0.15 doesn't support an array as value. To work around this limitation we
      // encode labels in a single string. Here we need to decode the string to get all labels.
      if (property === 'label') {
        const labels = [];
        valuesList.forEach((value) => {
          value.split('; ').forEach((label) => {
            // We don't want duplicates, so we check if it already exists.
            if (labels.indexOf(label) === -1) {
              labels.push(label);
            }
          });
        });
        return labels;
      }
      // Same workaround as above, but for roles.
      if (property === 'role') {
        const roles = [];
        valuesList.forEach((value) => {
          value.split(', ').forEach((role) => {
            // We don't want duplicates, so we check if it already exists.
            if (roles.indexOf(role) === -1) {
              roles.push(role);
            }
          });
        });
        return roles;
      }

      if (isDate(valuesList[0])) {
        return ['isDate'];
      } if (isNumber(valuesList[0])) { //  || isNumWithChars(valuesList[0]))
        valuesList = valuesList.filter((res) => (isNumber(res) || (!isNumber(res))) && res !== '');
        valuesList.sort((a, b) => parseInt(a, 10) - parseInt(b, 10));
        if (valuesList.length > 1) {
          return ['isNumber', valuesList[0], valuesList[valuesList.length - 1]];
        } if (valuesList.length === 1) {
          return ['isNumber', valuesList[0]];
        }
      }
    }
    return valuesList;
  }

  // WORKAROUND: This function divides the query to prevent hitting the maximum query size. The tradeoff is
  // a slower execution time. Long term we need to replace with a more efficient query or rethink how we enforce
  // rbac. Technical debt being tracked with: https://github.com/open-cluster-management/backlog/issues/6016
  async getRelationshipsWithSeparatedQueryWorkaround(withClause, returnClause) {
    const startTime = Date.now();
    let relatedQueries = [];
    relatedQueries = [
      `${withClause} MATCH (n)-[]-(r) ${returnClause}`,
      `${withClause} MATCH (n:Application)-->(:Subscription)<--(:Subscription)--(r) ${returnClause}`,
      `${withClause} MATCH (r:Application)-->(:Subscription)<--(:Subscription)--(n) ${returnClause}`,
      `${withClause} MATCH (n:Subscription)<--(:Subscription)--(r) ${returnClause}`,
      `${withClause} MATCH (r:Subscription)<--(:Subscription)--(n) ${returnClause}`,
      `${withClause} MATCH (n:Application)-->(:Subscription)--(r) ${returnClause}`,
      `${withClause} MATCH (r:Application)-->(:Subscription)--(n) ${returnClause}`,
    ];
    let results = await Promise.all(relatedQueries.map(async (q) => formatResult(await this.g.query(q))));
    // Compress to a single array (originally an array of arrays)
    results = _.flatten(results);
    // Need to ger rid of duplicate resources
    results = _.uniqBy(results, (item) => item._uid);
    logger.perfLog(startTime, 500, 'findRelationships() - Using separate query workaround');
    return results;
  }

  async findRelationships({ filters = [], countOnly = false, relatedKinds = [] } = {}) {
    const MAX_LENGTH_WITH_CLAUSE = 148500;
    if (this.rbac.length > 0) {
      const { withClause, whereClause } = await this.createWhereClause(filters, ['n', 'r']);
      const startTime = Date.now();
      let query = '';
      let returnClause = '';
      if (relatedKinds.length > 0) {
        const relatedClause = relatedKinds.map((kind) => `r.kind = '${kind}'`).join(' OR ');
        returnClause = `WHERE (${relatedClause}) AND ${whereClause.replace('WHERE ', '')} AND (r._uid <> n._uid) RETURN DISTINCT r`;
      } else {
        returnClause = `${whereClause} AND (r._uid <> n._uid) RETURN DISTINCT ${countOnly ? 'r._uid, r.kind' : 'r'}`;
      }
      // This is tech debt, tracking with: https://github.com/open-cluster-management/backlog/issues/6016
      if (withClause.length > MAX_LENGTH_WITH_CLAUSE) {
        return this.getRelationshipsWithSeparatedQueryWorkaround(withClause, returnClause);
      }
      query = `${withClause} MATCH (n)-[]-(r) ${returnClause}
      UNION ${withClause} MATCH (n:Application)-->(:Subscription)<--(:Subscription)--(r) ${returnClause}
      UNION ${withClause} MATCH (r:Application)-->(:Subscription)<--(:Subscription)--(n) ${returnClause}
      UNION ${withClause} MATCH (n:Subscription)<--(:Subscription)--(r) ${returnClause}
      UNION ${withClause} MATCH (r:Subscription)<--(:Subscription)--(n) ${returnClause}
      UNION ${withClause} MATCH (n:Application)-->(:Subscription)--(r) ${returnClause}
      UNION ${withClause} MATCH (r:Application)-->(:Subscription)--(n) ${returnClause}`;
      const result = await this.g.query(query);
      logger.perfLog(startTime, 300, 'findRelationships()');
      return formatResult(result);
    }
    return [];
  }
}
