import { ApolloLink, concat, execute } from '@apollo/client/link/core';
import { createUploadLink, isExtractableFile, formDataAppendFile } from 'apollo-upload-client';
import FormData from 'form-data';
import { fetch } from 'cross-fetch';
import { Observable } from '@apollo/client/utilities';
import { toPromise } from '@apollo/client/link/utils';
import { observableToAsyncIterable } from '@graphql-tools/utils';
import { GraphQLScalarType, GraphQLError } from 'graphql';
import isPromise from 'is-promise';

function getFinalPromise(object) {
    return Promise.resolve(object).then(resolvedObject => {
        if (resolvedObject == null) {
            return resolvedObject;
        }
        if (Array.isArray(resolvedObject)) {
            return Promise.all(resolvedObject.map(o => getFinalPromise(o)));
        }
        else if (typeof resolvedObject === 'object') {
            const keys = Object.keys(resolvedObject);
            return Promise.all(keys.map(key => getFinalPromise(resolvedObject[key]))).then(awaitedValues => {
                for (let i = 0; i < keys.length; i++) {
                    resolvedObject[keys[i]] = awaitedValues[i];
                }
                return resolvedObject;
            });
        }
        return resolvedObject;
    });
}
class AwaitVariablesLink extends ApolloLink {
    request(operation, forward) {
        return new Observable(observer => {
            let subscription;
            getFinalPromise(operation.variables)
                .then(resolvedVariables => {
                operation.variables = resolvedVariables;
                subscription = forward(operation).subscribe({
                    next: observer.next.bind(observer),
                    error: observer.error.bind(observer),
                    complete: observer.complete.bind(observer),
                });
            })
                .catch(observer.error.bind(observer));
            return () => {
                if (subscription != null) {
                    subscription.unsubscribe();
                }
            };
        });
    }
}

const createServerHttpLink = (options) => concat(new AwaitVariablesLink(), createUploadLink({
    ...options,
    fetch,
    FormData,
    isExtractableFile: (value) => isExtractableFile(value) || (value === null || value === void 0 ? void 0 : value.createReadStream),
    formDataAppendFile: (form, index, file) => {
        if (file.createReadStream != null) {
            form.append(index, file.createReadStream(), {
                filename: file.filename,
                contentType: file.mimetype,
            });
        }
        else {
            formDataAppendFile(form, index, file);
        }
    },
}));

const linkToExecutor = (link) => (params) => {
    const { document, variables, extensions, context, info } = params;
    return toPromise(execute(link, {
        query: document,
        variables: variables,
        context: {
            graphqlContext: context,
            graphqlResolveInfo: info,
            clientAwareness: {},
        },
        extensions,
    }));
};

const linkToSubscriber = (link) => async (params) => {
    const { document, variables, extensions, context, info } = params;
    return observableToAsyncIterable(execute(link, {
        query: document,
        variables,
        context: {
            graphqlContext: context,
            graphqlResolveInfo: info,
            clientAwareness: {},
        },
        extensions,
    }))[Symbol.asyncIterator]();
};

const GraphQLUpload = new GraphQLScalarType({
    name: 'Upload',
    description: 'The `Upload` scalar type represents a file upload.',
    parseValue: value => {
        if (value != null && isPromise(value.promise)) {
            // graphql-upload v10
            return value.promise;
        }
        else if (isPromise(value)) {
            // graphql-upload v9
            return value;
        }
        throw new GraphQLError('Upload value invalid.');
    },
    // serialization requires to support schema stitching
    serialize: value => value,
    parseLiteral: ast => {
        throw new GraphQLError('Upload literal unsupported.', ast);
    },
});

export { AwaitVariablesLink, GraphQLUpload, createServerHttpLink, linkToExecutor, linkToSubscriber };
//# sourceMappingURL=index.esm.js.map
