(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.graphqlWs = {}));
}(this, (function (exports) { 'use strict';

    /**
     *
     * protocol
     *
     */
    /** The WebSocket sub-protocol used for the [GraphQL over WebSocket Protocol](/PROTOCOL.md). */
    const GRAPHQL_TRANSPORT_WS_PROTOCOL = 'graphql-transport-ws';

    // Extremely small optimisation, reduces runtime prototype traversal
    const baseHasOwnProperty = Object.prototype.hasOwnProperty;
    function isObject(val) {
        return typeof val === 'object' && val !== null;
    }
    function areGraphQLErrors(obj) {
        return (Array.isArray(obj) &&
            // must be at least one error
            obj.length > 0 &&
            // error has at least a message
            obj.every((ob) => 'message' in ob));
    }
    function hasOwnProperty(obj, prop) {
        return baseHasOwnProperty.call(obj, prop);
    }
    function hasOwnObjectProperty(obj, prop) {
        return baseHasOwnProperty.call(obj, prop) && isObject(obj[prop]);
    }
    function hasOwnStringProperty(obj, prop) {
        return baseHasOwnProperty.call(obj, prop) && typeof obj[prop] === 'string';
    }

    /**
     *
     * message
     *
     */
    /** Types of messages allowed to be sent by the client/server over the WS protocol. */
    var MessageType;
    (function (MessageType) {
        MessageType["ConnectionInit"] = "connection_init";
        MessageType["ConnectionAck"] = "connection_ack";
        MessageType["Subscribe"] = "subscribe";
        MessageType["Next"] = "next";
        MessageType["Error"] = "error";
        MessageType["Complete"] = "complete";
    })(MessageType || (MessageType = {}));
    /** Checks if the provided value is a message. */
    function isMessage(val) {
        if (isObject(val)) {
            // all messages must have the `type` prop
            if (!hasOwnStringProperty(val, 'type')) {
                return false;
            }
            // validate other properties depending on the `type`
            switch (val.type) {
                case MessageType.ConnectionInit:
                    // the connection init message can have optional payload object
                    return (!hasOwnProperty(val, 'payload') ||
                        val.payload === undefined ||
                        isObject(val.payload));
                case MessageType.ConnectionAck:
                    // the connection ack message can have optional payload object too
                    return (!hasOwnProperty(val, 'payload') ||
                        val.payload === undefined ||
                        isObject(val.payload));
                case MessageType.Subscribe:
                    return (hasOwnStringProperty(val, 'id') &&
                        hasOwnObjectProperty(val, 'payload') &&
                        (!hasOwnProperty(val.payload, 'operationName') ||
                            val.payload.operationName === undefined ||
                            val.payload.operationName === null ||
                            typeof val.payload.operationName === 'string') &&
                        hasOwnStringProperty(val.payload, 'query') &&
                        (!hasOwnProperty(val.payload, 'variables') ||
                            val.payload.variables === undefined ||
                            val.payload.variables === null ||
                            hasOwnObjectProperty(val.payload, 'variables')));
                case MessageType.Next:
                    return (hasOwnStringProperty(val, 'id') &&
                        hasOwnObjectProperty(val, 'payload'));
                case MessageType.Error:
                    return hasOwnStringProperty(val, 'id') && areGraphQLErrors(val.payload);
                case MessageType.Complete:
                    return hasOwnStringProperty(val, 'id');
                default:
                    return false;
            }
        }
        return false;
    }
    /** Parses the raw websocket message data to a valid message. */
    function parseMessage(data) {
        if (isMessage(data)) {
            return data;
        }
        if (typeof data !== 'string') {
            throw new Error('Message not parsable');
        }
        const message = JSON.parse(data);
        if (!isMessage(message)) {
            throw new Error('Invalid message');
        }
        return message;
    }
    /** Stringifies a valid message ready to be sent through the socket. */
    function stringifyMessage(msg) {
        if (!isMessage(msg)) {
            throw new Error('Cannot stringify invalid message');
        }
        return JSON.stringify(msg);
    }

    /**
     *
     * client
     *
     */
    /** Creates a disposable GraphQL over WebSocket client. */
    function createClient(options) {
        const { url, connectionParams, lazy = true, onNonLazyError = console.error, keepAlive = 0, retryAttempts = 5, retryWait = async function randomisedExponentialBackoff(retries) {
            let retryDelay = 1000; // start with 1s delay
            for (let i = 0; i < retries; i++) {
                retryDelay *= 2;
            }
            await new Promise((resolve) => setTimeout(resolve, retryDelay +
                // add random timeout from 300ms to 3s
                Math.floor(Math.random() * (3000 - 300) + 300)));
        }, on, webSocketImpl, 
        /**
         * Generates a v4 UUID to be used as the ID using `Math`
         * as the random number generator. Supply your own generator
         * in case you need more uniqueness.
         *
         * Reference: https://stackoverflow.com/a/2117523/709884
         */
        generateID = function generateUUID() {
            return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
                const r = (Math.random() * 16) | 0, v = c == 'x' ? r : (r & 0x3) | 0x8;
                return v.toString(16);
            });
        }, } = options;
        let ws;
        if (webSocketImpl) {
            if (!isWebSocket(webSocketImpl)) {
                throw new Error('Invalid WebSocket implementation provided');
            }
            ws = webSocketImpl;
        }
        else if (typeof WebSocket !== 'undefined') {
            ws = WebSocket;
        }
        else if (typeof global !== 'undefined') {
            ws =
                global.WebSocket ||
                    // @ts-expect-error: Support more browsers
                    global.MozWebSocket;
        }
        else if (typeof window !== 'undefined') {
            ws =
                window.WebSocket ||
                    // @ts-expect-error: Support more browsers
                    window.MozWebSocket;
        }
        if (!ws) {
            throw new Error('WebSocket implementation missing');
        }
        const WebSocketImpl = ws;
        // websocket status emitter, subscriptions are handled differently
        const emitter = (() => {
            const listeners = {
                connecting: (on === null || on === void 0 ? void 0 : on.connecting) ? [on.connecting] : [],
                connected: (on === null || on === void 0 ? void 0 : on.connected) ? [on.connected] : [],
                closed: (on === null || on === void 0 ? void 0 : on.closed) ? [on.closed] : [],
            };
            return {
                on(event, listener) {
                    const l = listeners[event];
                    l.push(listener);
                    return () => {
                        l.splice(l.indexOf(listener), 1);
                    };
                },
                emit(event, ...args) {
                    for (const listener of listeners[event]) {
                        // @ts-expect-error: The args should fit
                        listener(...args);
                    }
                },
                reset() {
                    Object.keys(listeners).forEach((event) => {
                        listeners[event] = [];
                    });
                },
            };
        })();
        let state = {
            disposed: false,
            socket: null,
            acknowledged: false,
            locks: 0,
            retrying: false,
            retries: 0,
        };
        // all those waiting for the `retryWait` to resolve
        const retryWaiting = [];
        async function connect(cancellerRef, callDepth = 0) {
            var _a;
            if (callDepth) {
                // prevents too many recursive calls when reavaluating/re-connecting
                if (callDepth > 10) {
                    throw new Error('Kept trying to connect but the socket never settled.');
                }
                // wait a bit for socket state changes in recursive calls
                await new Promise((resolve) => setTimeout(resolve, callDepth * 50));
            }
            // retry wait strategy for all callers
            if (state.retrying) {
                if (retryWaiting.length) {
                    // if others are waiting for retry, I'll wait too
                    await new Promise((resolve) => retryWaiting.push(resolve));
                }
                else {
                    retryWaiting.push(() => {
                        /** fake waiter to lead following connects in the `retryWaiting` queue */
                    });
                    // use retry wait strategy
                    await retryWait(state.retries);
                    state = Object.assign(Object.assign({}, state), { retrying: false, retries: state.retries + 1 });
                    // complete all waiting and clear the queue
                    while (retryWaiting.length) {
                        (_a = retryWaiting.pop()) === null || _a === void 0 ? void 0 : _a();
                    }
                }
            }
            // socket already exists. can be ready or pending, check and behave accordingly
            if (state.socket) {
                switch (state.socket.readyState) {
                    case WebSocketImpl.OPEN: {
                        // if the socket is not acknowledged, wait a bit and reavaluate
                        if (!state.acknowledged)
                            return connect(cancellerRef, callDepth + 1);
                        return makeConnectReturn(state.socket, cancellerRef);
                    }
                    case WebSocketImpl.CONNECTING: {
                        // if the socket is in the connecting phase, wait a bit and reavaluate
                        return connect(cancellerRef, callDepth + 1);
                    }
                    case WebSocketImpl.CLOSED:
                        break; // just continue, we'll make a new one
                    case WebSocketImpl.CLOSING: {
                        // if the socket is in the closing phase, wait a bit and connect
                        return connect(cancellerRef, callDepth + 1);
                    }
                    default:
                        throw new Error(`Impossible ready state ${state.socket.readyState}`);
                }
            }
            // establish connection and assign to singleton
            const socket = new WebSocketImpl(url, GRAPHQL_TRANSPORT_WS_PROTOCOL);
            state = Object.assign(Object.assign({}, state), { acknowledged: false, socket });
            emitter.emit('connecting');
            await new Promise((resolve, reject) => {
                let cancelled = false;
                cancellerRef.current = () => (cancelled = true);
                const tooLong = setTimeout(() => {
                    socket.close(3408, 'Waited 5 seconds but socket connect never settled');
                }, 5 * 1000);
                /**
                 * `onerror` handler is unnecessary because even if an error occurs, the `onclose` handler will be called
                 *
                 * From: https://developer.mozilla.org/en-US/docs/Web/API/WebSockets_API/Writing_WebSocket_client_applications
                 * > If an error occurs while attempting to connect, first a simple event with the name error is sent to the
                 * > WebSocket object (thereby invoking its onerror handler), and then the CloseEvent is sent to the WebSocket
                 * > object (thereby invoking its onclose handler) to indicate the reason for the connection's closing.
                 */
                socket.onclose = (event) => {
                    socket.onclose = null;
                    clearTimeout(tooLong);
                    state = Object.assign(Object.assign({}, state), { acknowledged: false, socket: null });
                    emitter.emit('closed', event);
                    return reject(event);
                };
                socket.onmessage = (event) => {
                    socket.onmessage = null;
                    if (cancelled) {
                        socket.close(3499, 'Client cancelled the socket before connecting');
                        return;
                    }
                    try {
                        const message = parseMessage(event.data);
                        if (message.type !== MessageType.ConnectionAck) {
                            throw new Error(`First message cannot be of type ${message.type}`);
                        }
                        clearTimeout(tooLong);
                        state = Object.assign(Object.assign({}, state), { acknowledged: true, socket, retrying: false, retries: 0 });
                        emitter.emit('connected', socket, message.payload); // connected = socket opened + acknowledged
                        return resolve();
                    }
                    catch (err) {
                        socket.close(4400, err instanceof Error ? err.message : new Error(err).message);
                    }
                };
                // as soon as the socket opens and the connectionParams
                // resolve, send the connection initalisation request
                socket.onopen = () => {
                    socket.onopen = null;
                    if (cancelled) {
                        socket.close(3499, 'Client cancelled the socket before connecting');
                        return;
                    }
                    (async () => {
                        try {
                            socket.send(stringifyMessage({
                                type: MessageType.ConnectionInit,
                                payload: typeof connectionParams === 'function'
                                    ? await connectionParams()
                                    : connectionParams,
                            }));
                        }
                        catch (err) {
                            // even if not open, call close again to report error
                            socket.close(4400, err instanceof Error ? err.message : new Error(err).message);
                        }
                    })();
                };
            });
            return makeConnectReturn(socket, cancellerRef);
        }
        async function makeConnectReturn(socket, cancellerRef) {
            return [
                socket,
                (cleanup) => new Promise((resolve, reject) => {
                    if (socket.readyState === WebSocketImpl.CLOSED) {
                        return reject(new Error('Socket has already been closed'));
                    }
                    state.locks++;
                    socket.addEventListener('close', listener);
                    function listener(event) {
                        cancellerRef.current = null;
                        state.locks--;
                        socket.removeEventListener('close', listener);
                        return reject(event);
                    }
                    cancellerRef.current = () => {
                        cancellerRef.current = null;
                        cleanup === null || cleanup === void 0 ? void 0 : cleanup();
                        state.locks--;
                        if (!state.locks) {
                            if (keepAlive > 0 && isFinite(keepAlive)) {
                                // if the keepalive is set, allow for the specified calmdown
                                // time and then close. but only if no lock got created in the
                                // meantime and if the socket is still open
                                setTimeout(() => {
                                    if (!state.locks && socket.OPEN) {
                                        socket.close(1000, 'Normal Closure');
                                    }
                                }, keepAlive);
                            }
                            else {
                                // otherwise close immediately
                                socket.close(1000, 'Normal Closure');
                            }
                        }
                        socket.removeEventListener('close', listener);
                        return resolve();
                    };
                }),
            ];
        }
        /**
         * Checks the `connect` problem and evaluates if the client should
         * retry. If the problem is worth throwing, it will be thrown immediately.
         */
        function shouldRetryConnectOrThrow(errOrCloseEvent) {
            // throw non `CloseEvent`s immediately, something else is wrong
            if (!isLikeCloseEvent(errOrCloseEvent)) {
                throw errOrCloseEvent;
            }
            // some close codes are worth reporting immediately
            if ([
                1002,
                1011,
                4400,
                4401,
                4409,
                4429,
            ].includes(errOrCloseEvent.code)) {
                throw errOrCloseEvent;
            }
            // already disposed or normal closure, shouldnt try again
            if (state.disposed || errOrCloseEvent.code === 1000) {
                return false;
            }
            // user cancelled early, shouldnt try again
            if (errOrCloseEvent.code === 3499) {
                return false;
            }
            // retries are not allowed or we tried to many times, report error
            if (!retryAttempts || state.retries >= retryAttempts) {
                throw errOrCloseEvent;
            }
            // looks good, start retrying
            state.retrying = true;
            return true;
        }
        // in non-lazy (hot?) mode always hold one connection lock to persist the socket
        if (!lazy) {
            (async () => {
                for (;;) {
                    try {
                        const [, throwOnCloseOrWaitForCancel] = await connect({
                            current: null,
                        });
                        await throwOnCloseOrWaitForCancel();
                        // cancelled, shouldnt try again
                        return;
                    }
                    catch (errOrCloseEvent) {
                        try {
                            // return and report if shouldnt try again
                            if (!shouldRetryConnectOrThrow(errOrCloseEvent))
                                return onNonLazyError === null || onNonLazyError === void 0 ? void 0 : onNonLazyError(errOrCloseEvent);
                        }
                        catch (_a) {
                            // report thrown error, no further retries
                            return onNonLazyError === null || onNonLazyError === void 0 ? void 0 : onNonLazyError(errOrCloseEvent);
                        }
                    }
                }
            })();
        }
        // to avoid parsing the same message in each
        // subscriber, we memo one on the last received data
        let lastData, lastMessage;
        function memoParseMessage(data) {
            if (data !== lastData) {
                lastMessage = parseMessage(data);
                lastData = data;
            }
            return lastMessage;
        }
        return {
            on: emitter.on,
            subscribe(payload, sink) {
                const id = generateID();
                let completed = false;
                const cancellerRef = { current: null };
                const messageListener = ({ data }) => {
                    const message = memoParseMessage(data);
                    switch (message.type) {
                        case MessageType.Next: {
                            if (message.id === id) {
                                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                                sink.next(message.payload);
                            }
                            return;
                        }
                        case MessageType.Error: {
                            if (message.id === id) {
                                sink.error(message.payload);
                                // the canceller must be set at this point
                                // because you cannot receive a message
                                // if there is no existing connection
                                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                                cancellerRef.current();
                                // TODO-db-201025 calling canceller will complete the sink, meaning that both the `error` and `complete` will be
                                // called. neither promises or observables care; once they settle, additional calls to the resolvers will be ignored
                            }
                            return;
                        }
                        case MessageType.Complete: {
                            if (message.id === id) {
                                completed = true;
                                // the canceller must be set at this point
                                // because you cannot receive a message
                                // if there is no existing connection
                                // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
                                cancellerRef.current();
                                // calling canceller will complete the sink
                            }
                            return;
                        }
                    }
                };
                (async () => {
                    for (;;) {
                        try {
                            const [socket, throwOnCloseOrWaitForCancel] = await connect(cancellerRef);
                            socket.addEventListener('message', messageListener);
                            socket.send(stringifyMessage({
                                id: id,
                                type: MessageType.Subscribe,
                                payload,
                            }));
                            // either the canceller will be called and the promise resolved
                            // or the socket closed and the promise rejected
                            await throwOnCloseOrWaitForCancel(() => {
                                // if not completed already, send complete message to server on cancel
                                if (!completed) {
                                    socket.send(stringifyMessage({
                                        id: id,
                                        type: MessageType.Complete,
                                    }));
                                }
                            });
                            socket.removeEventListener('message', messageListener);
                            // cancelled, shouldnt try again
                            return;
                        }
                        catch (errOrCloseEvent) {
                            // return if shouldnt try again
                            if (!shouldRetryConnectOrThrow(errOrCloseEvent))
                                return;
                        }
                    }
                })()
                    .catch(sink.error)
                    .then(sink.complete); // resolves on cancel or normal closure
                return () => {
                    var _a;
                    (_a = cancellerRef.current) === null || _a === void 0 ? void 0 : _a.call(cancellerRef);
                };
            },
            dispose() {
                var _a;
                state.disposed = true;
                (_a = state.socket) === null || _a === void 0 ? void 0 : _a.close(1000, 'Normal Closure');
                emitter.reset();
            },
        };
    }
    function isLikeCloseEvent(val) {
        return isObject(val) && 'code' in val && 'reason' in val;
    }
    function isWebSocket(val) {
        return (typeof val === 'function' &&
            'constructor' in val &&
            'CLOSED' in val &&
            'CLOSING' in val &&
            'CONNECTING' in val &&
            'OPEN' in val);
    }

    exports.createClient = createClient;

    Object.defineProperty(exports, '__esModule', { value: true });

})));
