'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const paramCase = require('param-case');
const graphql = require('graphql');
const lodash = require('lodash');

function resolveExternalModuleAndFn(pointer) {
    // eslint-disable-next-line no-eval
    const importExternally = (moduleName) => eval(`require('${moduleName}')`);
    if (typeof pointer === 'function') {
        return pointer;
    }
    // eslint-disable-next-line prefer-const
    let [moduleName, functionName] = pointer.split('#');
    // Temp workaround until v2
    if (moduleName === 'change-case') {
        moduleName = paramCase.paramCase(functionName);
    }
    const { resolve } = importExternally('path');
    const localFilePath = resolve(process.cwd(), moduleName);
    const { existsSync } = importExternally('fs');
    const localFileExists = existsSync(localFilePath);
    const importFrom = importExternally('import-from');
    const loadedModule = localFileExists ? importExternally(localFilePath) : importFrom(process.cwd(), moduleName);
    if (!(functionName in loadedModule) && typeof loadedModule !== 'function') {
        throw new Error(`${functionName} couldn't be found in module ${moduleName}!`);
    }
    return loadedModule[functionName] || loadedModule;
}

function isComplexPluginOutput(obj) {
    return typeof obj === 'object' && obj.hasOwnProperty('content');
}

function mergeOutputs(content) {
    const result = { content: '', prepend: [], append: [] };
    if (Array.isArray(content)) {
        content.forEach(item => {
            if (typeof item === 'string') {
                result.content += item;
            }
            else {
                result.content += item.content;
                result.prepend.push(...(item.prepend || []));
                result.append.push(...(item.append || []));
            }
        });
    }
    return [...result.prepend, result.content, ...result.append].join('\n');
}
function isWrapperType(t) {
    return graphql.isListType(t) || graphql.isNonNullType(t);
}
function getBaseType(type) {
    if (isWrapperType(type)) {
        return getBaseType(type.ofType);
    }
    else {
        return type;
    }
}

function isOutputConfigArray(type) {
    return Array.isArray(type);
}
function isConfiguredOutput(type) {
    return typeof type === 'object' && type.plugins;
}
function normalizeOutputParam(config) {
    // In case of direct array with a list of plugins
    if (isOutputConfigArray(config)) {
        return {
            documents: [],
            schema: [],
            plugins: isConfiguredOutput(config) ? config.plugins : config,
        };
    }
    else if (isConfiguredOutput(config)) {
        return config;
    }
    else {
        throw new Error(`Invalid "generates" config!`);
    }
}
function normalizeInstanceOrArray(type) {
    if (Array.isArray(type)) {
        return type;
    }
    else if (!type) {
        return [];
    }
    return [type];
}
function normalizeConfig(config) {
    if (typeof config === 'string') {
        return [{ [config]: {} }];
    }
    else if (Array.isArray(config)) {
        return config.map(plugin => (typeof plugin === 'string' ? { [plugin]: {} } : plugin));
    }
    else if (typeof config === 'object') {
        return Object.keys(config).reduce((prev, pluginName) => [...prev, { [pluginName]: config[pluginName] }], []);
    }
    else {
        return [];
    }
}
function hasNullableTypeRecursively(type) {
    if (!graphql.isNonNullType(type)) {
        return true;
    }
    if (graphql.isListType(type) || graphql.isNonNullType(type)) {
        return hasNullableTypeRecursively(type.ofType);
    }
    return false;
}
function isUsingTypes(document, externalFragments, schema) {
    let foundFields = 0;
    const typesStack = [];
    graphql.visit(document, {
        SelectionSet: {
            enter(node, key, parent, anscestors) {
                const insideIgnoredFragment = anscestors.find((f) => f.kind && f.kind === 'FragmentDefinition' && externalFragments.includes(f.name.value));
                if (insideIgnoredFragment) {
                    return;
                }
                const selections = node.selections || [];
                if (schema && selections.length > 0) {
                    const nextTypeName = (() => {
                        if (parent.kind === graphql.Kind.FRAGMENT_DEFINITION) {
                            return parent.typeCondition.name.value;
                        }
                        else if (parent.kind === graphql.Kind.FIELD) {
                            const lastType = typesStack[typesStack.length - 1];
                            if (!lastType) {
                                throw new Error(`Unable to find parent type! Please make sure you operation passes validation`);
                            }
                            const field = lastType.getFields()[parent.name.value];
                            if (!field) {
                                throw new Error(`Unable to find field "${parent.name.value}" on type "${lastType}"!`);
                            }
                            return getBaseType(field.type).name;
                        }
                        else if (parent.kind === graphql.Kind.OPERATION_DEFINITION) {
                            if (parent.operation === 'query') {
                                return schema.getQueryType().name;
                            }
                            else if (parent.operation === 'mutation') {
                                return schema.getMutationType().name;
                            }
                            else if (parent.operation === 'subscription') {
                                return schema.getSubscriptionType().name;
                            }
                        }
                        else if (parent.kind === graphql.Kind.INLINE_FRAGMENT) {
                            if (parent.typeCondition) {
                                return parent.typeCondition.name.value;
                            }
                            else {
                                return typesStack[typesStack.length - 1].name;
                            }
                        }
                        return null;
                    })();
                    typesStack.push(schema.getType(nextTypeName));
                }
            },
            leave(node) {
                const selections = node.selections || [];
                if (schema && selections.length > 0) {
                    typesStack.pop();
                }
            },
        },
        Field: {
            enter: (node, key, parent, path, anscestors) => {
                if (node.name.value.startsWith('__')) {
                    return;
                }
                const insideIgnoredFragment = anscestors.find((f) => f.kind && f.kind === 'FragmentDefinition' && externalFragments.includes(f.name.value));
                if (insideIgnoredFragment) {
                    return;
                }
                const selections = node.selectionSet ? node.selectionSet.selections || [] : [];
                const relevantFragmentSpreads = selections.filter(s => s.kind === graphql.Kind.FRAGMENT_SPREAD && !externalFragments.includes(s.name.value));
                if (selections.length === 0 || relevantFragmentSpreads.length > 0) {
                    foundFields++;
                }
                if (schema) {
                    const lastType = typesStack[typesStack.length - 1];
                    if (lastType) {
                        if (graphql.isObjectType(lastType)) {
                            const field = lastType.getFields()[node.name.value];
                            if (!field) {
                                throw new Error(`Unable to find field "${node.name.value}" on type "${lastType}"!`);
                            }
                            const currentType = field.type;
                            // To handle `Maybe` usage
                            if (hasNullableTypeRecursively(currentType)) {
                                foundFields++;
                            }
                        }
                    }
                }
            },
        },
        enter: {
            VariableDefinition: (node, key, parent, path, anscestors) => {
                const insideIgnoredFragment = anscestors.find((f) => f.kind && f.kind === 'FragmentDefinition' && externalFragments.includes(f.name.value));
                if (insideIgnoredFragment) {
                    return;
                }
                foundFields++;
            },
            InputValueDefinition: (node, key, parent, path, anscestors) => {
                const insideIgnoredFragment = anscestors.find((f) => f.kind && f.kind === 'FragmentDefinition' && externalFragments.includes(f.name.value));
                if (insideIgnoredFragment) {
                    return;
                }
                foundFields++;
            },
        },
    });
    return foundFields > 0;
}

/**
 * Federation Spec
 */
const federationSpec = graphql.parse(/* GraphQL */ `
  scalar _FieldSet

  directive @external on FIELD_DEFINITION
  directive @requires(fields: _FieldSet!) on FIELD_DEFINITION
  directive @provides(fields: _FieldSet!) on FIELD_DEFINITION
  directive @key(fields: _FieldSet!) on OBJECT | INTERFACE
`);
/**
 * Adds `__resolveReference` in each ObjectType involved in Federation.
 * @param schema
 */
function addFederationReferencesToSchema(schema) {
    const typeMap = schema.getTypeMap();
    for (const typeName in typeMap) {
        const type = schema.getType(typeName);
        if (graphql.isObjectType(type) && isFederationObjectType(type)) {
            const typeConfig = type.toConfig();
            typeConfig.fields = {
                [resolveReferenceFieldName]: {
                    type,
                },
                ...typeConfig.fields,
            };
            const newType = new graphql.GraphQLObjectType(typeConfig);
            newType.astNode = newType.astNode || graphql.parse(graphql.printType(newType)).definitions[0];
            newType.astNode.fields.unshift({
                kind: graphql.Kind.FIELD_DEFINITION,
                name: {
                    kind: graphql.Kind.NAME,
                    value: resolveReferenceFieldName,
                },
                type: {
                    kind: graphql.Kind.NAMED_TYPE,
                    name: {
                        kind: graphql.Kind.NAME,
                        value: typeName,
                    },
                },
            });
            typeMap[typeName] = newType;
        }
    }
    return schema;
}
/**
 * Removes Federation Spec from GraphQL Schema
 * @param schema
 * @param config
 */
function removeFederation(schema) {
    const queryType = schema.getQueryType();
    const queryTypeFields = queryType.getFields();
    delete queryTypeFields._entities;
    delete queryTypeFields._service;
    const typeMap = schema.getTypeMap();
    delete typeMap._Service;
    delete typeMap._Entity;
    delete typeMap._Any;
    return schema;
}
const resolveReferenceFieldName = '__resolveReference';
class ApolloFederation {
    constructor({ enabled, schema }) {
        this.enabled = false;
        this.enabled = enabled;
        this.schema = schema;
        this.providesMap = this.createMapOfProvides();
    }
    /**
     * Excludes types definde by Federation
     * @param typeNames List of type names
     */
    filterTypeNames(typeNames) {
        return this.enabled ? typeNames.filter(t => t !== '_FieldSet') : typeNames;
    }
    /**
     * Excludes `__resolveReference` fields
     * @param fieldNames List of field names
     */
    filterFieldNames(fieldNames) {
        return this.enabled ? fieldNames.filter(t => t !== resolveReferenceFieldName) : fieldNames;
    }
    /**
     * Decides if directive should not be generated
     * @param name directive's name
     */
    skipDirective(name) {
        return this.enabled && ['external', 'requires', 'provides', 'key'].includes(name);
    }
    /**
     * Decides if scalar should not be generated
     * @param name directive's name
     */
    skipScalar(name) {
        return this.enabled && name === '_FieldSet';
    }
    /**
     * Decides if field should not be generated
     * @param data
     */
    skipField({ fieldNode, parentType }) {
        if (!this.enabled || !graphql.isObjectType(parentType) || !isFederationObjectType(parentType)) {
            return false;
        }
        return this.isExternalAndNotProvided(fieldNode, parentType);
    }
    isResolveReferenceField(fieldNode) {
        const name = typeof fieldNode.name === 'string' ? fieldNode.name : fieldNode.name.value;
        return this.enabled && name === resolveReferenceFieldName;
    }
    /**
     * Transforms ParentType signature in ObjectTypes involved in Federation
     * @param data
     */
    transformParentType({ fieldNode, parentType, parentTypeSignature, }) {
        if (this.enabled &&
            graphql.isObjectType(parentType) &&
            isFederationObjectType(parentType) &&
            (isTypeExtension(parentType) || fieldNode.name.value === resolveReferenceFieldName)) {
            const keys = getDirectivesByName('key', parentType);
            if (keys.length) {
                const outputs = [`{ __typename: '${parentType.name}' } &`];
                // Look for @requires and see what the service needs and gets
                const requires = getDirectivesByName('requires', fieldNode).map(this.extractKeyOrRequiresFieldSet);
                const requiredFields = this.translateFieldSet(lodash.merge({}, ...requires), parentTypeSignature);
                // @key() @key() - "primary keys" in Federation
                const primaryKeys = keys.map(def => {
                    const fields = this.extractKeyOrRequiresFieldSet(def);
                    return this.translateFieldSet(fields, parentTypeSignature);
                });
                const [open, close] = primaryKeys.length > 1 ? ['(', ')'] : ['', ''];
                outputs.push([open, primaryKeys.join(' | '), close].join(''));
                // include required fields
                if (requires.length) {
                    outputs.push(`& ${requiredFields}`);
                }
                return outputs.join(' ');
            }
        }
        return parentTypeSignature;
    }
    isExternalAndNotProvided(fieldNode, objectType) {
        return this.isExternal(fieldNode) && !this.hasProvides(objectType, fieldNode);
    }
    isExternal(node) {
        return getDirectivesByName('external', node).length > 0;
    }
    hasProvides(objectType, node) {
        const fields = this.providesMap[graphql.isObjectType(objectType) ? objectType.name : objectType.name.value];
        if (fields && fields.length) {
            return fields.includes(node.name.value);
        }
        return false;
    }
    translateFieldSet(fields, parentTypeRef) {
        return `GraphQLRecursivePick<${parentTypeRef}, ${JSON.stringify(fields)}>`;
    }
    extractKeyOrRequiresFieldSet(directive) {
        const arg = directive.arguments.find(arg => arg.name.value === 'fields');
        const value = arg.value.value;
        return graphql.visit(graphql.parse(`{${value}}`), {
            leave: {
                SelectionSet(node) {
                    return node.selections.reduce((accum, field) => {
                        accum[field.name] = field.selection;
                        return accum;
                    }, {});
                },
                Field(node) {
                    return {
                        name: node.name.value,
                        selection: node.selectionSet ? node.selectionSet : true,
                    };
                },
                Document(node) {
                    return node.definitions.find((def) => def.kind === 'OperationDefinition' && def.operation === 'query').selectionSet;
                },
            },
        });
    }
    extractProvidesFieldSet(directive) {
        const arg = directive.arguments.find(arg => arg.name.value === 'fields');
        const value = arg.value.value;
        if (/[{}]/gi.test(value)) {
            throw new Error('Nested fields in _FieldSet is not supported in the @provides directive');
        }
        return value.split(/\s+/g);
    }
    createMapOfProvides() {
        const providesMap = {};
        Object.keys(this.schema.getTypeMap()).forEach(typename => {
            const objectType = this.schema.getType(typename);
            if (graphql.isObjectType(objectType)) {
                Object.values(objectType.getFields()).forEach(field => {
                    const provides = getDirectivesByName('provides', field.astNode)
                        .map(this.extractProvidesFieldSet)
                        .reduce((prev, curr) => [...prev, ...curr], []);
                    const ofType = getBaseType(field.type);
                    if (!providesMap[ofType.name]) {
                        providesMap[ofType.name] = [];
                    }
                    providesMap[ofType.name].push(...provides);
                });
            }
        });
        return providesMap;
    }
}
/**
 * Checks if Object Type is involved in Federation. Based on `@key` directive
 * @param node Type
 */
function isFederationObjectType(node) {
    const definition = graphql.isObjectType(node)
        ? node.astNode || graphql.parse(graphql.printType(node)).definitions[0]
        : node;
    const name = definition.name.value;
    const directives = definition.directives;
    const isNotRoot = !['Query', 'Mutation', 'Subscription'].includes(name);
    const isNotIntrospection = !name.startsWith('__');
    const hasKeyDirective = directives.some(d => d.name.value === 'key');
    return isNotRoot && isNotIntrospection && hasKeyDirective;
}
/**
 * Extracts directives from a node based on directive's name
 * @param name directive name
 * @param node ObjectType or Field
 */
function getDirectivesByName(name, node) {
    let astNode;
    if (graphql.isObjectType(node)) {
        astNode = node.astNode;
    }
    else {
        astNode = node;
    }
    if (astNode && astNode.directives) {
        return astNode.directives.filter(d => d.name.value === name);
    }
    return [];
}
/**
 * Checks if the Object Type extends a federated type from a remote schema.
 * Based on if any of its fields contain the `@external` directive
 * @param node Type
 */
function isTypeExtension(node) {
    var _a;
    const definition = graphql.isObjectType(node)
        ? node.astNode || graphql.parse(graphql.printType(node)).definitions[0]
        : node;
    return (_a = definition.fields) === null || _a === void 0 ? void 0 : _a.some(field => getDirectivesByName('external', field).length);
}

class DetailedError extends Error {
    constructor(message, details, source) {
        super(message);
        this.message = message;
        this.details = details;
        this.source = source;
        Object.setPrototypeOf(this, DetailedError.prototype);
        Error.captureStackTrace(this, DetailedError);
    }
}
function isDetailedError(error) {
    return error.details;
}

exports.ApolloFederation = ApolloFederation;
exports.DetailedError = DetailedError;
exports.addFederationReferencesToSchema = addFederationReferencesToSchema;
exports.federationSpec = federationSpec;
exports.getBaseType = getBaseType;
exports.hasNullableTypeRecursively = hasNullableTypeRecursively;
exports.isComplexPluginOutput = isComplexPluginOutput;
exports.isConfiguredOutput = isConfiguredOutput;
exports.isDetailedError = isDetailedError;
exports.isOutputConfigArray = isOutputConfigArray;
exports.isUsingTypes = isUsingTypes;
exports.isWrapperType = isWrapperType;
exports.mergeOutputs = mergeOutputs;
exports.normalizeConfig = normalizeConfig;
exports.normalizeInstanceOrArray = normalizeInstanceOrArray;
exports.normalizeOutputParam = normalizeOutputParam;
exports.removeFederation = removeFederation;
exports.resolveExternalModuleAndFn = resolveExternalModuleAndFn;
//# sourceMappingURL=index.cjs.js.map
