/*
 * Copyright 2019 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package orca

import (
	"strings"
	"testing"

	orcapb "github.com/cncf/udpa/go/udpa/data/orca/v1"
	"github.com/golang/protobuf/proto"
	"github.com/google/go-cmp/cmp"
	"google.golang.org/grpc/internal/grpctest"
	"google.golang.org/grpc/metadata"
)

var (
	testMessage = &orcapb.OrcaLoadReport{
		CpuUtilization: 0.1,
		MemUtilization: 0.2,
		RequestCost:    map[string]float64{"ccc": 3.4},
		Utilization:    map[string]float64{"ttt": 0.4},
	}
	testBytes, _ = proto.Marshal(testMessage)
)

type s struct {
	grpctest.Tester
}

func Test(t *testing.T) {
	grpctest.RunSubTests(t, s{})
}

func (s) TestToMetadata(t *testing.T) {
	tests := []struct {
		name string
		r    *orcapb.OrcaLoadReport
		want metadata.MD
	}{{
		name: "nil",
		r:    nil,
		want: nil,
	}, {
		name: "valid",
		r:    testMessage,
		want: metadata.MD{
			strings.ToLower(mdKey): []string{string(testBytes)},
		},
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := ToMetadata(tt.r); !cmp.Equal(got, tt.want) {
				t.Errorf("ToMetadata() = %v, want %v", got, tt.want)
			}
		})
	}
}

func (s) TestFromMetadata(t *testing.T) {
	tests := []struct {
		name string
		md   metadata.MD
		want *orcapb.OrcaLoadReport
	}{{
		name: "nil",
		md:   nil,
		want: nil,
	}, {
		name: "valid",
		md: metadata.MD{
			strings.ToLower(mdKey): []string{string(testBytes)},
		},
		want: testMessage,
	}}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := FromMetadata(tt.md); !cmp.Equal(got, tt.want, cmp.Comparer(proto.Equal)) {
				t.Errorf("FromMetadata() = %v, want %v", got, tt.want)
			}
		})
	}
}
