"use strict";
/**********************************************************************
 * Copyright (c) 2018-2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 ***********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const yargs = require("yargs");
const generate_assembly_1 = require("./generate-assembly");
const link_1 = require("./link");
const update_dependencies_1 = require("./update-dependencies");
const cdn_1 = require("./cdn");
const clean_1 = require("./clean");
const cli_error_1 = require("./cli-error");
const init_1 = require("./init");
const init_sources_1 = require("./init-sources");
const logger_1 = require("./logger");
const production_1 = require("./production");
const ASSSEMBLY_PATH = 'examples/assembly';
/**
 * Entry point of this helper script
 * @author Florent Benoit
 */
const commandArgs = yargs
    .usage('$0 <cmd> [args]')
    .command({
    command: 'init',
    describe: 'Initialize current theia to behave like a Che/Theia',
    builder: init_sources_1.InitSources.argBuilder,
    handler: async (args) => {
        try {
            const assemblyFolder = path.resolve(process.cwd(), ASSSEMBLY_PATH);
            const packagesFolder = path.resolve(process.cwd(), 'packages');
            const pluginsFolder = path.resolve(process.cwd(), 'plugins');
            const cheFolder = path.resolve(process.cwd(), 'che');
            const init = new init_1.Init(process.cwd(), assemblyFolder, cheFolder, pluginsFolder);
            const version = await init.getCurrentVersion();
            await init.generate();
            await init.updatePluginsConfiguration();
            const initSources = new init_sources_1.InitSources(process.cwd(), packagesFolder, pluginsFolder, cheFolder, assemblyFolder, version);
            await initSources.initSourceLocationAliases(args.alias);
            await initSources.readConfigurationAndGenerate(args.config, args.dev);
            await init.updadeBuildConfiguration(initSources.extensions);
        }
        catch (err) {
            handleError(err);
        }
    },
})
    .command({
    command: 'production',
    describe: 'Copy Theia to a production directory',
    handler: async () => {
        try {
            const assemblyFolder = path.resolve(process.cwd(), ASSSEMBLY_PATH);
            const production = new production_1.Production(process.cwd(), assemblyFolder, 'production');
            await production.create();
        }
        catch (err) {
            handleError(err);
        }
    },
})
    .command({
    command: 'cdn',
    describe: 'Add or update the CDN support configuration',
    builder: cdn_1.Cdn.argBuilder,
    handler: async (argv) => {
        try {
            const assemblyFolder = path.resolve(process.cwd(), ASSSEMBLY_PATH);
            const cdn = new cdn_1.Cdn(assemblyFolder, argv.theia, argv.monaco);
            await cdn.create();
        }
        catch (err) {
            handleError(err);
        }
    },
})
    .command({
    command: 'clean',
    describe: 'Clean Theia repository',
    handler: async () => {
        try {
            const assemblyFolder = path.resolve(process.cwd(), ASSSEMBLY_PATH);
            const packagesFolder = path.resolve(process.cwd(), 'packages');
            const pluginsFolder = path.resolve(process.cwd(), 'plugins');
            const cheFolder = path.resolve(process.cwd(), 'che');
            const nodeModules = path.resolve(process.cwd(), 'node_modules');
            const clean = new clean_1.Clean(assemblyFolder, cheFolder, packagesFolder, pluginsFolder, nodeModules);
            clean.cleanCheTheia();
        }
        catch (err) {
            handleError(err);
        }
    },
})
    // commands related to the "linkless" mode of building che-theia
    .command({
    command: 'link',
    describe: 'Yarn link to a given theia source tree',
    handler: link_1.handleCommand,
    builder: link_1.builder,
})
    .command({
    command: 'generate',
    describe: 'Generate the che-theia assembly folder',
    handler: generate_assembly_1.handleCommand,
    builder: generate_assembly_1.builder,
})
    .command({
    command: 'update-dependencies',
    describe: 'Update the theia version in all package.json files',
    handler: update_dependencies_1.handleCommand,
    builder: update_dependencies_1.builder,
})
    .help()
    .strict()
    .demandCommand().argv;
if (!commandArgs) {
    yargs.showHelp();
}
// eslint-disable-next-line @typescript-eslint/no-explicit-any
function handleError(error) {
    if (error instanceof cli_error_1.CliError) {
        logger_1.Logger.error('=> 🚒 ' + error.message);
    }
    else {
        logger_1.Logger.error(error);
    }
    process.exit(1);
}
//# sourceMappingURL=yargs.js.map