package cf

import (
	"github.com/Apicurio/apicurio-registry-operator/controllers/common"
	"github.com/Apicurio/apicurio-registry-operator/controllers/loop"
	"github.com/Apicurio/apicurio-registry-operator/controllers/loop/context"
	"github.com/Apicurio/apicurio-registry-operator/controllers/loop/services"
	"github.com/Apicurio/apicurio-registry-operator/controllers/svc/client"
	"github.com/Apicurio/apicurio-registry-operator/controllers/svc/resources"
	meta "k8s.io/apimachinery/pkg/apis/meta/v1"
	"os"
)

var _ loop.ControlFunction = &OperatorPodCF{}

type OperatorPodCF struct {
	ctx       *context.LoopContext
	podExists bool
	clients   *client.Clients
}

// Read the operator pod into the resource cache
func NewOperatorPodCF(ctx *context.LoopContext, services *services.LoopServices) loop.ControlFunction {
	return &OperatorPodCF{
		ctx:       ctx,
		podExists: false,
		clients:   services.GetClients(),
	}
}

func (this *OperatorPodCF) Describe() string {
	return "OperatorPodCF"
}

func (this *OperatorPodCF) Sense() {
	// Observation #1
	_, this.podExists = this.ctx.GetResourceCache().Get(resources.RC_KEY_OPERATOR_POD)
}

func (this *OperatorPodCF) Compare() bool {
	// Condition #1
	return !this.podExists
}

func (this *OperatorPodCF) Respond() {
	namespace := os.Getenv("POD_NAMESPACE")
	name := os.Getenv("POD_NAME")

	if namespace == "" || name == "" {
		panic("Operator could not determine name and namespace of its own pod. " +
			"Make sure that both environment variables POD_NAMESPACE and POD_NAME are present in the operators Deployment.")
	}

	// Response #1
	pod, err := this.clients.Kube().GetPod(common.Namespace(namespace), common.Name(name), &meta.GetOptions{})
	if err == nil && pod.GetObjectMeta().GetDeletionTimestamp() == nil {
		this.ctx.GetResourceCache().Set(resources.RC_KEY_OPERATOR_POD, resources.NewResourceCacheEntry(common.Name(name), pod))
	} else {
		this.ctx.GetLog().WithValues("type", "Warning", "error", err).
			Info("Could not read operator's Pod resource. Will retry.")
	}
}

func (this *OperatorPodCF) Cleanup() bool {
	// No cleanup
	return true
}
