/*
Copyright 2017 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package cache

import (
	"sync"

	"k8s.io/klog/v2"

	v1 "k8s.io/api/core/v1"
)

// VolumeCache keeps all the PersistentVolumes that have been created by this provisioner.
// It is periodically updated by the Populator.
// The Deleter and Discoverer use the VolumeCache to check on created PVs
type VolumeCache struct {
	mutex   sync.Mutex
	pvs     map[string]*v1.PersistentVolume
	cleaned map[string]bool
}

// NewVolumeCache creates a new PV cache object for storing PVs created by this provisioner.
func NewVolumeCache() *VolumeCache {
	return &VolumeCache{
		pvs:     map[string]*v1.PersistentVolume{},
		cleaned: map[string]bool{},
	}
}

// GetPV returns the PV object given the PV name
func (cache *VolumeCache) GetPV(pvName string) (*v1.PersistentVolume, bool) {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	pv, exists := cache.pvs[pvName]
	return pv, exists
}

// AddPV adds the PV object to the cache
func (cache *VolumeCache) AddPV(pv *v1.PersistentVolume) {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	cache.pvs[pv.Name] = pv
	cache.cleaned[pv.Name] = false
	klog.Infof("Added pv %q to cache", pv.Name)
}

// UpdatePV updates the PV object in the cache
func (cache *VolumeCache) UpdatePV(pv *v1.PersistentVolume) {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	cache.pvs[pv.Name] = pv
	klog.Infof("Updated pv %q to cache", pv.Name)
}

// DeletePV deletes the PV object from the cache
func (cache *VolumeCache) DeletePV(pvName string) {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	delete(cache.pvs, pvName)
	delete(cache.cleaned, pvName)
	klog.Infof("Deleted pv %q from cache", pvName)
}

// ListPVs returns a list of all the PVs in the cache
func (cache *VolumeCache) ListPVs() []*v1.PersistentVolume {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	pvs := []*v1.PersistentVolume{}
	for _, pv := range cache.pvs {
		pvs = append(pvs, pv)
	}
	return pvs
}

// CleanPV marks the PV object as cleaned in the cache
func (cache *VolumeCache) CleanPV(pv *v1.PersistentVolume) {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	cache.cleaned[pv.Name] = true
	klog.Infof("Marked pv %q as cleaned in the cache", pv.Name)
}

// SuccessfullyCleanedPV returns true if the PV was already cleaned once
// since the cache entry was created, or if the cache entry no longer exists.
func (cache *VolumeCache) SuccessfullyCleanedPV(pv *v1.PersistentVolume) bool {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()

	cleaned, exists := cache.cleaned[pv.Name]
	return !exists || cleaned
}

// LookupPVsByPath returns a list of all of the PVs in the cache with a given local path
// Note: The PVs in the cache have been filtered by the populator so that the cache only
// contains volumes created by the local-static-provisioner running on this knode.
func (cache *VolumeCache) LookupPVsByPath(filePath string) []string {
	cache.mutex.Lock()
	defer cache.mutex.Unlock()
	var matches = make([]string, 0, 1)

	for _, pv := range cache.pvs {
		lvs := pv.Spec.Local
		if lvs != nil {
			if lvs.Path == filePath {
				matches = append(matches, pv.ObjectMeta.Name)
			}
		}
	}

	return matches
}
