package builds

import (
	"fmt"

	g "github.com/onsi/ginkgo"
	o "github.com/onsi/gomega"

	exutil "github.com/openshift/origin/test/extended/util"
)

var _ = g.Describe("[sig-builds][Feature:Builds] build without output image", func() {
	defer g.GinkgoRecover()
	var (
		dockerImageFixture = exutil.FixturePath("testdata", "builds", "test-docker-no-outputname.json")
		s2iImageFixture    = exutil.FixturePath("testdata", "builds", "test-s2i-no-outputname.json")
		oc                 = exutil.NewCLI("build-no-outputname")
	)

	g.Context("", func() {

		g.BeforeEach(func() {
			exutil.PreTestDump()
		})

		g.AfterEach(func() {
			if g.CurrentGinkgoTestDescription().Failed {
				exutil.DumpPodStates(oc)
				exutil.DumpConfigMapStates(oc)
				exutil.DumpPodLogsStartingWith("", oc)
			}
		})

		g.Describe("building from templates", func() {
			g.It(fmt.Sprintf("should create an image from a docker template without an output image reference defined"), func() {
				err := oc.Run("create").Args("-f", dockerImageFixture).Execute()
				o.Expect(err).NotTo(o.HaveOccurred())

				g.By("expecting build to pass without an output image reference specified")
				br, err := exutil.StartBuildAndWait(oc, "test-docker")
				br.AssertSuccess()

				g.By("verifying the build test-docker-1 output")
				buildLog, err := br.Logs()
				fmt.Fprintf(g.GinkgoWriter, "\nBuild log:\n%s\n", buildLog)
				o.Expect(err).NotTo(o.HaveOccurred())
				o.Expect(buildLog).Should(o.ContainSubstring(`Build complete, no image push requested`))
			})

			g.It(fmt.Sprintf("should create an image from a S2i template without an output image reference defined"), func() {
				err := oc.Run("create").Args("-f", s2iImageFixture).Execute()
				o.Expect(err).NotTo(o.HaveOccurred())

				g.By("expecting build to pass without an output image reference specified")
				br, err := exutil.StartBuildAndWait(oc, "test-sti")
				o.Expect(err).NotTo(o.HaveOccurred())
				br.AssertSuccess()

				g.By("verifying the build test-sti-1 output")
				buildLog, err := br.Logs()
				fmt.Fprintf(g.GinkgoWriter, "\nBuild log:\n%s\n", buildLog)
				o.Expect(err).NotTo(o.HaveOccurred())

				o.Expect(buildLog).Should(o.ContainSubstring(`Build complete, no image push requested`))
			})
		})
	})
})
