// Copyright ©2015 The Gonum Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package testlapack

import (
	"math"
	"testing"

	"golang.org/x/exp/rand"

	"gonum.org/v1/gonum/floats"
)

type Dlasq2er interface {
	Dgetrfer
	Dlasq2(n int, z []float64) (info int)
}

func Dlasq2Test(t *testing.T, impl Dlasq2er) {
	dTol := 1e-6
	// Answers from calling the netlib Dlasq2 function directly.
	for c, test := range []struct {
		n    int
		z    []float64
		info int
		zOut []float64
	}{
		{
			z:    []float64{0.9975595900926172e+00, 0.5668247076112733e+00, 0.9659153754961249e+00, 0.7479276854714322e+00, 0.3673908973747557e+00, 0.4806368987547315e+00, 0.7375426363398452e-01, 0.5355229277727247e-02, 0.3470812885180155e+00, 0.3422438160728350e+00, 0.2179517263384726e+00, 0.1331604100136593e+00, 0.9005245144218924e+00, 0.3867660104574036e+00, 0.4454822893878481e+00, 0.6619321808958428e+00, 0.1610830043055933e-01, 0.6508548361039168e+00, 0.6464088254838254e+00, 0.3229872909405558e+00, 0.8556924028853313e+00, 0.4012869193638139e+00, 0.2068743292187569e+00, 0.9685394642165999e+00, 0.5983995343181346e+00, 0.6729807327762632e+00, 0.4568823106729620e+00, 0.3300151235733747e+00, 0.1003829265021773e+00, 0.7554533047597268e+00, 0.6056932669802318e+00, 0.7190479134084530e+00, 0.8973346038865299e+00, 0.6582291204811077e+00, 0.1507168371301046e+00, 0.6123149049967000e+00, 0.9786602381321465e+00, 0.9991422694268032e+00, 0.2567979861831603e+00, 0.5508654031552206e+00, 0.6590475178936379e+00, 0.5540051353968156e+00, 0.9777600986094505e+00, 0.9019233044604550e+00, 0.6579246844202109e+00, 0.7288585083995637e+00, 0.4024552650632751e+00, 0.9286276631540373e+00, 0.1478351900438915e+00, 0.6745292990637881e+00, 0.7696143092758962e+00, 0.3393225521457851e+00, 0.1158188549489931e+00, 0.6143691843615144e+00, 0.8206171394658319e+00, 0.9470946559240657e+00, 0.7311286518535550e+00, 0.4976039064630389e+00, 0.3748017407872005e+00, 0.4215058560045335e+00, 0.5529030382557849e+00, 0.9979192791781042e+00, 0.9903947480415133e+00, 0.7463096553073095e+00, 0.9537590617479789e+00, 0.9327469033343772e-01, 0.7340236871158429e+00, 0.7517616159256260e+00, 0.9468485056759987e+00, 0.7061763632511533e+00, 0.8138096664824992e+00, 0.5585945195965548e+00, 0.6170557687507783e-01, 0.4803807801853603e+00, 0.5976897721963292e+00, 0.1375319188363359e+00, 0.5873952004066820e+00, 0.5199682629163997e+00, 0.8858783448129579e+00, 0.3038101719904888e+00, 0.6696573039722583e+00, 0.6649400915297697e+00, 0.5036768993979404e+00, 0.2615751204119269e+00},
			n:    21,
			zOut: []float64{0.2550031738262204e+01, 0.2480977513501848e+01, 0.2366602269912148e+01, 0.2169993432366266e+01, 0.1810817848712033e+01, 0.1681677309759878e+01, 0.1303743375476047e+01, 0.1202161769544433e+01, 0.1183377192742860e+01, 0.1094206688544886e+01, 0.9154376259418607e+00, 0.8241395430971566e+00, 0.7318275341991307e+00, 0.5186198053161721e+00, 0.4143051093784424e+00, 0.2112065329503869e+00, 0.1093987140067686e+00, 0.9751785856405315e-01, 0.2818174710670554e-01, 0.1697846193036144e-01, 0.2117542506861687e-04, 0.5241368559131172e-22, 0.4369342452764876e-18, 0.4084491703488284e-18, 0.1303743375476047e+01, 0.0000000000000000e+00, 0.2325140442247288e-18, 0.2353545363971710e-18, 0.1202161769544433e+01, 0.0000000000000000e+00, 0.2163565864913247e-18, 0.2222159192042978e-18, 0.1183377192742860e+01, 0.2033579915108999e-22, 0.2652482201353177e-18, 0.2438256017573345e-18, 0.1094206688544886e+01, 0.3167883469916549e-16, 0.9806070694382835e-18, 0.1224486171222500e-17, 0.9154376259418607e+00, 0.2764930279233778e-14, 0.2171122324673871e+02, 0.2171122324673871e+02, 0.1020000000000000e+03, 0.3081632653061225e+01, 0.2941176470588236e+01, 0.2548071704976161e-16, 0.7318275341991307e+00, 0.4959852501050381e-23, 0.8676862906242004e-16, 0.1414000300319855e-15, 0.5186198053161721e+00, 0.1503295986001297e-14, 0.3452895805257589e-14, 0.7981257539768321e-14, 0.4143051093784424e+00, 0.1303656847202082e-15, 0.9855809897129084e-12, 0.2614986238618434e-11, 0.2112065329503869e+00, 0.8411884198867843e-17, 0.2293573303077261e-09, 0.9833566024906726e-10, 0.1093987140067686e+00, 0.2696165428113804e-17, 0.5825676764620370e-08, 0.2638661645752538e-08, 0.9751785856405315e-01, 0.6617412588681544e-21, 0.2809066689524106e-07, 0.6165545233233256e-07, 0.2818174710670554e-01, 0.1396210563637443e-18, 0.2854399814229969e-05, 0.1344352644992036e-05, 0.1697846193036144e-01, 0.6162975822039155e-31, 0.3724766654883956e-05, 0.4695180568393632e-05, 0.2117542506861687e-04, 0.0000000000000000e+00, 0.9895328911616120e-03, 0.1620493249248586e-02},
			info: 0,
		},
		{
			z:    []float64{0.7655950151081736e-01, 0.1012496627944287e+00, 0.5492657397218519e+00, 0.3755849474908193e+00, 0.1514950257902281e-01, 0.7929154460744389e+00, 0.6208775449015512e+00, 0.7736035774285512e+00, 0.9535807609862423e+00, 0.1142443721338974e+00, 0.3184626428755636e+00, 0.5968198462235605e+00, 0.4815290260558158e-01, 0.1142057780273510e+00, 0.2159649191761220e+00, 0.1005733924928321e+00, 0.7334180249993905e-01, 0.2468617397276878e+00, 0.4433842667962828e+00, 0.2083675732544600e+00, 0.5669983427348256e+00, 0.2431239969651688e-01, 0.4202905770712168e+00, 0.3978530241169832e+00, 0.9765854254920880e+00, 0.6926050329928320e+00, 0.4943367486777372e-02, 0.1299210324451839e+00, 0.4677725963527957e-01, 0.8397777412847708e+00, 0.6784888222113696e+00, 0.5819508167748705e+00, 0.7335259194567719e+00, 0.1160427446694747e+00, 0.8402996546275487e+00, 0.8349959735427709e+00, 0.7465363962886192e+00, 0.8432008930683078e+00, 0.5288390014350658e+00, 0.6654846664813405e+00, 0.7307365692392301e+00, 0.4106042636492306e+00, 0.3557215961646439e+00, 0.7353770423280176e+00, 0.4713176648251016e+00, 0.4626254343486430e+00, 0.7596917071958065e+00, 0.7024594192654096e+00, 0.2579658084846748e+00, 0.9377050325544740e+00, 0.4561035840049426e+00, 0.8084892970067921e+00, 0.9088480529888660e+00, 0.6948766633235142e+00, 0.2194885062923195e+00, 0.8549545559228027e+00, 0.7443966896835119e+00, 0.3011130612705175e+00, 0.6719685452983997e+00, 0.6187140363783860e+00, 0.9675736114028010e+00, 0.9902393027740470e+00, 0.3380065169449795e+00, 0.9207672475882130e+00, 0.3390733399571427e+00, 0.9309435300662920e+00, 0.5465285509796513e+00, 0.4655125893217942e+00, 0.1769140958718970e+00, 0.3779940975005719e+00, 0.1752206228227682e+00, 0.3568582675012224e+00, 0.6656764250906758e+00, 0.6185492680730227e+00, 0.4689472437795331e+00, 0.3162518610221317e+00, 0.2688799086902824e+00, 0.1999212438248075e+00, 0.4154279763213168e+00, 0.9793029133347251e+00, 0.5432115888768534e+00, 0.1295182752745038e+00, 0.8047416637896615e+00, 0.8458210244967665e+00},
			n:    21,
			zOut: []float64{0.2649710614371106e+01, 0.2321564474027070e+01, 0.2090779203479937e+01, 0.2001510674733695e+01, 0.1702794694134603e+01, 0.1077066053646038e+01, 0.1060140274732043e+01, 0.9894235909971354e+00, 0.9539342071687115e+00, 0.8046649468928653e+00, 0.7009142227469247e+00, 0.5502651835254770e+00, 0.4423863025187732e+00, 0.3697086796938907e+00, 0.1446051340026323e+00, 0.1110032523123295e+00, 0.7513603923341917e-01, 0.6860214953971246e-01, 0.2434478048112329e-01, 0.6989818532012803e-03, 0.7811996215926567e-04, 0.7156177943897596e-01, 0.2292199980814605e-03, 0.1899018778701386e-03, 0.1060140274732043e+01, 0.8660746506696473e-01, 0.1357005210961402e-39, 0.1331360138522907e-23, 0.9894235909971354e+00, 0.8804208964992894e-17, 0.8660746506696473e-01, 0.8679736700028205e-01, 0.9539342071687115e+00, 0.2761013168273541e-29, 0.1040577915698430e-22, 0.8107757314001177e-24, 0.8046649468928653e+00, 0.5098326619997980e-22, 0.1416795225784663e-10, 0.1216165317638265e-10, 0.7009142227469247e+00, 0.1183291357831518e-29, 0.1813933158005285e+02, 0.1813933158005285e+02, 0.1090000000000000e+03, 0.3231292517006803e+01, 0.5504587155963303e+01, 0.4068037876491279e-10, 0.4423863025187732e+00, 0.2477754901417239e-20, 0.1737857614720001e-07, 0.5295826057530262e-07, 0.3697086796938907e+00, 0.5143373102040997e-26, 0.1312431380925897e-05, 0.1614374370413396e-05, 0.1446051340026323e+00, 0.9928287808749566e-25, 0.3777515963415321e-05, 0.2773141909621761e-05, 0.1110032523123295e+00, 0.0000000000000000e+00, 0.2532463507333992e-05, 0.2934028940292093e-05, 0.7513603923341917e-01, 0.5394210206791908e-19, 0.6032617175984252e-07, 0.2575740214720034e-06, 0.6860214953971246e-01, 0.7542232825258426e-19, 0.3784397549471832e-09, 0.1131370986389306e-09, 0.2434478048112329e-01, 0.1013559914197709e-18, 0.1483665133446019e-05, 0.1887408451311279e-05, 0.6989818532012803e-03, 0.9466330862652142e-28, 0.2539111990097616e-03, 0.9240506410060376e-03, 0.7811996215926567e-04, 0.3786532345060857e-28, 0.1283811140869274e-03, 0.1684766835109566e-03},
			info: 0,
		},
		{
			z:    []float64{0.7090328374865308e+00, 0.5289732604915972e+00, 0.6638408676820399e+00, 0.9007392577685345e+00, 0.9411642674278379e+00, 0.1144202631330921e+00, 0.8666675899972089e+00, 0.6323680737151134e+00, 0.2427877245566508e+00, 0.1603957810123527e+00, 0.2123565414665629e-01, 0.1902539934369850e+00, 0.3123401294447815e+00, 0.8096475663099788e+00, 0.5737184899962203e+00, 0.8262606712006769e+00, 0.2864548736211023e+00, 0.1605147759074373e+00, 0.3309241142010387e+00, 0.2658533720189140e+00, 0.5447811280109880e+00, 0.5535203163154195e+00, 0.2161483615344606e-01, 0.1487480590324270e+00, 0.1049033762317957e+00, 0.6142137665037214e+00, 0.4062909815669834e+00, 0.9015282863354186e+00, 0.9428381000165521e+00, 0.1566198551488233e+00, 0.9369593797444957e+00, 0.1354729715149707e+00, 0.3712821832808397e+00, 0.3215112250898227e+00, 0.7149642455474942e+00, 0.7297077774959745e+00, 0.3785927874981971e+00, 0.8289344072747380e+00, 0.9536643756705742e+00, 0.3084192540817629e+00, 0.5797406195506623e+00, 0.2334488720392539e+00, 0.8444034680728870e+00, 0.2581324672603945e+00, 0.8397436353763532e+00, 0.5376562757207516e+00, 0.6215703771401371e+00, 0.4908247191212207e+00, 0.2848423854658637e+00, 0.2355591822832676e+00, 0.6513799258897619e+00, 0.6071703403704332e+00, 0.3981458512505542e+00, 0.8487946917355450e+00, 0.6376169839751418e+00, 0.2588233361358196e+00, 0.3588309885453386e+00, 0.5618301121344036e+00, 0.1214685095104872e+00, 0.9784693864155333e+00, 0.3832063578469370e+00, 0.7206987579838013e+00, 0.3582640246022534e+00, 0.9098815720988161e+00, 0.5436872318254895e+00, 0.9596753969896468e+00, 0.9294728975416107e+00, 0.6333208483475120e+00, 0.2956089747729942e+00, 0.1747395639573214e+00, 0.8425785962238860e+00, 0.6185896542974498e+00, 0.7058928872472775e+00, 0.2424399357290958e+00, 0.7979610473510427e+00, 0.3910140128664733e+00, 0.1099592891010870e+00, 0.7536879342329168e+00, 0.2979717756246736e-01, 0.7006607890734408e+00, 0.5252033671714352e+00, 0.7525598235593517e+00, 0.6476778890126167e+00, 0.8710847630132640e+00},
			n:    21,
			zOut: []float64{0.2486733497028020e+01, 0.2359071020584524e+01, 0.2276435093308226e+01, 0.2061969496713965e+01, 0.1674052496256098e+01, 0.1634406821090099e+01, 0.1324515540477209e+01, 0.1213036064691536e+01, 0.1142092570196908e+01, 0.8479728747156923e+00, 0.7723287409164926e+00, 0.6845846156476076e+00, 0.4690003902541462e+00, 0.3969394481970619e+00, 0.3209839311891622e+00, 0.1762493640751034e+00, 0.1552318325824889e+00, 0.1299582823757977e+00, 0.4568511390642456e-01, 0.2058742849236605e-01, 0.6687292092604351e-04, 0.0000000000000000e+00, 0.9075510178548721e-16, 0.1809089799756099e-15, 0.1324515540477209e+01, 0.1972152263052530e-28, 0.7207148989396829e-15, 0.1135837106477965e-14, 0.1213036064691536e+01, 0.6865185493883853e-20, 0.3862483738583121e-14, 0.5795548067006594e-14, 0.1142092570196908e+01, 0.1237761291638161e-15, 0.1761387669893985e-13, 0.1204467761940304e-13, 0.8479728747156923e+00, 0.5980780785048348e-20, 0.5961915529495600e-12, 0.8546407799162654e-12, 0.7723287409164926e+00, 0.9983823616477125e-26, 0.2019190149561986e+02, 0.2019190149561985e+02, 0.1120000000000000e+03, 0.3267573696145125e+01, 0.7142857142857143e+01, 0.2981196239393922e-10, 0.4690003902541462e+00, 0.1411215361455214e-23, 0.1291839253366726e-09, 0.2114581039065388e-09, 0.3969394481970619e+00, 0.3215018396446338e-24, 0.8992249144513522e-09, 0.1460279739604385e-08, 0.3209839311891622e+00, 0.5301710738582062e-20, 0.5956459688113839e-08, 0.9509260151006140e-08, 0.1762493640751034e+00, 0.3305918938014538e-18, 0.2145126200057182e-06, 0.1391189925457742e-06, 0.1552318325824889e+00, 0.0000000000000000e+00, 0.8222867095540534e-06, 0.8520250895881005e-06, 0.1299582823757977e+00, 0.1946507846527860e-19, 0.4299487364203470e-05, 0.2670451838837521e-05, 0.4568511390642456e-01, 0.1783600337601646e-17, 0.1688330183620908e-03, 0.2840434656464124e-03, 0.2058742849236605e-01, 0.2127174768075152e-17, 0.5266290662212281e-04, 0.3082204001196310e-04, 0.6687292092604351e-04, 0.4725269249209171e-22, 0.3479992598793783e-02, 0.7157534417412210e-03},
			info: 0,
		},
		{
			z:    []float64{0.8307212987388154e+00, 0.9571804405037874e+00, 0.2222674360399970e+00, 0.5794927951917870e+00, 0.9907605078898842e+00, 0.8399673535531696e+00, 0.1748054629244838e+00, 0.7095417514667766e+00, 0.3015507353664195e+00, 0.2504155371033889e-01, 0.9167396887269817e+00, 0.6888065712047682e+00, 0.7567577995550097e+00, 0.9327280452266753e+00, 0.5500991579404824e-01, 0.4989499454693689e+00, 0.5949898652751251e-01, 0.1718660951302574e+00, 0.6883620036166904e+00, 0.3066205390931208e+00, 0.7351901912310475e+00, 0.7521684164853142e+00, 0.3957210266430871e+00, 0.8778502662051324e-01, 0.4896297811280748e-01, 0.7793618951247419e+00, 0.6807766665219679e+00, 0.3115699624095645e+00, 0.5227646665950632e+00, 0.5057964692468060e+00, 0.4385260593432981e+00, 0.5876392780963172e+00, 0.2973252261444674e+00, 0.2948995727695043e+00, 0.7378444260331429e-01, 0.9749457382350615e+00, 0.7281626398300706e+00, 0.7580903878987629e+00, 0.3091762186458183e+00, 0.2464983240671905e+00, 0.2568872936019900e+00, 0.6180627613788815e+00, 0.9377882352337240e+00, 0.9921824152100049e+00, 0.2248192100106184e+00, 0.9472840045361628e+00, 0.8477853668436538e+00, 0.7001263233958666e+00, 0.5544440083201581e+00, 0.4415839019184687e+00, 0.2491818172148126e+00, 0.8976627554942691e+00, 0.1457298176556392e+00, 0.4345936198183369e+00, 0.4573762249453559e+00, 0.3093805347522776e+00, 0.4809246108862507e+00, 0.6897426752489948e+00, 0.8107988692352123e+00, 0.3921644162683642e+00, 0.2931487419364392e+00, 0.5759780604710760e+00, 0.7253418085569345e+00, 0.1427733376597652e+00, 0.3335555091947459e+00, 0.1603897948564823e+00, 0.5345943097231743e+00, 0.7233253065718669e+00, 0.2281802915301432e+00, 0.1990064435881517e+00, 0.9726196083044836e+00, 0.9935713768557323e+00, 0.8195201715007894e-01, 0.2508806553004002e+00, 0.2158489797899930e+00, 0.1957771185505375e+00, 0.8502651710165565e+00, 0.4468235108377239e-01, 0.8639211642125266e+00, 0.6713373034638015e-01, 0.4273173462824215e+00, 0.1373188375801981e+00, 0.1190874985301729e+00, 0.3059495456429208e+00},
			n:    21,
			zOut: []float64{0.2399198387954965e+01, 0.2388283868878418e+01, 0.2234728187505421e+01, 0.1947641030296732e+01, 0.1845273105794422e+01, 0.1837097911259683e+01, 0.1471118007292610e+01, 0.1105651020532553e+01, 0.9883044021137968e+00, 0.9478841363642468e+00, 0.8550293065516678e+00, 0.7566674361156733e+00, 0.5310790726859680e+00, 0.3571217714437427e+00, 0.2893653184719416e+00, 0.2534414197270934e+00, 0.2173582726294167e+00, 0.4908458037260943e-01, 0.1246498683649276e-01, 0.5862142085722914e-02, 0.4704104342516643e-04, 0.1939986690620616e-19, 0.1034652479755287e-16, 0.1536368559490732e-16, 0.1471118007292610e+01, 0.5364686198585281e-13, 0.6088846084221820e-07, 0.1529604999486430e-06, 0.1105651020532553e+01, 0.2903008131213323e-27, 0.1160127138141562e-06, 0.1335479603764263e-06, 0.9883044021137968e+00, 0.2125606020077514e-18, 0.5906967539735199e-07, 0.6675815740532400e-07, 0.9478841363642468e+00, 0.1331202777560457e-29, 0.4438231138534822e-07, 0.3948328125271971e-07, 0.8550293065516678e+00, 0.5860341995390921e-20, 0.2049270140595660e+02, 0.2049270140595660e+02, 0.9800000000000000e+02, 0.2893424036281179e+01, 0.4081632653061225e+01, 0.1377712740565197e-07, 0.5310790726859680e+00, 0.1537961403283359e-19, 0.8011211497963711e-08, 0.8400248580143560e-08, 0.3571217714437427e+00, 0.5389497704469953e-25, 0.6965765014245226e-08, 0.7336584331089061e-08, 0.2893653184719416e+00, 0.6081961202185573e-17, 0.9973730165755502e-08, 0.8562226724273748e-08, 0.2534414197270934e+00, 0.3644137891043716e-17, 0.1552441635408352e-06, 0.1605842922922618e-07, 0.2173582726294167e+00, 0.1710250442519154e-26, 0.2805940935843235e-04, 0.1153299285109565e-04, 0.4908458037260943e-01, 0.1140499340513312e-15, 0.3458275194991493e-03, 0.2689727098393103e-03, 0.1246498683649276e-01, 0.4437342591868191e-30, 0.1429141842628293e-03, 0.1683368406483595e-03, 0.5862142085722914e-02, 0.2515327803596613e-22, 0.2394538487567536e-03, 0.1603201011413965e-03, 0.4704104342516643e-04, 0.0000000000000000e+00, 0.2149753858259932e-02, 0.4092589741085703e-02},
			info: 0,
		},
		{
			z:    []float64{0.6132370003914160e+00, 0.6636632940312478e+00, 0.8959100063178993e+00, 0.7536793777540497e+00, 0.6298705159481864e+00, 0.2209880035966978e+00, 0.4838373503885766e+00, 0.6150203746616348e+00, 0.9876587724998732e+00, 0.5591876171910803e+00, 0.3708840600033242e+00, 0.3375947236522073e+00, 0.7454250822233153e+00, 0.5697349010621601e+00, 0.4561367007431339e+00, 0.6481896738286699e+00, 0.6835960923418801e+00, 0.1703905177923075e+00, 0.2730661115049011e+00, 0.4407624886449348e+00, 0.4577220273293221e+00, 0.3952566732937318e+00, 0.6883342106097436e+00, 0.2988887099704991e+00, 0.2052274263883085e+00, 0.1922842703083140e+00, 0.4189999528960880e+00, 0.6906452535299822e+00, 0.1405554539793709e+00, 0.1437412601859918e+00, 0.6624635742176765e+00, 0.5756752017566100e+00, 0.7749505183416766e+00, 0.7752965769666669e+00, 0.7653541054136069e+00, 0.8765107194498384e+00, 0.9921378985956986e-01, 0.3345156485504346e+00, 0.6450936562634260e+00, 0.1316181443467312e+00, 0.8075436662342752e+00, 0.5418723210737144e+00, 0.2657525890099199e+00, 0.4588749915888913e+00, 0.5109250246010762e+00, 0.5048342930643721e+00, 0.9617739307304302e+00, 0.5502351700731637e+00, 0.1896288106400514e+00, 0.2011189112848212e+00, 0.2487376305081674e+00, 0.9700914054633822e+00, 0.7763920500140777e+00, 0.2727004477756484e+00, 0.4969507403482072e+00, 0.3987515903118115e+00, 0.6132963384905279e+00, 0.8526865910258985e-01, 0.7370021348269777e+00, 0.8628692345603857e+00, 0.3410351221174988e+00, 0.6944408188735030e+00, 0.6418832405583447e+00, 0.8549561267667016e-01, 0.7079842121947655e+00, 0.5924596383322266e+00, 0.2327865765542259e+00, 0.7519783901092147e-01, 0.3243359400926964e+00, 0.9130914950306788e+00, 0.2042866226175709e+00, 0.4527677275849958e+00, 0.7271733772634871e+00, 0.2835797318864963e-01, 0.5170298378180002e+00, 0.7676612860726464e+00, 0.8943646165240225e+00, 0.9444900692816881e+00, 0.2898061819682256e+00, 0.5314658042534958e+00, 0.2328255902743269e+00, 0.3127766212762262e+00, 0.1415213232426512e+00, 0.3425808715425687e+00},
			n:    21,
			zOut: []float64{0.2568497247033635e+01, 0.2474290441512907e+01, 0.2254355189310800e+01, 0.1987099258783412e+01, 0.1595738223681455e+01, 0.1520583769371047e+01, 0.1349481184037943e+01, 0.1286943522977408e+01, 0.1233633359467258e+01, 0.1105388350935419e+01, 0.8741663391031235e+00, 0.7042784684853729e+00, 0.6809050564171195e+00, 0.5312535309083547e+00, 0.3765999563195902e+00, 0.2982885955048700e+00, 0.1761408438030470e+00, 0.1021950592815966e+00, 0.5149411253987173e-01, 0.2409819364774552e-01, 0.3292801347387554e-02, 0.7649046082865931e-16, 0.2073367755475779e-07, 0.1625592945762753e-07, 0.1349481184037943e+01, 0.1183291357831518e-29, 0.4660703060512034e-13, 0.1807862490656452e-12, 0.1286943522977408e+01, 0.1649665924998180e-25, 0.1294990598064480e-10, 0.5231829186377068e-10, 0.1233633359467258e+01, 0.8526291632353489e-16, 0.3101844497115867e-08, 0.1140775990076438e-07, 0.1105388350935419e+01, 0.9624103043696344e-28, 0.5663389137423196e-06, 0.4346359020206590e-06, 0.8741663391031235e+00, 0.3067428736041242e-19, 0.2119872350446936e+02, 0.2119872350446936e+02, 0.1030000000000000e+03, 0.3056689342403628e+01, 0.2912621359223301e+01, 0.2039882393707806e-06, 0.6809050564171195e+00, 0.0000000000000000e+00, 0.3469744030301619e-06, 0.2644131172050035e-06, 0.5312535309083547e+00, 0.3043881270837009e-21, 0.5541811479797519e-05, 0.3827690844980446e-05, 0.3765999563195902e+00, 0.1909043390634849e-27, 0.3412067971678679e-04, 0.3418356678006945e-04, 0.2982885955048700e+00, 0.1167175485134229e-15, 0.1390598737761422e-03, 0.1685020271483295e-03, 0.1761408438030470e+00, 0.3311298373357662e-16, 0.2687183242739271e-03, 0.2970561145148615e-03, 0.1021950592815966e+00, 0.3861046454850253e-20, 0.3877059717439437e-03, 0.4274183271611622e-03, 0.5149411253987173e-01, 0.3171094621243632e-24, 0.5614281929126997e-03, 0.6201503766753644e-03, 0.2409819364774552e-01, 0.5854827030937197e-31, 0.8580439238194122e-03, 0.9937175751003719e-03, 0.3292801347387554e-02, 0.1687712113428427e-19, 0.6404816195044538e-02, 0.3574263865391779e-02},
			info: 0,
		},
		{
			z:    []float64{0.1535280608392028e+00, 0.8492164242958209e+00, 0.9720508422713467e+00, 0.4409240313091006e+00, 0.7744413453832578e+00, 0.4562143200311385e+00, 0.8645004927526174e+00, 0.7279750979364787e+00, 0.6373358699411871e+00, 0.7240388495630282e+00, 0.6042124182518795e+00, 0.2629417492305242e-01, 0.2927658696806527e+00, 0.1971917075100318e+00, 0.9486620805007744e+00, 0.5318461733579811e+00, 0.1035922758398233e-01, 0.4521355876782263e+00, 0.9635828765019094e+00, 0.9861179636072996e+00, 0.3409131837267612e+00, 0.9592913933135179e+00, 0.3267081303471464e+00, 0.7042329006712417e+00, 0.7904477014671945e+00, 0.7235636505410947e+00, 0.7289557043239185e+00, 0.4453819137782510e+00, 0.3817792285714174e+00, 0.9222246883547156e+00, 0.7484233302016036e+00, 0.4267300781414476e+00, 0.2174272124496083e+00, 0.7294135193889231e+00, 0.7969150359898325e+00, 0.5584233158827989e+00, 0.8854167010119880e+00, 0.3831257236275667e+00, 0.2407966333760054e+00, 0.3150832526072347e+00, 0.4329046703236793e+00, 0.4798261053116015e-01, 0.4240797212710790e+00, 0.7373271503520912e+00, 0.8727856342792233e+00, 0.3079871491200393e+00, 0.8433176604823556e+00, 0.8160920845908722e+00, 0.4068006968345205e+00, 0.8649881745683563e+00, 0.8522504948442566e+00, 0.5120820890311298e+00, 0.7764141319706099e+00, 0.4928600452500731e+00, 0.2048536231221674e+00, 0.4451308747598662e+00, 0.4335546033385644e+00, 0.5471256397417612e-01, 0.7004960984988220e+00, 0.9192372420579460e+00, 0.3890133989146303e+00, 0.2488782917619309e+00, 0.3663470187625480e+00, 0.9986636866778884e+00, 0.6620792031393874e+00, 0.1797345205931197e+00, 0.8723761011602340e+00, 0.2439575243714007e+00, 0.1163919437436586e+00, 0.8182195270597429e+00, 0.5356403226658167e+00, 0.9880537276300997e+00, 0.4163057033602857e+00, 0.8081980088045720e+00, 0.2399383760782623e+00, 0.9702581513812658e+00, 0.3763181880939181e+00, 0.6334004896983517e+00, 0.1368995930243956e+00, 0.7684724492197631e+00, 0.1011989548860721e+00, 0.5519462180485674e+00, 0.8316516728108886e+00, 0.3772586994144635e+00},
			n:    21,
			zOut: []float64{0.2579297838383781e+01, 0.2566242251743589e+01, 0.2402206059499122e+01, 0.2314572748831938e+01, 0.2210246243389976e+01, 0.2073991263591532e+01, 0.1647852770173070e+01, 0.1458062583915517e+01, 0.1424747680105817e+01, 0.1229594627898745e+01, 0.8682818248784862e+00, 0.7800126440184885e+00, 0.7769870810337002e+00, 0.4018986220147585e+00, 0.3451236345722463e+00, 0.2059421010707241e+00, 0.1665395041671736e+00, 0.1384541468623413e+00, 0.4569625338134105e-01, 0.3572364569397507e-01, 0.7785678858960618e-04, 0.1325624140040768e-19, 0.1464254707155794e-03, 0.9424803187834284e-04, 0.1647852770173070e+01, 0.1564186274351218e-15, 0.3473328524848360e-05, 0.4592867365821283e-05, 0.1458062583915517e+01, 0.3231174267785264e-26, 0.1403657959312935e-04, 0.9634257574341414e-05, 0.1424747680105817e+01, 0.1020140793606197e-15, 0.1723203566561169e-05, 0.6067379842882497e-05, 0.1229594627898745e+01, 0.2663149152121406e-17, 0.1061013981937667e-04, 0.1674993264096194e-04, 0.8682818248784862e+00, 0.8099755562101574e-24, 0.2367155138201492e+02, 0.2367155138201491e+02, 0.1110000000000000e+03, 0.3174603174603174e+01, 0.5405405405405405e+01, 0.8398559432747079e-06, 0.7769870810337002e+00, 0.0000000000000000e+00, 0.6532651675838807e-05, 0.1393736055608104e-04, 0.4018986220147585e+00, 0.1488702756444268e-16, 0.4786924897797409e-03, 0.2484818164296786e-03, 0.3451236345722463e+00, 0.1393994753571567e-18, 0.7967956438667165e-06, 0.1566848631115136e-05, 0.2059421010707241e+00, 0.1872521839860791e-17, 0.1674828806776016e-04, 0.3829209207735075e-04, 0.1665395041671736e+00, 0.5127595883936577e-29, 0.1145979873394449e-04, 0.6763541499702777e-05, 0.1384541468623413e+00, 0.6162975822039155e-31, 0.1509781169748205e-03, 0.4934238518442497e-04, 0.4569625338134105e-01, 0.3815207438965640e-25, 0.3298388086972868e-02, 0.1367005987525658e-02, 0.3572364569397507e-01, 0.2773930797150972e-21, 0.3955985102398275e-03, 0.3433403686203228e-03, 0.7785678858960618e-04, 0.7790054198074833e-21, 0.1914938157793930e-01, 0.1330976133014081e-01},
			info: 0,
		},
		{
			z:    []float64{0.8677677383100214e+00, 0.6246622741727026e+00, 0.5523136647126087e+00, 0.2116103484147348e+00, 0.8227197076777661e+00, 0.3636019443697256e+00, 0.2305222034118074e+00, 0.3291143648196290e+00, 0.6800926583943011e+00, 0.4290450477605455e+00, 0.4523048516285796e+00, 0.9276084588261152e+00, 0.4462311938877914e+00, 0.6672491542978151e+00, 0.7808537049355564e+00, 0.7908271145868994e+00, 0.1977402045683964e+00, 0.2563794060302639e+00, 0.6768044668199432e+00, 0.8985309013607360e+00, 0.3750140409512910e+00, 0.2003663066929584e-01, 0.4020022629723622e+00, 0.9909625073113403e+00, 0.9747495116121061e+00, 0.9702779584867087e+00, 0.9688371740876045e+00, 0.3747269639045615e+00, 0.9728403214040160e-01, 0.7507234588709335e+00, 0.8245099981459856e+00, 0.7244448753186800e+00, 0.3670523289486082e+00, 0.6220520718362722e+00, 0.3063226611493952e+00, 0.3012769332894331e+00, 0.6763821017483275e+00, 0.8300640433662996e+00, 0.5563237542291373e+00, 0.5594449054752113e+00, 0.3278038518373678e+00, 0.7307041435680680e+00, 0.1582529760475658e+00, 0.5328751876026443e+00, 0.7841944053171677e+00, 0.2157818394558657e+00, 0.3253134136288985e+00, 0.5014502641306768e+00, 0.2229178930043155e+00, 0.4664985859426845e+00, 0.1867987929192785e+00, 0.1951091673564507e+00, 0.2069185156156168e+00, 0.9058058542218750e+00, 0.7999188612304056e+00, 0.7349484002829904e+00, 0.4153480749961962e+00, 0.8109879258422723e+00, 0.9883741464101338e+00, 0.4911202582082937e+00, 0.2193166361224579e+00, 0.7618820222564749e+00, 0.9073087210515056e+00, 0.5896383620860655e+00, 0.8706198942119541e+00, 0.3860710050052940e-01, 0.9380336544251486e+00, 0.4690283060543987e-01, 0.8970820572284144e+00, 0.1974579201509297e-01, 0.9992442335759629e+00, 0.9801967887432729e+00, 0.6116617864086051e+00, 0.5875675950916066e+00, 0.9622537584002233e+00, 0.8934200351761762e+00, 0.2022182729782195e+00, 0.1273316669652236e-01, 0.2549904925850485e+00, 0.8904280763656908e+00, 0.2695748922419082e+00, 0.8890196843116763e+00, 0.5191439214393291e+00, 0.3672061831861048e+00},
			n:    21,
			zOut: []float64{0.3071910136434907e+01, 0.2366684893081943e+01, 0.2247985022656176e+01, 0.2134160618983571e+01, 0.1929380862422316e+01, 0.1895668448583188e+01, 0.1735809456784568e+01, 0.1390623822795548e+01, 0.1372853012616850e+01, 0.1197489438156041e+01, 0.1106600205023489e+01, 0.8564547950868329e+00, 0.4949238197552136e+00, 0.3889452841496929e+00, 0.3405947975938995e+00, 0.2516597920662951e+00, 0.2113890461921598e+00, 0.1169778386549868e+00, 0.1072268106836703e+00, 0.7646784687889400e-02, 0.1286588928027629e-02, 0.1194767075191949e-22, 0.1162094363453686e-09, 0.8691558890693288e-10, 0.1735809456784568e+01, 0.8134816760001801e-14, 0.1019665656815186e-08, 0.1906916241197346e-08, 0.1390623822795548e+01, 0.0000000000000000e+00, 0.1594323424098234e-05, 0.3312168640703357e-05, 0.1372853012616850e+01, 0.0000000000000000e+00, 0.3830524268846776e-04, 0.1577534719508939e-04, 0.1197489438156041e+01, 0.1592850300841939e-16, 0.1256296423282912e-03, 0.1149042579564614e-03, 0.1106600205023489e+01, 0.6530365529382261e-18, 0.2322627147533726e+02, 0.2322627147533726e+02, 0.1020000000000000e+03, 0.2963718820861678e+01, 0.3921568627450980e+01, 0.6483516393063832e-03, 0.4949238197552136e+00, 0.1972152263052530e-28, 0.5430985229870038e-04, 0.4447663749968261e-04, 0.3889452841496929e+00, 0.1329049288162406e-17, 0.2738942590248176e-04, 0.1582812372822289e-04, 0.3405947975938995e+00, 0.1187508759187252e-18, 0.7465410105999781e-04, 0.1255487400488340e-03, 0.2516597920662951e+00, 0.4141519752410312e-29, 0.2152266118694742e-04, 0.6710408995051273e-05, 0.2113890461921598e+00, 0.3333294884283764e-18, 0.6132111109262542e-03, 0.4277764486586744e-03, 0.1169778386549868e+00, 0.2711709361697228e-30, 0.3832079371139971e-03, 0.3904887652242632e-03, 0.1072268106836703e+00, 0.2176254805972020e-18, 0.4306247975784355e-03, 0.4948147025217374e-03, 0.7646784687889400e-02, 0.1110488679882733e-19, 0.1569670832436511e-02, 0.2092370053382882e-02, 0.1286588928027629e-02, 0.5442848430598922e-18, 0.4460307636637516e-02, 0.7046212350403941e-02},
			info: 0,
		},
		{
			z:    []float64{0.9838241499564321e+00, 0.8769514477589457e+00, 0.6814467524843054e+00, 0.3624618995089947e+00, 0.1981908649829056e+00, 0.1872278407669582e+00, 0.9876554444003832e+00, 0.1826583579385833e+00, 0.1183269767614670e-01, 0.9437175277915953e+00, 0.4919320768587329e+00, 0.9805291749478650e+00, 0.6132842150855770e+00, 0.3626870797541525e+00, 0.5323780753704027e+00, 0.1457937981484730e+00, 0.1328300737138376e+00, 0.3811835111034733e+00, 0.7003413504474413e+00, 0.5497308136521228e+00, 0.8255713255718768e+00, 0.5373393799690840e+00, 0.6623327475759663e+00, 0.5316913920782962e-01, 0.1714649183335913e+00, 0.6186592957613974e+00, 0.7587327126921758e+00, 0.9100586255702885e+00, 0.7140000863381626e+00, 0.7400119974078728e+00, 0.3567025022071233e+00, 0.8218612151637966e+00, 0.5520556331010515e+00, 0.6508027169553372e+00, 0.1875664464519897e+00, 0.8183092398415237e+00, 0.2239662369231676e-01, 0.6571790154721793e+00, 0.1196415656369646e+00, 0.3853512775546503e+00, 0.2426450009067070e+00, 0.9653447575661293e+00, 0.4852906340727067e+00, 0.9923496603563310e+00, 0.2812811891225394e+00, 0.9784300144088514e+00, 0.9528689986126391e+00, 0.2431080020200448e+00, 0.4166208849723857e+00, 0.7357806826601304e+00, 0.1952831632199464e+00, 0.6393407699957093e+00, 0.6296804002658308e+00, 0.1028694374663701e+00, 0.1387452591062114e+00, 0.4549213107961220e+00, 0.9843671628976580e+00, 0.4249152858158138e+00, 0.7048043599695387e+00, 0.3653998904622074e+00, 0.2003765309007774e+00, 0.4095912484595889e+00, 0.2392233396621621e+00, 0.7050044605645847e+00, 0.7908086424110450e-01, 0.2038177493661538e+00, 0.1735168081909322e-01, 0.3343100300667304e+00, 0.2848572906345944e+00, 0.8454138442968477e+00, 0.4944340369822147e+00, 0.7101083458004501e+00, 0.2546477176633619e+00, 0.8227235336957786e+00, 0.2816146807741101e+00, 0.6247995550265835e+00, 0.7458069042729408e+00, 0.8739345864578790e+00, 0.1278937298671606e+00, 0.2791833532060612e+00, 0.9234762152279486e+00, 0.9863012584972185e+00, 0.7892913846591531e+00, 0.3479266835120939e+00},
			n:    21,
			zOut: []float64{0.2623495482601064e+01, 0.2337512212773574e+01, 0.2310508039367694e+01, 0.2188663096012427e+01, 0.1954141717332891e+01, 0.1371476701077329e+01, 0.1267869950064541e+01, 0.1203016320141301e+01, 0.1132845050819559e+01, 0.9979737678355206e+00, 0.8633053234635751e+00, 0.6603430282082061e+00, 0.6064350433288983e+00, 0.4855760962077864e+00, 0.4280373135604714e+00, 0.3659370670152315e+00, 0.1453682093766173e+00, 0.9617861239701422e-01, 0.7192949634365248e-01, 0.1892143303127712e-02, 0.3946528731286962e-05, 0.0000000000000000e+00, 0.9258243476772570e-07, 0.9000418438492766e-07, 0.1267869950064541e+01, 0.0000000000000000e+00, 0.2500234135491857e-07, 0.1856395032302831e-07, 0.1203016320141301e+01, 0.2658426139847422e-18, 0.4168972835699797e-07, 0.5531467635213317e-07, 0.1132845050819559e+01, 0.2518044009465470e-24, 0.1194059466661935e-08, 0.2159625808467239e-08, 0.9979737678355206e+00, 0.1047506307937635e-23, 0.4478743649191460e-07, 0.9991281120631680e-07, 0.8633053234635751e+00, 0.5895762127714383e-22, 0.2111250861775921e+02, 0.2111250861775921e+02, 0.9900000000000000e+02, 0.2927437641723356e+01, 0.4040404040404041e+01, 0.9244672049175682e-07, 0.6064350433288983e+00, 0.1817535525629211e-26, 0.9398974933478891e-07, 0.9547946669183887e-07, 0.4855760962077864e+00, 0.3185418565406740e-17, 0.9937648506460343e-07, 0.9792626502078532e-07, 0.4280373135604714e+00, 0.2958228394578794e-30, 0.9921525775570031e-07, 0.1006425128472401e-06, 0.3659370670152315e+00, 0.4461863922899316e-21, 0.1047488573169732e-06, 0.1032270990738791e-06, 0.1453682093766173e+00, 0.6504493884088473e-15, 0.1102576940374778e-06, 0.1180293023862048e-06, 0.9617861239701422e-01, 0.6467797587852522e-18, 0.1844536574504796e-06, 0.1488485108626942e-06, 0.7192949634365248e-01, 0.2114103686638959e-13, 0.8531410278849863e-06, 0.1408572284551695e-05, 0.1892143303127712e-02, 0.0000000000000000e+00, 0.6242841403373791e-04, 0.3522115697192072e-04, 0.3946528731286962e-05, 0.6203854594147708e-24, 0.1209929086462260e-02, 0.4906802871097585e-03},
			info: 0,
		},
		{
			z:    []float64{0.7643247442799757e+00, 0.1930914554442843e+00, 0.6441117687067544e-01, 0.4522061669974708e-01, 0.1465182320932497e+00, 0.8069033698425149e+00, 0.6560188128523874e+00, 0.2446749124420051e+00, 0.3243649544497533e+00, 0.9836733438970116e+00, 0.6137527621157332e+00, 0.2925225554635034e+00, 0.4955619768673408e+00, 0.7361289066349539e+00, 0.3953262269487445e-01, 0.1565854250658760e-01, 0.4785236355428808e+00, 0.7766165270899886e+00, 0.9069259340739989e+00, 0.9151936268801151e+00, 0.3338292872636017e+00, 0.2993135005490550e+00, 0.8287022194584887e+00, 0.4593628480546942e+00, 0.1484428934315618e-01, 0.3390977660295040e+00, 0.5819746692101802e-01, 0.9277352020779983e-01, 0.6517310862845033e+00, 0.3951866136249272e+00, 0.6199300068296295e+00, 0.8252199884043853e+00, 0.3390764160478065e+00, 0.3084679704183562e+00, 0.3609211283483548e+00, 0.4116857210304998e+00, 0.4609476939442378e+00, 0.3824129844545763e+00, 0.6431169626236315e+00, 0.2416519334621993e+00, 0.7156780768158061e+00, 0.8568018406565006e+00, 0.1639408452444666e+00, 0.7313040585903831e+00, 0.5099183572592548e+00, 0.2801641590611897e+00, 0.9930562568266591e+00, 0.7612468327164370e+00, 0.3271886985411110e-01, 0.7066605591101006e+00, 0.6720867845388043e+00, 0.9004209067318458e-01, 0.4611918168927077e+00, 0.2638384667560661e+00, 0.3554909044606102e+00, 0.7231384727137510e+00, 0.2729268177075039e-01, 0.5848397641916390e+00, 0.6942450598380807e+00, 0.1106927451590289e+00, 0.5262572286481777e+00, 0.9936768911750095e+00, 0.7262536107933111e+00, 0.7604873714570974e+00, 0.2036182130165832e+00, 0.8632527217998969e+00, 0.1924957421132846e+00, 0.8815968660524154e-01, 0.2412813036310424e+00, 0.9105232147744077e+00, 0.8298587923387578e+00, 0.7754575363227978e-01, 0.8714764594177653e+00, 0.5571901679066804e+00, 0.3971870017409277e+00, 0.9993097753535422e+00, 0.7700188328643205e+00, 0.7856615172810383e+00, 0.7533953600915489e+00, 0.5602208064288483e+00, 0.6944234372397261e+00, 0.8756090822068926e+00, 0.1433700749181099e-01, 0.2857413018245216e+00},
			n:    21,
			zOut: []float64{0.2546753248931182e+01, 0.2051980365170268e+01, 0.1958366389726797e+01, 0.1631238683423845e+01, 0.1586034790683645e+01, 0.1538534425510682e+01, 0.1255412586747036e+01, 0.1121690972560896e+01, 0.9718147643766369e+00, 0.9049110982817965e+00, 0.7946671649858114e+00, 0.5884376759528481e+00, 0.4050217422474118e+00, 0.3189813042957475e+00, 0.2502079076659038e+00, 0.1378021633219403e+00, 0.1230588999356998e+00, 0.5024634530670342e-01, 0.4118371112977081e-01, 0.5279820157992808e-02, 0.1421284452028254e-03, 0.2680065184671810e-21, 0.2301930285321580e-14, 0.5132436462785431e-14, 0.1255412586747036e+01, 0.2424007516335156e-18, 0.1913029885324846e-10, 0.9021761757831104e-11, 0.1121690972560896e+01, 0.1977581519346623e-22, 0.4602280167506019e-08, 0.2460881124989248e-08, 0.9718147643766369e+00, 0.1104484640438811e-18, 0.4278245612761057e-07, 0.2387897413722468e-07, 0.9049110982817965e+00, 0.2768434971275651e-16, 0.3015076539683485e-07, 0.2961312447326530e-07, 0.7946671649858114e+00, 0.1262042385269225e-19, 0.1828176618885781e+02, 0.1828176618885782e+02, 0.1040000000000000e+03, 0.3122448979591837e+01, 0.3846153846153846e+01, 0.1851926648424158e-06, 0.4050217422474118e+00, 0.9293221980374701e-17, 0.1284245841610550e-05, 0.2164676294343609e-05, 0.3189813042957475e+00, 0.1378649105642637e-16, 0.4610281885819925e-06, 0.1003327305702036e-05, 0.2502079076659038e+00, 0.2177821011994986e-13, 0.3677246803752887e-04, 0.3095541926811936e-04, 0.1378021633219403e+00, 0.9210656120777790e-20, 0.9401458780737957e-05, 0.1039122642078444e-04, 0.1230588999356998e+00, 0.4930380657631324e-30, 0.6528493871226699e-06, 0.1524653341983116e-05, 0.5024634530670342e-01, 0.6838275466788171e-23, 0.2104276631320477e-03, 0.3160258883556432e-03, 0.4118371112977081e-01, 0.6162975822039155e-32, 0.9318133168194381e-04, 0.1261364640367337e-03, 0.5279820157992808e-02, 0.7005084838362585e-27, 0.1903158606362315e-02, 0.7288242235414376e-03, 0.1421284452028254e-03, 0.7703719777548943e-33, 0.7890744155827079e-03, 0.6129022301104231e-03},
			info: 0,
		},
		{
			z:    []float64{0.1813251202119042e+00, 0.8077156210855166e+00, 0.5525396662071069e+00, 0.4858391136124270e+00, 0.5367389425969793e+00, 0.4534294101162717e-01, 0.1438386197304258e+00, 0.7801305810912117e+00, 0.7706244548712898e+00, 0.1595286891781422e+00, 0.4689597797472277e+00, 0.1630160981693277e+00, 0.5488525733069047e+00, 0.3050379405729475e+00, 0.1544160351990825e+00, 0.5750210099163514e+00, 0.1440876969429373e+00, 0.1932969959898051e+00, 0.6910796187912660e+00, 0.9501882645686455e+00, 0.1314109921701710e+00, 0.7099941843070685e+00, 0.3797313713209320e+00, 0.3979808845174081e+00, 0.1006018923100482e-02, 0.4131896358268045e+00, 0.8038132966802857e+00, 0.8778643630407806e+00, 0.1407889867849830e+00, 0.4305134083308382e-01, 0.5656496703997997e+00, 0.5820188952601091e-01, 0.8713537916178680e+00, 0.3443162753417257e+00, 0.4898335360866699e+00, 0.2822165738696382e+00, 0.2385629061714932e+00, 0.5496478420286478e+00, 0.7138724344268773e+00, 0.9028268569137332e+00, 0.7573748459251011e+00, 0.5392259606733280e+00, 0.6459213128573325e+00, 0.7769497211063184e+00, 0.8790154346127051e+00, 0.6027389165195913e+00, 0.6151696559086139e-01, 0.4391208291942791e+00, 0.8705350585502258e-02, 0.2995750933531418e+00, 0.9735013653226658e+00, 0.9551710085419519e+00, 0.2132875677034638e+00, 0.6739668529657764e+00, 0.5361543284664445e+00, 0.3596977928604983e+00, 0.1373006735837845e-02, 0.8494363479416891e+00, 0.4214720423022895e+00, 0.7478449906096684e+00, 0.1164429527473354e-01, 0.6143683285709396e+00, 0.4444360064738268e+00, 0.6626608299302873e+00, 0.6887459689591686e+00, 0.9391262494647765e+00, 0.7167310461528731e+00, 0.4516657019045726e+00, 0.6345059624902227e+00, 0.2241941493026407e+00, 0.3981621310014529e+00, 0.8300976429090324e+00, 0.1390103241004258e+00, 0.7559319384422415e+00, 0.7929892786129927e+00, 0.2279061631835571e+00, 0.5280867615120838e+00, 0.1374499172030500e+00, 0.1739503966093568e+00, 0.1475298919309330e+00, 0.8100098524713260e+00, 0.2514850916075489e+00, 0.4100580488172028e+00, 0.9565966467338131e+00},
			n:    21,
			zOut: []float64{0.2319179815575576e+01, 0.1965907124623829e+01, 0.1887317422176422e+01, 0.1806195743608332e+01, 0.1738584542532610e+01, 0.1518839817892385e+01, 0.1233018482483247e+01, 0.1147944957333420e+01, 0.8847632334488245e+00, 0.7964611162721277e+00, 0.7478364459856081e+00, 0.6404526251941426e+00, 0.4533452491188792e+00, 0.4218888103472739e+00, 0.2727838365998446e+00, 0.2511907405802137e+00, 0.9874721307137806e-01, 0.7639943902083833e-01, 0.4105860224136690e-01, 0.2835039177122555e-01, 0.1849635761425839e-05, 0.5687744349931984e-17, 0.4733652108368680e-16, 0.8474188088642906e-16, 0.1233018482483247e+01, 0.2772075531849652e-16, 0.2588283290162890e-14, 0.3835910828290052e-14, 0.1147944957333420e+01, 0.1246926305809672e-21, 0.2331804915191085e-13, 0.1628473983494628e-13, 0.8847632334488245e+00, 0.1029638495848151e-18, 0.6869501877456999e-13, 0.5177641249986829e-13, 0.7964611162721277e+00, 0.2366582715663035e-28, 0.1931319748910825e-12, 0.1483226506162133e-12, 0.7478364459856081e+00, 0.0000000000000000e+00, 0.1833026745951331e+02, 0.1833026745951330e+02, 0.1060000000000000e+03, 0.3281179138321995e+01, 0.4716981132075472e+01, 0.1032982741787823e-11, 0.4533452491188792e+00, 0.1306947070833229e-13, 0.7648022419924529e-11, 0.9563864191611805e-11, 0.4218888103472739e+00, 0.2958228394578794e-30, 0.1901963755802408e-10, 0.2450915092704452e-10, 0.2727838365998446e+00, 0.2459678163240430e-25, 0.5533379126041796e-09, 0.3623624255644135e-09, 0.2511907405802137e+00, 0.1890110728909544e-26, 0.3358805149198491e-08, 0.5375184613165106e-08, 0.9874721307137806e-01, 0.5174927538249837e-27, 0.5300164302540724e-07, 0.8261446474891677e-07, 0.7639943902083833e-01, 0.5686866711287438e-23, 0.3075058441398368e-06, 0.1980286028193141e-06, 0.4105860224136690e-01, 0.3391723239215845e-21, 0.3233943307567427e-08, 0.7908141105983331e-09, 0.2835039177122555e-01, 0.0000000000000000e+00, 0.6263230510933575e-04, 0.3843833760247782e-04, 0.1849635761425839e-05, 0.1263270231490984e-18, 0.3749684628997549e-02, 0.9345197933294263e-02},
			info: 0,
		},
		{
			z:    []float64{0.7197398584131909e+00, 0.8353840049537183e+00, 0.7768334206137515e+00, 0.5378835466874095e+00, 0.6112887154160760e+00, 0.6941229566444685e+00, 0.6006831303610110e+00, 0.9634869426512738e-01, 0.5491965547787625e+00, 0.2928965321959978e+00, 0.2058564636772363e+00, 0.8430652032339261e+00, 0.6514967955084195e+00, 0.9630832188959448e+00, 0.6099558369768557e+00, 0.4038812997556026e+00, 0.1286146012481256e+00, 0.1377932216741019e+00, 0.2151249629422850e+00, 0.5230096632076910e+00, 0.7097266420339532e+00, 0.5223889086485353e+00, 0.1844557902493219e+00, 0.8630886970819189e+00, 0.3143018526841452e+00, 0.3064747008129198e-02, 0.9995777893655772e+00, 0.8804003743833377e+00, 0.2437734590599224e+00, 0.5439224193002447e+00, 0.7609946860078259e+00, 0.6816222186177212e+00, 0.4213100187092527e+00, 0.5947299218072758e+00, 0.4464510683518146e+00, 0.9444991162538406e+00, 0.1663546170185037e+00, 0.9745722490509305e+00, 0.8691156687582791e-01, 0.1511122923287322e+00, 0.9099013034006386e+00, 0.2055280668016559e+00, 0.2215435147609097e+00, 0.1341458340087506e+00, 0.6974464577249216e+00, 0.9770585050570383e+00, 0.2836484389736937e-02, 0.4942205711890229e+00, 0.2476589071081227e+00, 0.9025674475802441e+00, 0.1043173067377164e+00, 0.7752297952628227e+00, 0.1045812556997563e+00, 0.6207135868564088e+00, 0.3343332589884191e+00, 0.9777410164371825e+00, 0.6919265640807399e+00, 0.8276791372331573e+00, 0.9681277631171376e+00, 0.2774385945770376e+00, 0.3660516291700039e+00, 0.1542243233614283e+00, 0.9809065840498956e+00, 0.9317468521553756e+00, 0.2342755587307467e+00, 0.1233924367977165e+00, 0.3662228319745224e+00, 0.8394462754756703e+00, 0.1257927855780998e+00, 0.9899742295380913e+00, 0.4800112259347978e+00, 0.9614338190585195e+00, 0.9605383026678687e+00, 0.7532278237974727e+00, 0.6455994058946259e+00, 0.6648840697262024e+00, 0.8792931963163917e+00, 0.7296338860815629e+00, 0.6962759488463338e+00, 0.5343477930729980e+00, 0.7488075021642351e+00, 0.8267598422017227e+00, 0.4128864062622362e-02, 0.2248794313377519e+00},
			n:    21,
			zOut: []float64{0.2440396187223568e+01, 0.2429593845031884e+01, 0.2248977224883395e+01, 0.1983141106497864e+01, 0.1775047097539426e+01, 0.1685850120713433e+01, 0.1685238074648001e+01, 0.1228212776246503e+01, 0.1167582065653490e+01, 0.1163468828895948e+01, 0.9840288601088477e+00, 0.8446765744020336e+00, 0.5858419229944851e+00, 0.5828733061329966e+00, 0.4321746091262585e+00, 0.2506332278777725e+00, 0.1601952029039821e+00, 0.9519704710728188e-01, 0.5380488026634178e-01, 0.2070966044404248e-02, 0.4144953892415127e-03, 0.1577721810442024e-28, 0.2622007349295965e-13, 0.1959395597612919e-13, 0.1685238074648001e+01, 0.5298116057209151e-24, 0.4690429322435832e-13, 0.6275308499802102e-13, 0.1228212776246503e+01, 0.3786532345060857e-26, 0.6968636880972768e-12, 0.1483661956038748e-11, 0.1167582065653490e+01, 0.4135903062764930e-23, 0.1852086238235612e-10, 0.7750996975985970e-11, 0.1163468828895948e+01, 0.1171277614412583e-17, 0.1085363071939462e-09, 0.2643024434742409e-09, 0.9840288601088477e+00, 0.6878867093527223e-27, 0.2179941841968715e+02, 0.2179941841968716e+02, 0.1010000000000000e+03, 0.3140589569160998e+01, 0.4950495049504950e+01, 0.1077573597463830e-06, 0.5858419229944851e+00, 0.3627601229823987e-15, 0.8463992810044351e-06, 0.7647582718993818e-06, 0.5828733061329966e+00, 0.5798443197736525e-25, 0.5805124866712180e-06, 0.5336279095847515e-06, 0.4321746091262585e+00, 0.1164890830118439e-21, 0.3270430863981586e-06, 0.3120015969970153e-06, 0.2506332278777725e+00, 0.1344740577327136e-22, 0.6612009870063018e-06, 0.4716796838663449e-06, 0.1601952029039821e+00, 0.0000000000000000e+00, 0.1909143391170965e-04, 0.1029106625513428e-04, 0.9519704710728188e-01, 0.5127595883936577e-28, 0.3722911957200607e-04, 0.5334790194682025e-04, 0.5380488026634178e-01, 0.4916199468524512e-16, 0.8145434425645190e-04, 0.4879722252791062e-03, 0.2070966044404248e-02, 0.2051115390772406e-31, 0.1886522673573371e-02, 0.1146190938097804e-02, 0.4144953892415127e-03, 0.1134788402846185e-20, 0.5837570815253673e-02, 0.1435831510150291e-01},
			info: 0,
		},
		{
			z:    []float64{0.4141124863190200e+00, 0.6286592513493726e+00, 0.4875442526190354e-01, 0.6631999174821102e+00, 0.1778328741285358e+00, 0.5820997881750413e+00, 0.9610329112936564e+00, 0.2860180457593013e+00, 0.2800413372286658e+00, 0.8782990052007664e+00, 0.6769848820880126e+00, 0.2361981145928655e+00, 0.1864129841856804e+00, 0.6207802268756868e-01, 0.1668832170172669e+00, 0.2136536864861034e+00, 0.4266698341219798e+00, 0.9672331991204097e-01, 0.2350195655513042e+00, 0.1297162551474343e+00, 0.9428942697229301e+00, 0.3355426683568981e+00, 0.6977568272344947e-01, 0.4926170514192555e+00, 0.6966571762613959e+00, 0.7422949877167778e+00, 0.1216493993328296e+00, 0.5414775352707472e+00, 0.4507250901570525e+00, 0.6851601412475350e+00, 0.8339584749808495e+00, 0.9321374665508209e+00, 0.7664819871766777e+00, 0.5935455015911359e+00, 0.4047285985374618e+00, 0.9996987445838635e+00, 0.1347158425141065e+00, 0.3601395592420920e+00, 0.1128763847009464e+00, 0.6316809196633902e+00, 0.1559195263655234e+00, 0.4041853050912981e+00, 0.1854167847278504e+00, 0.6852913644854558e+00, 0.6126092157077416e+00, 0.2444516268314137e+00, 0.1447238269517732e+00, 0.7179956614587130e+00, 0.8198652148251639e+00, 0.1186997025611126e+00, 0.1267596520196094e+00, 0.2863431547875360e+00, 0.9867539057590858e+00, 0.2454746550565152e+00, 0.4891318294681372e+00, 0.2969454310391295e-01, 0.2349636207332649e+00, 0.4537731552146703e+00, 0.2883801124165087e+00, 0.7150664117598651e+00, 0.4359050919721921e+00, 0.6116081525473241e+00, 0.5619840590438748e+00, 0.3431726287099235e+00, 0.7715898362788249e+00, 0.4981359837313185e+00, 0.4537925078699018e+00, 0.6956466175473529e+00, 0.1814680828887681e+00, 0.4413903687143481e+00, 0.4367593306147978e+00, 0.5875066776157845e-01, 0.3688522862246468e+00, 0.4308468783440871e+00, 0.1986553761412286e+00, 0.3949957841484223e+00, 0.2468986938324821e+00, 0.6216444107315144e-01, 0.9076510210089642e+00, 0.5197394332656491e+00, 0.8460559187475114e-01, 0.3959477165848089e+00, 0.1014864219123556e+00, 0.4789226780612066e-01},
			n:    21,
			zOut: []float64{0.2746085040517442e+01, 0.2006646947029628e+01, 0.1843368381126182e+01, 0.1738309804669461e+01, 0.1678478075610122e+01, 0.1408432532104903e+01, 0.1219097982435216e+01, 0.1121569511965291e+01, 0.9315294884679637e+00, 0.7914377647612258e+00, 0.7334412101899261e+00, 0.6113376828240070e+00, 0.4134386595159317e+00, 0.3303980566325673e+00, 0.2866331058299493e+00, 0.2287206720574312e+00, 0.1408981800627584e+00, 0.6649497607659796e-01, 0.5733628497886722e-01, 0.1387598234001070e-02, 0.2497701489633820e-04, 0.3487740233155725e-22, 0.3361270559917294e-05, 0.1214091727263821e-05, 0.1219097982435216e+01, 0.2843433355651033e-23, 0.3877657623657789e-06, 0.2861458561047538e-06, 0.1121569511965291e+01, 0.1501849799666580e-22, 0.7980776610337536e-07, 0.1296132789777516e-06, 0.9315294884679637e+00, 0.1397279453391396e-14, 0.3797866386428551e-06, 0.5462183171930727e-06, 0.7914377647612258e+00, 0.1231982124821006e-21, 0.6463973892160826e-06, 0.4555968092429713e-06, 0.7334412101899261e+00, 0.9423416829408119e-25, 0.1835506693210437e+02, 0.1835506693210436e+02, 0.9800000000000000e+02, 0.2884353741496599e+01, 0.2040816326530612e+01, 0.6355562713716383e-07, 0.4134386595159317e+00, 0.4284860196510039e-21, 0.4931868922851139e-05, 0.2760082849901006e-05, 0.3303980566325673e+00, 0.1448288531514775e-16, 0.4631299886341806e-05, 0.9149811670123540e-05, 0.2866331058299493e+00, 0.1558000287811498e-28, 0.1337090019087612e-05, 0.8008074996380075e-06, 0.2287206720574312e+00, 0.2366582715663035e-29, 0.4303808308110317e-06, 0.3181248709867330e-06, 0.1408981800627584e+00, 0.2443697032165075e-19, 0.2973006759321362e-05, 0.5017304713868616e-05, 0.6649497607659796e-01, 0.1131941655598205e-16, 0.8328041136745409e-04, 0.8552508571764750e-04, 0.5733628497886722e-01, 0.9398173280441046e-25, 0.5097763200650848e-04, 0.5644224660768206e-04, 0.1387598234001070e-02, 0.8888394955180400e-22, 0.1868721561136477e-03, 0.1091351925463373e-03, 0.2497701489633820e-04, 0.0000000000000000e+00, 0.1020791029288490e-01, 0.2791888375404797e-02},
			info: 0,
		},
		{
			z:    []float64{0.2396228092919251e+00, 0.6434601049058037e+00, 0.5488667726527494e+00, 0.1843504645657519e+00, 0.4269001369847240e+00, 0.6276093013166625e+00, 0.8724336393628729e-01, 0.5371039952903035e+00, 0.8447460457667564e+00, 0.4094766929502263e+00, 0.2031514124499976e+00, 0.8271428312997948e+00, 0.9658284302973903e+00, 0.3777353969048113e+00, 0.6471334022587341e+00, 0.6777435175679861e+00, 0.4456890557292859e+00, 0.1720826572683752e+00, 0.3520207432901141e+00, 0.3291396508341311e+00, 0.7468262262373744e+00, 0.2715104496960554e+00, 0.5612917863396235e+00, 0.1324429469286776e+00, 0.7983766064014711e+00, 0.2059599586019641e+00, 0.2655510256425775e-01, 0.1962390264484870e+00, 0.7845246896371560e-02, 0.8217737559711210e+00, 0.9444498218704492e+00, 0.4679957695831033e+00, 0.1135918053039552e+00, 0.1604948252571132e+00, 0.8452381101613837e+00, 0.8537070814716049e+00, 0.3013932596816266e+00, 0.2511715406702125e+00, 0.2008647638036438e+00, 0.2607897573444329e+00, 0.2571050910078490e+00, 0.6286174024463315e+00, 0.4739701849723611e+00, 0.8698947534844018e+00, 0.6486079589595961e+00, 0.8894321904301987e+00, 0.7952475840185558e+00, 0.8801661721215410e+00, 0.2147272194381030e+00, 0.9518356646476019e+00, 0.7191152289414877e+00, 0.1106524971173308e+00, 0.9574587230734148e+00, 0.6469474230679395e+00, 0.5818260757443584e+00, 0.1292202837075540e-01, 0.2761722913438078e+00, 0.8004535925585585e+00, 0.3278973855590038e+00, 0.5794037438664289e+00, 0.2396155443104175e-01, 0.6699048475753794e+00, 0.1814492937128086e+00, 0.2866184936222242e+00, 0.6584557212506269e+00, 0.5889651227775431e+00, 0.6369766519060738e-01, 0.1328193529598596e+00, 0.7416631417262871e+00, 0.5003318278706693e+00, 0.3144077296395376e-01, 0.9907393156403193e+00, 0.3423696594914594e-02, 0.6992922072850826e+00, 0.7590519617064543e+00, 0.2542182514116359e+00, 0.3314657184555089e+00, 0.5728659684019645e+00, 0.9915854796414423e+00, 0.4896749730105238e+00, 0.5611369720796141e+00, 0.6064142215043516e+00, 0.8567590164766914e-01, 0.2581731932678185e+00},
			n:    21,
			zOut: []float64{0.2319327880258489e+01, 0.2079713994229665e+01, 0.1953873370330741e+01, 0.1755147049127663e+01, 0.1569664109844580e+01, 0.1497177879850755e+01, 0.1442939549198070e+01, 0.1142654640257259e+01, 0.1037259380018118e+01, 0.7336163063097807e+00, 0.6300572204242147e+00, 0.3966542874116493e+00, 0.3925821156107781e+00, 0.2721798695257625e+00, 0.2365431978248397e+00, 0.2044065204974935e+00, 0.1570073942488483e+00, 0.8918832674569614e-01, 0.5938710196407761e-01, 0.3241742658191688e-02, 0.4778146591607052e-04, 0.1325286320771300e-27, 0.1485620696270319e-14, 0.4653485276256090e-14, 0.1442939549198070e+01, 0.1846131417899111e-22, 0.1162592678154001e-11, 0.3263011919854044e-11, 0.1142654640257259e+01, 0.4412642697382792e-11, 0.1735169830505648e-08, 0.3333560963091213e-08, 0.1037259380018118e+01, 0.1055565179678919e-18, 0.1940309609380032e-07, 0.3465529731850831e-07, 0.7336163063097807e+00, 0.4440642298320086e-11, 0.3810071749579748e-06, 0.5916980508399715e-06, 0.6300572204242147e+00, 0.4658681208910640e-20, 0.1797266971780259e+02, 0.1797266971780259e+02, 0.1110000000000000e+03, 0.3253968253968254e+01, 0.3603603603603604e+01, 0.2492194730770069e-13, 0.3925821156107781e+00, 0.0000000000000000e+00, 0.7667233942003248e-12, 0.2416678934089916e-12, 0.2721798695257625e+00, 0.2051915459060795e-16, 0.1487877135020801e-08, 0.6143545671036994e-08, 0.2365431978248397e+00, 0.1972152263052530e-29, 0.1960096418178738e-05, 0.2274829676605902e-05, 0.2044065204974935e+00, 0.2021903259098794e-21, 0.8745843979356665e-06, 0.9251106153042376e-06, 0.1570073942488483e+00, 0.5482898835648120e-25, 0.7118492322907369e-04, 0.3316595178521346e-03, 0.8918832674569614e-01, 0.1022725463125710e-19, 0.2167108979032774e-02, 0.1228336641518804e-02, 0.5938710196407761e-01, 0.1994240368398718e-26, 0.2711464061500906e-02, 0.2561834534417173e-02, 0.3241742658191688e-02, 0.1814380082008327e-27, 0.4237092909216320e-02, 0.4420407214578293e-02, 0.4778146591607052e-04, 0.2596519942803954e-20, 0.1820843269063370e-02, 0.8929397931439835e-02},
			info: 0,
		},
		{
			z:    []float64{0.4704013808869884e+00, 0.8133248937317098e+00, 0.7246209022332254e+00, 0.9930301346620946e+00, 0.6565279241713462e-01, 0.7480498761899035e-01, 0.8311943018468732e+00, 0.4127487814312358e+00, 0.6620288516904083e+00, 0.2540390589975015e+00, 0.4830377447593585e+00, 0.6124023244542695e+00, 0.6830062885714996e+00, 0.4006497306883100e+00, 0.2108017907398356e+00, 0.9099126499549155e+00, 0.4756780627224959e+00, 0.8563186613679700e+00, 0.9168447467108382e+00, 0.5153963449488133e+00, 0.7188966629304918e+00, 0.2875525393671987e+00, 0.6200986280097742e-01, 0.3765186242141257e+00, 0.2589923271177115e+00, 0.5404093081826918e+00, 0.5343437371907505e+00, 0.3104227497255196e-01, 0.4221838478889377e+00, 0.4233113354732956e+00, 0.6486058235731677e+00, 0.1590032665029245e+00, 0.3877113621801469e+00, 0.6038849620024899e+00, 0.5400341407425849e+00, 0.7060013907006117e+00, 0.1173345165878200e+00, 0.8548311704210352e+00, 0.5804532700087333e+00, 0.5231399784962363e+00, 0.4754841524148748e+00, 0.3395282932287647e-01, 0.7668594432348432e+00, 0.8184682560752043e+00, 0.5633264054575460e-01, 0.2127176392631261e+00, 0.7432372685121402e+00, 0.3098453278874418e+00, 0.1150335122169536e+00, 0.8681635561145716e+00, 0.3593935949259885e+00, 0.9085822737263088e+00, 0.2122810259380176e+00, 0.3589456483065394e+00, 0.9437026007300053e-01, 0.6518875210051764e+00, 0.9046367728621729e+00, 0.7638617707854546e+00, 0.8924521343092110e+00, 0.8353112965912926e+00, 0.5076189975461921e+00, 0.4955137299922637e+00, 0.5041148053921253e+00, 0.7686114107944846e+00, 0.3833696141187980e+00, 0.9181102072361824e-01, 0.9260679284685853e+00, 0.7081687340916837e+00, 0.8312629527805896e+00, 0.8010881636433624e+00, 0.5891143654107377e+00, 0.8149927954305255e+00, 0.7192672792238343e+00, 0.9981622901025261e+00, 0.9368513325443091e+00, 0.6909941993236977e+00, 0.7208744872845105e+00, 0.6334330703247726e+00, 0.8856108044732016e+00, 0.7612967011342276e+00, 0.6840498257884192e+00, 0.8627400929877541e+00, 0.2970500386847904e+00, 0.9828991929119790e+00},
			n:    21,
			zOut: []float64{0.2548749189884266e+01, 0.2318283655447516e+01, 0.1949991272828373e+01, 0.1903113092077109e+01, 0.1681173723989745e+01, 0.1621144658162279e+01, 0.1435953736019784e+01, 0.1373215273744876e+01, 0.1276544145324249e+01, 0.8508650942792525e+00, 0.7433578075251207e+00, 0.7333566617981635e+00, 0.6327824956484905e+00, 0.5146506379994326e+00, 0.4497218333883862e+00, 0.2736876649003249e+00, 0.2262999758546705e+00, 0.6320618886076086e-01, 0.1413547856395308e-01, 0.6182735080657475e-02, 0.1223662826415605e-02, 0.9844984097158227e-26, 0.2664250906052687e-13, 0.8143635875649387e-14, 0.1435953736019784e+01, 0.7237830359838992e-24, 0.1147364015889613e-09, 0.4819137169184198e-10, 0.1373215273744876e+01, 0.5490471900338242e-27, 0.1654936529088367e-07, 0.7460880307437913e-08, 0.1276544145324249e+01, 0.2958228394578794e-30, 0.1455672577071588e-05, 0.7169454013791148e-06, 0.8508650942792525e+00, 0.4041314959719700e-19, 0.1917383253612269e-05, 0.1657209462004936e-05, 0.7433578075251207e+00, 0.2357464745776125e-22, 0.2061763898420382e+02, 0.2061763898420383e+02, 0.9600000000000000e+02, 0.2888888888888889e+01, 0.3125000000000000e+01, 0.6718966036888534e-06, 0.6327824956484905e+00, 0.1006441314168352e-20, 0.4252254757420258e-06, 0.3750418335727261e-06, 0.5146506379994326e+00, 0.1125208173090593e-20, 0.1634173128014842e-06, 0.1777130114194832e-06, 0.4497218333883862e+00, 0.1397861524051633e-26, 0.1373153840978587e-06, 0.1424351942740530e-06, 0.2736876649003249e+00, 0.3055476705078142e-15, 0.2530372171390620e-06, 0.3073674645773779e-06, 0.2262999758546705e+00, 0.3987473575125276e-15, 0.5814400581048744e-06, 0.7370753629310487e-06, 0.6320618886076086e-01, 0.9597492304117215e-21, 0.2453687283764395e-05, 0.3193610655869065e-05, 0.1413547856395308e-01, 0.8528167491936360e-20, 0.7231103044220720e-05, 0.9769896858142881e-05, 0.6182735080657475e-02, 0.0000000000000000e+00, 0.3039337033448658e-04, 0.5367078032707280e-04, 0.1223662826415605e-02, 0.6617444900424221e-23, 0.7353916438463039e-03, 0.2144171525794732e-02},
			info: 0,
		},
		{
			z:    []float64{0.9714680453643365e-01, 0.6357804703219754e+00, 0.6664990519488578e-01, 0.1729636879767130e+00, 0.4638342338705743e+00, 0.4390808892462035e+00, 0.8041041328438946e+00, 0.4288731182982508e+00, 0.7624625505229141e+00, 0.7083386603826051e+00, 0.2464792096944004e+00, 0.9011775040224685e+00, 0.6339539510596026e-01, 0.1795036824346197e-01, 0.9473751738931280e+00, 0.4886196718733559e+00, 0.3845982510172906e+00, 0.8175566437299375e-01, 0.4796454941504319e+00, 0.1995303497008669e+00, 0.3542217290845850e+00, 0.4313075601907782e+00, 0.9062634832132850e+00, 0.9362490171108290e+00, 0.7454458344912314e+00, 0.9678028365928690e+00, 0.8785588729275070e+00, 0.2582842558253936e+00, 0.7937433110799214e+00, 0.7912406502001545e-01, 0.1813501620770241e-01, 0.3340784700607168e+00, 0.3348119644971841e+00, 0.5042866134297835e+00, 0.3808798817625770e+00, 0.8932448815066745e+00, 0.2612469692343123e+00, 0.4943911382967600e+00, 0.7396437014435053e+00, 0.6180267747680381e+00, 0.6768529193184074e+00, 0.7346572400466492e+00, 0.8658232798365260e+00, 0.6590888814314796e+00, 0.9020629495832085e+00, 0.2719526227756762e+00, 0.9337960905099231e+00, 0.1402012445044727e+00, 0.3363383773836534e+00, 0.4741712787560632e+00, 0.1270352327896518e-02, 0.2317652069389444e+00, 0.9583986396877475e+00, 0.8857657675309066e+00, 0.5564654172742324e+00, 0.3373121694331064e+00, 0.1866413892025005e+00, 0.3829716065530064e+00, 0.6660176769447556e+00, 0.9140619394089379e+00, 0.4126532961354976e+00, 0.9883322088463480e+00, 0.1678362296131076e+00, 0.1246866166129172e+00, 0.5083959276848560e+00, 0.4743484764552197e+00, 0.1561141239638767e+00, 0.4867628897662705e+00, 0.8068909470080686e+00, 0.9878597697456749e+00, 0.8535449450676436e-01, 0.9609706407601909e+00, 0.7015171470525342e+00, 0.4808513401395343e+00, 0.4421870922261545e+00, 0.2476919441028158e-01, 0.2417065613871903e+00, 0.1174043678151601e+00, 0.4259626380136439e-02, 0.2614829171642307e+00, 0.5973781466265196e+00, 0.3195524674053074e-01, 0.4778949570977242e+00, 0.2741401812748340e+00},
			n:    21,
			zOut: []float64{0.2956562879292357e+01, 0.2160920112738185e+01, 0.2057048821491083e+01, 0.1826665062315388e+01, 0.1626476426276149e+01, 0.1617659828407750e+01, 0.1458559932420392e+01, 0.1037845007643388e+01, 0.9436387491729998e+00, 0.8889042931978449e+00, 0.8147110102488896e+00, 0.7938854029224469e+00, 0.5240642869032099e+00, 0.4281570616174346e+00, 0.4032231719469492e+00, 0.2134769254146812e+00, 0.1570954054223686e+00, 0.5771888278451791e-01, 0.2771650645038780e-01, 0.1168397530048623e-02, 0.8626671344136174e-03, 0.6837796509826355e-18, 0.5416843762303492e-12, 0.3045298824248355e-12, 0.1458559932420392e+01, 0.8473754517266856e-24, 0.4783112378014045e-11, 0.8163747631331082e-11, 0.1037845007643388e+01, 0.8077935669463161e-27, 0.2442833758035109e-10, 0.3576901446935589e-10, 0.9436387491729998e+00, 0.1292469707114106e-25, 0.1060154010453575e-09, 0.7148292157772074e-10, 0.8889042931978449e+00, 0.5995342879679690e-28, 0.2265830666026058e-09, 0.3413067902462323e-09, 0.8147110102488896e+00, 0.6874131484520357e-19, 0.1999636083133089e+02, 0.1999636083133089e+02, 0.1000000000000000e+03, 0.3002267573696145e+01, 0.2000000000000000e+01, 0.2441378744031031e-06, 0.5240642869032099e+00, 0.2169367489357782e-29, 0.9564347597274963e-06, 0.8629398155870994e-06, 0.4281570616174346e+00, 0.1533177238774536e-16, 0.2198723423439569e-05, 0.1852641299824983e-05, 0.4032231719469492e+00, 0.1742814620686677e-24, 0.6898774546312495e-05, 0.1242988026413374e-04, 0.2134769254146812e+00, 0.1303198215425112e-25, 0.3054946518379762e-04, 0.2458010895625246e-04, 0.1570954054223686e+00, 0.2480178686014861e-25, 0.5573684427056079e-04, 0.2879079609909993e-04, 0.5771888278451791e-01, 0.0000000000000000e+00, 0.4895501114079361e-04, 0.6615663653270374e-04, 0.2771650645038780e-01, 0.6559433690090112e-18, 0.2090848897194263e-03, 0.3843856621792604e-03, 0.1168397530048623e-02, 0.0000000000000000e+00, 0.2396113765895983e-02, 0.4353699886144585e-02, 0.8626671344136174e-03, 0.1420984816841217e-20, 0.3572874909786609e-04, 0.2591037812233495e-05},
			info: 0,
		},
		{
			z:    []float64{0.7761482897976135e-01, 0.7690133527529974e-01, 0.8589697242168557e+00, 0.5718901484414876e+00, 0.8450349347421171e-01, 0.5688250545090190e+00, 0.6202247495663940e+00, 0.7388100008042665e+00, 0.9890328265375476e+00, 0.2969659976080158e+00, 0.5442392738534707e-01, 0.9562602329262000e+00, 0.8530135563879812e+00, 0.5471012261519459e-01, 0.9710076793297340e+00, 0.4869447961696205e+00, 0.4659851368281548e+00, 0.7647605444900396e-01, 0.7872766307526106e+00, 0.3237029299700311e+00, 0.7194522935712468e+00, 0.7270865543961295e+00, 0.5183429612108851e+00, 0.3145688411100903e+00, 0.9096514649777033e+00, 0.7816253592835525e+00, 0.4809049584349555e+00, 0.2052154618933422e-01, 0.1481963756328628e+00, 0.8742358520807045e+00, 0.4565878430880338e+00, 0.1977913919695908e+00, 0.6526813957349178e+00, 0.8032154703264597e+00, 0.2356108621359847e+00, 0.4857415898450358e+00, 0.2259727396620309e+00, 0.1176413330041139e+00, 0.8335507369731385e+00, 0.4539477133958770e+00, 0.8910876835374724e+00, 0.1559757847548392e+00, 0.1885199219441583e+00, 0.1520953639100613e+00, 0.6035948595957944e+00, 0.9286538294716984e+00, 0.9676298024186124e+00, 0.1209629819848071e+00, 0.1660874781174386e+00, 0.8622065054293394e+00, 0.7485077140876945e+00, 0.9420716698171637e+00, 0.2904751962009996e+00, 0.2180437045655230e+00, 0.6692025851955981e+00, 0.9258369042460884e+00, 0.6243742234586596e+00, 0.6760203577391158e+00, 0.2199572443863818e-02, 0.5981650670220221e+00, 0.8252506688545779e+00, 0.9433813762695346e+00, 0.5320093960532474e+00, 0.3846966829839934e+00, 0.6727780061271659e+00, 0.4937802596181958e+00, 0.1177802539787526e+00, 0.2963448754612064e-01, 0.2479547713122239e+00, 0.2694003173993550e+00, 0.8186090816580105e+00, 0.2886041049393229e-01, 0.7384561600560133e+00, 0.8345282295575730e+00, 0.1740871089016751e+00, 0.7946006936544379e+00, 0.5640836203667244e+00, 0.4951346622832815e+00, 0.4981407974176422e+00, 0.2513094341381288e+00, 0.4316077257561814e+00, 0.1685190107506288e+00, 0.3264570829830372e+00, 0.1676643753771256e-01},
			n:    21,
			zOut: []float64{0.2309990065237698e+01, 0.2261037149894265e+01, 0.1946075797700671e+01, 0.1868880050926239e+01, 0.1838164096767709e+01, 0.1792848619787804e+01, 0.1601503570675395e+01, 0.1526383850720226e+01, 0.1395433670210245e+01, 0.1010109387319946e+01, 0.8644978203099237e+00, 0.7632668975571170e+00, 0.4552388959212355e+00, 0.4364144823342648e+00, 0.2696980874891977e+00, 0.1302983706884531e+00, 0.1278582769210706e+00, 0.7956979307866878e-01, 0.7389266586143965e-01, 0.8102308371415692e-02, 0.2690335013870282e-02, 0.5752609692135112e-13, 0.2842400234743069e-06, 0.1045980487121886e-05, 0.1601503570675395e+01, 0.1668093198756260e-18, 0.5190756964721459e-05, 0.2367180586416983e-05, 0.1526383850720226e+01, 0.1262177448353619e-27, 0.1974011051185169e-03, 0.4728862593194215e-03, 0.1395433670210245e+01, 0.2337388813815525e-19, 0.1632969150852556e-06, 0.4026698297028720e-06, 0.1010109387319946e+01, 0.3802651360818291e-21, 0.1380334146666910e-03, 0.1568641113059460e-03, 0.8644978203099237e+00, 0.2788260408793746e-19, 0.2076195419278686e+02, 0.2076195419278686e+02, 0.1070000000000000e+03, 0.3183673469387755e+01, 0.2803738317757009e+01, 0.1273578837393323e-04, 0.4552388959212355e+00, 0.1016579123433529e-20, 0.9426551662020599e-04, 0.5179870206653002e-04, 0.4364144823342648e+00, 0.7336406418555410e-28, 0.1112142557098828e-05, 0.1460057881679474e-05, 0.2696980874891977e+00, 0.8909623382378835e-18, 0.3232451044433717e-03, 0.3229988347563871e-03, 0.1302983706884531e+00, 0.0000000000000000e+00, 0.2867336104451064e-03, 0.2890982620847502e-03, 0.1278582769210706e+00, 0.6875938841518755e-23, 0.2127348925826359e-06, 0.9655734242129450e-07, 0.7956979307866878e-01, 0.3291208348307372e-17, 0.5717743824937758e-07, 0.2183606505996598e-06, 0.7389266586143965e-01, 0.7801771243763389e-24, 0.1012732311499954e-04, 0.2863554462185051e-05, 0.8102308371415692e-02, 0.5719241562852336e-29, 0.6292939995324734e-04, 0.3382281194073554e-04, 0.2690335013870282e-02, 0.4951522129891247e-25, 0.3008856418189283e-03, 0.2691027365699742e-03},
			info: 0,
		},
		{
			z:    []float64{0.1834451937168287e+00, 0.4501020603239488e+00, 0.9673873583575473e+00, 0.2207688184907047e+00, 0.4205773270263516e+00, 0.6767264871143511e+00, 0.2493506785870784e+00, 0.7809638015955807e+00, 0.8297195492174936e+00, 0.1617611869044215e+00, 0.9206032300762146e+00, 0.1914715918877316e+00, 0.7976268296797403e+00, 0.9839522457816396e+00, 0.8841932579084628e+00, 0.6979540605797908e+00, 0.8049859576106227e+00, 0.8117504351099887e+00, 0.3842107296635322e+00, 0.6368881208030270e+00, 0.2369530469547876e+00, 0.8005533377467571e+00, 0.7105152962140180e+00, 0.5356376472112734e-02, 0.2192323464228777e+00, 0.8992715082939398e+00, 0.5988055508193698e+00, 0.4081645962203215e+00, 0.1002944447483383e+00, 0.1052431248059748e+00, 0.5730054733733171e+00, 0.2386396328544268e+00, 0.6503058109411604e+00, 0.9019164849722401e+00, 0.4702244241412763e+00, 0.5759947448793571e+00, 0.7588959717342861e+00, 0.4126392985506339e+00, 0.9987799941641187e+00, 0.5598172962702971e+00, 0.2151524348024290e+00, 0.7511708041336773e-01, 0.1539602741757972e+00, 0.4584191954304165e+00, 0.8937659116858414e+00, 0.6186897030318780e+00, 0.6386630452844105e+00, 0.9151803046984296e+00, 0.3614526997644965e+00, 0.9143967867261915e+00, 0.1523575506386343e+00, 0.9745493424336691e+00, 0.4149640117372425e+00, 0.7418302113477876e+00, 0.3041602774831942e-01, 0.4852660298208696e+00, 0.9115870107661288e+00, 0.9024516441581734e+00, 0.9201550825657324e+00, 0.5200815862126855e+00, 0.7353784634707106e+00, 0.8264932398040087e+00, 0.7035838297463828e+00, 0.6738634069153341e+00, 0.7657166064092835e-01, 0.4678396966074577e+00, 0.9927685693816569e+00, 0.5845157970225999e+00, 0.5638346985777254e+00, 0.5850492500516753e+00, 0.3856054034589584e+00, 0.8134648529141429e+00, 0.4174766661784889e+00, 0.3336151222017159e+00, 0.5355530597844248e+00, 0.9274106499154580e+00, 0.7442750408730826e+00, 0.2948331405701974e+00, 0.5696937454611060e+00, 0.3104181547964658e+00, 0.4694844160987418e+00, 0.1538578615150018e+00, 0.5969883537795497e+00, 0.3873507919637014e+00},
			n:    21,
			zOut: []float64{0.2914952592069403e+01, 0.2353566179964282e+01, 0.2062119351173516e+01, 0.1965819938313762e+01, 0.1942714243081098e+01, 0.1812271492844174e+01, 0.1712144141310849e+01, 0.1666892005264074e+01, 0.1208030687686805e+01, 0.1159871012571561e+01, 0.9581193432281622e+00, 0.7738358894690665e+00, 0.7454032361827709e+00, 0.3901865930581417e+00, 0.3240765847275939e+00, 0.2173673669709887e+00, 0.1778074780703053e+00, 0.5660863840842725e-01, 0.3038731391621440e-01, 0.1466323002394464e-01, 0.7362797481955845e-02, 0.1496427221292385e-15, 0.2798385737764086e-17, 0.6348906733176861e-18, 0.1712144141310849e+01, 0.2625127144183791e-23, 0.5842516095868784e-15, 0.1699577922475928e-15, 0.1666892005264074e+01, 0.2934562567422164e-27, 0.7233314809955462e-13, 0.2286859678494404e-12, 0.1208030687686805e+01, 0.8467561517631538e-12, 0.2143114506868008e-08, 0.4305471816838367e-08, 0.1159871012571561e+01, 0.4930380657631324e-31, 0.3270528389239616e-07, 0.6410715811146478e-07, 0.9581193432281622e+00, 0.8911174733768286e-23, 0.2249420011581710e+02, 0.2249420011581710e+02, 0.1050000000000000e+03, 0.3197278911564626e+01, 0.2857142857142857e+01, 0.6896683202589802e-07, 0.7454032361827709e+00, 0.3255610023185390e-23, 0.5315428740895598e-04, 0.2875762635390457e-03, 0.3901865930581417e+00, 0.4610239445276015e-22, 0.9146669702187219e-04, 0.4183987855699404e-04, 0.3240765847275939e+00, 0.4688433862556419e-23, 0.8204008598914952e-05, 0.1523341313649621e-04, 0.2173673669709887e+00, 0.4570683409889788e-20, 0.9316307389873694e-05, 0.2741587710667867e-05, 0.1778074780703053e+00, 0.3706159570762342e-17, 0.9605798472890332e-04, 0.3099752510959849e-03, 0.5660863840842725e-01, 0.5638682600030866e-16, 0.1627758109301002e-02, 0.5977814903797443e-03, 0.3038731391621440e-01, 0.7161878634670464e-18, 0.4935537640644100e-05, 0.4281293465082162e-05, 0.1466323002394464e-01, 0.9343665470806783e-19, 0.1384059308707374e-05, 0.7139695062761456e-05, 0.7362797481955845e-02, 0.1252221392641006e-22, 0.1398979151771859e-02, 0.1830558814226920e-02},
			info: 0,
		},
		{
			z:    []float64{0.1901253435444810e+00, 0.1005117877448146e+00, 0.7431497892893435e-02, 0.4180244733822649e+00, 0.5723993917742813e+00, 0.5092276894610764e+00, 0.6914570054828403e+00, 0.6343440208263138e+00, 0.9714240100252226e+00, 0.9708780190932921e+00, 0.7346948492781401e+00, 0.9261658900466149e+00, 0.6500321763704298e-01, 0.2101988980308164e+00, 0.2569765082490436e+00, 0.6872855071573251e+00, 0.7393915007490531e+00, 0.5293076476804583e+00, 0.5736721160165409e+00, 0.6249080544922647e+00, 0.1805969142240157e+00, 0.6235867578362418e+00, 0.3156395994838945e+00, 0.6768110107418109e+00, 0.5036038244498190e+00, 0.2096885591435443e+00, 0.2471056170621776e+00, 0.4420201221816504e+00, 0.5954916898507195e+00, 0.6849385183400085e+00, 0.6865376402638398e+00, 0.5015659601935997e+00, 0.9646916309223035e+00, 0.1976274617660123e+00, 0.6173831761654865e+00, 0.9161785290588952e+00, 0.8303766381148641e+00, 0.4163407208236638e+00, 0.2406183750735502e-01, 0.9033217043220854e+00, 0.3435327703530300e-01, 0.4851293431546745e+00, 0.1308952665690980e+00, 0.1639949900839996e+00, 0.6530528840204755e+00, 0.4007776795656349e+00, 0.6520732821596675e-01, 0.2482448577499247e+00, 0.6417548063561368e+00, 0.6287927677318972e+00, 0.2491821746122613e+00, 0.1309766085785375e+00, 0.1799023200797401e+00, 0.6166520678731939e+00, 0.4068347418426869e+00, 0.8022499378251485e+00, 0.2090643045880468e+00, 0.9552039287477454e-01, 0.3405285833145572e+00, 0.9010017729234689e+00, 0.8087026571264999e-01, 0.6784547103017031e+00, 0.5001579210546183e+00, 0.9044563563261202e+00, 0.6248337185292430e-01, 0.7231216956646513e+00, 0.3330924448715603e+00, 0.5906941805835393e+00, 0.6144619726434308e-01, 0.4494963607638514e-01, 0.5817524116985039e+00, 0.4323943825883398e+00, 0.9871518362913079e-01, 0.9577530955877981e+00, 0.7720928198042669e+00, 0.1473259710759589e+00, 0.7227644962870149e+00, 0.9964485210410867e+00, 0.3235582385765219e+00, 0.5984954533580267e-01, 0.1531719633524309e-01, 0.8969070444410814e+00, 0.8320043732823273e+00, 0.7541623800831910e+00},
			n:    21,
			zOut: []float64{0.2943190889471759e+01, 0.2404733889291933e+01, 0.2164120218254414e+01, 0.2101040443717650e+01, 0.1828309865967689e+01, 0.1647179310188885e+01, 0.1393092850251358e+01, 0.1217421086377993e+01, 0.1157068329170861e+01, 0.9693609911235493e+00, 0.7556225164779018e+00, 0.6051010122910800e+00, 0.5244111548905432e+00, 0.4204021043595824e+00, 0.3294230503008973e+00, 0.2905705444026201e+00, 0.1318042053167392e+00, 0.6748792703931331e-01, 0.3115549452672914e-01, 0.3850968595106850e-02, 0.1766035472385010e-05, 0.2996366283941816e-12, 0.8018633237482923e-17, 0.2249911398990393e-16, 0.1393092850251358e+01, 0.8588781736260544e-19, 0.1768526861021460e-14, 0.8068659675231548e-15, 0.1217421086377993e+01, 0.2465190328815662e-30, 0.6841595655394777e-14, 0.1344694857995563e-13, 0.1157068329170861e+01, 0.2714422906896024e-19, 0.9771202940807984e-13, 0.1895423136689650e-12, 0.9693609911235493e+00, 0.1093011646009194e-15, 0.5148224565067754e-11, 0.9904113235317579e-11, 0.7556225164779018e+00, 0.6752649348691861e-27, 0.2098534861805207e+02, 0.2098534861805208e+02, 0.9600000000000000e+02, 0.2832199546485261e+01, 0.2083333333333333e+01, 0.3437245789676428e-09, 0.5244111548905432e+00, 0.2665011951551765e-23, 0.6566411319039389e-08, 0.3635282930812000e-08, 0.4204021043595824e+00, 0.1249353725478346e-22, 0.6690013285699780e-07, 0.1185995569029548e-06, 0.3294230503008973e+00, 0.3250106929510569e-26, 0.6264577969246473e-06, 0.1086745069147475e-05, 0.2905705444026201e+00, 0.1121066009627684e-24, 0.5553814969713904e-05, 0.9554866377781883e-05, 0.1318042053167392e+00, 0.1875595688253478e-25, 0.3215548179944659e-03, 0.4036498498040293e-03, 0.6748792703931331e-01, 0.4043956767801041e-20, 0.1191830515693705e-02, 0.1851908589351717e-02, 0.3115549452672914e-01, 0.1124126789939942e-28, 0.7898349055483677e-03, 0.3617649115749287e-02, 0.3850968595106850e-02, 0.4455486392688275e-26, 0.3971779475237804e-01, 0.1814591379891354e-01, 0.1766035472385010e-05, 0.2780981219403279e-20, 0.1014029004767683e-01, 0.8718057102094371e-02},
			info: 0,
		},
		{
			z:    []float64{0.3437994630115073e+00, 0.2969584004763312e+00, 0.5428789456218400e+00, 0.3643219910857877e+00, 0.8842011736515154e-01, 0.4426793535323088e+00, 0.9851327949196382e+00, 0.9928836193844874e+00, 0.9294888334528270e+00, 0.5131906939013953e+00, 0.8372743749579292e-01, 0.3799113781856175e+00, 0.9980979619320958e+00, 0.1940068951405882e+00, 0.1265916571930514e+00, 0.7533120158726621e+00, 0.7604310151358937e+00, 0.9521608240404056e-01, 0.6048411980078113e+00, 0.8887307551694945e-01, 0.9109406880842844e+00, 0.9457532858494480e+00, 0.7192031611135805e+00, 0.5051759890546635e+00, 0.1927673350758418e+00, 0.6784576615085791e+00, 0.4582307655806892e+00, 0.4110913566635671e+00, 0.5843861752552199e+00, 0.1849014446101248e+00, 0.5286425989283670e+00, 0.9425316584065953e+00, 0.9757360470900053e+00, 0.4697986765545260e+00, 0.5423082602059015e-01, 0.4516134263086808e+00, 0.2022962345269688e+00, 0.4899666677522230e-01, 0.9657583521001409e+00, 0.7147044884479304e+00, 0.8698989147298346e+00, 0.1620630683580001e+00, 0.4917083507892124e+00, 0.6587971895356357e+00, 0.2830046270176211e+00, 0.8907428232631098e-01, 0.3385480253038764e-01, 0.3808208973559474e+00, 0.6301790073720519e+00, 0.7283330529106518e+00, 0.8754045015579843e+00, 0.5691786300850922e+00, 0.2221459247333947e+00, 0.3807253530058278e+00, 0.5099681617674874e+00, 0.8815981352292609e+00, 0.2670679959789202e-02, 0.3061419069919761e+00, 0.9088634553510815e+00, 0.3137439308670030e+00, 0.3256630871489109e+00, 0.7397175549546020e-01, 0.8600700672393357e+00, 0.2944165267259768e-01, 0.1962740610392520e+00, 0.7223088666910011e+00, 0.1555262415199788e+00, 0.7683934132272936e+00, 0.9837306979871404e+00, 0.4307236017454147e+00, 0.2763099157748476e+00, 0.5623054633156477e-01, 0.1414438705425911e+00, 0.3910643421231946e+00, 0.3414855836828583e+00, 0.4833767934540261e+00, 0.1874948602631823e+00, 0.6456680649630069e+00, 0.5377739930676261e+00, 0.4911199998049517e+00, 0.3925147326457343e+00, 0.1205990368593943e+00, 0.6208170182205421e+00, 0.4308681300066786e+00},
			n:    21,
			zOut: []float64{0.2838978926644850e+01, 0.2478331275466363e+01, 0.2456585262660691e+01, 0.2180185738821122e+01, 0.1712066680562540e+01, 0.1602701074250422e+01, 0.1503969746560551e+01, 0.1233768461187729e+01, 0.9787455830966086e+00, 0.9668289373592873e+00, 0.7729605529095109e+00, 0.7035556930615693e+00, 0.5295938697567051e+00, 0.4165425267136870e+00, 0.3781386605204551e+00, 0.3701077812928417e+00, 0.1575477292510020e+00, 0.7328369731218511e-01, 0.3974472017556452e-01, 0.4256064253578745e-02, 0.1985700463370783e-02, 0.0000000000000000e+00, 0.2182718854229073e-04, 0.3091215918753839e-04, 0.1503969746560551e+01, 0.7889046064910921e-15, 0.8525655623819580e-10, 0.5987167574554257e-09, 0.1233768461187729e+01, 0.1850899338648654e-19, 0.6965867087478826e-06, 0.6532556049979120e-06, 0.9787455830966086e+00, 0.3101927297073854e-24, 0.3608646891594891e-06, 0.2907657476508735e-06, 0.9668289373592873e+00, 0.5936688180990121e-20, 0.1421522625242826e-07, 0.7063390122609291e-08, 0.7729605529095109e+00, 0.1231338578304599e-17, 0.2139987868232064e+02, 0.2139987868232063e+02, 0.9600000000000000e+02, 0.2854875283446712e+01, 0.2083333333333333e+01, 0.2575545978160743e-06, 0.5295938697567051e+00, 0.8537368260663878e-24, 0.6647760383095535e-05, 0.6726768703723720e-05, 0.4165425267136870e+00, 0.5898327163094257e-18, 0.6007815560144262e-05, 0.6070984870789135e-05, 0.3781386605204551e+00, 0.1932242212135588e-23, 0.5780348596252855e-05, 0.5727260717649887e-05, 0.3701077812928417e+00, 0.2291245128765917e-16, 0.5589406660782675e-05, 0.5617175435749363e-05, 0.1575477292510020e+00, 0.3855952104720306e-26, 0.8404567502440466e-05, 0.1060626986994499e-04, 0.7328369731218511e-01, 0.1176980470589750e-26, 0.3402042835087641e-05, 0.5518338211116594e-05, 0.3974472017556452e-01, 0.3697785493223493e-30, 0.6423930978115149e-05, 0.5690659965930955e-05, 0.4256064253578745e-02, 0.1813010266067178e-20, 0.3330330670135602e-05, 0.3509207052336586e-05, 0.1985700463370783e-02, 0.9911627066431288e-23, 0.2656536456367949e-04, 0.9644853918428974e-05},
			info: 0,
		},
		{
			z:    []float64{0.6087347824331316e+00, 0.2344592930815549e+00, 0.1839180152431182e+00, 0.8946895357068063e-01, 0.3789275576065679e+00, 0.1028144130339237e-01, 0.8432962228323871e+00, 0.5720035990048252e+00, 0.9674750777949459e+00, 0.1387850858951470e+00, 0.1049983295079311e+00, 0.1869597635484084e+00, 0.2049847474830355e+00, 0.9353902309354832e+00, 0.7711597045807418e+00, 0.6677931698783810e+00, 0.5710020050743191e+00, 0.7559352115290241e+00, 0.8799700817290574e+00, 0.2989086447618220e+00, 0.7223221852829088e+00, 0.9172044263776469e-01, 0.8222766733565161e+00, 0.4357755436106698e+00, 0.3174399370719186e+00, 0.3716323007311311e+00, 0.5710261889832198e+00, 0.5875728000816982e-01, 0.1970825867540514e+00, 0.7514821290356389e+00, 0.4768968814440278e+00, 0.1414806690779697e+00, 0.5126469713172739e+00, 0.2618595999903317e-01, 0.3414455658478163e+00, 0.7845736413672471e+00, 0.3193997824316201e+00, 0.5244207174429636e+00, 0.7114985438553419e+00, 0.6380953409710879e+00, 0.5110544105167975e+00, 0.6987353947091863e+00, 0.2359552404464268e+00, 0.1389779414569779e+00, 0.4283565484470430e+00, 0.5978343235372734e+00, 0.8941068622870674e+00, 0.9993918591572127e+00, 0.9276104816334276e+00, 0.6840640077048410e+00, 0.1906252952212643e+00, 0.3896055694019390e+00, 0.8889540838698040e-01, 0.6972017776398595e+00, 0.7403854978121883e+00, 0.9624319129844994e-01, 0.7107468555423346e-01, 0.2167351328914763e+00, 0.5250363507339506e-01, 0.3012298012035964e+00, 0.7467053513110199e+00, 0.6311877906561809e+00, 0.6465457722939830e+00, 0.9462483307457270e-01, 0.1615629644808589e+00, 0.3714392567485900e+00, 0.5909086992570731e+00, 0.2147078824711308e+00, 0.8083150503324107e+00, 0.1136496291628873e+00, 0.3556622878263437e+00, 0.1687673308056986e+00, 0.2663369573375586e+00, 0.9992292880074174e+00, 0.4743706799728067e+00, 0.3784791276003248e+00, 0.9641428139214289e+00, 0.4526980047510465e-01, 0.5291384859598749e+00, 0.9114927597067145e+00, 0.2015740390806801e+00, 0.9111109510473957e+00, 0.4797378470164849e+00, 0.4344842342843944e+00},
			n:    21,
			zOut: []float64{0.2535248843080872e+01, 0.2027297538476471e+01, 0.1999341423958793e+01, 0.1905819580237666e+01, 0.1558994176508815e+01, 0.1443542425236611e+01, 0.1321312714967014e+01, 0.1044341128979687e+01, 0.9175998827905956e+00, 0.8543653487373744e+00, 0.6128068085852322e+00, 0.5723484802692980e+00, 0.4865842293721045e+00, 0.4743009418296827e+00, 0.4424121716041568e+00, 0.2157744105271115e+00, 0.1401155020588904e+00, 0.9545054691199920e-01, 0.4846378139764250e-01, 0.2597131641839770e-01, 0.9574417588705024e-02, 0.7983827718913915e-23, 0.5897957913656464e-08, 0.4905965358471980e-08, 0.1321312714967014e+01, 0.1437620113674772e-24, 0.2330664696165742e-08, 0.2036082149427908e-08, 0.1044341128979687e+01, 0.7099748146989106e-29, 0.1191493821660667e-08, 0.1294122655697948e-08, 0.9175998827905956e+00, 0.7069460558567906e-16, 0.2320495180901029e-08, 0.3242686959161780e-08, 0.8543653487373744e+00, 0.3414430154917050e-13, 0.2143028595220802e-07, 0.3181347787023606e-07, 0.6128068085852322e+00, 0.0000000000000000e+00, 0.1873166566953712e+02, 0.1873166566953712e+02, 0.1040000000000000e+03, 0.3224489795918367e+01, 0.2884615384615385e+01, 0.3533200143891392e-06, 0.4865842293721045e+00, 0.4552522969733916e-17, 0.1776353409716398e-09, 0.6419018055060889e-09, 0.4743009418296827e+00, 0.4679164786798698e-18, 0.4782181507340309e-06, 0.1264406900548246e-06, 0.4424121716041568e+00, 0.1728289684559415e-20, 0.3046857378867568e-04, 0.2182730305393890e-04, 0.2157744105271115e+00, 0.1120182485413837e-27, 0.4003877877583194e-03, 0.2588513428463001e-03, 0.1401155020588904e+00, 0.0000000000000000e+00, 0.6916990427934659e-04, 0.5501406364660823e-04, 0.9545054691199920e-01, 0.6162975822039155e-32, 0.1059944134272042e-03, 0.1421052287262650e-03, 0.4846378139764250e-01, 0.1755514705909767e-17, 0.4499204819889415e-04, 0.2567010249779503e-03, 0.2597131641839770e-01, 0.1143848312570467e-28, 0.5289175880660015e-04, 0.5437725505075631e-04, 0.9574417588705024e-02, 0.3697785493223493e-31, 0.2606391166909807e-05, 0.6294847321319143e-06},
			info: 0,
		},
		{
			z:    []float64{0.3257261020965838e+00, 0.9499222527145429e+00, 0.3597416524070001e+00, 0.7608369220069154e+00, 0.3348964244425234e-01, 0.9918240230913730e+00, 0.2435073049094116e-01, 0.2927090414647138e-01, 0.2724578324033788e+00, 0.7486167980650003e+00, 0.2854897136111678e+00, 0.2267361867398041e+00, 0.4294380831801550e+00, 0.7722184192395405e+00, 0.5889718788171796e+00, 0.5201154937785833e+00, 0.1341644467607439e-01, 0.3251983707424297e+00, 0.3712514383296180e+00, 0.8346822920283823e+00, 0.2613611110008908e+00, 0.7377020224374462e+00, 0.4290238209924468e-01, 0.8422463567888154e+00, 0.1390406385354674e+00, 0.8899155813346852e+00, 0.5366293959824918e+00, 0.1311808378334547e+00, 0.5127829646320820e+00, 0.3806825827063995e+00, 0.9008244723505014e+00, 0.8397173164045483e+00, 0.8005434668537068e+00, 0.4188784622408616e+00, 0.4815341947551696e+00, 0.2004130034689415e+00, 0.9620560381224039e+00, 0.9936712626034268e+00, 0.7996240982628953e+00, 0.9066259300527590e+00, 0.7119745510778175e+00, 0.4009593231138309e+00, 0.7503038869147766e+00, 0.2307657656692345e+00, 0.5796381959344894e+00, 0.4564585857276309e+00, 0.3087632483431468e-01, 0.4925813444884515e+00, 0.2548189626174928e+00, 0.7548007003619551e+00, 0.2595357873879023e+00, 0.2576161595637174e+00, 0.6530834734960895e+00, 0.9659056802808288e+00, 0.4073417244577716e+00, 0.7973054858347279e+00, 0.1098542135402614e-03, 0.5270152208809449e+00, 0.9503221531609122e+00, 0.3389760481404548e+00, 0.2936463197963648e+00, 0.1532837593240226e+00, 0.7039965742744211e+00, 0.1538522429841973e+00, 0.4565500206514618e+00, 0.4890306322657354e+00, 0.7406911389536464e+00, 0.4609110724077861e+00, 0.3010957214922741e+00, 0.4706875322819619e+00, 0.9245411658881026e+00, 0.1225486958523536e-02, 0.7513538867599376e+00, 0.7480092012350148e+00, 0.1668544624863488e+00, 0.6027497133760584e+00, 0.4060848284780354e+00, 0.7777661676033926e+00, 0.5476587806685121e+00, 0.3318235867866506e+00, 0.4865355710680780e+00, 0.9789804028805801e+00, 0.6449091593541079e-01, 0.3386899175207851e+00},
			n:    21,
			zOut: []float64{0.2916446272919928e+01, 0.2518523111789073e+01, 0.1881620740030630e+01, 0.1802535552376618e+01, 0.1705712498790935e+01, 0.1630934300031556e+01, 0.1493206464268041e+01, 0.1289863129801601e+01, 0.1141879970494692e+01, 0.1064939765404915e+01, 0.9148995873510519e+00, 0.6667419164223021e+00, 0.5793746811807822e+00, 0.5763823737203533e+00, 0.5577209455074719e+00, 0.2258972104173268e+00, 0.1775772659782911e+00, 0.1224839196229167e+00, 0.7643325286481288e-01, 0.1088888144792382e-01, 0.1013218159024319e-07, 0.1529626957613473e-16, 0.4760904670371191e-26, 0.1685698503770275e-25, 0.1493206464268041e+01, 0.4740164255731978e-15, 0.6203207773025231e-22, 0.1697026618649852e-21, 0.1289863129801601e+01, 0.1555744028694447e-17, 0.2220966892754199e-18, 0.8352563633635413e-19, 0.1141879970494692e+01, 0.8283039504820624e-29, 0.9137114046044313e-17, 0.2186983724354431e-16, 0.1064939765404915e+01, 0.7560947786617519e-24, 0.1476623491156538e-14, 0.6521525514986541e-15, 0.9148995873510519e+00, 0.3956572890903056e-23, 0.2135406185055340e+02, 0.2135406185055340e+02, 0.1110000000000000e+03, 0.3462585034013606e+01, 0.7207207207207207e+01, 0.3107640874581820e-11, 0.5793746811807822e+00, 0.0000000000000000e+00, 0.5611258671513064e-10, 0.1000932867794445e-09, 0.5763823737203533e+00, 0.2935229715008717e-19, 0.5869279282596528e-09, 0.3232392517855140e-09, 0.5577209455074719e+00, 0.7161080495560552e-17, 0.6771347567635994e-08, 0.1273942735381484e-07, 0.2258972104173268e+00, 0.2531843062293473e-19, 0.3233128964696781e-05, 0.1839431244951091e-05, 0.1775772659782911e+00, 0.1025519176787315e-28, 0.1632093348006395e-04, 0.1394944946706296e-04, 0.1224839196229167e+00, 0.3147671031182890e-20, 0.4090539240175316e-04, 0.3499438731982919e-04, 0.7643325286481288e-01, 0.4418024966021139e-23, 0.7751221398283570e-04, 0.6522231732461535e-04, 0.1088888144792382e-01, 0.0000000000000000e+00, 0.1168516922729128e-02, 0.6798960389683572e-03, 0.1013218159024319e-07, 0.8311253061997563e-15, 0.2266806408535672e-02, 0.8228900118596701e-02},
			info: 0,
		},
		{
			z:    []float64{0.9013837094783728e-01, 0.4093071371456288e+00, 0.7563443593246222e-01, 0.6047712831067179e+00, 0.3616734158068488e+00, 0.3179352368154293e+00, 0.4877726790168605e+00, 0.7283754332956183e+00, 0.6564126870433257e+00, 0.7426267273836333e+00, 0.9575970808689874e+00, 0.2328163542816707e+00, 0.9804488839127946e+00, 0.5880329556580101e+00, 0.6720101508999355e+00, 0.3029641470022006e+00, 0.9176198736416905e+00, 0.9273523415722473e-01, 0.7356501689695268e+00, 0.3516482504433937e+00, 0.5613309946663773e+00, 0.8064085547776983e+00, 0.2421909924481950e+00, 0.4321291285180304e+00, 0.7195726435630079e+00, 0.4201959590716675e-01, 0.6971840239025084e-03, 0.6931587576362976e+00, 0.4512374813335629e+00, 0.9105433528826555e+00, 0.8233348499205092e+00, 0.5270028711667427e+00, 0.5092902528068642e+00, 0.6527825111058926e+00, 0.8077059946216197e+00, 0.5037249441803524e+00, 0.7474524841642615e+00, 0.8361353459662745e+00, 0.6675800963161582e+00, 0.5004244151247563e+00, 0.7976263900002633e+00, 0.1983499023889042e+00, 0.7522366994446783e+00, 0.3654861843844912e+00, 0.8247542543249066e+00, 0.9124235995852386e+00, 0.1818459043196227e+00, 0.8196813052173683e+00, 0.5538880208895494e+00, 0.9080443570480472e+00, 0.2981632101337024e+00, 0.6044133722200155e+00, 0.1883025355842233e+00, 0.2434431972046255e+00, 0.3420561860884004e+00, 0.3312086041097358e+00, 0.8261796050670067e-01, 0.2597483126975232e+00, 0.9037350022806145e+00, 0.5691251256015616e+00, 0.5433137600205407e-01, 0.2948423088781353e+00, 0.7125818959241866e+00, 0.1168360183447441e+00, 0.2271931321361120e-01, 0.5832011197836875e+00, 0.4321285977614958e+00, 0.9214755199664483e+00, 0.2929460789941122e+00, 0.9911839954873096e+00, 0.9352897353900299e+00, 0.6549195608155846e+00, 0.7733376794047198e-01, 0.7345459966842849e+00, 0.2997108680399072e+00, 0.6923968300089707e+00, 0.9895794156524929e+00, 0.2694202749089550e+00, 0.2419924375765692e-01, 0.2728256718882756e+00, 0.7688978378093437e+00, 0.6794051971520565e-01, 0.8211384369868754e+00, 0.7581877817293103e-01},
			n:    21,
			zOut: []float64{0.2587082763537058e+01, 0.2518438946693926e+01, 0.2381638511714923e+01, 0.2064758819607097e+01, 0.1861546331198453e+01, 0.1823771413694203e+01, 0.1462413605342042e+01, 0.1295483717305927e+01, 0.1204792654692795e+01, 0.1153126691944095e+01, 0.9646786889507554e+00, 0.8876075724637406e+00, 0.7328920239265601e+00, 0.5128961040525114e+00, 0.4187721082034697e+00, 0.3198505150609477e+00, 0.1661143074789019e+00, 0.1249876230738883e+00, 0.5554047196429147e-01, 0.2079344839078052e-02, 0.4713171572265143e-04, 0.1874281408120013e-19, 0.2104034717604745e-09, 0.2619972849686678e-09, 0.1462413605342042e+01, 0.9259545558536950e-18, 0.2890370685025388e-10, 0.3242632456342165e-10, 0.1295483717305927e+01, 0.1323488980084844e-22, 0.2279224462592833e-10, 0.2555931367214743e-10, 0.1204792654692795e+01, 0.2978738778114541e-26, 0.7485181957332586e-10, 0.4566030880095840e-10, 0.1153126691944095e+01, 0.6733803272339514e-21, 0.2633277318950464e-08, 0.1440235383519861e-08, 0.9646786889507554e+00, 0.7499216124683276e-17, 0.2253851934746039e+02, 0.2253851934746038e+02, 0.1040000000000000e+03, 0.3111111111111111e+01, 0.5769230769230769e+01, 0.2633869415622400e-06, 0.7328920239265601e+00, 0.6980417424693642e-13, 0.1603044445805627e-06, 0.1255532606981617e-06, 0.5128961040525114e+00, 0.6981404369947554e-21, 0.8484909168369356e-07, 0.7367232980661418e-07, 0.4187721082034697e+00, 0.2465190328815662e-30, 0.1470967366725541e-06, 0.2274394647752198e-06, 0.3198505150609477e+00, 0.9424343740971753e-25, 0.8932721831570202e-05, 0.5259727349591482e-05, 0.1661143074789019e+00, 0.1697763391846713e-18, 0.4712327641645974e-05, 0.7305995006514358e-05, 0.1249876230738883e+00, 0.9266645908481002e-19, 0.3596318120009657e-04, 0.2001742175196116e-04, 0.5554047196429147e-01, 0.5985976337495067e-16, 0.4811066675748340e-03, 0.3416912105856511e-03, 0.2079344839078052e-02, 0.8697917232094458e-24, 0.5710266423841766e-02, 0.2981491759065521e-02, 0.4713171572265143e-04, 0.0000000000000000e+00, 0.3288287603570621e-02, 0.1038987759212308e-02},
			info: 0,
		},
		{
			z:    []float64{0.3453921918700114e+00, 0.8256584009865165e+00, 0.5027306146870630e+00, 0.5014619412470180e+00, 0.9910264683396075e+00, 0.7385152238869899e+00, 0.9936695869989243e+00, 0.6206355260490602e+00, 0.6835539321104728e+00, 0.3987727521414011e+00, 0.2155721742355131e+00, 0.8212295891891395e+00, 0.7021626430853991e+00, 0.3753314442932942e+00, 0.2575710525141791e+00, 0.6551328468445583e+00, 0.3738398426943970e+00, 0.8805730558289802e+00, 0.6848909798416613e+00, 0.5520069040756137e-01, 0.8817067703079440e+00, 0.3238117712584516e+00, 0.1001825089826291e+00, 0.7369585910327309e+00, 0.7601318359400422e+00, 0.4312449588193277e+00, 0.4932891154015395e+00, 0.1099211239093930e+00, 0.5764729343614433e+00, 0.1286539516629047e+00, 0.5170000684627285e+00, 0.4006756226519864e+00, 0.5897018197291505e+00, 0.3774659871747915e+00, 0.4211183018281148e+00, 0.6260997273722376e+00, 0.7810234892670644e+00, 0.9934154489086249e+00, 0.5116204079460294e+00, 0.7948452919719275e+00, 0.5998807734230309e+00, 0.4868691071952969e-01, 0.7977221292955683e+00, 0.4907992372099745e+00, 0.4337635562764541e-01, 0.4865658084478622e+00, 0.7887982210525304e+00, 0.6218441262679077e+00, 0.1807768964120138e-01, 0.7341707407508724e+00, 0.4738048273586776e+00, 0.9324751870816922e+00, 0.4643019354540665e+00, 0.2401905271601299e+00, 0.6072757133952530e+00, 0.6257046236339610e+00, 0.1662572295885112e-01, 0.3360107439465247e+00, 0.5925152832499899e+00, 0.4206234399601897e+00, 0.1828405120649758e+00, 0.8923354266778754e+00, 0.6211472795399554e+00, 0.3215240169291051e+00, 0.3967102459115966e+00, 0.4376502563326659e+00, 0.4571232280132043e-01, 0.2973405440159681e+00, 0.1737124961451384e+00, 0.1123665207380958e+00, 0.8069742345287393e+00, 0.5302931605046994e+00, 0.2541675285808478e+00, 0.9099997182043015e+00, 0.7562530283478575e-01, 0.2521110146593245e+00, 0.2525672524245413e+00, 0.8930826245328107e+00, 0.8545394186204489e+00, 0.1856025965008533e+00, 0.8541401621096930e+00, 0.7406868214039234e+00, 0.5851270108744718e+00, 0.7554853143122133e+00},
			n:    21,
			zOut: []float64{0.2818806710434786e+01, 0.2596766072486540e+01, 0.2041400548479265e+01, 0.1919012388179902e+01, 0.1900863080415201e+01, 0.1778136512482050e+01, 0.1649297429111022e+01, 0.1375943901620400e+01, 0.1287933944237744e+01, 0.1246950647018126e+01, 0.9346701076388835e+00, 0.8619836761278706e+00, 0.6928534822715838e+00, 0.5816592049742084e+00, 0.4759893592204403e+00, 0.2537953742738920e+00, 0.2205935192530503e+00, 0.8350022078832350e-01, 0.3307087507037688e-01, 0.2105652896299196e-01, 0.3857874617179747e-02, 0.9177981866196648e-17, 0.6043035417783778e-14, 0.7682115486038763e-14, 0.1649297429111022e+01, 0.9047287949798740e-23, 0.2120842457393041e-14, 0.2617862539054627e-14, 0.1375943901620400e+01, 0.1499552906513639e-17, 0.3376573256769321e-15, 0.2886563968474245e-15, 0.1287933944237744e+01, 0.1966621906344823e-17, 0.2937881411718043e-15, 0.4577311861870994e-15, 0.1246950647018126e+01, 0.2742371736879744e-17, 0.2913637797754287e-14, 0.1483948533056680e-14, 0.9346701076388835e+00, 0.5800058106323822e-18, 0.2277814145766384e+02, 0.2277814145766384e+02, 0.9800000000000000e+02, 0.2997732426303855e+01, 0.1020408163265306e+01, 0.9030119244814722e-12, 0.6928534822715838e+00, 0.1471303281095442e-16, 0.2634125396797654e-10, 0.5071585384120369e-10, 0.5816592049742084e+00, 0.1365675999118616e-25, 0.3358753759221612e-09, 0.6258446588335648e-09, 0.4759893592204403e+00, 0.1479114197289397e-30, 0.3734630956704811e-08, 0.6716634735981551e-08, 0.2537953742738920e+00, 0.7467396771867359e-25, 0.3072344915986320e-06, 0.1829961043616119e-06, 0.2205935192530503e+00, 0.4437342591868191e-30, 0.2385309167517414e-05, 0.3978543929472447e-05, 0.8350022078832350e-01, 0.2436129622029919e-19, 0.4589207629691226e-04, 0.2836221395611648e-04, 0.3307087507037688e-01, 0.3081487911019577e-32, 0.1362072635971665e-04, 0.2978962617014088e-04, 0.2105652896299196e-01, 0.2711709361697228e-30, 0.2444625596237189e-05, 0.7827983423195974e-05, 0.3857874617179747e-02, 0.8925769947935872e-20, 0.9826965253502863e-02, 0.1259209175531780e-01},
			info: 0,
		},
		{
			z:    []float64{0.9560856702670685e+00, 0.5885863857778361e+00, 0.8071258175651005e+00, 0.5175476341605639e+00, 0.1941426739297742e+00, 0.8451490614471896e+00, 0.9052305950784733e+00, 0.6791526261185319e+00, 0.6480230038264980e+00, 0.8438040949137990e+00, 0.3034844043967853e+00, 0.3090582723248356e+00, 0.3282769832503399e+00, 0.7489751201458807e+00, 0.6672021557248126e+00, 0.1697989591956236e+00, 0.5723860504781838e+00, 0.3660621222567975e+00, 0.1505291330148173e-01, 0.7436839329892437e+00, 0.3679212455376497e+00, 0.5305714072096270e+00, 0.5792221354999488e+00, 0.3031533452885045e+00, 0.9514596930492207e+00, 0.3278419583960311e+00, 0.5742176036064726e+00, 0.7216680447710266e+00, 0.6608941225668634e+00, 0.3887216422692293e+00, 0.6952172825541724e+00, 0.8072929471269218e+00, 0.8776672848787970e-01, 0.9363773133978057e+00, 0.3470717327786855e+00, 0.9438495003134180e+00, 0.5511880105145929e+00, 0.5826637033513504e+00, 0.3416802173197317e+00, 0.5650234571673005e+00, 0.2927187699119284e+00, 0.4936451925034285e+00, 0.7496439056333554e+00, 0.8462697295115110e+00, 0.4661250827466905e+00, 0.3912590593193351e+00, 0.9140228064649625e+00, 0.4016314378444454e+00, 0.7105479239000361e+00, 0.3751330850058705e+00, 0.7309804928539319e+00, 0.1378037023247644e+00, 0.7527056268407295e+00, 0.1640478774378548e+00, 0.6030857470066908e+00, 0.7427119073248016e+00, 0.9996405952923426e-01, 0.7406467144237725e+00, 0.3762498316380406e+00, 0.8295145255291534e+00, 0.8322845800948830e-01, 0.6101670185797291e+00, 0.9451596878306588e+00, 0.1176379504958335e+00, 0.7940514845464750e+00, 0.3068100097277620e+00, 0.2371282369406802e+00, 0.2163781309135768e+00, 0.4705542393165623e+00, 0.4003463805482307e+00, 0.2731285781388088e+00, 0.8862626881898287e+00, 0.5854390777280626e+00, 0.4661955367614972e+00, 0.1993415035442425e+00, 0.9213995968359628e+00, 0.1542739762296447e+00, 0.9419076789558214e+00, 0.9611493561784724e+00, 0.7596360150269755e+00, 0.6590873675145426e+00, 0.3894853423829039e+00, 0.8407621320868611e+00, 0.4517123559923821e+00},
			n:    21,
			zOut: []float64{0.2583735261373388e+01, 0.2239759728480232e+01, 0.2157858012928955e+01, 0.2146305201359927e+01, 0.1853050031878152e+01, 0.1788497392649839e+01, 0.1558387970436693e+01, 0.1494017846927041e+01, 0.1323723239351723e+01, 0.1188003233397794e+01, 0.9827022795744828e+00, 0.8335582164080391e+00, 0.7854672559750093e+00, 0.6236176061997519e+00, 0.4677974993044671e+00, 0.2968385185647548e+00, 0.2304456569541352e+00, 0.1793008998018142e+00, 0.2471364396669566e-01, 0.7537422761275628e-02, 0.3241997301555496e-04, 0.1184063810429910e-21, 0.1281976731077423e-10, 0.2849227344309828e-10, 0.1558387970436693e+01, 0.3001369155282220e-19, 0.2940080271049715e-08, 0.1685793775386140e-08, 0.1494017846927041e+01, 0.1696366490587264e-24, 0.4557030850938984e-08, 0.5166797185591864e-08, 0.1323723239351723e+01, 0.1862197338326049e-12, 0.1908610209474282e-08, 0.1816527091730617e-08, 0.1188003233397794e+01, 0.1613415991578665e-17, 0.2801618631477443e-08, 0.2177314380835442e-08, 0.9827022795744828e+00, 0.1073695634537585e-17, 0.2276534933826718e+02, 0.2276534933826718e+02, 0.1060000000000000e+03, 0.3113378684807256e+01, 0.3773584905660377e+01, 0.2107018980320373e-07, 0.7854672559750093e+00, 0.7395570986446986e-31, 0.5134101120680572e-07, 0.6949862017183880e-07, 0.6236176061997519e+00, 0.1700003115737185e-12, 0.4917011909790346e-06, 0.6444201346122313e-06, 0.4677974993044671e+00, 0.1676329423594650e-29, 0.2178752696080751e-05, 0.1697685142168825e-05, 0.2968385185647548e+00, 0.2903323170704433e-20, 0.9062193610690628e-05, 0.7062135859641892e-05, 0.2304456569541352e+00, 0.9860761315262648e-30, 0.1460945060339096e-04, 0.1904220434906198e-04, 0.1793008998018142e+00, 0.4606947686490709e-27, 0.4451684144215141e-07, 0.1728984935219646e-06, 0.2471364396669566e-01, 0.1407327854914285e-26, 0.1942342839475662e-02, 0.3052712133602924e-02, 0.7537422761275628e-02, 0.9865964111678957e-20, 0.3214451590395407e-02, 0.5215635567340069e-02, 0.3241997301555496e-04, 0.0000000000000000e+00, 0.1184307710713820e-01, 0.9427403294935964e-02},
			info: 0,
		},
		{
			z:    []float64{0.4101594051788160e+00, 0.1002974419660251e+00, 0.8434067081152193e+00, 0.7258916354493068e+00, 0.9256853341957688e+00, 0.8379265467529036e+00, 0.1564408324797836e+00, 0.8292039738515364e+00, 0.3771190822561328e+00, 0.8236097732630416e+00, 0.3320638300046872e+00, 0.6850750205938005e-01, 0.5616647045816037e+00, 0.8305938719521835e+00, 0.6006351465721590e+00, 0.8469146468608777e+00, 0.1179160781320751e+00, 0.6975716721259332e+00, 0.3237957597653003e+00, 0.1297318227605904e-01, 0.8416519209661123e+00, 0.2623987624579764e+00, 0.3323289069826861e+00, 0.9848995259497869e+00, 0.1295798367899781e-01, 0.4196318718239949e+00, 0.8791346827157692e+00, 0.9015205418757779e+00, 0.2557984197164971e+00, 0.7295221077613672e+00, 0.8796719157178668e+00, 0.6983240767110626e+00, 0.6461186804049626e+00, 0.5003600760145155e-01, 0.3411701822058120e+00, 0.6026846298507913e+00, 0.4649210011845201e+00, 0.4169632595817018e+00, 0.9745024564197420e+00, 0.9159749674174896e+00, 0.3798779762430529e+00, 0.6757581567027395e+00, 0.1501600637708167e+00, 0.8841295845033917e+00, 0.1700642059156726e+00, 0.5745602516925833e+00, 0.7146078194346678e+00, 0.7876321314233041e+00, 0.3623958693011297e+00, 0.1316504845867333e+00, 0.6150819295249010e+00, 0.9948396700221377e+00, 0.2186788175334711e+00, 0.3433450548510013e+00, 0.4434886569716980e+00, 0.6596978167473024e+00, 0.7067569387667516e+00, 0.2710992196058543e+00, 0.2531644393448058e+00, 0.6179743389634106e+00, 0.6395694224823617e+00, 0.6233992467844496e-01, 0.9270082862838847e-01, 0.6813409948026068e+00, 0.8489230752840353e+00, 0.4248777519526148e+00, 0.9258241367507150e+00, 0.7591306919901064e-01, 0.1337231221457533e+00, 0.6673452115706187e+00, 0.3161361894802919e+00, 0.9287930266484576e+00, 0.9211411715651807e+00, 0.7110889130885073e+00, 0.6783920645346327e+00, 0.5798290672446252e+00, 0.6003988035906362e+00, 0.8338909998640337e+00, 0.1150264810560550e+00, 0.8268177524919460e+00, 0.2868686997384918e+00, 0.7734675037869032e+00, 0.8495624932870276e+00, 0.7382167415731139e+00},
			n:    21,
			zOut: []float64{0.2565771617602568e+01, 0.2464815559540473e+01, 0.2377249642737961e+01, 0.2182027799722737e+01, 0.1994511663794419e+01, 0.1777930797653110e+01, 0.1532836111328556e+01, 0.1298104160091711e+01, 0.1162548328490141e+01, 0.1031426069242260e+01, 0.9067767207978197e+00, 0.7702116437984168e+00, 0.7023507576840374e+00, 0.5846573732521254e+00, 0.3836750731694669e+00, 0.3732780616075560e+00, 0.2146097506115993e+00, 0.5462411591703480e-01, 0.3208967690429450e-01, 0.2788258306628742e-02, 0.1838228532938987e-03, 0.1703684557466520e-17, 0.8886730161731292e-14, 0.1725592004801531e-13, 0.1532836111328556e+01, 0.2310659810285515e-15, 0.8778995686431983e-12, 0.1396745951735637e-11, 0.1298104160091711e+01, 0.0000000000000000e+00, 0.8497749401208094e-11, 0.5925398902948847e-11, 0.1162548328490141e+01, 0.1292469707114106e-25, 0.1515360700076446e-10, 0.2085752134421418e-10, 0.1031426069242260e+01, 0.1195331315767287e-17, 0.8898419915788489e-10, 0.1186621585503233e-09, 0.9067767207978197e+00, 0.1888339631531495e-12, 0.2241246700510621e+02, 0.2241246700510621e+02, 0.1190000000000000e+03, 0.3539682539682540e+01, 0.7563025210084033e+01, 0.2596357213906324e-08, 0.7023507576840374e+00, 0.1416275483756143e-17, 0.5106480833143134e-08, 0.6438584684288567e-08, 0.5846573732521254e+00, 0.1317619996636992e-15, 0.1896058302301025e-07, 0.2367797858508706e-07, 0.3836750731694669e+00, 0.9199951564453634e-11, 0.8967796171107713e-07, 0.1296821709728121e-06, 0.3732780616075560e+00, 0.1615587133892578e-26, 0.3088999981101691e-06, 0.2343043411729366e-06, 0.2146097506115993e+00, 0.2055968734232262e-28, 0.9134179982896841e-05, 0.6280501592969192e-05, 0.5462411591703480e-01, 0.1901142159808155e-23, 0.3768301572228235e-05, 0.8777513601148953e-05, 0.3208967690429450e-01, 0.1479114197289397e-29, 0.1428515450408294e-03, 0.5208612044311805e-04, 0.2788258306628742e-02, 0.2057124874537592e-17, 0.2063479623178836e-02, 0.3566698154744510e-02, 0.1838228532938987e-03, 0.3273772756667199e-28, 0.8538542526496483e-03, 0.1571849612061994e-03},
			info: 0,
		},
		{
			z:    []float64{0.1589037341524596e+00, 0.1455226073481288e+00, 0.1938395273812957e+00, 0.1962691615573132e+00, 0.1687822326883440e+00, 0.6650981615477494e+00, 0.8270615916362424e+00, 0.1710502783654970e+00, 0.2206713720159043e+00, 0.4369721197399372e+00, 0.9301314325299896e+00, 0.5722466234411457e+00, 0.1135089451836189e+00, 0.5846068541971138e+00, 0.9216025315561343e+00, 0.9066267701445474e+00, 0.5584279552920539e+00, 0.4343925455049338e+00, 0.9858452060735702e+00, 0.6563046209578444e+00, 0.7077011175365495e+00, 0.4947310926689127e+00, 0.7385820400647596e+00, 0.5937728681009750e+00, 0.9762402194896037e+00, 0.3450302019452143e+00, 0.3805099852593911e+00, 0.7529397554746390e+00, 0.7330092480109731e+00, 0.3330086450808912e+00, 0.9038518043983117e+00, 0.6574079221280790e+00, 0.2509077957372493e+00, 0.6249580721461656e+00, 0.2757992349681960e+00, 0.5001238277403597e+00, 0.9621194298806066e+00, 0.3061711826567661e+00, 0.1119838022409911e+00, 0.2976298035182879e+00, 0.6363137835930621e-01, 0.4930031667447814e+00, 0.1865676338649569e-01, 0.4250211458984191e+00, 0.1095501188334868e+00, 0.2763357459722963e+00, 0.7075737347301593e+00, 0.5363987747039417e+00, 0.3986581101260644e+00, 0.2692140175173962e+00, 0.5588052030209328e+00, 0.5539805437492590e+00, 0.6882419411200217e+00, 0.6525011142322502e+00, 0.2351213214810641e+00, 0.4862287407564582e+00, 0.9494457572755843e-01, 0.2775605674126462e+00, 0.6088686305117397e-01, 0.1602712446759722e+00, 0.6726078044316741e-01, 0.1771280849255007e+00, 0.9403546682132294e+00, 0.1182787468569617e+00, 0.6084837514797026e+00, 0.4725411123129805e+00, 0.2891346904124859e-01, 0.3056197468639377e-01, 0.3597281640784106e+00, 0.3047108802830497e+00, 0.9462624965987684e+00, 0.6795048656132132e+00, 0.3142878274718968e+00, 0.2255775176392846e+00, 0.5741746909618360e+00, 0.6125119194512041e+00, 0.8636232492271716e+00, 0.5595270150362563e-01, 0.4894380312911951e+00, 0.2945948126181174e+00, 0.4782956027971728e+00, 0.1100404727801326e+00, 0.2692801555328298e+00, 0.4567347029351195e+00},
			n:    21,
			zOut: []float64{0.2662663085210597e+01, 0.2404979690430721e+01, 0.2214109781172517e+01, 0.2010332580463078e+01, 0.1879931283999763e+01, 0.1831239586461748e+01, 0.1664682288858752e+01, 0.1512516241203824e+01, 0.1081258277181201e+01, 0.8471473481460485e+00, 0.6221178288483303e+00, 0.5493156782308360e+00, 0.4496825320738197e+00, 0.3768653101376996e+00, 0.2833465507506064e+00, 0.2013980912694376e+00, 0.1418989984144841e+00, 0.7096946004385345e-01, 0.4542843847166723e-01, 0.6747056508451696e-02, 0.1343590842619022e-02, 0.4141519752410312e-28, 0.4845145585707669e-05, 0.3343357997526164e-05, 0.1664682288858752e+01, 0.8527270841077049e-25, 0.1655046287299363e-04, 0.2231546414257453e-04, 0.1512516241203824e+01, 0.5692210911982269e-19, 0.1383329297784762e-03, 0.1072264803352635e-03, 0.1081258277181201e+01, 0.1459392674658872e-28, 0.4158771817821724e-03, 0.3910579975830214e-03, 0.8471473481460485e+00, 0.2949199315750023e-16, 0.2216793811640345e-03, 0.2203201078239036e-03, 0.6221178288483303e+00, 0.2055113688275276e-18, 0.2085797369872005e+02, 0.2085797369872006e+02, 0.9000000000000000e+02, 0.2732426303854875e+01, 0.0000000000000000e+00, 0.1772659828830871e-05, 0.4496825320738197e+00, 0.4437342591868191e-30, 0.2138076899966292e-04, 0.1340116970023588e-04, 0.3768653101376996e+00, 0.2904017873172006e-24, 0.4668906275608195e-04, 0.7182837805013731e-04, 0.2833465507506064e+00, 0.4888968615550842e-22, 0.1290326696826671e-05, 0.6479808231987577e-06, 0.2013980912694376e+00, 0.4436553730962970e-26, 0.1673384055036320e-03, 0.7106703019344439e-04, 0.1418989984144841e+00, 0.3443985257364436e-16, 0.1415746442632383e-02, 0.1286112519803737e-02, 0.7096946004385345e-01, 0.1380506584136771e-29, 0.1898294559619081e-02, 0.3536485589649533e-02, 0.4542843847166723e-01, 0.9860761315262648e-31, 0.1085724134073318e-03, 0.2838967874935677e-03, 0.6747056508451696e-02, 0.2742900553697359e-16, 0.1307657950440773e-02, 0.4192938789348323e-02, 0.1343590842619022e-02, 0.2423380700838948e-25, 0.4340570868086748e-02, 0.9691995934364128e-02},
			info: 0,
		},
		{
			z:    []float64{0.6603307676446488e+00, 0.5215490552441340e+00, 0.9279560414420993e+00, 0.3871591216009305e+00, 0.4489001158072897e+00, 0.2115798060938074e+00, 0.6013990647859857e-01, 0.6977506014884622e+00, 0.8981757899514635e+00, 0.6366966965554082e+00, 0.5447619366248987e+00, 0.2926906196279220e+00, 0.9538633412864880e+00, 0.4052441866103130e+00, 0.9970193775088028e+00, 0.1550480233350559e+00, 0.9697068099139655e+00, 0.3748402815554551e+00, 0.3906465669660910e+00, 0.4103562008634933e+00, 0.9514404866677418e+00, 0.5608812699221558e+00, 0.6226680723027547e+00, 0.8653188982275728e+00, 0.3999642050403411e+00, 0.8968004947108832e+00, 0.1007886665847859e+00, 0.2055140520989622e+00, 0.8479005785073866e+00, 0.4772281284198989e+00, 0.3664785333935638e+00, 0.9311163393120341e+00, 0.2051534177095988e+00, 0.5588930379235971e+00, 0.9385948972790567e+00, 0.1380523811906209e+00, 0.9076293986285472e+00, 0.9937542903014829e+00, 0.5330397746663563e+00, 0.5373590340921116e+00, 0.2667502750074380e-01, 0.9996497580467543e+00, 0.7460509377370035e+00, 0.5868152625248035e+00, 0.2352631583529476e+00, 0.2062979890255305e+00, 0.1864397664760209e-02, 0.1562262993494207e+00, 0.1880894213999632e+00, 0.5369024951393064e+00, 0.6560943776892021e+00, 0.9558063214302399e+00, 0.1463497248954505e+00, 0.6977928179134164e+00, 0.8204108917709737e+00, 0.9530026100161404e+00, 0.7191901603465490e+00, 0.4974721533134605e+00, 0.8445129869563219e+00, 0.9007488855711614e+00, 0.4164322161987429e+00, 0.5750362072910759e+00, 0.1935672755254650e+00, 0.2663393354536251e+00, 0.3913931132271733e+00, 0.7724103066167269e+00, 0.7871420607985310e+00, 0.3766724710952808e+00, 0.9224124235283799e+00, 0.1080194858843619e+00, 0.3512364719709385e+00, 0.9868075881536598e-02, 0.9893144531298202e+00, 0.5983612801716819e+00, 0.2882889058269555e-01, 0.2759662054778022e+00, 0.4865673169207868e+00, 0.4317793431152178e+00, 0.4039098350536063e-01, 0.7594353556100967e+00, 0.6702102151313558e+00, 0.5757953377080282e+00, 0.7036908158147406e+00, 0.3078985083785768e+00},
			n:    21,
			zOut: []float64{0.2422149931211175e+01, 0.2394118175176597e+01, 0.2305882752563478e+01, 0.2031709011458775e+01, 0.2015634640599460e+01, 0.1933740155888154e+01, 0.1715509223328114e+01, 0.1559424733880939e+01, 0.1425500044444135e+01, 0.1021162532677627e+01, 0.8999999270989295e+00, 0.8371987758218183e+00, 0.7469639623683311e+00, 0.6494523294995699e+00, 0.4251816116188064e+00, 0.2881024603279346e+00, 0.2247792687845145e+00, 0.8318936570759092e-01, 0.1886599847837073e-01, 0.1029977709531541e-01, 0.8015530498900421e-03, 0.8380373580927862e-22, 0.1621276583815588e-04, 0.5889624784943332e-05, 0.1715509223328114e+01, 0.2616454581976124e-17, 0.2563506401168800e-03, 0.3048038004625151e-03, 0.1559424733880939e+01, 0.5629222562364776e-20, 0.1824080863602310e-03, 0.8021857862878124e-04, 0.1425500044444135e+01, 0.1078233452249475e-15, 0.2986624164553499e-03, 0.4306384056300427e-03, 0.1021162532677627e+01, 0.2425965640253176e-21, 0.3415477187080772e-04, 0.2730971166672021e-04, 0.8999999270989295e+00, 0.1110716154551185e-26, 0.2300966623107952e+02, 0.2300966623107953e+02, 0.9800000000000000e+02, 0.2877551020408163e+01, 0.3061224489795918e+01, 0.5341669529353412e-07, 0.7469639623683311e+00, 0.1147369787651374e-23, 0.3199917396615717e-04, 0.1114773640824220e-03, 0.6494523294995699e+00, 0.1324417262847731e-16, 0.4143040953885707e-03, 0.3815591968277185e-03, 0.4251816116188064e+00, 0.4294702353329713e-20, 0.5586157832836348e-03, 0.4985099391803953e-03, 0.2881024603279346e+00, 0.3449343154346125e-19, 0.3018840928486242e-04, 0.4804099248496092e-04, 0.2247792687845145e+00, 0.8223054085459871e-16, 0.6535536077614123e-03, 0.5704051254743707e-03, 0.8318936570759092e-01, 0.1269822990867506e-16, 0.2750660948852983e-03, 0.2439371709019565e-03, 0.1886599847837073e-01, 0.2261821987449685e-25, 0.1693307421164339e-03, 0.1879460762504110e-03, 0.1029977709531541e-01, 0.2207699818464282e-23, 0.8362752373159153e-06, 0.5564443135649789e-05, 0.8015530498900421e-03, 0.1142002837193949e-22, 0.2005722186927794e-03, 0.1790331282576109e-02},
			info: 0,
		},
		{
			z:    []float64{0.7180522604364612e+00, 0.1926580568606098e+00, 0.9998274874952506e-01, 0.5945840846677781e+00, 0.6814004826706310e-01, 0.6532586662527488e+00, 0.6162535781196632e+00, 0.6323752897874746e+00, 0.6913197528064650e+00, 0.8218961980054600e+00, 0.2391583485552943e+00, 0.6893465701234240e+00, 0.8646254741258573e+00, 0.4399082441653793e+00, 0.7657207697437258e+00, 0.6442902814401660e+00, 0.2997828758931008e+00, 0.3311754429773528e+00, 0.4602753981904284e-01, 0.2937498413335546e+00, 0.9438033090912972e+00, 0.6689413829611792e+00, 0.3690605709328096e+00, 0.1297746871929878e+00, 0.5761771518385747e+00, 0.9470476076636891e+00, 0.6065387228873672e+00, 0.3279182193602236e+00, 0.7521029317260848e+00, 0.3448791591170053e-01, 0.8260379882894462e+00, 0.3381289908082724e+00, 0.6350192899014672e+00, 0.6181098759446546e+00, 0.7961594478307139e+00, 0.9929813242449080e+00, 0.4678192142649594e+00, 0.8317252674327552e+00, 0.1301328932823079e+00, 0.9710090231649569e+00, 0.1522128889234372e+00, 0.8594558260240532e+00, 0.6013715693159983e+00, 0.8742383713574672e+00, 0.7309721379377087e+00, 0.5021385621750218e+00, 0.9190365975533138e+00, 0.8027800227596282e-01, 0.1149439815205374e+00, 0.3491372322884074e+00, 0.3638810075690350e+00, 0.7397316797734275e+00, 0.6765505976038609e+00, 0.6412509132149137e+00, 0.6636796276324973e+00, 0.2769301898862327e+00, 0.6293983439129569e+00, 0.7661949976862148e+00, 0.1787796367414251e+00, 0.2134901744318570e+00, 0.8213730837735125e+00, 0.5732135541056227e+00, 0.8956530012694059e+00, 0.5095442527120628e+00, 0.8428538487299666e+00, 0.6837140150023939e+00, 0.1256023460764972e+00, 0.1828667371827231e-01, 0.3922785557614588e+00, 0.9906794824083128e+00, 0.6528984219237121e+00, 0.2175815548938115e+00, 0.5215238469901794e+00, 0.6874084104558049e+00, 0.4144717036123762e+00, 0.5901719497674505e+00, 0.3178975712304457e+00, 0.7061784208820305e+00, 0.6028763180221938e+00, 0.6012903376000152e+00, 0.6812913355997376e+00, 0.7728807542206401e+00, 0.4888911469874758e+00, 0.1012713084879602e+00},
			n:    21,
			zOut: []float64{0.2588311953706596e+01, 0.2370344164465735e+01, 0.2152772432177807e+01, 0.2121020324458281e+01, 0.1935798000017230e+01, 0.1639966751245229e+01, 0.1447991497477954e+01, 0.1303796318075928e+01, 0.1122178474206467e+01, 0.1028684266176079e+01, 0.9663785372526962e+00, 0.8401839996503420e+00, 0.6468021461987292e+00, 0.5148128374433424e+00, 0.4081393344645305e+00, 0.3744044838513741e+00, 0.1951863401557736e+00, 0.1222936930600034e+00, 0.3610852087288927e-01, 0.2215175692702855e-02, 0.1055251342481684e-03, 0.5816113682013476e-24, 0.3490790538048844e-06, 0.5279190866861155e-06, 0.1447991497477954e+01, 0.6749793798367842e-21, 0.1187240552619153e-05, 0.8669686727541259e-06, 0.1303796318075928e+01, 0.2818270319791895e-16, 0.1425495821091504e-08, 0.4238958678865879e-08, 0.1122178474206467e+01, 0.1974523135472784e-17, 0.8140039386639262e-06, 0.3278912117172865e-06, 0.1028684266176079e+01, 0.3004196424688596e-17, 0.2459469608149132e-06, 0.1027360875624831e-06, 0.9663785372526962e+00, 0.8319175140295936e-21, 0.2181749477578394e+02, 0.2181749477578394e+02, 0.1040000000000000e+03, 0.3040816326530612e+01, 0.3846153846153846e+01, 0.3006532860022149e-05, 0.6468021461987292e+00, 0.2129924444096732e-28, 0.3097317061779137e-05, 0.3301766384461962e-05, 0.5148128374433424e+00, 0.5841522899747606e-17, 0.6589564866042633e-05, 0.8825655271690012e-05, 0.4081393344645305e+00, 0.5522026336547083e-29, 0.6929345622607317e-04, 0.1121666398022498e-03, 0.3744044838513741e+00, 0.4071868297361024e-19, 0.4964849153130707e-03, 0.6240579194345771e-03, 0.1951863401557736e+00, 0.3096999756315481e-22, 0.1538823843362176e-02, 0.5557701536471033e-03, 0.1222936930600034e+00, 0.9508642973366516e-17, 0.9344625728185554e-03, 0.1303932473154565e-02, 0.3610852087288927e-01, 0.2671050208464689e-19, 0.5349955893117279e-02, 0.9991286391872665e-02, 0.2215175692702855e-02, 0.2465190328815662e-30, 0.2737788875312504e-02, 0.2378589653250575e-02, 0.1055251342481684e-03, 0.0000000000000000e+00, 0.1487915832966980e-01, 0.9945584373260818e-02},
			info: 0,
		},
		{
			z:    []float64{0.2443960928887767e+00, 0.5306690011327519e+00, 0.8844980893594125e+00, 0.8505050348613927e+00, 0.1179365025679910e+00, 0.6787232231004032e+00, 0.4781425309482933e+00, 0.8596658025119301e+00, 0.7267566040500681e+00, 0.7988986860541384e+00, 0.5066697395752245e+00, 0.9256193759419001e+00, 0.1614174453766865e+00, 0.7938782145253169e+00, 0.3207377639351938e+00, 0.5936236877473402e+00, 0.5294512496859689e+00, 0.3341300959064973e+00, 0.8739309914591177e+00, 0.1643518115289022e-01, 0.9695643985202173e+00, 0.9508789318813814e+00, 0.6599781146531454e+00, 0.8013437877388673e+00, 0.1929493765752847e+00, 0.7655052376601326e+00, 0.6319580998438735e+00, 0.6831519307583876e+00, 0.3834927697748418e+00, 0.7094548193343077e+00, 0.7020510393185050e+00, 0.6520146205437408e+00, 0.6562814689912274e+00, 0.5240471964184201e+00, 0.2362557689024753e+00, 0.4347621644220048e+00, 0.1660979681267404e+00, 0.2422730270980831e+00, 0.7891782502389337e+00, 0.5007812255281535e+00, 0.4444633531981611e+00, 0.3026778693869716e+00, 0.9990331636150707e-02, 0.4097233632314936e+00, 0.9940240633376396e+00, 0.1626467187830966e+00, 0.3461071508544706e+00, 0.6476433090511620e+00, 0.7204399230581238e-01, 0.6885344672911693e+00, 0.1941295874559149e+00, 0.5883115695765473e+00, 0.7359955011333836e+00, 0.4277388472789430e+00, 0.4017294206583021e+00, 0.7838762945853397e+00, 0.9128155969033611e+00, 0.4856080402782125e+00, 0.4763260467232836e+00, 0.5586410048770161e+00, 0.8031586091432461e+00, 0.6068401118620381e-01, 0.9247455618544214e+00, 0.5857147883918616e+00, 0.7467203590494284e+00, 0.1453563221196804e+00, 0.2297646010524917e+00, 0.8950965613541395e+00, 0.3335127494359967e+00, 0.9671078623732762e+00, 0.8023166958816280e+00, 0.1059134943829219e+00, 0.7122485680883537e+00, 0.3211310961264330e+00, 0.1626983074059779e+00, 0.4042697895935750e+00, 0.4799615975845870e+00, 0.3426607739026810e-02, 0.5147088911567483e+00, 0.7533023355653040e+00, 0.9693400349159813e+00, 0.7251923493536844e+00, 0.9132309180417886e-01, 0.4001770994110867e+00},
			n:    21,
			zOut: []float64{0.2574587707602998e+01, 0.2548516143888565e+01, 0.2328646361106399e+01, 0.2152335028165169e+01, 0.1900054508136636e+01, 0.1821843561986577e+01, 0.1641395708303600e+01, 0.1593513640807486e+01, 0.1254245828754644e+01, 0.1184500984874290e+01, 0.9407518766665146e+00, 0.8321813659237318e+00, 0.6961683120499664e+00, 0.5315544763312857e+00, 0.4082285577131941e+00, 0.4022023658419822e+00, 0.2892256147411429e+00, 0.1412172326671640e+00, 0.7919841586115514e-01, 0.2055312514153672e-02, 0.1458583715266580e-03, 0.5995342879679690e-28, 0.2082901809770958e-07, 0.9383395476751434e-08, 0.1641395708303600e+01, 0.1321835131711384e-15, 0.1191741688360734e-04, 0.3695487724817346e-05, 0.1593513640807486e+01, 0.5607617048340495e-15, 0.5232768419257040e-05, 0.4082864434107817e-05, 0.1254245828754644e+01, 0.1850697850686327e-14, 0.2071860298001858e-05, 0.2753629328588578e-05, 0.1184500984874290e+01, 0.3549874073494553e-29, 0.9441520198112501e-05, 0.5871348343515868e-05, 0.9407518766665146e+00, 0.3656881477565973e-23, 0.2332256886230818e+02, 0.2332256886230818e+02, 0.1000000000000000e+03, 0.2984126984126984e+01, 0.3000000000000000e+01, 0.1218265160373847e-03, 0.6961683120499664e+00, 0.3847797347882626e-19, 0.2944801840696127e-03, 0.4565874591840146e-03, 0.5315544763312857e+00, 0.2366582715663035e-29, 0.5951942600827748e-04, 0.7082415940475930e-04, 0.4082285577131941e+00, 0.1744834104604043e-24, 0.4087064324555941e-04, 0.4051735857075661e-04, 0.4022023658419822e+00, 0.5055621275172978e-20, 0.1443306547898801e-05, 0.5013891517830428e-06, 0.2892256147411429e+00, 0.1446770900175336e-26, 0.7439355829951562e-04, 0.1741194257531676e-03, 0.1412172326671640e+00, 0.3448013628188340e-14, 0.2774107535050150e-02, 0.3338675080262731e-02, 0.7919841586115514e-01, 0.2413026984508734e-18, 0.7157138061395222e-03, 0.6803125903293225e-03, 0.2055312514153672e-02, 0.3994185778863994e-23, 0.5722078415717462e-03, 0.5910838594703896e-03, 0.1458583715266580e-03, 0.1183291357831518e-27, 0.8602246198024520e-03, 0.1402897408133881e-02},
			info: 0,
		},
		{
			z:    []float64{0.6756942057876089e+00, 0.6579919202857232e+00, 0.8168130848194743e+00, 0.2483963371195050e+00, 0.2866280219451008e+00, 0.7172808579862799e-03, 0.2411898945446896e+00, 0.5534360794587689e-02, 0.3878248310175912e+00, 0.4256983490856848e+00, 0.7118512164413080e+00, 0.8762976594477806e+00, 0.6850772131038755e+00, 0.4603991232581716e+00, 0.6546622425118448e+00, 0.3103854094723100e+00, 0.7913005955185648e+00, 0.4864788053022765e+00, 0.6214717223070496e+00, 0.7079028836241165e+00, 0.2446362485461124e+00, 0.3062353821816216e+00, 0.8766788389923219e-01, 0.4381267847541642e+00, 0.4459912790878073e-01, 0.9432264671740954e+00, 0.3830574008366420e+00, 0.3381025391982955e+00, 0.6585260539196298e+00, 0.1004337858849739e+00, 0.9320890528134474e+00, 0.3322930037041504e+00, 0.7282334590214989e-02, 0.8367863647557111e+00, 0.9736104021592913e+00, 0.1613168640897539e+00, 0.6409545774446285e+00, 0.8441218931570360e-01, 0.9793937208019250e+00, 0.8303681522642254e+00, 0.1022316826967604e+00, 0.3021608972738009e+00, 0.1238379032543057e+00, 0.3839961366681232e+00, 0.8871553342676223e+00, 0.2909019998361990e+00, 0.7955411369197152e-01, 0.6476593477808288e+00, 0.1739219408315845e+00, 0.6498034164943688e+00, 0.3405210004266120e+00, 0.5459816771585898e+00, 0.2574285667818188e-01, 0.8691155035114945e+00, 0.4320450067500660e+00, 0.6967311316847606e+00, 0.9186339034853966e+00, 0.9491091410944081e+00, 0.2046793386144015e+00, 0.4114169869296881e+00, 0.9170199522695309e+00, 0.1514288291023066e+00, 0.2203325773257230e+00, 0.7891126546748299e+00, 0.4738113210301682e+00, 0.6729225031084131e+00, 0.2382373075875677e+00, 0.4180455054749687e+00, 0.9185761229203592e+00, 0.8030412579733187e+00, 0.4789612026295414e+00, 0.5038982716350845e+00, 0.9959829239394621e+00, 0.3087119711461705e-01, 0.6615773194242809e+00, 0.4461280013353816e+00, 0.1982838436639324e+00, 0.7583555141388076e+00, 0.7946309632008934e+00, 0.6749216068143323e+00, 0.9948312782688508e+00, 0.2854132428873886e+00, 0.8405396060835862e+00, 0.3412432527462372e-01},
			n:    21,
			zOut: []float64{0.2393193272196224e+01, 0.2018136210599260e+01, 0.1995725670346335e+01, 0.1962470040389077e+01, 0.1880885332732760e+01, 0.1640866198530974e+01, 0.1381237309251015e+01, 0.1349378962361067e+01, 0.9454836406584929e+00, 0.7636867441308103e+00, 0.6820433637129892e+00, 0.6091604279437030e+00, 0.5856982990037397e+00, 0.4957297472312800e+00, 0.2553870576469542e+00, 0.2360257010865909e+00, 0.1371029529316844e+00, 0.8102847375475340e-01, 0.5521596879227348e-01, 0.9135542089709688e-02, 0.7426079491367038e-04, 0.1040308236944517e-14, 0.7301351780522185e-10, 0.7774740781328628e-10, 0.1381237309251015e+01, 0.2169367489357782e-29, 0.6020076428884570e-10, 0.5781767346304832e-10, 0.1349378962361067e+01, 0.1885108385497271e-16, 0.6878646083705261e-10, 0.6530919047819172e-10, 0.9454836406584929e+00, 0.5778430642028005e-17, 0.9207835097245334e-10, 0.7637719283517905e-10, 0.7636867441308103e+00, 0.1387778780781446e-16, 0.1259777267883337e-09, 0.1656656563258317e-09, 0.6820433637129892e+00, 0.0000000000000000e+00, 0.1947766517618461e+02, 0.1947766517618461e+02, 0.1070000000000000e+03, 0.3197278911564626e+01, 0.3738317757009346e+01, 0.5169611983503601e-09, 0.5856982990037397e+00, 0.2460562659624748e-15, 0.4628473083370263e-08, 0.9345989594028561e-08, 0.4957297472312800e+00, 0.2615714135774493e-18, 0.2264320058754590e-06, 0.2929148984854635e-06, 0.2553870576469542e+00, 0.1029104326196136e-18, 0.2640422579955958e-07, 0.2108896012270181e-07, 0.2360257010865909e+00, 0.3005450327091600e-17, 0.6822175517251369e-08, 0.8519370047734521e-08, 0.1371029529316844e+00, 0.3395257336071235e-24, 0.2687435952714477e-08, 0.2177398712709520e-08, 0.8102847375475340e-01, 0.0000000000000000e+00, 0.5343374415472690e-09, 0.5384901342094013e-09, 0.5521596879227348e-01, 0.2914620163738177e-24, 0.6457803167637477e-08, 0.2258738450739571e-07, 0.9135542089709688e-02, 0.5326388832052272e-25, 0.9576323906120589e-06, 0.3243967823107194e-05, 0.7426079491367038e-04, 0.3164925372525892e-18, 0.6752170208452239e-05, 0.3302115586194402e-04},
			info: 0,
		},
		{
			z:    []float64{0.1859840927709887e+00, 0.1326482709507555e+00, 0.1830793816152183e+00, 0.4955347339015181e+00, 0.7174832323677909e+00, 0.1239131234434754e-01, 0.8960455455727024e+00, 0.1883195594308053e+00, 0.3701760109540279e+00, 0.5189832590024491e+00, 0.8746518205444311e+00, 0.5127960794741733e+00, 0.2715029219143491e+00, 0.3130984190329523e+00, 0.6427011945330067e+00, 0.2728531634457195e+00, 0.4849915812764480e+00, 0.4474855195890282e-01, 0.1232624016522375e+00, 0.4054507321320544e+00, 0.8168182420396084e+00, 0.5779650563825861e+00, 0.3336575737681243e+00, 0.4793345429794458e+00, 0.3323762775436829e+00, 0.4013874798394268e-01, 0.3250840079396182e+00, 0.3339528328323188e+00, 0.6534745789256358e+00, 0.1592419685588734e+00, 0.4882641395765897e-01, 0.3378473153079410e+00, 0.1498873352536064e+00, 0.9366519936638189e+00, 0.8628314489246045e+00, 0.9035188141748132e+00, 0.3376603415628452e+00, 0.8754867721780122e+00, 0.4314313357227496e+00, 0.6689080802643724e+00, 0.7427192946427073e+00, 0.8408966332922378e+00, 0.3923135283693152e+00, 0.9856346160535667e+00, 0.5253798946684216e+00, 0.4589666030610524e+00, 0.3627813920321372e+00, 0.6650592661455799e+00, 0.5488555419923284e+00, 0.8723519563008553e+00, 0.5759829964509324e+00, 0.8314551283704829e+00, 0.8033910216639006e-01, 0.5533270198466882e+00, 0.4579854660280547e-01, 0.8805162884999327e+00, 0.7798601625088321e+00, 0.3998529604552336e+00, 0.2290634020126769e-01, 0.4087780821380217e+00, 0.8084384129630373e-01, 0.1651045317338867e+00, 0.5138876592921692e+00, 0.1502237554834691e+00, 0.3139451210283801e+00, 0.4951417832103121e+00, 0.4305502836911147e+00, 0.4221379680427187e+00, 0.6629276271381638e+00, 0.6032939209531929e+00, 0.7486429236694788e+00, 0.4603187644190643e+00, 0.9910999698454684e+00, 0.9830579802970648e+00, 0.1327788029232968e+00, 0.7906300391462903e+00, 0.4870686063503010e+00, 0.9107501765395865e-01, 0.5847999740468798e+00, 0.5237055378474927e+00, 0.1949033413503392e+00, 0.6706570830152104e+00, 0.4839742423553839e+00, 0.6933746141255293e+00},
			n:    21,
			zOut: []float64{0.2594702985921864e+01, 0.1963381202215380e+01, 0.1899121104965353e+01, 0.1852857657580134e+01, 0.1352179381057759e+01, 0.1316420368068040e+01, 0.1223907530772361e+01, 0.1104722051320913e+01, 0.1042814535040686e+01, 0.9392359492223399e+00, 0.5879944547218380e+00, 0.5343064059199153e+00, 0.3616892068472140e+00, 0.3474187565935542e+00, 0.3132932870269020e+00, 0.2394766491234008e+00, 0.1924620369224269e+00, 0.5796610832720670e-01, 0.4900919199482741e-01, 0.2087594658763894e-01, 0.6804292520862544e-03, 0.2186403793390500e-19, 0.2668268315891067e-16, 0.1216320938093713e-16, 0.1223907530772361e+01, 0.3077822220048293e-18, 0.9690290545884895e-18, 0.1677534925257981e-18, 0.1104722051320913e+01, 0.3294338264035954e-18, 0.1270509199570000e-13, 0.4678488073918506e-13, 0.1042814535040686e+01, 0.2740439675865377e-23, 0.1047809983111396e-11, 0.5781643460797313e-12, 0.9392359492223399e+00, 0.3379215452805658e-16, 0.8350045830190690e-10, 0.5730922211948388e-10, 0.5879944547218380e+00, 0.6938893903907228e-17, 0.1799451523948185e+02, 0.1799451523948184e+02, 0.1150000000000000e+03, 0.3578231292517007e+01, 0.6086956521739131e+01, 0.8630088713732289e-10, 0.3616892068472140e+00, 0.1664229231319872e-20, 0.9644621511794884e-10, 0.9354199679969641e-10, 0.3474187565935542e+00, 0.0000000000000000e+00, 0.1015403419286529e-09, 0.1046840761330470e-09, 0.3132932870269020e+00, 0.6366107505133565e-27, 0.3071873418074744e-11, 0.5342885575500614e-12, 0.2394766491234008e+00, 0.2599072586631376e-18, 0.1837414559126179e-09, 0.2303923049314304e-09, 0.1924620369224269e+00, 0.3400814535759169e-17, 0.9192916981159684e-08, 0.4803632388224207e-08, 0.5796610832720670e-01, 0.0000000000000000e+00, 0.1565526746702558e-05, 0.8408463868497472e-06, 0.4900919199482741e-01, 0.7888609052210118e-30, 0.1903794116022181e-04, 0.1014775117962413e-04, 0.2087594658763894e-01, 0.9734740470109229e-22, 0.4676468030348868e-06, 0.4248047233272650e-05, 0.6804292520862544e-03, 0.3161038140634776e-18, 0.1818811163938070e-02, 0.3653457173655013e-02},
			info: 0,
		},
		{
			z:    []float64{0.2029039404458555e-01, 0.4529725864625287e+00, 0.6412066909421453e+00, 0.6139288440336396e+00, 0.3661028502807454e+00, 0.9178254749803582e+00, 0.4307551173824371e+00, 0.4985808870906128e+00, 0.8171615510934167e+00, 0.8279058065788196e+00, 0.3035261047197422e+00, 0.7877147223127390e+00, 0.7688130520483442e+00, 0.4834409462671818e-01, 0.3121145202501183e+00, 0.5269232357652766e+00, 0.7600633371694686e+00, 0.5251092877920901e+00, 0.8263235554592012e+00, 0.6894073332954654e+00, 0.3521429062439027e+00, 0.6447189811681343e+00, 0.1105592548668879e+00, 0.1049454508922576e+00, 0.7381247076050562e+00, 0.7320584094686099e+00, 0.7003114013559573e+00, 0.9638534557041758e+00, 0.5727311775052889e+00, 0.3930768970870516e+00, 0.7640649329763241e+00, 0.3221407324893386e+00, 0.9725443946255905e+00, 0.6922592425787558e+00, 0.3449642467056757e-01, 0.1384627086161467e+00, 0.4398391723165511e+00, 0.8406456486574913e+00, 0.4126144976348992e+00, 0.5351779876797247e-01, 0.6669497026260232e+00, 0.5746288223886410e+00, 0.4589594741804619e+00, 0.7484833637036891e+00, 0.3274476209575072e-01, 0.7067960763848782e+00, 0.8759136475280368e+00, 0.3284828537280239e+00, 0.2536109230678856e+00, 0.8799264273691800e+00, 0.8159607815416694e+00, 0.5729670098854700e-01, 0.8884200881604851e+00, 0.5249116128877267e+00, 0.6498061599007876e-01, 0.6075292761444269e+00, 0.2428201413403417e+00, 0.2668460043818345e+00, 0.1429723702275580e+00, 0.6942338078813951e+00, 0.9271225003121289e+00, 0.8931957344289477e+00, 0.6420336733834545e+00, 0.3786225606645894e+00, 0.9421368123010410e-01, 0.9476439119845459e+00, 0.6497389318978118e+00, 0.3841806572586668e+00, 0.1329188317035728e+00, 0.6545507077812880e+00, 0.4430247695845350e+00, 0.1753161904868739e+00, 0.2413409779703910e+00, 0.1476409499735053e+00, 0.1179852848482287e+00, 0.3704631210033814e+00, 0.1967445635592234e+00, 0.2100970549835486e+00, 0.5744473538919733e+00, 0.1093206314301038e+00, 0.5132572818416121e+00, 0.4964557538937117e+00, 0.7518888363275744e-02, 0.9720615373003737e+00},
			n:    21,
			zOut: []float64{0.2563191374817756e+01, 0.2359364796169898e+01, 0.2296884847589588e+01, 0.2056943321577226e+01, 0.1949112826206492e+01, 0.1629835138023691e+01, 0.1562766390641567e+01, 0.1422208590143927e+01, 0.1368481726990307e+01, 0.1150114414099739e+01, 0.8482773777570239e+00, 0.7665061168435710e+00, 0.7135198036652551e+00, 0.3486547062556205e+00, 0.2456691453437445e+00, 0.2203009969871996e+00, 0.1404417091680581e+00, 0.1160909181211330e+00, 0.2112854362928769e-01, 0.5069559443178203e-02, 0.5650407111696261e-03, 0.3717112585401408e-26, 0.2316306425767847e-08, 0.1632477182259275e-08, 0.1562766390641567e+01, 0.6446192664231602e-24, 0.9634960729667980e-08, 0.6524712961691138e-08, 0.1422208590143927e+01, 0.1932709217791479e-28, 0.4932058767625490e-07, 0.7860293908453283e-07, 0.1368481726990307e+01, 0.1238511621196989e-27, 0.2778987342481399e-06, 0.5558918854266694e-06, 0.1150114414099739e+01, 0.2156683454185074e-19, 0.1662595987219070e-04, 0.2580299679298412e-04, 0.8482773777570239e+00, 0.7914569980025776e-17, 0.2178512734418544e+02, 0.2178512734418543e+02, 0.1000000000000000e+03, 0.2934240362811791e+01, 0.1000000000000000e+01, 0.1715129124192563e-05, 0.7135198036652551e+00, 0.1613909508212798e-19, 0.2836076293001813e-04, 0.1595224464977716e-04, 0.3486547062556205e+00, 0.3352658847189300e-29, 0.8812160938383347e-04, 0.1178320764067675e-03, 0.2456691453437445e+00, 0.0000000000000000e+00, 0.5251568647637462e-04, 0.4480459044047029e-04, 0.2203009969871996e+00, 0.6471183777709004e-25, 0.1114279427921990e-09, 0.2270576453660371e-10, 0.1404417091680581e+00, 0.2855676476900063e-27, 0.2886261726979772e-05, 0.5470704056451658e-06, 0.1160909181211330e+00, 0.3526208246337923e-27, 0.3588687949078309e-02, 0.5383294574404876e-02, 0.2112854362928769e-01, 0.2896602354076833e-17, 0.1732996915625309e-02, 0.1499310228751921e-02, 0.5069559443178203e-02, 0.1003594455767978e-17, 0.5081816639203317e-03, 0.1080637708244138e-02, 0.5650407111696261e-03, 0.0000000000000000e+00, 0.5066428376081889e-02, 0.3247326449492487e-02},
			info: 0,
		},
		{
			z:    []float64{0.7177806115181969e+00, 0.9619042524298227e+00, 0.9756290650460009e-01, 0.2695657197860104e+00, 0.5092382615580680e+00, 0.2216490915715645e+00, 0.7713962041593733e+00, 0.9019626308054164e+00, 0.1009043823109016e+00, 0.7164892891763703e+00, 0.6972336574704985e+00, 0.5604456603447616e-01, 0.4658944020177349e+00, 0.4677429402719713e+00, 0.9395972368396054e+00, 0.6567060928508761e-01, 0.6711470087762763e+00, 0.3572615865847095e-01, 0.7841795004355456e-01, 0.2100489732227784e+00, 0.1528790147771258e+00, 0.2944728557463859e+00, 0.6929796239262105e+00, 0.5593689694083593e+00, 0.5600220347177359e+00, 0.6885800745012647e+00, 0.1441376161827218e+00, 0.2468678391528564e+00, 0.2219056216366488e+00, 0.7828887069679423e+00, 0.1781536622262658e-01, 0.9712383516530864e+00, 0.1031071887927941e+00, 0.5428204596104211e+00, 0.9920375605622551e+00, 0.2231614148449633e+00, 0.1774197365731667e+00, 0.1667803739895276e+00, 0.2293956181863850e+00, 0.8453959149417756e+00, 0.2211895679989639e+00, 0.5610288802043042e+00, 0.3749535737701304e-01, 0.7418144847444434e+00, 0.2593245755300763e+00, 0.5358881543370908e-01, 0.9118629582226685e+00, 0.6483736004795430e+00, 0.7595140552166778e+00, 0.2981894989055883e+00, 0.6091779707233183e+00, 0.7798268953767704e+00, 0.1682114110436058e+00, 0.3801835867597201e+00, 0.6380508025759659e+00, 0.7892946000460455e-01, 0.7607559424299619e+00, 0.3271484239841950e+00, 0.7700240256278714e+00, 0.5894107219393652e+00, 0.6207117138108632e+00, 0.4725019923733027e+00, 0.3616574024313524e+00, 0.6744122205664662e+00, 0.5489632027065386e+00, 0.1532930834122788e+00, 0.1692473010839316e+00, 0.7515070367293223e+00, 0.1186783750638719e+00, 0.1028479850139022e+00, 0.3468642488228025e+00, 0.5278752643808988e+00, 0.3849250707234438e+00, 0.7889573536656195e+00, 0.4738712276132427e+00, 0.7058571493185843e+00, 0.8948240863202030e+00, 0.3494029097065446e+00, 0.4916843795342892e+00, 0.1993321046695898e+00, 0.3650043431665774e+00, 0.2403686388300026e+00, 0.5474155466298688e+00, 0.6157434374908201e+00},
			n:    21,
			zOut: []float64{0.1955717973339200e+01, 0.1915826728710446e+01, 0.1780143898982495e+01, 0.1751081083154649e+01, 0.1679171531306522e+01, 0.1398798787154148e+01, 0.1289579288306139e+01, 0.1171599632002814e+01, 0.9950381700653156e+00, 0.9008548767119643e+00, 0.7944341357698450e+00, 0.7347990252582277e+00, 0.4053391427425183e+00, 0.3415476013970046e+00, 0.2769292856200630e+00, 0.2086988709391624e+00, 0.7594980741000955e-01, 0.5721582845751132e-01, 0.5152297043277422e-01, 0.6186350115045150e-02, 0.5774958135565486e-05, 0.6722096271457334e-14, 0.6635533119216432e-06, 0.1048982691930868e-05, 0.1289579288306139e+01, 0.0000000000000000e+00, 0.9172850657907766e-08, 0.6527900227304331e-08, 0.1171599632002814e+01, 0.0000000000000000e+00, 0.3386861611239913e-06, 0.1115072165646295e-06, 0.9950381700653156e+00, 0.1044578056257455e-24, 0.5793689136913799e-11, 0.1625904890507595e-10, 0.9008548767119643e+00, 0.8883435102679349e-21, 0.3220671712964998e-09, 0.8453946299986560e-09, 0.7944341357698450e+00, 0.1653452457343241e-26, 0.1779044076283400e+02, 0.1779044076283399e+02, 0.1120000000000000e+03, 0.3267573696145125e+01, 0.3571428571428572e+01, 0.1211740269624600e-06, 0.4053391427425183e+00, 0.5298753945086544e-14, 0.1515410194583574e-06, 0.2069357347527298e-06, 0.3415476013970046e+00, 0.0000000000000000e+00, 0.2579333883676927e-05, 0.8435987645777294e-05, 0.2769292856200630e+00, 0.2465190328815662e-30, 0.1658852164453815e-08, 0.3207072615633299e-09, 0.2086988709391624e+00, 0.2017246551328580e-19, 0.6091457019925236e-05, 0.6297972233022177e-05, 0.7594980741000955e-01, 0.2553131970482608e-15, 0.2736366650719638e-04, 0.2348362771705736e-04, 0.5721582845751132e-01, 0.1141311335899276e-23, 0.6030351005218485e-04, 0.3385365250818500e-04, 0.5152297043277422e-01, 0.0000000000000000e+00, 0.4442856906399686e-04, 0.3980460155412817e-04, 0.6186350115045150e-02, 0.2711709361697228e-30, 0.1322896557202734e-03, 0.3209571887096705e-03, 0.5774958135565486e-05, 0.0000000000000000e+00, 0.9248295446812529e-03, 0.1317994001905306e-02},
			info: 0,
		},
		{
			z:    []float64{0.7669589114699304e-01, 0.2959120838012549e+00, 0.8617075527885690e+00, 0.3774472194278293e+00, 0.7430204868341380e+00, 0.3337487204025202e-01, 0.2115072947611989e+00, 0.8195300149220009e+00, 0.4610153087687133e+00, 0.1049580467270205e+00, 0.6553761031646719e+00, 0.2044425657813468e+00, 0.1419536829746254e+00, 0.8338399989758465e+00, 0.9917105482281005e+00, 0.3698082853107363e+00, 0.5656210986926330e+00, 0.5849265049217818e+00, 0.9018145207330325e+00, 0.7089264704350354e+00, 0.2562038341655152e+00, 0.1524378342317656e-01, 0.9206210351037002e+00, 0.1877259832276945e+00, 0.3637451600309541e+00, 0.9979742594017312e+00, 0.8919698496914060e+00, 0.3716780472994426e+00, 0.3142548907151147e+00, 0.4170439734847903e+00, 0.3197415298896636e+00, 0.7285365914169133e+00, 0.5784968714882697e+00, 0.6831919512327895e+00, 0.6057174197537164e+00, 0.6413693987683566e-01, 0.8827799219498907e+00, 0.4192891003797022e+00, 0.6040167945472836e+00, 0.6469271762278970e+00, 0.7359461974470041e+00, 0.5539488596393002e+00, 0.4023966166720030e+00, 0.9915288950117843e+00, 0.3043860170301459e+00, 0.4917889743094436e+00, 0.1179472550216760e+00, 0.9125202427370891e+00, 0.8786441005384636e+00, 0.4634730997209351e+00, 0.1080276946115265e+00, 0.6187727001119313e+00, 0.6709766557170562e+00, 0.3818949179452977e+00, 0.8906457783485427e+00, 0.8010289089804480e+00, 0.6931910498827129e+00, 0.8914072225833563e+00, 0.3822463769530011e+00, 0.5898102885291842e+00, 0.1092049166271940e+00, 0.8496011938807149e+00, 0.2304394290731624e+00, 0.5949006075515944e+00, 0.3290638194228065e+00, 0.1971699984758039e+00, 0.7443144303888384e+00, 0.4653555333092598e+00, 0.8207632269304853e+00, 0.4152482306441556e+00, 0.1689785233215235e+00, 0.1721084345877374e+00, 0.3931596107353640e+00, 0.5403345228237123e+00, 0.6387555392172577e+00, 0.1804790096977364e+00, 0.5783025205396422e+00, 0.7395837791384520e+00, 0.5701597209798811e+00, 0.1952734055752668e+00, 0.9661114908130567e+00, 0.4893113195434146e+00, 0.6562980654604210e+00, 0.1879394621701417e+00},
			n:    21,
			zOut: []float64{0.2390558120678448e+01, 0.2326196956724044e+01, 0.2071676597874671e+01, 0.1995424877356871e+01, 0.1973795189776371e+01, 0.1728273264358535e+01, 0.1492322130169846e+01, 0.1216694895544233e+01, 0.1149617898796123e+01, 0.1007011094632661e+01, 0.9207056739913982e+00, 0.7404938152461292e+00, 0.6082352700464410e+00, 0.5965144149455339e+00, 0.2276577044879657e+00, 0.1776272681811473e+00, 0.1662286875439616e+00, 0.8310804130734020e-01, 0.4604735683437314e-01, 0.1806926893972028e-01, 0.1257133375345320e-01, 0.3600386373101646e-15, 0.6280777483216298e-03, 0.1060573000233681e-02, 0.1492322130169846e+01, 0.1292469707114105e-25, 0.5799349861429450e-06, 0.5584709508928944e-06, 0.1216694895544233e+01, 0.2047280287874868e-18, 0.1299955932331242e-09, 0.8534229075832313e-10, 0.1149617898796123e+01, 0.5088714046955349e-16, 0.6001732451841635e-09, 0.4081989685050919e-09, 0.1007011094632661e+01, 0.5578719308559292e-21, 0.1498995168909837e-07, 0.5625367692005608e-08, 0.9207056739913982e+00, 0.2728742825015995e-21, 0.2094882986118927e+02, 0.2094882986118927e+02, 0.9800000000000000e+02, 0.3004535147392290e+01, 0.4081632653061225e+01, 0.6223967900246754e-04, 0.6082352700464410e+00, 0.4038967834731580e-26, 0.1122996207538654e-05, 0.1754126012223728e-05, 0.5965144149455339e+00, 0.4386207914826929e-19, 0.1334483593164094e-04, 0.6982705884252015e-05, 0.2276577044879657e+00, 0.3148538924591216e-18, 0.9085150621800872e-04, 0.4628114404518947e-04, 0.1776272681811473e+00, 0.1149284743081177e-19, 0.3934684725502468e-05, 0.5044161242313764e-05, 0.1662286875439616e+00, 0.3944304526105059e-30, 0.1149340461723492e-04, 0.8510588625472410e-05, 0.8310804130734020e-01, 0.1540743955509789e-32, 0.1285467894756909e-02, 0.9239857033877289e-03, 0.4604735683437314e-01, 0.2169367489357782e-29, 0.2808730938050547e-03, 0.3455059327639833e-03, 0.1806926893972028e-01, 0.4614836295542919e-28, 0.7683442768713957e-06, 0.1846813906304985e-05, 0.1257133375345320e-01, 0.2538554393001216e-26, 0.4028502005942143e-04, 0.8466712427867894e-04},
			info: 0,
		},
		{
			z:    []float64{0.5568197491282034e+00, 0.7338530212513171e+00, 0.1711270247791036e+00, 0.6696990680288049e+00, 0.1107644593582661e+00, 0.1487844153251054e+00, 0.6221478836712087e+00, 0.3739707210550620e+00, 0.6142936216832375e+00, 0.4504419047617665e+00, 0.1390832371836795e+00, 0.9602056283222130e+00, 0.4128383897877478e+00, 0.6202590221465013e+00, 0.5716294881431405e+00, 0.7009876531280159e+00, 0.8556346552408018e+00, 0.4300043005510307e+00, 0.5625488786064613e+00, 0.5236571943623558e+00, 0.2035297706440273e+00, 0.5324677179555473e+00, 0.9854023908952125e+00, 0.8088580870995794e+00, 0.7902887218423563e+00, 0.4196418643524230e+00, 0.5294914665193529e+00, 0.5947417442754066e-01, 0.2753919335549286e+00, 0.8807083336864044e+00, 0.6016279529290567e+00, 0.4144655693858035e-01, 0.4808953029705748e+00, 0.8506686680287934e-01, 0.2145404015834380e+00, 0.3020133878086849e+00, 0.8967140938263458e+00, 0.5344538485865815e+00, 0.2536120629120923e+00, 0.2552452828610304e+00, 0.9211096671641958e+00, 0.9028760680202415e+00, 0.3005599522329846e+00, 0.1197919551028395e+00, 0.3772579707039786e+00, 0.4157715274081910e+00, 0.2203607291065889e+00, 0.1084188329562713e+00, 0.5463193166123409e+00, 0.8107359194268960e+00, 0.6676190072779817e+00, 0.6711454730587799e+00, 0.6360708014875704e-01, 0.8242026716736568e+00, 0.3687006444230088e+00, 0.2216898043026083e-01, 0.4978612409817640e+00, 0.3144452507777135e+00, 0.3412289603988730e+00, 0.3266626310182044e+00, 0.4581662376370765e+00, 0.6985218227047190e+00, 0.6594479039269319e-01, 0.3938130402504401e+00, 0.9907371819490932e+00, 0.2478499678343852e+00, 0.3565435772734814e+00, 0.8420455744018336e+00, 0.2692247454903065e+00, 0.7327570909626056e+00, 0.1752776205164243e+00, 0.3569190164542581e+00, 0.2122621782757903e+00, 0.2056257129895300e+00, 0.6641092684756426e+00, 0.5386917539530447e+00, 0.8420713652275945e+00, 0.3587226239611645e+00, 0.9561844063661247e+00, 0.9126272291387975e+00, 0.5768542567999925e+00, 0.1972168939311342e+00, 0.5763304496236371e+00, 0.9478337554439876e+00},
			n:    21,
			zOut: []float64{0.2593664459139222e+01, 0.2338172956520739e+01, 0.1783072679058981e+01, 0.1768637196043293e+01, 0.1688808722758796e+01, 0.1535694950136988e+01, 0.1488691522387158e+01, 0.1290354539321546e+01, 0.1187570436840380e+01, 0.1077679082742636e+01, 0.7925423484101771e+00, 0.6549022645335129e+00, 0.6315653598883190e+00, 0.5184441284206909e+00, 0.3791491409939438e+00, 0.3105722740860607e+00, 0.1128189367670211e+00, 0.7036440937731858e-01, 0.4999153992819697e-01, 0.2730914559941030e-01, 0.7241089204639656e-03, 0.7344040350490801e-18, 0.1828200693350630e-07, 0.2226187377323797e-07, 0.1488691522387158e+01, 0.4586948101010510e-17, 0.3930142187345605e-07, 0.5206893736347286e-07, 0.1290354539321546e+01, 0.6658190007940994e-20, 0.1499933495859733e-06, 0.7910043275590841e-07, 0.1187570436840380e+01, 0.2370536763713858e-16, 0.1192422432592963e-05, 0.6065280015592916e-06, 0.1077679082742636e+01, 0.6803770690793804e-23, 0.5360141282897325e-04, 0.3254990522417721e-04, 0.7925423484101771e+00, 0.8404284270509473e-23, 0.2030073020187486e+02, 0.2030073020187486e+02, 0.9700000000000000e+02, 0.2975056689342404e+01, 0.1030927835051546e+01, 0.1762395614510388e-04, 0.6315653598883190e+00, 0.2034005362457094e-16, 0.3178492515415974e-08, 0.1141868824523976e-08, 0.5184441284206909e+00, 0.1631704231140345e-20, 0.1154759308961600e-04, 0.3555504775624820e-05, 0.3791491409939438e+00, 0.9251916923707874e-19, 0.1302644751785016e-04, 0.1874642271188771e-04, 0.3105722740860607e+00, 0.1235671721938193e-25, 0.8697586673438179e-05, 0.7595546859022038e-05, 0.1128189367670211e+00, 0.1175733067685735e-18, 0.2174688695641498e-03, 0.1411064498403114e-03, 0.7036440937731858e-01, 0.5916456789157589e-30, 0.4997004849851516e-03, 0.7625846015345374e-03, 0.4999153992819697e-01, 0.3710108611279200e-20, 0.2898431160817185e-04, 0.2329024376647231e-03, 0.2730914559941030e-01, 0.1593079290931123e-17, 0.5731904499848677e-04, 0.7071520517918782e-04, 0.7241089204639656e-03, 0.1043820749788443e-25, 0.2096656950463827e-03, 0.4915700720935459e-03},
			info: 0,
		},
		{
			z:    []float64{0.2129092067945103e+00, 0.6174796583507793e+00, 0.2209275310088966e+00, 0.1770681976597738e+00, 0.8468210663920229e+00, 0.7900600194799612e+00, 0.8319584509547915e+00, 0.8077183726155964e+00, 0.7964773509287093e+00, 0.2479492923952727e+00, 0.5169395370002006e-01, 0.6312063121285433e+00, 0.5688486960732374e+00, 0.3049379579822397e+00, 0.3779887662042721e+00, 0.4551165943302187e+00, 0.8807128836121972e+00, 0.1333615670826408e+00, 0.6901278826487529e+00, 0.8360350720177558e+00, 0.4190492004641316e+00, 0.9067077556412378e+00, 0.9762584494767094e+00, 0.9959763915912888e+00, 0.8503724779590973e+00, 0.6671116075685212e+00, 0.8841807167789617e+00, 0.6917331852931945e+00, 0.3414398718690443e+00, 0.2211364241075178e+00, 0.8057250135329493e+00, 0.4405475768508057e+00, 0.9321382353755537e+00, 0.6976358515081519e+00, 0.3254605598548291e+00, 0.1144237265035558e+00, 0.8547099978727912e-01, 0.3131314226614652e+00, 0.6940029038044814e+00, 0.2327469211945017e+00, 0.3523896044562020e+00, 0.6443116162172926e+00, 0.5382708884344491e+00, 0.9378966459649967e+00, 0.7274262519928346e+00, 0.4882092650189528e+00, 0.4497457043638882e+00, 0.8803511251742836e+00, 0.1077837527475992e+00, 0.4760275900323754e+00, 0.9725944612716640e+00, 0.4013370227296387e+00, 0.8188706727711300e+00, 0.7549128555022213e+00, 0.3045373312490215e+00, 0.9776368127163177e+00, 0.5516094182757485e+00, 0.8449472155130459e+00, 0.9309695118211208e+00, 0.6985346927019656e+00, 0.3270836933831586e+00, 0.2305805980953167e+00, 0.9135598850313159e+00, 0.2886818807914165e+00, 0.6607756604738726e+00, 0.7992086772120486e+00, 0.9191714969823802e+00, 0.9956222713515444e+00, 0.1646632133753003e+00, 0.8114783440114911e+00, 0.9650128806412454e+00, 0.1382218488636167e-01, 0.8464890579713791e+00, 0.2866238720970538e+00, 0.4485743907736028e-01, 0.5384705384174762e+00, 0.2996807710312754e-01, 0.7020716858392958e+00, 0.6280049430389886e+00, 0.2750308929159380e+00, 0.1648004007672321e-01, 0.8720570971091830e+00, 0.9962373992422624e+00, 0.8022759164194093e-01},
			n:    21,
			zOut: []float64{0.3179557858974794e+01, 0.2694788878259719e+01, 0.2281326093492730e+01, 0.2196001939032204e+01, 0.1828355236773963e+01, 0.1586247682528960e+01, 0.1454772213395289e+01, 0.1286809233158695e+01, 0.1269332746846940e+01, 0.1099355321362049e+01, 0.9971216943768625e+00, 0.8310914612054856e+00, 0.4047131081238095e+00, 0.3869245862172090e+00, 0.3328473924133467e+00, 0.2908704078018756e+00, 0.1880200745280021e+00, 0.5784266149949692e-01, 0.4139216712761701e-01, 0.1651732519409763e-01, 0.3149646326524349e-02, 0.0000000000000000e+00, 0.5227593374295754e-11, 0.1202138413496143e-10, 0.1454772213395289e+01, 0.1393199790273629e-16, 0.1121878363020271e-09, 0.5353121918082171e-10, 0.1286809233158695e+01, 0.8659547037664508e-24, 0.1645814283637519e-08, 0.8601359367531547e-09, 0.1269332746846940e+01, 0.2823844361652584e-22, 0.5929700276768749e-08, 0.1126902625809321e-07, 0.1099355321362049e+01, 0.2358945595467410e-22, 0.2764360194400850e-05, 0.1553001254800959e-05, 0.9971216943768625e+00, 0.4450651748190101e-21, 0.2242703772863968e+02, 0.2242703772863967e+02, 0.1000000000000000e+03, 0.3015873015873016e+01, 0.3000000000000000e+01, 0.2632919892231646e-04, 0.4047131081238095e+00, 0.5820888909768840e-10, 0.2209233784503674e-04, 0.4136287504408153e-04, 0.3869245862172090e+00, 0.6462348535570008e-26, 0.3088379129783557e-03, 0.1528857353804532e-03, 0.3328473924133467e+00, 0.5549599819233453e-15, 0.8893453420684614e-08, 0.8741044673362129e-07, 0.2908704078018756e+00, 0.2289589891313465e-25, 0.5532514209995071e-04, 0.2183152999074702e-03, 0.1880200745280021e+00, 0.6113672015462841e-28, 0.9851023787951559e-03, 0.2769770083415459e-02, 0.5784266149949692e-01, 0.3571962178840741e-24, 0.3360295855593957e-02, 0.3874576703777444e-02, 0.4139216712761701e-01, 0.1555002616371658e-25, 0.3047908004675713e-02, 0.4283880536865950e-02, 0.1651732519409763e-01, 0.1498533807074276e-21, 0.9218134151523294e-06, 0.6993690185531638e-05, 0.3149646326524349e-02, 0.5476587948406352e-25, 0.9434027266146764e-03, 0.1032678112098284e-02},
			info: 0,
		},
		{
			z:    []float64{0.5325973817915391e+00, 0.8075384328404875e+00, 0.2063920576624930e+00, 0.4895050697482526e+00, 0.3224859392283662e+00, 0.4597919629142447e+00, 0.5480240110168966e-01, 0.5228709929272914e+00, 0.9956836568223112e+00, 0.3978480622803672e+00, 0.9124816118920143e-01, 0.3247027355683270e+00, 0.9224658967837341e+00, 0.9984579111484309e+00, 0.8533112174943273e+00, 0.8907006920892125e+00, 0.3659539504394835e+00, 0.5561036191465061e+00, 0.7570794287369270e+00, 0.7781602931894585e+00, 0.8583979385494303e+00, 0.5883490719908564e+00, 0.2958788793535505e+00, 0.8441503484168636e+00, 0.7171074610327245e+00, 0.6324186559935309e+00, 0.6889002812298057e+00, 0.9679449699589238e-01, 0.8071604187540067e+00, 0.5036464295967858e+00, 0.3075050973746345e+00, 0.4354181299061508e+00, 0.5397482510302705e+00, 0.3266316370831254e+00, 0.7127302469392831e+00, 0.7822329043112050e+00, 0.1787867076882754e+00, 0.5059399114486356e+00, 0.1635357104384569e+00, 0.7923616792404551e+00, 0.8527619409672080e+00, 0.6624868870738707e+00, 0.6395827198572863e+00, 0.5045303635733072e+00, 0.5207053335438128e-01, 0.3872927271413512e+00, 0.8590236531191382e+00, 0.8644737695336893e+00, 0.5447620155822054e+00, 0.3420775682173337e+00, 0.1787889858336624e+00, 0.2962040723342502e+00, 0.1197091589896203e+00, 0.2582729465177200e+00, 0.8564224654785235e+00, 0.1003255861450059e+00, 0.5792751794645187e+00, 0.3553877787422808e+00, 0.3515051629979192e+00, 0.7026347035497048e+00, 0.7851727410421285e+00, 0.9451470000488936e+00, 0.1482591966327134e+00, 0.8035088707163867e+00, 0.8018983228501475e-02, 0.5992223740738118e+00, 0.3369698862429809e+00, 0.9971168486495201e+00, 0.7449267587097297e+00, 0.5929357935363798e+00, 0.8575900212486389e+00, 0.8115297023854162e+00, 0.3380359989630263e+00, 0.8328174310878843e+00, 0.8255713836908419e+00, 0.7399149789721748e+00, 0.1725333812438860e+00, 0.6828663265380140e+00, 0.6234367510076171e+00, 0.5628070366267671e+00, 0.8501068267959022e-01, 0.9315797050119701e+00, 0.1842029704669090e+00, 0.9504814303233714e+00},
			n:    21,
			zOut: []float64{0.2923425454171414e+01, 0.2535311958925162e+01, 0.2192080092128743e+01, 0.1925146455902099e+01, 0.1841700496719334e+01, 0.1753280727277585e+01, 0.1582442080550360e+01, 0.1581069533140902e+01, 0.1420113611638973e+01, 0.1070153147168429e+01, 0.9980373856163754e+00, 0.9531244298503998e+00, 0.7077665812605668e+00, 0.5659990522281487e+00, 0.3146390113583965e+00, 0.2516605740660456e+00, 0.1663945631562664e+00, 0.1260010860507715e+00, 0.4255237190875141e-01, 0.7100815132372204e-02, 0.1566331927046113e-03, 0.4122332130840440e-22, 0.4572750287882177e-06, 0.8150029788456509e-06, 0.1582442080550360e+01, 0.6246970091107327e-19, 0.1184058245159526e-05, 0.5957129004529113e-06, 0.1581069533140902e+01, 0.2712893915232508e-21, 0.4684400026881307e-05, 0.9374804625040887e-05, 0.1420113611638973e+01, 0.4814409269321697e-22, 0.2425457953812455e-04, 0.3059673263223507e-04, 0.1070153147168429e+01, 0.4457872866361589e-11, 0.2849702090992769e-05, 0.4079444611357521e-05, 0.9980373856163754e+00, 0.2148022614677702e-19, 0.2295815606144379e+02, 0.2295815606144380e+02, 0.9700000000000000e+02, 0.2981859410430839e+01, 0.2061855670103093e+01, 0.4336961109868880e-06, 0.7077665812605668e+00, 0.3355216099479021e-19, 0.2178215603335118e-03, 0.1596478432727446e-03, 0.5659990522281487e+00, 0.3777948407501058e-19, 0.1732350582817826e-03, 0.1687341431970761e-03, 0.3146390113583965e+00, 0.4215672677501087e-24, 0.8708765796225133e-03, 0.5753505295482449e-03, 0.2516605740660456e+00, 0.8872339250080977e-20, 0.2809442395187102e-02, 0.2330568324677941e-02, 0.1663945631562664e+00, 0.1848892746611746e-31, 0.7424955400801435e-06, 0.1353946060853610e-06, 0.1260010860507715e+00, 0.7099748146989106e-27, 0.5320431162642944e-03, 0.1085271179151602e-03, 0.4255237190875141e-01, 0.2808116620904140e-21, 0.1709028753951538e-02, 0.8528114742759547e-03, 0.7100815132372204e-02, 0.4930380657631324e-31, 0.2904056487242609e-02, 0.3026916823887611e-02, 0.1566331927046113e-03, 0.0000000000000000e+00, 0.2549645590833870e-02, 0.2592045976400073e-02},
			info: 0,
		},
		{
			z:    []float64{0.6141085085904113e+00, 0.3421028526671319e+00, 0.2523050636037050e-01, 0.6894939888024587e+00, 0.1106696902059259e+00, 0.3232796216228777e+00, 0.5026462674631121e+00, 0.4061906185674803e+00, 0.2489870821257593e+00, 0.5655528330838598e+00, 0.8215445521378404e+00, 0.6847122058387792e+00, 0.1058838690465073e+00, 0.3150742731806708e+00, 0.6299058795873502e-01, 0.2792122534089967e+00, 0.6156682980944264e+00, 0.6784924553414444e+00, 0.5548029179057333e+00, 0.9875334352131669e+00, 0.7114388615166174e+00, 0.4680838251435693e+00, 0.9939378516921551e+00, 0.9317334719497590e+00, 0.7166727913665860e+00, 0.9133735504519750e+00, 0.6652039093977162e+00, 0.8687642647047388e+00, 0.5616945762819415e+00, 0.3985759546461669e+00, 0.7431799484571856e+00, 0.7126935860242257e+00, 0.1006984098734091e+00, 0.3341210154632034e+00, 0.8752290131259591e+00, 0.9044047683690323e+00, 0.2748574881470656e+00, 0.9153414825361147e+00, 0.1879700367171477e-01, 0.4139829057070108e+00, 0.6163962814716032e+00, 0.4758310984958897e+00, 0.7283265374189019e-01, 0.4293121596195046e+00, 0.1627494487778676e+00, 0.6477819718948226e+00, 0.1769525517070647e-01, 0.8315284238294101e+00, 0.3951374239126071e-01, 0.8607842403091001e+00, 0.4542147501441236e+00, 0.7533183963616871e+00, 0.3860688952521878e+00, 0.7194066659377851e+00, 0.5625319388225433e+00, 0.2666855367146020e+00, 0.1952117588514313e+00, 0.4351982558470283e+00, 0.5735131371596182e+00, 0.6135067797741378e-01, 0.6627925797553079e+00, 0.5958408703801275e+00, 0.1272760770553625e+00, 0.9764806486481753e+00, 0.3738230266231765e+00, 0.8498786384730982e+00, 0.9509621583287799e+00, 0.5442964253426497e+00, 0.1274761736897801e+00, 0.8749459887021185e+00, 0.5319902581507450e+00, 0.4050330112540623e-01, 0.4679445276719574e-01, 0.6398654173496683e+00, 0.1619398995392787e+00, 0.2834692561637309e-01, 0.7369067773846130e+00, 0.2655208002136908e+00, 0.1902643003388997e+00, 0.7054378485643029e+00, 0.8869999325574263e+00, 0.1700725746174213e+00, 0.7638558771240449e+00, 0.3965145662986846e+00},
			n:    21,
			zOut: []float64{0.3026266675500349e+01, 0.2485098580281992e+01, 0.2208517136068298e+01, 0.2089817225148914e+01, 0.1987605159106218e+01, 0.1671782272803828e+01, 0.1288709839574181e+01, 0.1226108028801468e+01, 0.1094715872236073e+01, 0.9893293804315401e+00, 0.9470004635866569e+00, 0.8354737908875075e+00, 0.7678170961167213e+00, 0.5139691139127377e+00, 0.3402258729602133e+00, 0.2640682732152568e+00, 0.1814642316568930e+00, 0.9210283120697343e-01, 0.5017295169510767e-01, 0.1311272184039149e-01, 0.2605821149103687e-06, 0.8821141249794507e-26, 0.1703621403540070e-06, 0.1022946688596755e-06, 0.1288709839574181e+01, 0.0000000000000000e+00, 0.7771811135093639e-06, 0.1129025861285597e-05, 0.1226108028801468e+01, 0.8850832554089860e-22, 0.6461044606060110e-07, 0.3214361230688769e-07, 0.1094715872236073e+01, 0.5589172803550431e-19, 0.3128365184192157e-08, 0.1088448901615084e-07, 0.9893293804315401e+00, 0.4249640396990678e-22, 0.2862531707476509e-05, 0.1047704834723032e-05, 0.9470004635866569e+00, 0.2336393132634786e-16, 0.2207335777761344e+02, 0.2207335777761343e+02, 0.1020000000000000e+03, 0.3043083900226757e+01, 0.9803921568627451e+00, 0.1031792375203169e-03, 0.7678170961167213e+00, 0.8633839815190009e-18, 0.1747705279299585e-04, 0.1213546251050483e-04, 0.5139691139127377e+00, 0.2016356967855187e-17, 0.7704963297163309e-04, 0.2063636533685701e-03, 0.3402258729602133e+00, 0.5916456789157589e-30, 0.2417908304509407e-03, 0.2207145059751390e-03, 0.2640682732152568e+00, 0.1479114197289397e-30, 0.1687694376453652e-02, 0.8888286778685235e-03, 0.1814642316568930e+00, 0.1064934658613210e-15, 0.3591290150683059e-02, 0.7604711042910346e-02, 0.9210283120697343e-01, 0.2042102137240287e-23, 0.1463034917919357e-02, 0.1882622569903463e-02, 0.5017295169510767e-01, 0.2551739098626718e-22, 0.3961160948815250e-03, 0.9438210467320612e-03, 0.1311272184039149e-01, 0.1972152263052530e-29, 0.6693623055505279e-03, 0.9046122259092353e-03, 0.2605821149103687e-06, 0.3100321587303656e-14, 0.3978735902689679e-02, 0.4253573989104679e-02},
			info: 0,
		},
		{
			z:    []float64{0.8778106868894964e+00, 0.4777882175260769e+00, 0.1820603962716897e+00, 0.7891460918977841e+00, 0.7131176407472852e+00, 0.2166796106605611e+00, 0.7093860568912320e+00, 0.6747559124413653e+00, 0.1755802732664898e+00, 0.2538529139601842e-01, 0.3839736304139417e+00, 0.4972278527690542e+00, 0.2219918935874782e+00, 0.6587468361459490e+00, 0.6959875726535614e+00, 0.8785259347357933e+00, 0.4385001119617188e+00, 0.2992124106963234e-01, 0.9853525010355352e+00, 0.1842422327604198e-01, 0.7939453301916002e+00, 0.8502867854292200e+00, 0.6683895767567380e+00, 0.8502503508041696e+00, 0.2705991244761674e-02, 0.2273590703395093e+00, 0.1996606140173212e+00, 0.5828845765712893e+00, 0.7928614954547730e+00, 0.1675870051229429e+00, 0.3078809727828886e+00, 0.6073537805183471e-01, 0.3498986727103111e+00, 0.4266607326484094e+00, 0.7790595486765419e+00, 0.6274781693151706e+00, 0.3468808495278470e+00, 0.4002308549977796e+00, 0.8927740819756569e+00, 0.5380209790008849e+00, 0.9944530064668308e+00, 0.8382845348910617e+00, 0.6578833875204938e+00, 0.6784609218227333e-01, 0.9880398702345439e-01, 0.1591591396324092e+00, 0.3625625292925116e-01, 0.7834491156107602e+00, 0.8935128283958205e-01, 0.6534249936105881e+00, 0.9608003610697393e+00, 0.9122649849010867e+00, 0.3064782192618438e+00, 0.6165812021330105e+00, 0.3942978367050161e+00, 0.2389206118855702e+00, 0.4357310309586615e+00, 0.6366539061757281e+00, 0.2487697657706114e+00, 0.3158114775243555e+00, 0.4359459725257834e+00, 0.8521387333399649e+00, 0.7376171834812397e+00, 0.7198918826028659e+00, 0.3787355957929660e+00, 0.5132345781976363e+00, 0.5880413889667653e+00, 0.8394688090887562e+00, 0.3673771632022454e+00, 0.1291928261630060e+00, 0.6552015571219838e+00, 0.7918941075374089e+00, 0.3759434916876330e+00, 0.2131514444401432e+00, 0.5997464643577372e+00, 0.8626895671971565e+00, 0.1962050188754781e+00, 0.6291330853690040e+00, 0.6873926281549517e+00, 0.8949666870308315e-01, 0.1384684788210775e+00, 0.8959871983479050e+00, 0.5967626810459625e+00, 0.6838507108780668e+00},
			n:    21,
			zOut: []float64{0.2344484145674817e+01, 0.2205594465347172e+01, 0.2180337141043826e+01, 0.1936893457336740e+01, 0.1819297464222486e+01, 0.1595314798543036e+01, 0.1497965290687764e+01, 0.1192635586559562e+01, 0.1109297670462514e+01, 0.1021894664697417e+01, 0.8474592078637809e+00, 0.7996737801504824e+00, 0.5357931354846659e+00, 0.5077230661246940e+00, 0.4005808269588222e+00, 0.2350964781455809e+00, 0.1732854339186359e+00, 0.5510322151786835e-01, 0.3865442753785928e-01, 0.1308309165665133e-01, 0.1986637888070920e-03, 0.1267902442921281e-19, 0.1632713022477316e-10, 0.9197304681744396e-11, 0.1497965290687764e+01, 0.6540946849634405e-22, 0.1644600422568816e-11, 0.2569557641484267e-11, 0.1192635586559562e+01, 0.1886570321166912e-23, 0.1155544514329289e-09, 0.3178796095468434e-10, 0.1109297670462514e+01, 0.2958228394578794e-30, 0.1814581278982536e-06, 0.5763166468812303e-07, 0.1021894664697417e+01, 0.1648133134119960e-16, 0.5817023408243420e-07, 0.2381874475610022e-07, 0.8474592078637809e+00, 0.5828670879282072e-15, 0.2051036601772319e+02, 0.2051036601772318e+02, 0.1250000000000000e+03, 0.3639455782312925e+01, 0.8800000000000001e+01, 0.1964091724818299e-07, 0.5357931354846659e+00, 0.3518203713570956e-14, 0.1200151291154190e-07, 0.2251426423705210e-07, 0.5077230661246940e+00, 0.6882401190382613e-24, 0.1803872146382089e-06, 0.4763453660743515e-06, 0.4005808269588222e+00, 0.1559915108945621e-18, 0.4978941410546894e-05, 0.5275156660739611e-05, 0.2350964781455809e+00, 0.1004043408045437e-17, 0.5647304839276210e-04, 0.2873291365503010e-04, 0.1732854339186359e+00, 0.2485883728081419e-23, 0.9907264920144563e-04, 0.1293007080503375e-03, 0.5510322151786835e-01, 0.0000000000000000e+00, 0.9623294785827948e-04, 0.7898847596644510e-04, 0.3865442753785928e-01, 0.0000000000000000e+00, 0.2279226510196514e-03, 0.3149968919986460e-03, 0.1308309165665133e-01, 0.4893402802699089e-27, 0.4145854568122489e-03, 0.3196372933697030e-03, 0.1986637888070920e-03, 0.1930680189793120e-17, 0.2183385452652873e-03, 0.3171631588309268e-03},
			info: 0,
		},
		{
			z:    []float64{0.3053504558052776e+00, 0.2234484687205406e+00, 0.6531459952011753e+00, 0.3122060875846019e+00, 0.9991431676363117e+00, 0.2727966396486101e+00, 0.9325466197866266e+00, 0.9368849148298319e+00, 0.8561644447997885e+00, 0.1054815122266185e+00, 0.4923718332155202e-01, 0.9268095624277023e+00, 0.1888864280722891e+00, 0.7655148720572605e+00, 0.1232767826602820e+00, 0.5903563091717590e+00, 0.9248190356569348e+00, 0.5395288290327440e+00, 0.9028402566412551e+00, 0.9500520891471187e-01, 0.8953008781613688e+00, 0.2933361506740548e+00, 0.8453264440546789e+00, 0.5481580781657376e+00, 0.3587646761786403e+00, 0.5176903549368316e+00, 0.7562422108585272e+00, 0.9675296263022207e+00, 0.8986594456301542e+00, 0.5421338625853966e-01, 0.1201688601088308e+00, 0.6190380352364769e+00, 0.5038255434020116e+00, 0.6721007162690862e+00, 0.5702376374225947e+00, 0.9284095682432886e+00, 0.3695163217482177e+00, 0.6039359484498441e-01, 0.4652604684312921e+00, 0.8651698808632446e-01, 0.7546245127264203e+00, 0.7397909213681499e+00, 0.1212064081570199e+00, 0.3507242515197573e+00, 0.1903823153835104e+00, 0.7217789084869874e-01, 0.4631739813227773e+00, 0.4692891642215747e+00, 0.4583968848602870e+00, 0.1862358222844885e+00, 0.2939086301666586e+00, 0.5739509914073745e+00, 0.2602639918141684e+00, 0.7265362515535626e+00, 0.2180031380994948e+00, 0.1132710387386646e+00, 0.7997129355612416e+00, 0.7503609043894605e+00, 0.7921549516519859e+00, 0.2820213454373731e+00, 0.3757885220124352e+00, 0.7700990985029622e+00, 0.7233316318551536e+00, 0.1305854233773539e+00, 0.7245124401476205e+00, 0.5999815977622437e+00, 0.3886468370487757e+00, 0.5608149286383455e+00, 0.4298380566755162e+00, 0.6436590616913128e+00, 0.3743035898091585e-01, 0.4516061963302198e-01, 0.3504820364503235e+00, 0.4814816352089226e+00, 0.5678245050510763e+00, 0.5711961955150159e+00, 0.9549329198325482e+00, 0.8527165734568565e-01, 0.4185783350739758e-01, 0.5702414021975877e-01, 0.2302271730247050e+00, 0.7442529870754924e+00, 0.5366636699040384e+00, 0.8085683101567075e+00},
			n:    21,
			zOut: []float64{0.2590499366480712e+01, 0.2541239603587953e+01, 0.2259909426243320e+01, 0.2181782566945434e+01, 0.1886341857097904e+01, 0.1651483296972034e+01, 0.1427705031981996e+01, 0.1248256205386413e+01, 0.1112814709657656e+01, 0.9343995210628773e+00, 0.8379224813887226e+00, 0.8181328249547786e+00, 0.6165308413237942e+00, 0.4939038343854810e+00, 0.4603621358717260e+00, 0.3639045182225083e+00, 0.3137819362841643e+00, 0.1577326270698492e+00, 0.8198100142281199e-01, 0.9578206270736807e-02, 0.4942793252781514e-03, 0.1292469707114106e-22, 0.1980335658106781e-04, 0.1681298419390782e-04, 0.1427705031981996e+01, 0.2584939414228211e-25, 0.8200031816835676e-05, 0.9102296987148495e-05, 0.1248256205386413e+01, 0.4930380657631324e-30, 0.4104488055426074e-09, 0.2576051753309951e-09, 0.1112814709657656e+01, 0.6452508963690269e-19, 0.2339325112754308e-08, 0.4010758969857423e-08, 0.9343995210628773e+00, 0.2041003537989240e-20, 0.4754834042401540e-06, 0.1343807701186157e-06, 0.8379224813887226e+00, 0.2942145729531872e-22, 0.2198875627193616e+02, 0.2198875627193615e+02, 0.1030000000000000e+03, 0.3158730158730159e+01, 0.5825242718446602e+01, 0.2325395492704734e-05, 0.6165308413237942e+00, 0.1475671084242936e-18, 0.4982404846255511e-06, 0.2497087821784679e-06, 0.4939038343854810e+00, 0.1295808538938520e-16, 0.6559194735079054e-05, 0.3408173383041987e-05, 0.4603621358717260e+00, 0.5811344985782108e-17, 0.9559725090946887e-05, 0.1189458663431952e-04, 0.3639045182225083e+00, 0.2913839059670682e-18, 0.4863259217476146e-04, 0.7411068946771063e-04, 0.3137819362841643e+00, 0.2338183723075079e-26, 0.4219303942492233e-03, 0.5983359002845232e-03, 0.1577326270698492e+00, 0.3451266460341927e-30, 0.4370512454739224e-03, 0.2953761682364422e-03, 0.8198100142281199e-01, 0.2482356093574565e-20, 0.7632038538424591e-04, 0.1389505478745866e-03, 0.9578206270736807e-02, 0.1680537859656637e-20, 0.1168521819007177e-01, 0.1200804951998557e-01, 0.4942793252781514e-03, 0.1967697241141140e-19, 0.1184783913560175e-01, 0.1590443390723593e-01},
			info: 0,
		},
		{
			z:    []float64{0.9721612501531746e+00, 0.9226086145236946e+00, 0.6294635256706198e+00, 0.3023176111891041e+00, 0.6049371291087220e+00, 0.2862933293136216e+00, 0.8819078559307578e+00, 0.5258944749310531e+00, 0.1024274135843443e+00, 0.4745865163816484e+00, 0.3569366663070145e+00, 0.5081536020377282e+00, 0.6994675492692721e+00, 0.3184330055525583e+00, 0.4062572718658636e+00, 0.8497577069296732e+00, 0.7686323719242980e+00, 0.8873781649289114e+00, 0.5445815352331720e+00, 0.5465168091072755e+00, 0.2655612328242208e+00, 0.7149116271405493e+00, 0.2779393895548951e+00, 0.1344927254674255e-01, 0.4544708454847078e+00, 0.5332505496545747e-01, 0.3223583007255554e+00, 0.4314145703973596e+00, 0.6194759383350579e+00, 0.9673744275195654e+00, 0.5445933229599376e-01, 0.2359803760372408e+00, 0.4349989237395262e+00, 0.3983243124260842e+00, 0.1549760127081995e+00, 0.6483062457720113e+00, 0.9796384288763583e+00, 0.2088098624466561e-01, 0.7266086220595113e+00, 0.7787182393455840e+00, 0.2714545750078465e+00, 0.3281766266219532e-01, 0.4443206765749519e+00, 0.7385116967559627e+00, 0.4742748966298079e+00, 0.2719801119596308e+00, 0.5535252469642412e+00, 0.6819653774052530e+00, 0.5802197659205470e+00, 0.8706014802153047e+00, 0.7899732601913489e-01, 0.6149130585720997e+00, 0.1184080138409910e+00, 0.7335155686110397e+00, 0.3161056790243246e+00, 0.4751009230144272e+00, 0.6074228340613933e+00, 0.1834459697241099e+00, 0.5602196728537751e+00, 0.6036792416269905e+00, 0.4996379576165632e+00, 0.5762117563802562e+00, 0.4313302593954876e+00, 0.2357090997139660e+00, 0.5266696040444221e+00, 0.4943568418910921e+00, 0.5194970579745682e+00, 0.8729901711287622e+00, 0.1253423978225398e+00, 0.3249303186427334e+00, 0.1476841747547597e+00, 0.8115004890141919e+00, 0.3358761522855614e+00, 0.6267425544884889e+00, 0.8684373664078195e+00, 0.3410509649937432e+00, 0.1223171934609151e+00, 0.6692100819077175e+00, 0.8714060240466885e+00, 0.8435328911476530e+00, 0.3613173587051273e+00, 0.5970888673100956e+00, 0.4814113673780336e+00, 0.9518076426657822e+00},
			n:    21,
			zOut: []float64{0.2554276088974017e+01, 0.2353122577016648e+01, 0.1866240795386955e+01, 0.1792222013488891e+01, 0.1776522631286352e+01, 0.1671843747376224e+01, 0.1608565757744856e+01, 0.1413448668253747e+01, 0.9978397039940720e+00, 0.9776727877490632e+00, 0.8701050228770406e+00, 0.6522479060818760e+00, 0.6295532782699175e+00, 0.4076513128864766e+00, 0.3045303763029391e+00, 0.2026925943270836e+00, 0.1597713759311400e+00, 0.1149331809326307e+00, 0.5501161781795570e-01, 0.3472328989596482e-02, 0.1615352262160075e-02, 0.2261810287929078e-15, 0.1434223756681676e-21, 0.1189989751754647e-20, 0.1608565757744856e+01, 0.1068494074045434e-17, 0.1018285523195584e-14, 0.1812446834981097e-15, 0.1413448668253747e+01, 0.8361925595342725e-26, 0.5806129122189708e-11, 0.1848580748818003e-11, 0.9978397039940720e+00, 0.5222833230866609e-21, 0.9842396145073251e-07, 0.4923018478143697e-07, 0.9776727877490632e+00, 0.9540591325261466e-21, 0.1504532523487471e-05, 0.7619936176069947e-06, 0.8701050228770406e+00, 0.2405710216561998e-25, 0.2041333911794964e+02, 0.2041333911794964e+02, 0.1050000000000000e+03, 0.3108843537414966e+01, 0.6666666666666667e+01, 0.1010063659896995e-05, 0.6295532782699175e+00, 0.2958228394578794e-30, 0.7231781167780556e-06, 0.8027785243996724e-06, 0.4076513128864766e+00, 0.2327139670401985e-28, 0.4311361064418766e-06, 0.4365718303993244e-06, 0.3045303763029391e+00, 0.3511198893363720e-17, 0.5813968676396444e-06, 0.4990623034093460e-06, 0.2026925943270836e+00, 0.5787194672316798e-22, 0.1414410322798341e-05, 0.1084116171450400e-05, 0.1597713759311400e+00, 0.3368436065293720e-27, 0.4438807230072854e-05, 0.3295442179394680e-05, 0.1149331809326307e+00, 0.3973698114522315e-22, 0.1450590122472916e-04, 0.1074561246336364e-04, 0.5501161781795570e-01, 0.1153335343143272e-20, 0.9954779132989017e-07, 0.6305695341872095e-07, 0.3472328989596482e-02, 0.7183968878891587e-15, 0.2465868832870137e-03, 0.3524118878925340e-03, 0.1615352262160075e-02, 0.1517768381645227e-26, 0.5410549418889337e-03, 0.6396403869804795e-03},
			info: 0,
		},
		{
			z:    []float64{0.9945523629868341e+00, 0.6311483525653909e+00, 0.7029190400645571e+00, 0.6452581068575963e+00, 0.9353553331334358e+00, 0.1547193049011500e+00, 0.5087511216875095e+00, 0.8821879483245458e+00, 0.5873463835171173e+00, 0.3391476900726194e+00, 0.3121839954924539e+00, 0.2777103145160921e+00, 0.2404463284577694e+00, 0.9114535183484027e+00, 0.4707329974301702e+00, 0.8965398451252625e+00, 0.6082343132413309e+00, 0.6509204493235183e+00, 0.1573499033176421e+00, 0.7416865701488552e+00, 0.1617007503826062e-01, 0.9896605969885931e+00, 0.3427350027671039e+00, 0.9677837053324486e+00, 0.1744936477417883e+00, 0.1063275349300943e+00, 0.7956875801511568e+00, 0.8998967378241718e-01, 0.9957418429488507e-01, 0.2587083053394715e+00, 0.3753593926504107e+00, 0.7537263520315082e+00, 0.1726167393116790e+00, 0.1420490749456633e+00, 0.1475360304745418e+00, 0.3281044300775052e+00, 0.4011897283558843e+00, 0.5039659382748809e+00, 0.8444659776686902e-01, 0.6685320205580927e+00, 0.8425776598223370e+00, 0.6334616337862548e+00, 0.6754035119469365e+00, 0.3594856315188868e+00, 0.4308279252773916e+00, 0.2170629535054914e+00, 0.5040120070238915e+00, 0.3780998968579012e+00, 0.9176188652711103e+00, 0.1603892455353655e+00, 0.2475008720560291e+00, 0.4414989878913728e+00, 0.3466658552370731e+00, 0.3098329951977107e+00, 0.8940921934240968e+00, 0.6686136942966417e+00, 0.4049936818549904e-01, 0.1308695446239941e+00, 0.8770734618597430e+00, 0.7917220872288512e+00, 0.5736823795257117e+00, 0.5474219864141461e+00, 0.3320663514826834e+00, 0.5824816531032581e+00, 0.6748067573570548e+00, 0.8139348519761679e+00, 0.1984641509226878e+00, 0.5557729841117627e+00, 0.1101812804319026e+00, 0.2169710965518002e+00, 0.5846617154920911e+00, 0.9784843983810533e+00, 0.8650267562586114e+00, 0.4041298540058539e+00, 0.9548165813531374e+00, 0.3839697059508718e+00, 0.4296171922210591e+00, 0.9617664542372570e+00, 0.2102779850721345e+00, 0.6135130181658475e+00, 0.3331715196673474e+00, 0.3177082868916457e+00, 0.2836520073686416e+00, 0.4452894940247868e+00},
			n:    21,
			zOut: []float64{0.2413112525759640e+01, 0.2309698811549223e+01, 0.1971065496840386e+01, 0.1839093033734385e+01, 0.1586321309986888e+01, 0.1384287501933647e+01, 0.1357349915568794e+01, 0.1342137050614370e+01, 0.1098398899504136e+01, 0.1013688381736709e+01, 0.8838922773071145e+00, 0.7023193003094995e+00, 0.6083268670387161e+00, 0.3967105732701523e+00, 0.3295901758335739e+00, 0.2610566591740404e+00, 0.2254040681461952e+00, 0.1107925081061023e+00, 0.7084089717166595e-01, 0.5790734022898784e-02, 0.1162539701808497e-05, 0.2217532248565769e-17, 0.1400266087055540e-09, 0.1806586784837133e-09, 0.1357349915568794e+01, 0.1160047396826257e-18, 0.1994113072724655e-16, 0.4228494360227003e-16, 0.1342137050614370e+01, 0.1172187884313513e-15, 0.1255075745859590e-14, 0.2503873040081153e-15, 0.1098398899504136e+01, 0.3711182370878063e-20, 0.5744299067797088e-08, 0.4873476684669810e-08, 0.1013688381736709e+01, 0.1283916659070448e-18, 0.9737119066622311e-08, 0.8382615528986061e-08, 0.8838922773071145e+00, 0.3385457556489824e-15, 0.1990987815014785e+02, 0.1990987815014784e+02, 0.1050000000000000e+03, 0.3324263038548753e+01, 0.5714285714285714e+01, 0.5531678096648157e-07, 0.6083268670387161e+00, 0.9327491343333244e-26, 0.3094358165149238e-06, 0.4992693882721107e-06, 0.3967105732701523e+00, 0.1615587133892632e-26, 0.5013465632663534e-06, 0.4474558637993164e-06, 0.3295901758335739e+00, 0.7527343574232552e-22, 0.6917755140723670e-06, 0.9191878134168280e-06, 0.2610566591740404e+00, 0.8992644540970212e-27, 0.3283328599477859e-08, 0.8980185396558612e-09, 0.2254040681461952e+00, 0.2761013168273541e-29, 0.1566859737304233e-05, 0.9651044230345755e-05, 0.1107925081061023e+00, 0.7853700447084989e-18, 0.2494389596499284e-03, 0.3103855763358142e-03, 0.7084089717166595e-01, 0.7910121402691465e-20, 0.9848486231185840e-03, 0.1285663115805527e-02, 0.5790734022898784e-02, 0.1925929944387236e-33, 0.1531134660067926e-02, 0.1103959511478054e-02, 0.1162539701808497e-05, 0.0000000000000000e+00, 0.3553926119432854e-02, 0.1161337136649311e-01},
			info: 0,
		},
		{
			z:    []float64{0.6912525216559012e+00, 0.7535513505900934e+00, 0.9471932644608124e+00, 0.3186447229836621e+00, 0.5324891284853083e+00, 0.8568362673509099e+00, 0.7683599820389093e+00, 0.5584553004793524e+00, 0.5571013534938567e+00, 0.9300192917069305e-01, 0.8580829209571639e+00, 0.7019126526839274e+00, 0.6637712341629651e+00, 0.1211000088170944e+00, 0.6340253546080570e+00, 0.8089912896293909e+00, 0.9056005723453709e+00, 0.8259750504337368e+00, 0.7787230128973186e+00, 0.5552567089885843e+00, 0.1575217143336131e+00, 0.6258385609742768e+00, 0.3943745586872103e+00, 0.2414131111537543e-02, 0.1491577729788018e-01, 0.3231570165180105e+00, 0.6629837458829935e+00, 0.3692715436026686e+00, 0.4217957028148089e+00, 0.9587355452830710e+00, 0.2048474370070816e+00, 0.9514275263988156e+00, 0.6824099372375696e+00, 0.4368303647429550e-01, 0.1779948598152153e+00, 0.8894462775192779e+00, 0.5101271266317630e+00, 0.9448652224292025e+00, 0.2265718177112338e+00, 0.3995811139403516e+00, 0.8380789267629041e+00, 0.6621441253102893e+00, 0.1548712624835822e+00, 0.6028850196944033e+00, 0.6938032609628844e+00, 0.6847684356722229e+00, 0.5160938472467536e+00, 0.2489894297094153e+00, 0.1603949345362032e-01, 0.8469682453172568e+00, 0.1332301650751776e+00, 0.4964641576870391e+00, 0.8955382322295532e+00, 0.8913224930710517e-01, 0.1632933980261509e+00, 0.7671629642421702e+00, 0.1918190416387825e+00, 0.4660642407804531e+00, 0.3421851388276426e+00, 0.6793284417353496e-03, 0.4856583571413033e+00, 0.6596584184483706e+00, 0.3999565195667565e-01, 0.4265147212719423e+00, 0.1721559491949431e-01, 0.4779030554385033e-01, 0.9628058707102425e+00, 0.4575395386200597e+00, 0.3526890078762277e+00, 0.7271726865364773e+00, 0.1741804541232641e+00, 0.3524640041347155e+00, 0.2143698389021181e+00, 0.9553499123803968e+00, 0.7803670081719427e+00, 0.7906509834069840e+00, 0.5939624443460740e+00, 0.5439294231719827e+00, 0.4026522884432556e+00, 0.2400403086558794e+00, 0.1695045416332096e+00, 0.8774151073165960e+00, 0.9528799076371641e-01, 0.6357307125994049e-01},
			n:    21,
			zOut: []float64{0.2768602537214622e+01, 0.2449987435840808e+01, 0.2103617452638331e+01, 0.2091492233971247e+01, 0.2040261542124423e+01, 0.1984511273424848e+01, 0.1507960384149227e+01, 0.1491386484166052e+01, 0.1283295633334833e+01, 0.1065640575633715e+01, 0.9760063516861833e+00, 0.9535053549649605e+00, 0.6117733318422185e+00, 0.5156356643703764e+00, 0.4217391763517565e+00, 0.2514332616796913e+00, 0.1056943786041289e+00, 0.6556299181229375e-01, 0.2554189252987301e-01, 0.1599343544217954e-01, 0.7248128851221257e-03, 0.1757582096832414e-26, 0.8938568486456872e-12, 0.7801064326234513e-12, 0.1507960384149227e+01, 0.1792073872399063e-19, 0.2167105506392983e-08, 0.2566615495401212e-08, 0.1491386484166052e+01, 0.2939740300877914e-17, 0.1513553473151668e-08, 0.1292098164447321e-08, 0.1283295633334833e+01, 0.5002192774679673e-18, 0.3422853236655498e-09, 0.3087733884567774e-09, 0.1065640575633715e+01, 0.2649562899583917e-24, 0.1824851422869684e-09, 0.1969539993879423e-09, 0.9760063516861833e+00, 0.4849588587138365e-15, 0.2273036620466689e+02, 0.2273036620466689e+02, 0.1110000000000000e+03, 0.3401360544217687e+01, 0.6306306306306307e+01, 0.7012703734578152e-09, 0.6117733318422185e+00, 0.2636779683484747e-15, 0.8832037497597917e-08, 0.1377654767646391e-07, 0.5156356643703764e+00, 0.0000000000000000e+00, 0.4403364729024244e-07, 0.2921450058151117e-07, 0.4217391763517565e+00, 0.5116160632510206e-16, 0.4627032593211780e-06, 0.6780935482482274e-06, 0.2514332616796913e+00, 0.3477119313860159e-18, 0.8396751469149006e-05, 0.1191099168408868e-04, 0.1056943786041289e+00, 0.3125614770312806e-14, 0.3873238345753621e-04, 0.1564927337192913e-03, 0.6556299181229375e-01, 0.2145521217817673e-17, 0.9476510350371504e-03, 0.9104754488359111e-03, 0.2554189252987301e-01, 0.1508279173816323e-23, 0.1093077353977330e-02, 0.3191708039391989e-03, 0.1599343544217954e-01, 0.1319576415787468e-18, 0.1003863779460019e-02, 0.1271197433268307e-02, 0.7248128851221257e-03, 0.5169135658374832e-21, 0.1128681157512944e-02, 0.4574374031799749e-02},
			info: 0,
		},
		{
			z:    []float64{0.8514967554120231e+00, 0.1903564904561825e+00, 0.9315134585755248e+00, 0.6865565499502658e+00, 0.9874780893034819e+00, 0.4243732836164926e-01, 0.4493254789963134e+00, 0.1218497220886967e+00, 0.3032420422448456e-01, 0.4472006833337416e+00, 0.7764061193501844e+00, 0.2765680149425037e+00, 0.1540380964603436e+00, 0.8227084597523057e+00, 0.1281873340097673e+00, 0.2797045229662546e+00, 0.1723485697940061e+00, 0.5407468670158238e+00, 0.3703892253237170e+00, 0.3275545813562609e+00, 0.1384768695152372e+00, 0.8683661690082241e+00, 0.2417038393999236e+00, 0.1573965961648072e+00, 0.7549703050710239e+00, 0.4197135409081310e+00, 0.5565257749798536e-01, 0.7494783707245707e+00, 0.8342833407960080e+00, 0.7207413539297494e+00, 0.3936911272324267e+00, 0.8307872407436112e-01, 0.5696677800088412e-01, 0.2580569153061851e-01, 0.3976149332803511e+00, 0.4082340292723502e+00, 0.3620800929220623e+00, 0.2078006460748600e+00, 0.6810562489816266e+00, 0.8414616295760738e+00, 0.2053319202340824e+00, 0.7745582586901971e+00, 0.8243869237197288e-01, 0.7513634029365589e+00, 0.8658814490552211e+00, 0.7014202441287165e-01, 0.8786892218281559e+00, 0.2415414918645031e-01, 0.1648367770784782e+00, 0.5852629690413500e+00, 0.6558826026732849e+00, 0.3505171007322015e+00, 0.5959744796861849e+00, 0.6000199298251039e+00, 0.5156931460104713e+00, 0.1010243586499049e-01, 0.2804608587164645e+00, 0.7088599878489240e+00, 0.3159377781412633e+00, 0.1196632783263437e+00, 0.8053993759743566e+00, 0.9537751230670044e+00, 0.6890540876884894e+00, 0.7327967809533995e+00, 0.1752177926983489e-01, 0.1176294169286665e+00, 0.1417165464977135e+00, 0.1128117051181837e+00, 0.4431718138373526e+00, 0.6383761487884151e+00, 0.4394388749172047e+00, 0.4496324720961747e+00, 0.2336818223819791e+00, 0.8283237197456694e+00, 0.5968446077081969e+00, 0.2123688346867825e-03, 0.2316409292069651e+00, 0.4041203764036865e+00, 0.5989557684304492e-01, 0.1751475090786662e+00, 0.3300532749209990e+00, 0.5257653351639615e+00, 0.9423173850128755e+00, 0.5110067889308013e+00},
			n:    21,
			zOut: []float64{0.2298367449128524e+01, 0.2183508003369306e+01, 0.1768578955898148e+01, 0.1497740650820804e+01, 0.1451044171656134e+01, 0.1248562072819587e+01, 0.1096241131844911e+01, 0.1047761158237631e+01, 0.1036921434023463e+01, 0.1001878203872264e+01, 0.6028997776208512e+00, 0.5966927468392602e+00, 0.3787547972649039e+00, 0.3164597916816877e+00, 0.2770907680746649e+00, 0.1378928396715414e+00, 0.1008983167754413e+00, 0.7523836829300481e-01, 0.4549396861483578e-01, 0.2905908538794771e-01, 0.1164397397714791e-04, 0.2593449025859535e-11, 0.4028878361793870e-19, 0.4523833078775027e-19, 0.1096241131844911e+01, 0.2254118867995285e-20, 0.6781252758190456e-19, 0.6132859990189332e-19, 0.1047761158237631e+01, 0.2208810534618833e-28, 0.7941399228881527e-19, 0.8755467647203911e-19, 0.1036921434023463e+01, 0.2697904295855860e-27, 0.1193508568348155e-18, 0.1375302458176128e-18, 0.1001878203872264e+01, 0.1808754486438286e-19, 0.1585460437287443e-17, 0.7110442921296878e-18, 0.6028997776208512e+00, 0.1121217470921487e-23, 0.1719109533586889e+02, 0.1719109533586889e+02, 0.1020000000000000e+03, 0.3002267573696145e+01, 0.7843137254901960e+01, 0.6126841018734423e-14, 0.3787547972649039e+00, 0.2514494135391975e-29, 0.3694268515215640e-12, 0.2142184401680204e-12, 0.3164597916816877e+00, 0.2495880173472380e-22, 0.1052194788283320e-11, 0.1801542652052392e-11, 0.2770907680746649e+00, 0.2042405059827892e-23, 0.3416315088185822e-10, 0.1784450556067769e-10, 0.1378928396715414e+00, 0.2231720057459842e-17, 0.5936850868223322e-09, 0.2792901329469146e-09, 0.1008983167754413e+00, 0.4009937753419447e-25, 0.1438568455567306e-07, 0.6376284135652434e-08, 0.7523836829300481e-01, 0.1673899732006569e-23, 0.3811420638095229e-06, 0.1677978014843256e-06, 0.4549396861483578e-01, 0.4108428387966160e-20, 0.9969899927958168e-05, 0.2238705130236833e-04, 0.2905908538794771e-01, 0.5621738354967019e-25, 0.1622697124847995e-03, 0.1340889241809050e-03, 0.1164397397714791e-04, 0.4733165431326071e-29, 0.2205086015306072e-02, 0.1109731617446849e-02},
			info: 0,
		},
		{
			z:    []float64{0.9566341326420642e+00, 0.1368890791339401e+00, 0.1041033686995702e+00, 0.1843288381431366e+00, 0.9983016155690853e+00, 0.1786866722264480e+00, 0.3401743522788719e+00, 0.3857333052582377e+00, 0.5001620282112933e+00, 0.9013417473823726e+00, 0.9483009872337210e+00, 0.8066933876879568e+00, 0.5299205541713562e+00, 0.1746250429119467e-01, 0.9676426134678857e+00, 0.9451210738656991e+00, 0.4964770237154315e+00, 0.7837324008636358e+00, 0.9355959680316486e+00, 0.6160646886859618e+00, 0.3400424465599898e+00, 0.5332164761252830e+00, 0.3086260018745091e+00, 0.7161810741898500e+00, 0.9586563909920308e+00, 0.5072089025298385e+00, 0.5345428289191819e+00, 0.3564033548208417e+00, 0.5073508962435702e-01, 0.5457190349920044e+00, 0.5924351087865416e+00, 0.4551284321479383e+00, 0.1212070325395470e+00, 0.4136992969230449e+00, 0.9668715153070689e+00, 0.5158905905227948e+00, 0.3815356588296225e+00, 0.9643666102790432e+00, 0.2243124357596349e-01, 0.1865949320717719e+00, 0.7081123336136068e+00, 0.8847286955309422e+00, 0.8075237017566694e+00, 0.7058935338718864e+00, 0.8930194548226011e-01, 0.5968683124630468e+00, 0.6423123320238798e+00, 0.4264046421741549e+00, 0.6535060636263267e+00, 0.8848885990305457e+00, 0.1727496532173951e+00, 0.2233116713730792e+00, 0.5179656160482806e+00, 0.8833427955005373e+00, 0.5463392573098168e+00, 0.2339495215057854e+00, 0.2531599269911875e+00, 0.8693933445310196e+00, 0.9904673923253784e+00, 0.5001765262594373e+00, 0.8475066901059425e+00, 0.7747140538451288e+00, 0.8097617518836179e+00, 0.6177833422496881e+00, 0.4294196301939426e+00, 0.8754149021622922e+00, 0.1154777972031917e+00, 0.4002319777705416e+00, 0.8948519482752643e+00, 0.2680092666078437e-01, 0.4982362166797213e+00, 0.8946688039099321e+00, 0.6004540842339657e+00, 0.7864972660235388e+00, 0.5343742612900907e+00, 0.6804570163829448e+00, 0.8534533212497195e+00, 0.6588190749186591e+00, 0.3987256626502571e+00, 0.9641448737959348e+00, 0.8049200655531427e+00, 0.8429466638354991e+00, 0.3173537092974966e+00, 0.2969758130071765e+00},
			n:    21,
			zOut: []float64{0.2691092090580979e+01, 0.2651656165852700e+01, 0.2273892452779949e+01, 0.2046491657483657e+01, 0.1856589708219707e+01, 0.1474606896148613e+01, 0.1449970763396692e+01, 0.1157826920722246e+01, 0.1131422765724169e+01, 0.1103316690199991e+01, 0.1081571209422854e+01, 0.8863514547656711e+00, 0.7487688552248520e+00, 0.4946957959671176e+00, 0.3261665336572834e+00, 0.2704116388140369e+00, 0.1035778920187539e+00, 0.8274904671505576e-01, 0.4712842456001845e-01, 0.3463118683574901e-01, 0.5254769424391846e-04, 0.2373014771939845e-22, 0.2238209824420786e-09, 0.1516830937674482e-09, 0.1449970763396692e+01, 0.6496080927593321e-18, 0.4705401432274776e-09, 0.6877602609145650e-09, 0.1157826920722246e+01, 0.9485735114056482e-11, 0.6765447972194981e-08, 0.9075269590011981e-08, 0.1131422765724169e+01, 0.0000000000000000e+00, 0.2115471642425146e-07, 0.1586124232313620e-07, 0.1103316690199991e+01, 0.1685550134725853e-21, 0.9651672077875321e-13, 0.7184159896265294e-12, 0.1081571209422854e+01, 0.5505416081326749e-23, 0.2191297069678434e+02, 0.2191297069678434e+02, 0.1080000000000000e+03, 0.3308390022675737e+01, 0.4629629629629630e+01, 0.3615019372146533e-05, 0.7487688552248520e+00, 0.2711157159063573e-25, 0.1185348717221526e-04, 0.1538970705040117e-04, 0.4946957959671176e+00, 0.7711572887860419e-23, 0.5228762878644057e-04, 0.2140066476035995e-04, 0.3261665336572834e+00, 0.1027012207743053e-16, 0.1058102716324823e-03, 0.1097923206605328e-03, 0.2704116388140369e+00, 0.1419949629397821e-27, 0.3075139383959133e-03, 0.2041016121130936e-03, 0.1035778920187539e+00, 0.1569288801032274e-18, 0.2245437021601444e-03, 0.2161294383881767e-03, 0.8274904671505576e-01, 0.1322963914266329e-22, 0.4098429023844965e-03, 0.2685224923308173e-03, 0.4712842456001845e-01, 0.1972152263052530e-27, 0.1445743106447529e-03, 0.3591079434142046e-04, 0.3463118683574901e-01, 0.1109335647967048e-30, 0.1752604588321185e-04, 0.1168961103966469e-04, 0.5254769424391846e-04, 0.0000000000000000e+00, 0.7596773881065606e-02, 0.6101958366533248e-02},
			info: 0,
		},
		{
			z:    []float64{0.8353120998595153e+00, 0.2762691216596785e+00, 0.4005263075088337e+00, 0.2013066591993183e+00, 0.7159087664363259e+00, 0.6826958051178053e+00, 0.3940626911632166e+00, 0.4990834556070268e+00, 0.6760530863202496e-02, 0.5490926358579395e-01, 0.1673564807759778e+00, 0.6884305772394704e+00, 0.3902830106720901e+00, 0.9394502319309270e+00, 0.1453732258871473e+00, 0.6913382271536774e+00, 0.4797608951630542e+00, 0.8813081041297155e+00, 0.7179153476502249e+00, 0.8684015860527726e+00, 0.4912919359735076e+00, 0.2734537604788182e+00, 0.6687443482830383e+00, 0.1554406884207427e+00, 0.2919860915775756e+00, 0.2642733542167044e+00, 0.8429147529046822e+00, 0.2888320698097314e+00, 0.8826173246454528e+00, 0.8977386905269689e+00, 0.7688524109189127e+00, 0.9833521147086286e+00, 0.9216258197795713e-01, 0.4886212819310254e-01, 0.2266462269156035e+00, 0.3411440109054314e+00, 0.3008601997419669e-01, 0.6174499545648404e+00, 0.5004878789063633e+00, 0.6820549240543587e+00, 0.2697209119346977e-01, 0.5561910637447186e+00, 0.5496066776611622e+00, 0.5185759009159874e+00, 0.1533641991284778e+00, 0.9896294286837857e+00, 0.3752499036369343e+00, 0.6748924776803822e-01, 0.2328081038279309e+00, 0.6721552304404512e+00, 0.5472950056303504e+00, 0.9949773164461492e+00, 0.5646629720515278e+00, 0.4280389734679516e+00, 0.4488723317267336e+00, 0.7263390986042261e+00, 0.2371171913738990e-01, 0.2980895305269504e-01, 0.2561842200752050e+00, 0.2554059142478259e+00, 0.5952758446711104e-01, 0.7921877816291150e+00, 0.2045719212105177e+00, 0.9215889426626128e+00, 0.7130733301970400e+00, 0.1420514661017834e+00, 0.8292002693849506e+00, 0.5327095640926792e+00, 0.9857821333545339e+00, 0.6686900428841760e+00, 0.1963263366455671e+00, 0.2881353528210300e+00, 0.7077457532403378e+00, 0.9075905470440924e+00, 0.1600253949359417e+00, 0.1037543259304946e+00, 0.3343707004174388e+00, 0.7287034220491817e-01, 0.8402900560224008e+00, 0.1343632243665364e+00, 0.8746515522185468e+00, 0.4102434537961821e+00, 0.1922774302984470e+00, 0.7386986269659255e+00},
			n:    21,
			zOut: []float64{0.2705701250380865e+01, 0.2517516394575531e+01, 0.1823532784589120e+01, 0.1787562059778101e+01, 0.1543666662169401e+01, 0.1517718677289282e+01, 0.1380407836894782e+01, 0.1248997815711926e+01, 0.1005808190406691e+01, 0.8278526045363968e+00, 0.6708485238929687e+00, 0.6082664705845506e+00, 0.5949155529076431e+00, 0.3419285270749396e+00, 0.2930861537880754e+00, 0.2677590880174420e+00, 0.1897681878114907e+00, 0.4498262444281959e-01, 0.4027859329477797e-01, 0.1212960387301111e-02, 0.4788271754899720e-05, 0.2113479533223326e-17, 0.1072835726919963e-07, 0.2769573896400322e-08, 0.1380407836894782e+01, 0.2075077381359190e-19, 0.2509615479685493e-12, 0.8472237738633739e-13, 0.1248997815711926e+01, 0.8862433277732312e-18, 0.5565142852935547e-10, 0.1115246324111066e-09, 0.1005808190406691e+01, 0.1508363342119467e-20, 0.4766364300104371e-06, 0.1686414207934524e-06, 0.8278526045363968e+00, 0.2958228394578794e-30, 0.1339811587116281e-04, 0.9618739022097639e-05, 0.6708485238929687e+00, 0.5096041447727736e-27, 0.1941181574680586e+02, 0.1941181574680586e+02, 0.1100000000000000e+03, 0.3301587301587301e+01, 0.6363636363636363e+01, 0.2769598741382734e-05, 0.5949155529076431e+00, 0.1660823573641619e-23, 0.5008194680845295e-05, 0.3644576454263968e-05, 0.3419285270749396e+00, 0.8597611808520244e-19, 0.8197192371775147e-05, 0.1495511835801176e-04, 0.2930861537880754e+00, 0.6452008777913547e-22, 0.7613853051650506e-05, 0.9176969716117157e-05, 0.2677590880174420e+00, 0.1626694295438144e-23, 0.2172962611604878e-06, 0.9910144268854854e-06, 0.1897681878114907e+00, 0.5698278443334010e-17, 0.3051734820342141e-03, 0.3112883552318466e-03, 0.4498262444281959e-01, 0.6022708464481807e-24, 0.1205042765713422e-04, 0.1759326252954196e-04, 0.4027859329477797e-01, 0.1150776253123047e-20, 0.1426316484726719e-04, 0.9508927802446297e-05, 0.1212960387301111e-02, 0.3136662420452545e-23, 0.9384394253406367e-03, 0.4268700521479601e-03, 0.4788271754899720e-05, 0.0000000000000000e+00, 0.1227549320701017e-01, 0.5132215250850077e-02},
			info: 0,
		},
		{
			z:    []float64{0.6118263281058917e+00, 0.9095172470423057e+00, 0.3600259528878788e+00, 0.2816648927398081e+00, 0.1399164597780194e+00, 0.8570391268113827e+00, 0.6056165732881742e+00, 0.8852338629006543e+00, 0.5421670517489531e+00, 0.6780884449288360e+00, 0.8758212925230515e+00, 0.8288674605669587e+00, 0.6440226363310471e+00, 0.5894956271378661e+00, 0.5783475018163303e+00, 0.3156005123186516e+00, 0.8703974350876685e+00, 0.2617715240366199e+00, 0.8370578409687427e+00, 0.5560169751702698e+00, 0.7455275952513329e-01, 0.7082615127868872e+00, 0.1297707357428401e-02, 0.3064709066297203e+00, 0.8391571848933236e+00, 0.9253805128965463e+00, 0.1130984054668048e+00, 0.8319072101853607e+00, 0.7799276064931965e+00, 0.8579163369971375e+00, 0.5543120979067712e+00, 0.1549966158329672e+00, 0.2643029344048516e+00, 0.8314141615883028e+00, 0.5782452249644381e+00, 0.6319732449892587e+00, 0.4892864093075822e+00, 0.1584418066385224e+00, 0.1120577745932696e+00, 0.6617954786615949e+00, 0.1159882395495345e+00, 0.2876505945591634e+00, 0.7061058070010036e+00, 0.7252986915993946e+00, 0.6775944839786487e+00, 0.7230010551819884e+00, 0.6571938557215073e+00, 0.7482251296401533e+00, 0.9684494841571512e-01, 0.1940071502680237e+00, 0.8896785746779762e+00, 0.9170179284973872e+00, 0.8053995841969270e+00, 0.9321965525342302e+00, 0.7288973738137661e-01, 0.3849028957514653e+00, 0.4410947536600551e+00, 0.4770808300783691e-01, 0.3690162722923055e+00, 0.1908498348358704e+00, 0.8094133574022262e+00, 0.4809437445509840e+00, 0.1405808283203624e+00, 0.4482801388864215e+00, 0.7023724161235537e+00, 0.5041894616412674e+00, 0.2984594028552712e+00, 0.8510121345026274e+00, 0.7612412832182369e-01, 0.8890771840141870e+00, 0.8611357776125709e+00, 0.7702458212424681e+00, 0.8813091992976040e+00, 0.3838156614335277e+00, 0.1536911508311884e+00, 0.2763378258408019e+00, 0.7613349010649187e+00, 0.4228084592762277e+00, 0.5894002697827440e+00, 0.5744691724607953e-01, 0.3014767112059925e+00, 0.7221203248495245e-01, 0.7619553096876313e+00, 0.5154237548276358e+00},
			n:    21,
			zOut: []float64{0.2723457737035386e+01, 0.2426437846006860e+01, 0.2169822787916940e+01, 0.1997417054976096e+01, 0.1928996526776896e+01, 0.1891356815602834e+01, 0.1803418741268920e+01, 0.1442378108645599e+01, 0.1026854281138765e+01, 0.9766368850685674e+00, 0.8449303421617025e+00, 0.7575761304896602e+00, 0.7013909577306906e+00, 0.6362631409338654e+00, 0.3605885444762012e+00, 0.1912852882240109e+00, 0.1837209447646653e+00, 0.1221129878112117e+00, 0.2486838916706533e-01, 0.9765050532318768e-02, 0.3171294864219318e-06, 0.1371236688474761e-19, 0.2987011438498780e-15, 0.8087574764480620e-15, 0.1803418741268920e+01, 0.5451390847569379e-21, 0.7270236634641202e-11, 0.2896404140496822e-11, 0.1442378108645599e+01, 0.6954964380428437e-14, 0.4131395561099005e-08, 0.7275670618661885e-08, 0.1026854281138765e+01, 0.1292469707114106e-25, 0.1838064307199299e-06, 0.1238204560933622e-06, 0.9766368850685674e+00, 0.3388131789017201e-20, 0.3880169268725981e-06, 0.5681289313967714e-06, 0.8449303421617025e+00, 0.6785686819474261e-17, 0.2221927887785774e+02, 0.2221927887785774e+02, 0.1080000000000000e+03, 0.3244897959183673e+01, 0.4629629629629630e+01, 0.3025393053677348e-08, 0.7013909577306906e+00, 0.1341063538875720e-28, 0.3767849368753714e-07, 0.3184959175038880e-07, 0.6362631409338654e+00, 0.8874685183736383e-29, 0.5999352088776256e-07, 0.7114794232343430e-07, 0.3605885444762012e+00, 0.1582265649256097e-24, 0.2391907791779699e-03, 0.3344095791507466e-03, 0.1912852882240109e+00, 0.1130832107634320e-25, 0.4040482702647743e-03, 0.3567248396804949e-03, 0.1837209447646653e+00, 0.4930380657631324e-31, 0.1016317982961682e-02, 0.8970313840717793e-03, 0.1221129878112117e+00, 0.2076836816009399e-16, 0.1892596629513954e-02, 0.2329446640559320e-02, 0.2486838916706533e-01, 0.1713835026472478e-23, 0.3568672048225553e-04, 0.1599561599209249e-03, 0.9765050532318768e-02, 0.0000000000000000e+00, 0.3650779499194616e-03, 0.5712243726145556e-03, 0.3171294864219318e-06, 0.8599783076848839e-19, 0.2181668718924628e-02, 0.9598847336904798e-03},
			info: 0,
		},
		{
			z:    []float64{0.1133237741366075e+00, 0.6519297783748091e+00, 0.1419059404596466e+00, 0.6942573823108688e+00, 0.4899507169502382e-03, 0.4408585439270172e+00, 0.4089484333544855e+00, 0.8087151633238087e+00, 0.7299241994604067e-01, 0.7772039339909241e+00, 0.6160100823712519e+00, 0.1679565169487562e+00, 0.3276744683398485e+00, 0.8481565553508272e+00, 0.5614085421003249e+00, 0.5889702994117808e+00, 0.9865181317193454e+00, 0.5686958983255483e+00, 0.2173616127956169e+00, 0.6617740807852496e+00, 0.1324295788322836e+00, 0.5932751564887147e+00, 0.4417023572495267e+00, 0.8308006832392040e+00, 0.6963900525231828e+00, 0.7023649910778577e+00, 0.2540087053190563e+00, 0.5116605124111596e+00, 0.4580501208371790e+00, 0.9900881058795367e+00, 0.6233054379975522e+00, 0.5093215893644945e+00, 0.8805051397371310e+00, 0.5918749676473295e+00, 0.8024628157090720e+00, 0.1089848141144264e+00, 0.9385822032209372e+00, 0.9676954608704463e+00, 0.7567544231269694e+00, 0.7763472508927070e+00, 0.7544500156824916e+00, 0.7080578938468585e-02, 0.8622695184492650e+00, 0.1268681530179145e+00, 0.2408963669487220e+00, 0.1909503620360825e+00, 0.9251648557888137e-01, 0.4447315055648083e+00, 0.1504005134322962e+00, 0.6245252202415555e+00, 0.8432800589533600e+00, 0.1408178180805120e+00, 0.8538656063298837e+00, 0.6959250082173740e-01, 0.7452290984944354e-01, 0.8186895700534982e+00, 0.4817344889163616e+00, 0.8941802149688474e+00, 0.8277272846279742e+00, 0.3122760515919010e+00, 0.4631046821883726e+00, 0.9939521405533804e+00, 0.5067334594578294e+00, 0.3251044449945518e+00, 0.4514296322650755e+00, 0.9520015267726308e+00, 0.3811787795610073e+00, 0.6466473208499942e+00, 0.1035647165455822e-01, 0.3768739780085785e+00, 0.7301131130949717e+00, 0.6341150979687762e+00, 0.2006486990564113e+00, 0.7995641372737874e+00, 0.3501522989293299e+00, 0.7212117404145123e+00, 0.3724384783514179e+00, 0.2704745802524587e+00, 0.5954948132025037e+00, 0.8796174543237598e+00, 0.2533217117523398e+00, 0.2406431629179949e+00, 0.4238947154458014e+00, 0.1002294321190256e+00},
			n:    21,
			zOut: []float64{0.2763722715135843e+01, 0.2473753815483350e+01, 0.2367461218794171e+01, 0.2245079256722998e+01, 0.1833576927640889e+01, 0.1670289128561736e+01, 0.1457654822263786e+01, 0.1350493275779479e+01, 0.1277854548609735e+01, 0.1156481310792204e+01, 0.1106963375568883e+01, 0.8704957369830484e+00, 0.6845380809859644e+00, 0.4950109522946930e+00, 0.4198272059610658e+00, 0.2867476262636770e+00, 0.2180239597616727e+00, 0.1578141753335409e+00, 0.1153511714179556e+00, 0.2506658150747594e-01, 0.5048803894682267e-08, 0.8470329472540474e-20, 0.3080709680328647e-07, 0.2263725078605666e-07, 0.1457654822263786e+01, 0.6882142696441190e-21, 0.4511645736903135e-07, 0.5799182381327367e-07, 0.1350493275779479e+01, 0.3225542408567476e-17, 0.3673248914261325e-08, 0.2431637013515395e-08, 0.1277854548609735e+01, 0.1479114197289397e-30, 0.1457855695816495e-07, 0.1029378003116242e-07, 0.1156481310792204e+01, 0.9441087313685069e-24, 0.4178859307872738e-07, 0.3166226192774972e-07, 0.1106963375568883e+01, 0.1745440919131533e-19, 0.2297620589091097e+02, 0.2297620589091097e+02, 0.9600000000000000e+02, 0.3092970521541950e+01, 0.3125000000000000e+01, 0.2193086683474567e-06, 0.6845380809859644e+00, 0.6815668447404457e-13, 0.1173742870849872e-05, 0.2415236474615244e-05, 0.4950109522946930e+00, 0.4604423331594002e-25, 0.1089056660110428e-04, 0.1249035785040860e-04, 0.4198272059610658e+00, 0.2034864306780130e-21, 0.1807219455044531e-04, 0.1582973453084106e-04, 0.2867476262636770e+00, 0.1157544870450433e-16, 0.1788428346885188e-06, 0.1048647571833164e-05, 0.2180239597616727e+00, 0.2820966597070338e-26, 0.2675556367648033e-04, 0.1965631735302374e-04, 0.1578141753335409e+00, 0.4632909363938796e-19, 0.6657143729168815e-04, 0.2311126907684945e-04, 0.1153511714179556e+00, 0.7089566920435294e-17, 0.2519028315687161e-03, 0.1126597449160621e-03, 0.2506658150747594e-01, 0.1511791266627385e-18, 0.1819059120658101e-02, 0.1223404988252185e-02, 0.5048803894682267e-08, 0.5048803894682267e-08, 0.1176483960189416e-01, 0.7561758166988619e-02},
			info: 0,
		},
		{
			z:    []float64{0.6007483099411913e+00, 0.8757604992442719e+00, 0.3498665716095440e+00, 0.4974125174432915e+00, 0.3942177430818359e+00, 0.4655097178512825e+00, 0.7062255099400584e+00, 0.5955461614709365e+00, 0.5705117921786074e+00, 0.2580859610386106e+00, 0.5213970441770472e+00, 0.3227485215512098e+00, 0.8200431655548257e-01, 0.9521784777464899e+00, 0.9768302381970198e+00, 0.5401259459761943e+00, 0.2876249345369413e+00, 0.9928912621382637e+00, 0.9068004769556834e+00, 0.8136746797372185e+00, 0.6557906174128045e+00, 0.7593059043472016e+00, 0.7306718357524993e+00, 0.7872142480398446e+00, 0.2169453387448870e+00, 0.1236189910537125e+00, 0.4376414329540577e+00, 0.2475974410903709e+00, 0.5743712937757064e+00, 0.8451940172770922e+00, 0.2954362279165403e+00, 0.9333330371592371e+00, 0.7261884685452219e+00, 0.2905925760133674e+00, 0.3022535752338255e+00, 0.6907049310391071e+00, 0.2616870809800952e+00, 0.3758121789769751e+00, 0.7112612645989228e+00, 0.9602233026485287e+00, 0.7212734445945013e+00, 0.9266335266312882e+00, 0.7829804394259932e-01, 0.6683939253194693e+00, 0.3539007443404153e+00, 0.1629995906155713e+00, 0.2470024645126471e+00, 0.7250237828800384e+00, 0.9725422312806663e+00, 0.1983270881451602e+00, 0.2167236969811593e+00, 0.2406674239058443e+00, 0.3811046126380717e+00, 0.6358718370494476e-01, 0.6038903598347725e+00, 0.2622945348536884e+00, 0.4871669697876424e+00, 0.1853788955105938e+00, 0.3471794920367192e+00, 0.9520318129455714e+00, 0.9596274379126818e+00, 0.5893179749928269e+00, 0.8560240909608330e-01, 0.9435308110759013e+00, 0.2058424446004554e+00, 0.2877465111691004e+00, 0.6254635389850627e+00, 0.5518846905281638e-01, 0.1132608874973561e+00, 0.8191427866511807e+00, 0.1031178242713734e+00, 0.1909407653284053e+00, 0.5428362177906626e+00, 0.2973328401307228e+00, 0.6079142025065691e+00, 0.1448090006401505e+00, 0.5807148716343321e+00, 0.2207276122221573e+00, 0.4617526470734991e+00, 0.2133930631010816e-01, 0.8719991154365724e+00, 0.4224743304865443e+00, 0.7012315055540164e+00, 0.9970966526798359e+00},
			n:    21,
			zOut: []float64{0.2875734378878489e+01, 0.2344615299508293e+01, 0.2292398069608056e+01, 0.2200897502376554e+01, 0.2016889876654445e+01, 0.1997885510503641e+01, 0.1786196162676624e+01, 0.1505330793574375e+01, 0.1193246823746953e+01, 0.1153847743209180e+01, 0.9864509519829328e+00, 0.8076407462080507e+00, 0.6427729668463894e+00, 0.4808392927305387e+00, 0.3994667257867982e+00, 0.3273321022655395e+00, 0.1654363687174887e+00, 0.1176242694536732e+00, 0.4639514928819789e-01, 0.1592499252046327e-01, 0.3521629889976334e-03, 0.1784216669875992e-16, 0.7121431862167897e-10, 0.2812798494121821e-10, 0.1786196162676624e+01, 0.4108568523994154e-17, 0.4438281217193102e-07, 0.2274689950145052e-07, 0.1505330793574375e+01, 0.7499168293791558e-11, 0.5226296435976747e-05, 0.8641809444662033e-05, 0.1193246823746953e+01, 0.0000000000000000e+00, 0.2628223783350490e-04, 0.3987870710471355e-04, 0.1153847743209180e+01, 0.5865702946789929e-16, 0.1241308452869419e-07, 0.3894882997253744e-08, 0.9864509519829328e+00, 0.7673863643253590e-18, 0.2335727788952568e+02, 0.2335727788952568e+02, 0.1040000000000000e+03, 0.3058956916099773e+01, 0.2884615384615385e+01, 0.3948721236408985e-04, 0.6427729668463894e+00, 0.1028674620408199e-26, 0.2391232675375281e-03, 0.2086678697444007e-03, 0.4808392927305387e+00, 0.3130051437900657e-19, 0.1426864276581422e-03, 0.1522473256880454e-03, 0.3994667257867982e+00, 0.2101777886998446e-24, 0.1526280399103512e-05, 0.7002070790936748e-06, 0.3273321022655395e+00, 0.1242239255406691e-15, 0.1253374424171423e-04, 0.2848254240665514e-04, 0.1654363687174887e+00, 0.2455402723507254e-17, 0.1659752591164777e-02, 0.8476830847116842e-03, 0.1176242694536732e+00, 0.1344218982496604e-26, 0.1195138829435961e-02, 0.1583065192028977e-02, 0.4639514928819789e-01, 0.2205655090997949e-26, 0.2795062437264197e-03, 0.9604240981222648e-03, 0.1592499252046327e-01, 0.6574583373762050e-21, 0.7008887495297913e-02, 0.4078689360266482e-02, 0.3521629889976334e-03, 0.2981894221735425e-26, 0.7785429009218255e-02, 0.8697865294154498e-02},
			info: 0,
		},
		{
			z:    []float64{0.3565428176835072e+00, 0.1947118362244993e+00, 0.7741156920152520e+00, 0.2422304602782238e+00, 0.6344560478781652e+00, 0.6454994272905118e+00, 0.1782236022643452e+00, 0.1761708296759242e+00, 0.4902923959906477e+00, 0.3501715044277058e+00, 0.8606282659620700e+00, 0.4671519599404039e-03, 0.9177061631942818e+00, 0.9869535184930249e+00, 0.9615604930014852e+00, 0.2283553018405985e+00, 0.2470322441623238e+00, 0.5789642646481346e+00, 0.9797717987449011e+00, 0.2628090453859127e+00, 0.1063995753795172e+00, 0.9446865880365994e+00, 0.8109285339456792e+00, 0.4534650679402863e-01, 0.9560542860825706e+00, 0.7338084401822241e+00, 0.4719995459594307e+00, 0.8981162685661518e+00, 0.6088366551058919e+00, 0.4061662134958367e+00, 0.6141507801286759e+00, 0.7683300427463863e+00, 0.2030033710532269e+00, 0.5109903203799075e+00, 0.1295039929316932e-01, 0.1313823831505738e+00, 0.4600541675974348e+00, 0.8052912479215125e+00, 0.7315674921483241e+00, 0.1311059905375703e+00, 0.9372521946935908e-01, 0.2804058432106148e+00, 0.1426401168120447e+00, 0.1302494454623315e-01, 0.9936953989155247e+00, 0.7109841901452127e+00, 0.8438933393801377e+00, 0.5599779486257196e+00, 0.6224094357036947e-01, 0.3722277721927331e+00, 0.2128496963914419e+00, 0.2132595627624145e+00, 0.1761474673379855e+00, 0.5003566260610350e+00, 0.5448249098276142e+00, 0.5701738373290982e+00, 0.5196161461339291e+00, 0.4684002904633344e-01, 0.3091316721875115e+00, 0.9311143262655500e+00, 0.5165589926587729e+00, 0.4230658512580241e+00, 0.1580095970610971e+00, 0.9213403630754643e+00, 0.8395102804227733e+00, 0.6279936637330286e-01, 0.2118369865058433e+00, 0.5654902136914255e+00, 0.8603345552081203e+00, 0.6355651085391135e+00, 0.9504788095745560e+00, 0.4928822316317183e+00, 0.8546738890428598e+00, 0.1506739079739667e+00, 0.7955004699434830e+00, 0.8623803468730415e+00, 0.3328976137523824e+00, 0.6282681071703561e+00, 0.5253150183233042e+00, 0.5508861108006231e+00, 0.8482246434328706e+00, 0.3438184421622726e+00, 0.4546750663905129e+00, 0.2702842147468514e+00},
			n:    21,
			zOut: []float64{0.2599043716461175e+01, 0.2392272222815941e+01, 0.1945154273268249e+01, 0.1882117601080557e+01, 0.1808214125087810e+01, 0.1701027479657859e+01, 0.1674631474676720e+01, 0.1432830957842129e+01, 0.1245386641860508e+01, 0.1003047986420639e+01, 0.5925499318269597e+00, 0.5737323778075372e+00, 0.4619197085153934e+00, 0.3795861721644865e+00, 0.2896752472940557e+00, 0.2024598813435771e+00, 0.1753335138277014e+00, 0.8865983947068434e-01, 0.4088280438555415e-01, 0.2271023207656241e-01, 0.3207012114254504e-03, 0.1328658858913301e-22, 0.6931093261583948e-12, 0.1962539051213948e-11, 0.1674631474676720e+01, 0.1972152263052530e-30, 0.1003060686577864e-09, 0.7786971971568132e-10, 0.1432830957842129e+01, 0.3552283923653146e-20, 0.1335875663368593e-08, 0.1098996752778210e-08, 0.1245386641860508e+01, 0.6078417508770261e-16, 0.6005521677860925e-08, 0.4565873133117504e-08, 0.1003047986420639e+01, 0.4907378230941548e-20, 0.6042756667162823e-07, 0.1337733817342293e-06, 0.5925499318269597e+00, 0.7849166006949067e-28, 0.2051155688909552e+02, 0.2051155688909552e+02, 0.1190000000000000e+03, 0.3632653061224490e+01, 0.6722689075630252e+01, 0.9823469248695270e-04, 0.4619197085153934e+00, 0.3582825487703231e-19, 0.4408755124729528e-13, 0.6425665938157879e-14, 0.3795861721644865e+00, 0.3049769924381334e-19, 0.2649002399507504e-08, 0.1284693180187630e-07, 0.2896752472940557e+00, 0.2730342256278547e-24, 0.3857186710932351e-04, 0.7729247272182434e-05, 0.2024598813435771e+00, 0.2411244474478362e-13, 0.9974443273860175e-04, 0.7715885583051736e-04, 0.1753335138277014e+00, 0.1003583340528496e-19, 0.1275804036051463e-05, 0.1384195629265487e-05, 0.8865983947068434e-01, 0.1976477763343393e-17, 0.9250271469637329e-05, 0.1838436200742544e-04, 0.4088280438555415e-01, 0.1570480975080192e-18, 0.1665813744256122e-03, 0.9368787794988337e-04, 0.2271023207656241e-01, 0.1120555887990158e-19, 0.1174000031728850e-02, 0.1217958466428532e-03, 0.3207012114254504e-03, 0.0000000000000000e+00, 0.1126152474028795e-01, 0.4219515300090987e-02},
			info: 0,
		},
		{
			z:    []float64{0.1431540885741494e+00, 0.8417471667062104e+00, 0.3721480902981644e-01, 0.9023772086815021e+00, 0.7374866167630612e-01, 0.8092071841305744e+00, 0.8019394206305236e+00, 0.4350675566733520e+00, 0.8704440278245326e+00, 0.8549104680538234e+00, 0.1422025173760633e+00, 0.8968456705882693e+00, 0.1647179447265090e-01, 0.8994077105608889e-01, 0.8596481603583839e-02, 0.3900328616396558e+00, 0.4012630844305533e+00, 0.8262908556770313e+00, 0.9026062467432411e+00, 0.6485345137550630e+00, 0.5598932241080331e+00, 0.6084992568220661e+00, 0.5039450314115669e+00, 0.3490854750626526e+00, 0.9921372250281116e+00, 0.1522469068827847e-01, 0.9395473455386270e+00, 0.3928498832803473e-01, 0.1672951108814763e+00, 0.9480670302336689e+00, 0.8743929741539138e+00, 0.9201765927311700e-01, 0.8641536356563365e+00, 0.8867562289118487e+00, 0.7156446870854873e+00, 0.5025184567500440e-01, 0.4878799838415181e+00, 0.5568327059782646e+00, 0.4596548235310455e+00, 0.6857902774944131e+00, 0.4795565553491499e-01, 0.4752906527216701e+00, 0.9288351773531449e+00, 0.7419018575576386e+00, 0.9987802353476521e+00, 0.8896105755435116e+00, 0.3190450046252536e+00, 0.9685806853440787e+00, 0.3396383774694021e+00, 0.9164401886915974e+00, 0.1269879571285023e+00, 0.7912318128907188e+00, 0.9987805345221650e+00, 0.4107567957671243e+00, 0.4798441760727139e+00, 0.2357309197085595e+00, 0.3404494185276096e+00, 0.7067865773496134e+00, 0.2931542927088892e+00, 0.6654441577727066e+00, 0.9279529535770864e+00, 0.9667208022109988e+00, 0.5518920376663735e+00, 0.1657487215954807e+00, 0.3897636058410684e+00, 0.9489706352102197e+00, 0.9626390247015670e+00, 0.1020775856599213e+00, 0.3545784768064359e+00, 0.3098089336816090e+00, 0.3791089991258285e+00, 0.6519489570740321e+00, 0.4763523952410913e+00, 0.3754524631080590e+00, 0.9792813089520809e+00, 0.1998301915332230e+00, 0.7618060057813935e+00, 0.1923436350325713e+00, 0.2507012719817848e+00, 0.9704520069999245e-01, 0.2277384851702763e+00, 0.7790727508043876e+00, 0.9105177383528497e+00, 0.9714469561729607e+00},
			n:    21,
			zOut: []float64{0.2499720115178021e+01, 0.2371028651009860e+01, 0.2298220416774619e+01, 0.1883765650093571e+01, 0.1735426237683963e+01, 0.1616050162695453e+01, 0.1545275083087796e+01, 0.1187762610175739e+01, 0.1136805332133151e+01, 0.9576839460725933e+00, 0.9184986446054146e+00, 0.7702442640767845e+00, 0.6433137908930019e+00, 0.6160049176831036e+00, 0.3208789970551323e+00, 0.1900871628722621e+00, 0.9670499780074775e-01, 0.7460708611062000e-01, 0.6486083888512703e-01, 0.9966295048259487e-02, 0.4462615269650136e-07, 0.2765885173224186e-23, 0.3446029508568731e-11, 0.2660256849760007e-11, 0.1545275083087796e+01, 0.1351675564459578e-15, 0.5238898174323723e-11, 0.6640890994279970e-11, 0.1187762610175739e+01, 0.1020402561847179e-21, 0.5048392892414130e-10, 0.4263729817754413e-10, 0.1136805332133151e+01, 0.1429794613494979e-24, 0.1140378416083416e-09, 0.8548580688939750e-10, 0.9576839460725933e+00, 0.1302866745903472e-12, 0.1207358062805810e-07, 0.1106598026337004e-07, 0.9184986446054146e+00, 0.3538135823224864e-23, 0.2093690524456137e+02, 0.2093690524456137e+02, 0.1130000000000000e+03, 0.3331065759637188e+01, 0.7964601769911504e+01, 0.4249897555997367e-08, 0.6433137908930019e+00, 0.4627357327667341e-16, 0.5702383148705814e-08, 0.8656383665454304e-08, 0.6160049176831036e+00, 0.0000000000000000e+00, 0.4551987331271016e-07, 0.8434005257711676e-07, 0.3208789970551323e+00, 0.2939872200043623e-13, 0.4894684795516590e-05, 0.8200734504818474e-05, 0.1900871628722621e+00, 0.9080407486043539e-23, 0.2659953011765320e-05, 0.2713538128245964e-05, 0.9670499780074775e-01, 0.1949336478520425e-16, 0.1287973346598181e-04, 0.3266830247798530e-04, 0.7460708611062000e-01, 0.0000000000000000e+00, 0.5588793940970657e-03, 0.8233963212342685e-03, 0.6486083888512703e-01, 0.1680331387617770e-17, 0.5510428852120968e-02, 0.6166917183278490e-02, 0.9966295048259487e-02, 0.9229672591085838e-28, 0.3019065027713110e-04, 0.7674568090533565e-04, 0.4462615269650136e-07, 0.1609276246650864e-27, 0.6212935138424260e-03, 0.6196304337665942e-02},
			info: 0,
		},
		{
			z:    []float64{0.2117689827794353e+00, 0.3469390325266096e+00, 0.8556157513093896e+00, 0.4757820823688597e-01, 0.8174618541686680e+00, 0.7502623797079592e+00, 0.3473865910309967e+00, 0.1102061280439136e+00, 0.3838366632541155e+00, 0.4335750892772081e+00, 0.8350483733926333e+00, 0.8472841724322083e+00, 0.7001411292294977e+00, 0.1602971622162627e+00, 0.8475208106005836e+00, 0.1606731136869689e+00, 0.1282020202063155e+00, 0.7021142414638628e+00, 0.6509542291315448e+00, 0.3403267711628978e+00, 0.2783715946342622e+00, 0.2391807909006515e+00, 0.9326061845277929e+00, 0.8200594229325990e+00, 0.5326312099842309e+00, 0.4390518762985745e+00, 0.9087332134146432e+00, 0.8586305843825174e+00, 0.5289458618923581e+00, 0.6098713986238019e+00, 0.4124733909450122e+00, 0.2730411694523917e+00, 0.8437528550291284e+00, 0.6776322145297138e+00, 0.6772761566308538e+00, 0.6457716484911746e+00, 0.4290441069454122e+00, 0.9486890113840626e+00, 0.1145429178800543e+00, 0.8512453512490206e+00, 0.6435458910126746e+00, 0.6571563420730420e-01, 0.6473933144723745e+00, 0.2355560081089225e+00, 0.1262112692360912e+00, 0.6892345322103945e+00, 0.9088034154559810e-01, 0.5725023743105110e+00, 0.8924086653580375e+00, 0.6645221244511262e+00, 0.7287357579298158e+00, 0.6462379994906295e+00, 0.8254375000545862e+00, 0.2402799002378904e+00, 0.2312499677892260e+00, 0.7164295349077132e-01, 0.1216505240532725e+00, 0.5279275209153104e+00, 0.2010432640871422e+00, 0.2335833224032272e+00, 0.7053869472451330e+00, 0.1271428628255256e-01, 0.5920854820119847e-01, 0.1973220586788875e+00, 0.9024202349843203e+00, 0.2696040474399301e+00, 0.2399100427530529e+00, 0.4936945945994136e+00, 0.7377918193661529e+00, 0.4454167008139350e+00, 0.6822934143143184e+00, 0.1980205964978332e+00, 0.9382259163150929e+00, 0.7952650736079295e+00, 0.2147267423149601e+00, 0.2160850567828982e+00, 0.2063316967709339e-02, 0.4613555480251579e+00, 0.2983741464914229e+00, 0.7653433710630531e+00, 0.9319255930520460e+00, 0.9624604653562387e+00, 0.7921958309630666e+00, 0.3349247737517954e+00},
			n:    21,
			zOut: []float64{0.2554012710728313e+01, 0.2350385283314499e+01, 0.2336543487584985e+01, 0.2078331003959482e+01, 0.1819938506810449e+01, 0.1751888604422140e+01, 0.1642765030717598e+01, 0.1391850148176030e+01, 0.1265614265169072e+01, 0.1192296172100378e+01, 0.1118290194286764e+01, 0.7471001949454404e+00, 0.6602310515087256e+00, 0.5223373748122110e+00, 0.3101567956620342e+00, 0.2634177676934227e+00, 0.1468471914237014e+00, 0.9669691105737251e-01, 0.6862527639473751e-01, 0.2251868425683227e-01, 0.2442899974694956e-02, 0.0000000000000000e+00, 0.3247526756203984e-07, 0.3301672636248404e-07, 0.1642765030717598e+01, 0.2849742554922193e-13, 0.2093860354423170e-09, 0.9504157619668633e-09, 0.1391850148176030e+01, 0.5471978423050590e-17, 0.2765846142955850e-06, 0.3763175370346248e-06, 0.1265614265169072e+01, 0.7099748146989106e-29, 0.8629250150360319e-06, 0.7218261937815504e-06, 0.1192296172100378e+01, 0.0000000000000000e+00, 0.1792585742344500e-05, 0.1391262462240339e-05, 0.1118290194286764e+01, 0.2158808112824856e-22, 0.2234228955499889e+02, 0.2234228955499888e+02, 0.1130000000000000e+03, 0.3253968253968254e+01, 0.7079646017699115e+01, 0.1273852622682608e-03, 0.6602310515087256e+00, 0.1370219837932689e-23, 0.6841106514156665e-04, 0.6915431692027581e-04, 0.5223373748122110e+00, 0.8086955524743426e-16, 0.3151324787101243e-03, 0.4936870820060986e-03, 0.3101567956620342e+00, 0.1690864291947714e-17, 0.5342613094053112e-03, 0.4525307423692253e-03, 0.2634177676934227e+00, 0.1569544193046020e-19, 0.8085628512848309e-03, 0.5320256570924908e-03, 0.1468471914237014e+00, 0.1725633230170963e-30, 0.2171574910172427e-04, 0.7393923467130197e-05, 0.9669691105737251e-01, 0.8185725058097063e-19, 0.6259416979424432e-04, 0.7993629240914341e-04, 0.6862527639473751e-01, 0.1144304443709360e-17, 0.8363460972147480e-05, 0.2145990923718838e-04, 0.2251868425683227e-01, 0.3423656328659191e-26, 0.4258124540743150e-02, 0.4456054742147899e-02, 0.2442899974694956e-02, 0.7553929676107889e-20, 0.1657093833399320e-01, 0.2123214045479585e-01},
			info: 0,
		},
		{
			z:    []float64{0.5143465832338334e-01, 0.9671338481452775e+00, 0.2594126021369855e+00, 0.4887933538896223e+00, 0.4809357095031462e+00, 0.1648852435705869e+00, 0.5382296047886386e+00, 0.2404093745709619e+00, 0.5749849863516346e+00, 0.8755417117410101e+00, 0.6479432781744229e+00, 0.8462387170859478e+00, 0.1030360983310576e+00, 0.2790419505795448e+00, 0.2643216914883461e+00, 0.5072751399665939e+00, 0.9813013168637370e+00, 0.5270325239254721e+00, 0.1341185443667956e+00, 0.1666757001787883e+00, 0.8421762037577841e+00, 0.6333707418037352e+00, 0.9714076539879462e+00, 0.8133030640726957e-01, 0.5961674478430062e+00, 0.9962348063653491e+00, 0.9690228851917282e-02, 0.4933763527820312e-01, 0.1005560439675963e+00, 0.9794838233988775e+00, 0.8412211319105695e+00, 0.4530689238841823e+00, 0.9824397821984348e+00, 0.9435184376619912e+00, 0.5899369853858024e+00, 0.3347062982138894e+00, 0.6634688706474758e+00, 0.7781534216536603e+00, 0.5213269680559092e+00, 0.2415498257414400e+00, 0.4153017903771141e+00, 0.6142268391585892e-01, 0.1582256219487426e+00, 0.5180630943028907e+00, 0.2531248086553088e+00, 0.4588063905612799e+00, 0.9850645955475579e-01, 0.4386688206276981e+00, 0.6723312445641013e+00, 0.2417358982121058e+00, 0.9102202020018957e+00, 0.4790528316718639e+00, 0.8645283277006380e+00, 0.5896385561561974e+00, 0.3547186358320912e+00, 0.2570702739547999e+00, 0.8872414069371877e+00, 0.7977277358714381e+00, 0.2283476919840042e+00, 0.5481349489617042e+00, 0.9565429487676439e+00, 0.3941453871538936e-01, 0.2005355998230810e+00, 0.6815820255993522e+00, 0.8827641753074383e+00, 0.3721138471690448e+00, 0.3473678385941080e+00, 0.5958558861949184e+00, 0.5289126915593071e+00, 0.7848410946975201e+00, 0.9380898148044153e+00, 0.4238368524709774e+00, 0.9095202310640094e-01, 0.5205916750079600e+00, 0.3388678441932766e+00, 0.9218715971576251e+00, 0.2785357832665022e+00, 0.7353529037048490e+00, 0.3430576699659011e+00, 0.1583386336396764e-03, 0.1574217518812531e+00, 0.2808549182652627e+00, 0.9165809855463820e+00, 0.2729923295285186e-01},
			n:    21,
			zOut: []float64{0.2740279952468604e+01, 0.2296796691149573e+01, 0.2162929780566866e+01, 0.2022438316208162e+01, 0.1971555203048247e+01, 0.1712998698314224e+01, 0.1596820997274643e+01, 0.1536702996895977e+01, 0.1112897445143912e+01, 0.8259900943746278e+00, 0.7751147071242228e+00, 0.6231241133597348e+00, 0.5263220334234422e+00, 0.4481765711163059e+00, 0.4215577615769598e+00, 0.1440581718410880e+00, 0.1296344814538628e+00, 0.5340807481344281e-01, 0.1871489185523312e-01, 0.3434703213692513e-02, 0.2376961512853157e-03, 0.2255238469879073e-22, 0.1711146112720571e-14, 0.5183409008081618e-15, 0.1596820997274643e+01, 0.0000000000000000e+00, 0.1075504130778587e-11, 0.4005497363076952e-12, 0.1536702996895977e+01, 0.1981618593915182e-26, 0.4700742807330894e-10, 0.1164621641126979e-09, 0.1112897445143912e+01, 0.4726602108594632e-24, 0.9459556816770867e-09, 0.1692486098434743e-08, 0.8259900943746278e+00, 0.0000000000000000e+00, 0.4245107391668450e-08, 0.4612248550244559e-08, 0.7751147071242228e+00, 0.8284213834597216e-21, 0.2112319338137411e+02, 0.2112319338137411e+02, 0.1000000000000000e+03, 0.2931972789115646e+01, 0.2000000000000000e+01, 0.1818678155237499e-07, 0.5263220334234422e+00, 0.4246772729828520e-23, 0.1119198730079850e-06, 0.4602199017952358e-06, 0.4481765711163059e+00, 0.1035379938102578e-29, 0.2940687130487931e-04, 0.4282452030972280e-04, 0.4215577615769598e+00, 0.5276311108751726e-20, 0.8277389737129399e-05, 0.6004055084531256e-05, 0.1440581718410880e+00, 0.0000000000000000e+00, 0.8355340113184283e-05, 0.1023986211128489e-04, 0.1296344814538628e+00, 0.1552281967638183e-19, 0.6343521312985942e-05, 0.6697527987420460e-05, 0.5340807481344281e-01, 0.6040133837432384e-17, 0.1556990792688111e-03, 0.3593183434334682e-03, 0.1871489185523312e-01, 0.1153562559317783e-18, 0.8237486079760113e-03, 0.4973426416094711e-03, 0.3434703213692513e-02, 0.0000000000000000e+00, 0.1008741049678482e-02, 0.1888495698860201e-02, 0.2376961512853157e-03, 0.2773339119917620e-31, 0.6098206042824776e-05, 0.1474144332182451e-03},
			info: 0,
		},
		{
			z:    []float64{0.1099031735712288e+00, 0.5724174842808358e+00, 0.3122550743498126e+00, 0.7231231021981217e+00, 0.6175771623314344e+00, 0.5814310193624382e+00, 0.6423493353686718e+00, 0.8309609356975209e+00, 0.6400193102877816e-01, 0.8728840273137624e+00, 0.8682274216885312e+00, 0.7829529646382269e+00, 0.4658588928084396e+00, 0.4997300418007194e+00, 0.5750522308672811e+00, 0.5282811502940782e+00, 0.7877382060309071e+00, 0.2242905591148447e-01, 0.9672168595892823e+00, 0.9229185948074735e+00, 0.7120977511666169e+00, 0.9513613307832994e+00, 0.6495350224590044e+00, 0.1796094611119584e+00, 0.6238202085518583e+00, 0.4056553414984787e-01, 0.2863733551163383e-01, 0.7740758620163646e-01, 0.1652368945937387e+00, 0.5017774327613052e+00, 0.6445125305703996e+00, 0.5670374114090049e-01, 0.1147723121006077e+00, 0.5126855432858525e+00, 0.9526888308153642e+00, 0.1724272641761350e+00, 0.9682495468226523e+00, 0.2579549866163475e+00, 0.5365442422680400e+00, 0.7055455249537391e+00, 0.6510869341601188e+00, 0.6910931676909811e+00, 0.8245356137796966e+00, 0.2467134555988069e+00, 0.2932965007679977e-01, 0.3174819406165880e+00, 0.1008447548759832e+00, 0.7998426243165985e+00, 0.6780870841109060e+00, 0.2896057920811994e+00, 0.9303619290394473e+00, 0.3607716194531815e+00, 0.5211419140438559e+00, 0.1046120770241560e+00, 0.7683871590101677e+00, 0.2147990043395044e+00, 0.6702055898403114e+00, 0.2446830800671760e+00, 0.1275992810549672e+00, 0.9234458715873217e+00, 0.6974016894347440e+00, 0.2461420250226486e+00, 0.2448954779360711e+00, 0.1956976010946661e+00, 0.8608131397075515e+00, 0.7387865940798233e-01, 0.5053692911917109e+00, 0.3354069180988316e+00, 0.4024076719528004e+00, 0.5167351886997615e+00, 0.7829885112810930e-01, 0.3735275576161866e+00, 0.9546718920527260e+00, 0.5321377922787256e+00, 0.6121034951800618e+00, 0.8028222695558183e+00, 0.3295254582685344e+00, 0.5907471536091985e+00, 0.8247143229089846e+00, 0.1329486496858557e+00, 0.9484305946158008e+00, 0.4510190965747323e+00, 0.1159598040227069e+00, 0.7290524870797256e+00},
			n:    21,
			zOut: []float64{0.2759844065448739e+01, 0.2469029941309663e+01, 0.2170232676307031e+01, 0.1960369893365743e+01, 0.1773994559660628e+01, 0.1654180064160601e+01, 0.1454572475268221e+01, 0.1398795476174613e+01, 0.1260788731803824e+01, 0.1011169812035994e+01, 0.9032283997666860e+00, 0.7344998632955592e+00, 0.6915613254033124e+00, 0.3097922830021908e+00, 0.2074097203460417e+00, 0.1750683611049406e+00, 0.1335859329381489e+00, 0.1070304609992355e+00, 0.6466361262660925e-01, 0.1024517112836532e-01, 0.4658519939343828e-03, 0.5916456789157589e-30, 0.2414839775472174e-07, 0.1563475838564645e-07, 0.1454572475268221e+01, 0.5710570490785520e-12, 0.1598956453701674e-05, 0.9938094799470723e-06, 0.1398795476174613e+01, 0.7078291130367095e-25, 0.1301105456376455e-06, 0.7650088290739393e-07, 0.1260788731803824e+01, 0.6660939091008025e-19, 0.3865722666202593e-07, 0.5454440035759107e-07, 0.1011169812035994e+01, 0.5414223942436753e-17, 0.3344051754361652e-04, 0.2635566037358100e-04, 0.9032283997666860e+00, 0.5272968833424410e-16, 0.2125052867814009e+02, 0.2125052867814008e+02, 0.1090000000000000e+03, 0.3179138321995465e+01, 0.2752293577981651e+01, 0.8667028517906969e-06, 0.6915613254033124e+00, 0.6016648435007899e-23, 0.2921486269254270e-05, 0.9773987035049091e-05, 0.3097922830021908e+00, 0.6500213859021137e-26, 0.6786203908623285e-05, 0.1231548092049969e-04, 0.2074097203460417e+00, 0.0000000000000000e+00, 0.7819652657144090e-06, 0.1217347772587842e-05, 0.1750683611049406e+00, 0.6582255393164123e-26, 0.1458540666485106e-06, 0.3789967922095048e-07, 0.1335859329381489e+00, 0.1232595164407831e-31, 0.4615373247666497e-05, 0.3177332815969559e-05, 0.1070304609992355e+00, 0.5947376905467984e-20, 0.9914288367235413e-05, 0.1492808588608919e-04, 0.6466361262660925e-01, 0.5865227530883812e-22, 0.3814164765172745e-05, 0.1095283410948310e-05, 0.1024517112836532e-01, 0.3761300629007363e-24, 0.3159730783704485e-04, 0.5907442037524074e-05, 0.4658519939343828e-03, 0.2094421310984266e-20, 0.1426408794386696e-02, 0.1350744215392459e-02},
			info: 0,
		},
		{
			z:    []float64{0.1337238926382778e+00, 0.3106335653974364e+00, 0.8951406789337562e+00, 0.2282076226383426e+00, 0.3559900270036094e+00, 0.1333708564991680e+00, 0.9225357561814306e+00, 0.6383337981459615e+00, 0.7455817572577521e+00, 0.9604635037498299e-01, 0.1362822625756884e+00, 0.7894982141438661e+00, 0.5371956247017955e+00, 0.2757551089818502e+00, 0.9298698100988576e+00, 0.8755382224220718e+00, 0.2542040210765101e+00, 0.1287904754285962e+00, 0.6810846780454294e+00, 0.6526638813043742e+00, 0.5955845877046446e+00, 0.4670096896697419e+00, 0.5475207984291373e+00, 0.6684162034597571e+00, 0.5235410539200527e+00, 0.7337617639827179e+00, 0.8903194008545992e+00, 0.8072217495300122e+00, 0.3881183186356038e+00, 0.9675156751063907e+00, 0.4869149855955625e+00, 0.4861129300486463e+00, 0.5200718178308463e+00, 0.7592798686765068e+00, 0.7809192005748844e+00, 0.9939952535261979e+00, 0.4314053553650160e+00, 0.3880099380329156e+00, 0.9349849386467751e+00, 0.5291872418075704e+00, 0.7655812944336832e+00, 0.7077741471317599e+00, 0.5275519688342463e+00, 0.1415092279196126e+00, 0.1904335663118598e+00, 0.3699272115054363e+00, 0.7445484920377089e+00, 0.8400049615316296e+00, 0.8734700661006028e+00, 0.8683217518669656e+00, 0.8515176384024258e+00, 0.6202080985416659e+00, 0.3604998919630682e+00, 0.2403115969743893e+00, 0.9554672796937618e+00, 0.1693873282831592e+00, 0.5994144523705257e+00, 0.1384422941659869e-01, 0.6596251817370012e+00, 0.5636231004812753e+00, 0.5389166856740800e+00, 0.2607457441602791e+00, 0.5920539595206202e+00, 0.2966504229935458e+00, 0.5816050348468149e+00, 0.9328164339475690e+00, 0.5805507576670368e+00, 0.6771712743677023e+00, 0.9823651336988842e+00, 0.9793282488782439e+00, 0.4667245203343464e+00, 0.7134684364121646e+00, 0.4880671978756712e-01, 0.8383078993270572e+00, 0.7936419849408084e+00, 0.5385800856643647e+00, 0.4634644976924968e+00, 0.3931501879746360e+00, 0.1248524879525928e+00, 0.3152310060096000e+00, 0.9363438862709352e+00, 0.3520470715344234e+00, 0.1429338846554544e+00, 0.1326946626315275e+00},
			n:    21,
			zOut: []float64{0.2657315323020249e+01, 0.2572905823883155e+01, 0.2192663626292596e+01, 0.2071095653426940e+01, 0.2044494413445205e+01, 0.1999166174223334e+01, 0.1788414821296782e+01, 0.1468335212647637e+01, 0.1398203839606685e+01, 0.1347197234417596e+01, 0.1032902470665773e+01, 0.6608948302945201e+00, 0.5911878145202527e+00, 0.5378924504889658e+00, 0.3142273934265400e+00, 0.2842363451796606e+00, 0.2019484989227465e+00, 0.1206949829709270e+00, 0.7426269792367300e-01, 0.2567909008435554e-01, 0.2199972943425967e-02, 0.1467781105980630e-22, 0.1166962122444304e-12, 0.3923264640094201e-13, 0.1788414821296782e+01, 0.4047127357551305e-19, 0.1945479305169457e-09, 0.9334440121571419e-10, 0.1468335212647637e+01, 0.0000000000000000e+00, 0.9722951216797492e-08, 0.1689489155058893e-07, 0.1398203839606685e+01, 0.0000000000000000e+00, 0.7669133926546234e-07, 0.4506074380002258e-07, 0.1347197234417596e+01, 0.5319335283638860e-18, 0.5446647026201709e-11, 0.3163064243829271e-10, 0.1032902470665773e+01, 0.4430813007685208e-16, 0.2338591866968101e+02, 0.2338591866968102e+02, 0.1020000000000000e+03, 0.3054421768707483e+01, 0.2941176470588236e+01, 0.1546723845733498e-03, 0.5911878145202527e+00, 0.3652829585034170e-19, 0.1702044288822486e-03, 0.2177596193354961e-03, 0.5378924504889658e+00, 0.1350864780951217e-19, 0.2554902108144373e-06, 0.3466691719880105e-07, 0.3142273934265400e+00, 0.2387932220615686e-11, 0.6740109938598231e-04, 0.1019437079103270e-03, 0.2842363451796606e+00, 0.4338734978715565e-28, 0.4438729696644533e-03, 0.6983868198931385e-03, 0.2019484989227465e+00, 0.2612058427213185e-17, 0.1559412966194519e-03, 0.4467949014161081e-04, 0.1206949829709270e+00, 0.7367960854764250e-27, 0.5572085070843593e-03, 0.6756387538188178e-03, 0.7426269792367300e-01, 0.4936220984925504e-18, 0.5280924471496804e-02, 0.4881438130898638e-02, 0.2567909008435554e-01, 0.4675428386128583e-22, 0.7885638578366546e-02, 0.6344653647355715e-02, 0.2199972943425967e-02, 0.1043187069080892e-15, 0.1144606019330652e-02, 0.5299576363060843e-02},
			info: 0,
		},
		{
			z:    []float64{0.2890643798273074e-01, 0.8912209063965303e+00, 0.7091793456137876e+00, 0.8245039023218966e+00, 0.5191833323605244e+00, 0.9894484508239253e+00, 0.5658659268235589e-01, 0.1289734762146678e+00, 0.3341027001588562e+00, 0.6587235353882195e+00, 0.7097532970603662e+00, 0.5025058684824796e+00, 0.5130835936402325e+00, 0.6692657853529054e+00, 0.2879265169895939e+00, 0.8181258306911487e-01, 0.2380388408780171e+00, 0.1056636973917433e+00, 0.4227669640232596e+00, 0.8862610364887391e+00, 0.3174262171725216e+00, 0.4110634725577023e+00, 0.6279625314936111e+00, 0.2325114979496041e+00, 0.5638889949786475e+00, 0.7596856267416864e+00, 0.8240081240110467e+00, 0.9973320050910897e+00, 0.5587421997876323e+00, 0.4288334013103122e+00, 0.8314043340979348e+00, 0.5650188659610876e+00, 0.2964679109455457e+00, 0.8407175605087841e+00, 0.1058970658656445e+00, 0.6883026296612506e+00, 0.9166830669002658e+00, 0.7992763288082885e+00, 0.5373538962118501e+00, 0.8167960120885054e-01, 0.9998997916538879e+00, 0.1975674901149639e+00, 0.7582212604798201e+00, 0.2860808530483561e+00, 0.5862675745609662e+00, 0.4334284293442484e+00, 0.9545191575190999e+00, 0.5542276060509370e+00, 0.9135493345204525e+00, 0.9597250519108470e+00, 0.4020434518393716e+00, 0.2147209304773570e+00, 0.8396072187357826e+00, 0.5371494282057584e+00, 0.9930506172699961e+00, 0.3818967451326610e+00, 0.8312670748053316e+00, 0.7518558266385443e+00, 0.7897281313428761e+00, 0.4898691976861856e+00, 0.9259118998644267e+00, 0.2411762039549979e+00, 0.3003769542607895e+00, 0.9868194969589331e+00, 0.1841003538786936e+00, 0.1211293694641126e+00, 0.7968556025054743e+00, 0.5500013093928259e+00, 0.1705581246551358e-01, 0.5695318221187962e-02, 0.1626907733376057e+00, 0.3178865594423397e-01, 0.6489383021126400e+00, 0.3661200730578906e-01, 0.9937238435410299e+00, 0.8693948290233534e+00, 0.8108974262339508e-01, 0.4858895290851201e+00, 0.5417752479164495e+00, 0.3742432440371155e+00, 0.4593696265617863e+00, 0.7058043658513553e+00, 0.3505553117224474e+00, 0.9647100571711842e+00},
			n:    21,
			zOut: []float64{0.2698643664756728e+01, 0.2430046536911446e+01, 0.2385739474636754e+01, 0.2028138720182182e+01, 0.1901894919230603e+01, 0.1736372633374239e+01, 0.1325608228839813e+01, 0.1300504895564844e+01, 0.1261345481029282e+01, 0.1136052475360616e+01, 0.1031365869480280e+01, 0.7689316276636458e+00, 0.5233321832080760e+00, 0.4817169321317593e+00, 0.3418348106633131e+00, 0.2652183763191593e+00, 0.1871042345023881e+00, 0.9160826173360032e-01, 0.4317601760647842e-01, 0.3303497213448481e-02, 0.1231458285303120e-03, 0.9226440439387837e-18, 0.2555628409159789e-14, 0.2440793084913596e-14, 0.1325608228839813e+01, 0.0000000000000000e+00, 0.5589422112493051e-13, 0.3085898188945034e-13, 0.1300504895564844e+01, 0.7444625512849138e-23, 0.1759654299048522e-12, 0.9781103208866350e-13, 0.1261345481029282e+01, 0.6802337789204302e-18, 0.3650786119252150e-16, 0.3306223266172732e-17, 0.1136052475360616e+01, 0.9063201483067583e-22, 0.4434055300668524e-09, 0.2614666143660316e-09, 0.1031365869480280e+01, 0.2213258132402469e-23, 0.2194206198623719e+02, 0.2194206198623719e+02, 0.1100000000000000e+03, 0.3090702947845805e+01, 0.2727272727272727e+01, 0.1493749053307339e-05, 0.5233321832080760e+00, 0.6795157417200880e-17, 0.4382394072074649e-04, 0.6871024518289093e-04, 0.4817169321317593e+00, 0.6227468015696590e-20, 0.2332676874935389e-03, 0.1644354889791431e-03, 0.3418348106633131e+00, 0.8455074291031222e-24, 0.5211421342985135e-03, 0.2133578004049512e-03, 0.2652183763191593e+00, 0.4572942646937810e-19, 0.1472671843690071e-04, 0.1084821769416695e-04, 0.1871042345023881e+00, 0.4930380657631324e-31, 0.2208706795255935e-03, 0.1179477310012027e-03, 0.9160826173360032e-01, 0.2210123199165121e-23, 0.1806566548732391e-03, 0.1927419907906541e-03, 0.4317601760647842e-01, 0.5416053917983313e-23, 0.1030445327331746e-02, 0.1902956416897231e-02, 0.3303497213448481e-02, 0.8065145389241560e-19, 0.2031248978330205e-02, 0.9946626682395031e-03, 0.1231458285303120e-03, 0.1613264727387662e-23, 0.1026826157383968e-01, 0.7685291223570192e-02},
			info: 0,
		},
		{
			z:    []float64{0.9180725347208115e+00, 0.2301506372544524e+00, 0.2276838071944142e+00, 0.3342400207211600e+00, 0.9639463843013925e+00, 0.5437724527901152e+00, 0.1580993234156807e+00, 0.7877490287014776e+00, 0.3507533849023979e+00, 0.8833112450245699e+00, 0.3185731691946991e-01, 0.4118134066759526e+00, 0.8069686780110494e+00, 0.8681061806796153e-01, 0.6859948674552803e+00, 0.3663178920382139e+00, 0.3720291416615038e+00, 0.8903287156706564e+00, 0.1324429304812152e+00, 0.1008541223140741e+00, 0.8373883396136816e-01, 0.6715628313581425e+00, 0.1594467086953334e+00, 0.5413131938455139e+00, 0.1818140526389971e+00, 0.8685248287187100e+00, 0.5132298455242696e+00, 0.6886095101515632e+00, 0.4005049001067575e+00, 0.6084237383755163e+00, 0.7877676299774720e+00, 0.7102138904901485e+00, 0.6423372229000449e+00, 0.5613660527307726e+00, 0.9158966894941798e+00, 0.4933711427035579e+00, 0.7338439214039514e+00, 0.9908978412238557e+00, 0.5798575644928616e-01, 0.8977291910051136e+00, 0.9253772304099678e+00, 0.3904911857418987e+00, 0.4422593302764081e+00, 0.8489177311215401e+00, 0.7212670509555793e+00, 0.9615617648004515e+00, 0.8226716687395509e+00, 0.9612871412036802e+00, 0.1291974168797438e-01, 0.4488531482063538e+00, 0.2469769340642032e+00, 0.3614360453900619e+00, 0.2074351009622584e+00, 0.7016685284881871e+00, 0.1548060648394445e-01, 0.7311687113133817e+00, 0.9296728393442250e+00, 0.9396166536686752e+00, 0.8501954423645642e+00, 0.4336964896981472e+00, 0.6496017727538979e+00, 0.4464862127725214e+00, 0.4878335970157573e+00, 0.3486784051750773e+00, 0.7455411052079824e+00, 0.5444043691163374e+00, 0.3832092245164885e+00, 0.2030848069834680e+00, 0.9785068556455849e+00, 0.4981273340936699e+00, 0.2968547769748885e+00, 0.8139137104995479e+00, 0.5979876314385963e+00, 0.7890959952347512e+00, 0.1396638856742113e+00, 0.6423356747667288e+00, 0.4204101650717844e+00, 0.3861511203891811e+00, 0.5255853602620404e+00, 0.8105960955427686e+00, 0.1919308380441287e-01, 0.5308532240470543e-01, 0.3836913823972644e+00, 0.5510156269652182e+00},
			n:    21,
			zOut: []float64{0.2573005090179951e+01, 0.2214810120454235e+01, 0.1922733254389488e+01, 0.1915272191491466e+01, 0.1796031325760082e+01, 0.1645710570695390e+01, 0.1570469265391087e+01, 0.1354841927194508e+01, 0.1294466452193999e+01, 0.1181212352080131e+01, 0.1084541549607215e+01, 0.7996017680870424e+00, 0.7671322760534200e+00, 0.5413194973828868e+00, 0.4525283650095193e+00, 0.2478901170080125e+00, 0.2160828479985811e+00, 0.1019188530491480e+00, 0.3154199687907010e-01, 0.6039114283538317e-02, 0.2585297601905087e-05, 0.1842779074596284e-25, 0.3846373205279405e-07, 0.1879918241657939e-07, 0.1570469265391087e+01, 0.5301607341005493e-20, 0.5291292043249480e-06, 0.9680604951014117e-06, 0.1354841927194508e+01, 0.4298088920574052e-25, 0.7835178780587823e-04, 0.5534134534297650e-04, 0.1294466452193999e+01, 0.1009741958682895e-27, 0.1198198722705974e-04, 0.2465537083987539e-04, 0.1181212352080131e+01, 0.6309019138365174e-17, 0.1614759348306573e-06, 0.9017894254302745e-07, 0.1084541549607215e+01, 0.2228464273087314e-11, 0.2171715152048637e+02, 0.2171715152048638e+02, 0.1110000000000000e+03, 0.3170068027210884e+01, 0.4504504504504505e+01, 0.2430910851731611e-05, 0.7671322760534200e+00, 0.2706512346053632e-23, 0.6377722122869933e-05, 0.4589261521941547e-05, 0.5413194973828868e+00, 0.6745076284001739e-23, 0.1892848367477632e-04, 0.2486261833151780e-04, 0.4525283650095193e+00, 0.4943696629711454e-24, 0.8902582280300076e-04, 0.6878116767917398e-04, 0.2478901170080125e+00, 0.2843433355651033e-22, 0.3448699135776589e-03, 0.2736188045956721e-03, 0.2160828479985811e+00, 0.6002787666751380e-18, 0.2577148216451160e-04, 0.6049628369117275e-04, 0.1019188530491480e+00, 0.1341063538875720e-28, 0.1448550797531803e-02, 0.1280708514177034e-02, 0.3154199687907010e-01, 0.6162975822039155e-32, 0.1183522807025889e-02, 0.1098677799367788e-02, 0.6039114283538317e-02, 0.2067951531379678e-24, 0.3575820084217755e-02, 0.2195872556012757e-02, 0.2585297601905087e-05, 0.1137373342260413e-23, 0.1063898509008824e-02, 0.9861644892506420e-03},
			info: 0,
		},
		{
			z:    []float64{0.9985296910189717e+00, 0.8831740589213332e+00, 0.2324113848518039e+00, 0.7533238559688371e+00, 0.3064992106368385e+00, 0.3932363212439300e+00, 0.5064743635322475e+00, 0.2720810167440399e+00, 0.1759478514683438e+00, 0.1390889491193166e+00, 0.1132483424706631e+00, 0.7690628781223635e+00, 0.9618570638759271e+00, 0.5440138404508711e+00, 0.4137500693527351e+00, 0.1092888165750733e+00, 0.3873107100859553e+00, 0.6511277223354393e+00, 0.2155123791160785e-02, 0.9398729888067584e+00, 0.6720352720616287e+00, 0.1535603342192503e+00, 0.1531061200685264e+00, 0.7545904619621174e+00, 0.7663253741916968e+00, 0.9523625338296001e+00, 0.7564452546028717e+00, 0.3871338859213032e+00, 0.8414924166180344e+00, 0.9656526120147547e+00, 0.8983142486291256e+00, 0.4931225086134777e+00, 0.2402429523967143e+00, 0.2566457485714786e+00, 0.6291482026295649e+00, 0.3102713820139572e+00, 0.8907835976003569e+00, 0.7576089622962475e+00, 0.4268794706004859e+00, 0.9679759265393412e+00, 0.8761096912299335e+00, 0.8367482459498927e+00, 0.6924039391525048e+00, 0.1390797616032549e+00, 0.4788493402292462e+00, 0.4619292003991998e+00, 0.6108276748986761e+00, 0.4236594210420574e-01, 0.1642264441572165e+00, 0.3019681501044491e+00, 0.6872347566608095e-01, 0.4797791709329366e+00, 0.3191437336153862e+00, 0.4231397520665843e+00, 0.9123119461602809e+00, 0.4032629658718287e+00, 0.3761450089120290e+00, 0.1152232563042578e+00, 0.9954630682646954e+00, 0.5373824202779058e+00, 0.6106178691724040e-01, 0.8364420731639322e-01, 0.8933845816890847e+00, 0.5581653322395922e+00, 0.2899731736068278e-01, 0.1770979452363362e+00, 0.5004901441174880e-01, 0.7090935149513330e+00, 0.6196074778714252e+00, 0.7620558730426811e+00, 0.3663579087238161e+00, 0.7799033559850937e+00, 0.5196335588513137e+00, 0.2674625162438964e+00, 0.2943854187199277e+00, 0.7770221064689762e+00, 0.1654873860303906e+00, 0.1664508117665333e+00, 0.4846990191351895e+00, 0.8900931539882498e+00, 0.1193560423223905e-01, 0.7974122776570580e+00, 0.1940680748090599e+00, 0.6554519288132757e+00},
			n:    21,
			zOut: []float64{0.2834962035267250e+01, 0.2457739633394971e+01, 0.2412615740112653e+01, 0.2205350258735166e+01, 0.2102070731548960e+01, 0.1691268502125131e+01, 0.1682341944223941e+01, 0.1365229052855181e+01, 0.1159748452086257e+01, 0.1104152264055237e+01, 0.9082896055849982e+00, 0.8350561719565637e+00, 0.6006322679527780e+00, 0.5944104151258589e+00, 0.3054714512857917e+00, 0.1934898777304675e+00, 0.1070106456037005e+00, 0.7977404013852835e-01, 0.4467492857453391e-01, 0.1796045849894196e-01, 0.1273912617121224e-04, 0.5392013224803094e-11, 0.7047397633000775e-04, 0.4975950204285101e-04, 0.1682341944223941e+01, 0.0000000000000000e+00, 0.1776257168565356e-04, 0.2478687660984077e-04, 0.1365229052855181e+01, 0.2904595710528663e-17, 0.2723859519581362e-08, 0.9106491605280572e-08, 0.1159748452086257e+01, 0.1011100673149366e-15, 0.1332628661445848e-04, 0.5871026575773219e-05, 0.1104152264055237e+01, 0.8308334390451641e-18, 0.8332699889408906e-05, 0.9212653513444334e-05, 0.9082896055849982e+00, 0.3518078502828044e-18, 0.2270226121598308e+02, 0.2270226121598308e+02, 0.1240000000000000e+03, 0.3505668934240363e+01, 0.5645161290322581e+01, 0.2479678732743906e-05, 0.6006322679527780e+00, 0.0000000000000000e+00, 0.7879659138115500e-06, 0.7199236433698600e-06, 0.5944104151258589e+00, 0.3024585909035025e-20, 0.8203337497349541e-06, 0.8765068833865939e-06, 0.3054714512857917e+00, 0.4733165431326071e-26, 0.3737214231038995e-06, 0.3923468144379782e-06, 0.1934898777304675e+00, 0.4260044778133448e-20, 0.3232147542049518e-06, 0.3376352641996194e-06, 0.1070106456037005e+00, 0.3308722450212072e-23, 0.5070207875063323e-06, 0.4107488228647395e-06, 0.7977404013852835e-01, 0.6162975822039155e-32, 0.1529214528521930e-08, 0.6332986682625979e-08, 0.4467492857453391e-01, 0.1074272623509048e-20, 0.2583088473438835e-04, 0.1663016813733052e-04, 0.1796045849894196e-01, 0.7053209140113412e-22, 0.1477449551257491e-03, 0.9547349784916723e-04, 0.1273912617121224e-04, 0.0000000000000000e+00, 0.2066074227952428e-02, 0.2903362394245088e-02},
			info: 0,
		},
		{
			z:    []float64{0.9711900205601530e+00, 0.1638861523026915e+00, 0.1833187318057834e+00, 0.9072755753374127e+00, 0.2283578849017619e+00, 0.7520216236020175e+00, 0.3999380553956507e+00, 0.2624503967963340e+00, 0.4831766333071612e+00, 0.5439216550253211e+00, 0.6414566703448968e+00, 0.4958417903972772e+00, 0.6484413600580574e+00, 0.7544273119845675e+00, 0.7061059155531373e+00, 0.9606326194792012e+00, 0.5067069535242709e+00, 0.3029510615023524e+00, 0.6987030441702768e+00, 0.9783205951302908e+00, 0.1548952654742521e+00, 0.9315036439248046e+00, 0.5516706717897903e+00, 0.5168328579759034e+00, 0.3596295330063849e+00, 0.4476574932894306e-01, 0.9741728325615778e+00, 0.2804678894509116e-01, 0.3902916772468864e+00, 0.8530466945184888e+00, 0.2969904800916723e-01, 0.1860020491995459e+00, 0.7523094177753908e-01, 0.1049676044274535e+00, 0.9260152605248726e+00, 0.4316992548719201e+00, 0.7469073285915341e+00, 0.9833774639100495e+00, 0.4974597252506658e+00, 0.4177634673241238e+00, 0.2822911131407428e+00, 0.8441017127353869e+00, 0.3634452570924520e+00, 0.3139202807660985e+00, 0.3006600055674361e+00, 0.3009046530704764e+00, 0.5749841775931973e+00, 0.8312625997674838e+00, 0.7353697939061690e+00, 0.5347208541042444e+00, 0.3745659295783104e+00, 0.2530774659476339e+00, 0.4186076940004666e+00, 0.9926397860987635e+00, 0.9744627902125954e+00, 0.5156862870399348e-01, 0.9609885826474012e+00, 0.1340044668459911e+00, 0.4269295726396694e+00, 0.1323606439861225e+00, 0.9547002490565327e+00, 0.3235500895339022e+00, 0.9408694232081565e+00, 0.5256315334453321e-01, 0.5435043012268493e+00, 0.8290694048290136e+00, 0.3690485876123201e+00, 0.1315626820984572e+00, 0.7782838224801890e+00, 0.4824080201008953e+00, 0.4904940544366522e+00, 0.7125669967024466e+00, 0.2079010288312985e+00, 0.3154779356788950e+00, 0.8039212621301359e+00, 0.1606576564727077e+00, 0.9470564071979923e+00, 0.9189825217362574e+00, 0.4332760074849629e+00, 0.2823180730446854e+00, 0.8869162317432342e+00, 0.9632623909785668e+00, 0.5010388463829042e+00, 0.4287967421178640e+00},
			n:    21,
			zOut: []float64{0.2578695081979337e+01, 0.2397185171350384e+01, 0.2122428254475021e+01, 0.1845603260474165e+01, 0.1680081614592041e+01, 0.1588339204725779e+01, 0.1309467974766432e+01, 0.1299839243529426e+01, 0.1209324813016549e+01, 0.1078225499420584e+01, 0.1021574489264402e+01, 0.8212075951361281e+00, 0.6252200086865959e+00, 0.5691142568406321e+00, 0.4167987738801949e+00, 0.2624912719101158e+00, 0.1530089161378868e+00, 0.6867147881453296e-01, 0.2505166907249862e-01, 0.2349256929716605e-02, 0.7151879759250424e-03, 0.1577721810442024e-29, 0.4511324342265551e-10, 0.2432911773654117e-10, 0.1309467974766432e+01, 0.0000000000000000e+00, 0.2950215161815511e-09, 0.4606013118502744e-09, 0.1299839243529426e+01, 0.0000000000000000e+00, 0.1717787641934460e-08, 0.1104907179077728e-08, 0.1209324813016549e+01, 0.2227112107619961e-25, 0.8047601721100021e-14, 0.3064349892439287e-13, 0.1078225499420584e+01, 0.1519533811285116e-12, 0.1082897652895214e-09, 0.2849698526408556e-09, 0.1021574489264402e+01, 0.9160719227198137e-18, 0.2107539302297835e+02, 0.2107539302297835e+02, 0.1110000000000000e+03, 0.3317460317460318e+01, 0.3603603603603604e+01, 0.1689224645784323e-04, 0.6252200086865959e+00, 0.1615587133892632e-26, 0.8276137867398754e-04, 0.1155304449854771e-03, 0.5691142568406321e+00, 0.1852884571782062e-20, 0.2098842504693311e-03, 0.1630830816462844e-03, 0.4167987738801949e+00, 0.7013543182763534e-16, 0.5750632050789642e-07, 0.3137720527277731e-06, 0.2624912719101158e+00, 0.1295300917428640e-14, 0.8049639874935534e-04, 0.1584688939577041e-03, 0.1530089161378868e+00, 0.3636550670570672e-17, 0.1206768167404478e-03, 0.2128462525136164e-03, 0.6867147881453296e-01, 0.4176548246217679e-19, 0.3626791442965051e-04, 0.6269592722343766e-04, 0.2505166907249862e-01, 0.8628166150854817e-31, 0.1144078296038157e-02, 0.1062123559861287e-02, 0.2349256929716605e-02, 0.1853343815383365e-22, 0.1348818609641394e-03, 0.1209095310981735e-04, 0.7151879759250424e-03, 0.2331746618088476e-23, 0.3639492207606013e-03, 0.1002254703741813e-03},
			info: 0,
		},
		{
			z:    []float64{0.5145122753774686e+00, 0.7453791781450037e+00, 0.4937919697078275e-01, 0.9600912461125237e+00, 0.5558575009096904e+00, 0.9202496102671529e+00, 0.1803962777705234e+00, 0.6732582413383450e+00, 0.3646300420523177e+00, 0.4351523227625338e+00, 0.5258615952095320e+00, 0.2618724553334706e-01, 0.6600255279090639e+00, 0.4323183918305084e+00, 0.2184526174816337e+00, 0.7758998738943313e+00, 0.3700044944947983e+00, 0.9075309916724112e+00, 0.7873590137849384e-02, 0.6422780210226944e+00, 0.7012374235504933e+00, 0.3225317189575561e-01, 0.9626413579028830e+00, 0.2957325301869758e+00, 0.9393912496594896e+00, 0.3122772343061888e+00, 0.3372582329155205e-01, 0.6694758703662361e+00, 0.5160501222108982e-03, 0.9689575293997177e-01, 0.3780626069308365e+00, 0.1243877469635841e+00, 0.3753610862140089e+00, 0.7686454249086220e+00, 0.8958999262270639e+00, 0.9484254659678725e+00, 0.7229120985603210e-01, 0.7282046499931183e+00, 0.1602916509824523e+00, 0.3708722215113680e+00, 0.5328491413896952e+00, 0.7547173918812869e+00, 0.4313535833192655e+00, 0.5849601017371611e+00, 0.5030126650939655e+00, 0.3345267020968724e+00, 0.8264894134567216e+00, 0.3997968176112487e+00, 0.7977852660508306e+00, 0.3214021056175872e+00, 0.4636510192825659e+00, 0.6796405156053968e+00, 0.6279823358542748e+00, 0.1573806073104234e-01, 0.6382889890358745e-01, 0.5417145287154319e+00, 0.2343304174607242e+00, 0.7662339681589646e+00, 0.6291880658725347e+00, 0.5139706179483768e+00, 0.1583956600060378e+00, 0.9391868842090931e+00, 0.2645051239078070e+00, 0.7814123021217941e+00, 0.8945112708258808e+00, 0.7464898217422987e+00, 0.3770460515219444e+00, 0.3622886345531428e+00, 0.2208273216958112e-01, 0.7102940678247979e+00, 0.2735151315490909e+00, 0.4039400600381490e+00, 0.9317363313474538e+00, 0.5754441557390587e+00, 0.5886999087747160e+00, 0.3030246870939485e+00, 0.8507721951782303e+00, 0.3043440840669203e+00, 0.6021496993262363e+00, 0.4505272689007913e+00, 0.3874595231430722e+00, 0.2566155965014733e+00, 0.2521494866083717e+00, 0.7091848125152684e+00},
			n:    21,
			zOut: []float64{0.2432728498446405e+01, 0.2127807422546525e+01, 0.1810603620342434e+01, 0.1754551066335428e+01, 0.1442239744319712e+01, 0.1358471286041730e+01, 0.1268104094970585e+01, 0.1166598518949770e+01, 0.1142749842956014e+01, 0.8038795081997708e+00, 0.7842646316799070e+00, 0.7340488722153766e+00, 0.6406325830516326e+00, 0.6326463399427553e+00, 0.4829120798428922e+00, 0.4021501496449236e+00, 0.3628067219099092e+00, 0.1315077965186685e-01, 0.3447035773652165e-02, 0.9829838131645125e-03, 0.5542446906743292e-07, 0.3726944967513355e-18, 0.7775649827661083e-15, 0.4707679240410040e-15, 0.1268104094970585e+01, 0.4204313080465905e-25, 0.2081271826241763e-14, 0.3549560565109884e-14, 0.1166598518949770e+01, 0.0000000000000000e+00, 0.3900682806423283e-13, 0.2423637210496787e-13, 0.1142749842956014e+01, 0.1408856084028437e-16, 0.9605918786201080e-13, 0.6043593726409546e-13, 0.8038795081997708e+00, 0.8160364745046778e-12, 0.9063031760991450e-12, 0.1255592191298957e-11, 0.7842646316799070e+00, 0.2274746684496083e-23, 0.1936477583605892e+02, 0.1936477583605892e+02, 0.1150000000000000e+03, 0.3673469387755102e+01, 0.7826086956521739e+01, 0.1055105767279683e-11, 0.6406325830516326e+00, 0.7534418653829021e-12, 0.2026915084594321e-10, 0.1938443898507311e-10, 0.6326463399427553e+00, 0.7428954317778491e-22, 0.1822158222411987e-10, 0.2310105649781206e-10, 0.4829120798428922e+00, 0.6818472407494482e-21, 0.4806621018591031e-08, 0.3430760494577760e-08, 0.4021501496449236e+00, 0.9407638730741502e-13, 0.4934881466368944e-07, 0.7682525823543034e-07, 0.3628067219099092e+00, 0.4641733924717305e-17, 0.3006332568302334e-06, 0.4841414710438772e-06, 0.1315077965186685e-01, 0.0000000000000000e+00, 0.4649166787994157e-03, 0.6126205080519521e-03, 0.3447035773652165e-02, 0.0000000000000000e+00, 0.8654037114492079e-03, 0.1518309118546661e-02, 0.9829838131645125e-03, 0.2948446519354054e-25, 0.2136365904689920e-10, 0.9493111427282873e-09, 0.5542446906743292e-07, 0.0000000000000000e+00, 0.8210503952597113e-02, 0.1866280792966489e-03},
			info: 0,
		},
		{
			z:    []float64{0.1382149566425072e+00, 0.6575474867299980e+00, 0.6348472674088155e+00, 0.2171300387013793e+00, 0.8255967246227341e+00, 0.3006539504418347e+00, 0.2997341048143420e+00, 0.8472159875058289e+00, 0.6738978429295159e+00, 0.5424646868297128e+00, 0.4120014569672862e+00, 0.8343380028244362e+00, 0.3003911043596160e+00, 0.4710045367840567e+00, 0.4862652198104067e-01, 0.7471894777046170e+00, 0.1793049886163456e+00, 0.7023511329140396e+00, 0.7338019696975340e-02, 0.6396982308318109e+00, 0.9907774162255600e-01, 0.9295761484947797e+00, 0.9648060362942152e+00, 0.9280158186031068e+00, 0.2707915909911263e-01, 0.3838735516497237e+00, 0.1682677327524866e+00, 0.2113105759433432e+00, 0.9045861188128749e+00, 0.7471063495001966e+00, 0.8342180430544732e-01, 0.1412979915701199e+00, 0.8279925714974835e+00, 0.1155895711297670e+00, 0.9231783969041454e+00, 0.3728782651185074e+00, 0.6830046202574890e+00, 0.9069008465878121e+00, 0.9754648591267837e+00, 0.8893150707687680e-01, 0.6771275111272185e+00, 0.9913978119103384e+00, 0.7253101374756428e+00, 0.9967636737323063e+00, 0.8627812047783159e+00, 0.3179382291892185e-01, 0.3971194218951257e+00, 0.1929798662828303e+00, 0.8983406738830656e+00, 0.8696986427722838e+00, 0.1626550211323166e+00, 0.5446758738044107e+00, 0.7791897500950196e+00, 0.5495589328822522e+00, 0.2598162756138545e+00, 0.4878822167962611e+00, 0.1063554708684851e+00, 0.9205986422545666e+00, 0.2525019548845375e+00, 0.6202404757545382e+00, 0.4398012123976176e+00, 0.3367811528280380e+00, 0.2846930465290612e+00, 0.3637854092863660e+00, 0.2816012898088578e+00, 0.8253973364224151e+00, 0.1068693035499244e+00, 0.6335234891165292e+00, 0.4457012705011152e+00, 0.3822128202770037e-01, 0.8044679788500431e-02, 0.1644333417433916e+00, 0.8243501026702226e+00, 0.3724822902257573e+00, 0.3284619434713851e+00, 0.8284230195063381e+00, 0.9834160248288059e+00, 0.5390643127606534e+00, 0.4874602694948874e+00, 0.3393327340444893e+00, 0.9478697182248690e+00, 0.9005921446100461e+00, 0.6870582101064761e+00, 0.8753727823211164e+00},
			n:    21,
			zOut: []float64{0.2520205603715150e+01, 0.2467591671122728e+01, 0.2156156465592955e+01, 0.1830603796963098e+01, 0.1688980285573207e+01, 0.1475729352347043e+01, 0.1457403847234258e+01, 0.1216096979409916e+01, 0.1020221539309743e+01, 0.9109792709218910e+00, 0.7786685137433046e+00, 0.6943983149907305e+00, 0.5952732619368740e+00, 0.5747064141061984e+00, 0.4193783470766816e+00, 0.2849493660130012e+00, 0.2000335007304226e+00, 0.1879518797669419e+00, 0.9250201866750386e-01, 0.6720326503698931e-01, 0.2522304426258675e-08, 0.0000000000000000e+00, 0.6146929522129176e-12, 0.1808448588630822e-11, 0.1457403847234258e+01, 0.4890937612370273e-28, 0.1829381779873047e-10, 0.6063479072736104e-11, 0.1216096979409916e+01, 0.3707772472283591e-24, 0.7129364201262979e-09, 0.6841733798178873e-09, 0.1020221539309743e+01, 0.2138261883449577e-21, 0.5257033338238628e-09, 0.5072654023074907e-09, 0.9109792709218910e+00, 0.7953858577580093e-22, 0.4258628064439383e-09, 0.4402488234184115e-09, 0.7786685137433046e+00, 0.2050608176139579e-11, 0.2063903369678094e+02, 0.2063903369678095e+02, 0.1070000000000000e+03, 0.3256235827664399e+01, 0.7476635514018692e+01, 0.2680505535092421e-09, 0.5952732619368740e+00, 0.3490920289236852e-22, 0.2767075275933556e-09, 0.2579359628684292e-09, 0.5747064141061984e+00, 0.6550779105727953e-17, 0.5565909614599560e-09, 0.4340922452181151e-09, 0.4193783470766816e+00, 0.3307688474446419e-20, 0.4344481061547019e-08, 0.2950068756973984e-08, 0.2849493660130012e+00, 0.2114500635677851e-23, 0.2105683914952377e-07, 0.1386409244069912e-07, 0.2000335007304226e+00, 0.2552627671550359e-24, 0.1231406445198996e-06, 0.7724199195397155e-07, 0.1879518797669419e+00, 0.2762617809804602e-19, 0.3259696859933768e-06, 0.5428848805276070e-06, 0.9250201866750386e-01, 0.2465190328815662e-31, 0.5308529636258477e-04, 0.3674504775836459e-04, 0.6720326503698931e-01, 0.4910144116114772e-20, 0.8312610965835068e-03, 0.3682729181838852e-03, 0.2522304426258675e-08, 0.0000000000000000e+00, 0.2850162708346282e-01, 0.1272924532804709e-01},
			info: 0,
		},
		{
			z:    []float64{0.8107097465650837e+00, 0.6284212325752714e-02, 0.8801609468679439e-01, 0.1317925189426976e+00, 0.7816238529622888e+00, 0.5138440552812117e+00, 0.8491910920486024e+00, 0.5354109734403600e-01, 0.9922209940223927e+00, 0.5302203883752287e+00, 0.7634020308181488e-01, 0.1679415943232965e+00, 0.2329711998757648e+00, 0.1414305919938111e+00, 0.1841480420952157e-01, 0.3417588803467335e+00, 0.2555749764808386e+00, 0.5793635436480560e-01, 0.5867939998055902e+00, 0.9699556000711096e+00, 0.2545002439558014e+00, 0.6318762794888357e+00, 0.3917398730545906e+00, 0.5624711052874073e+00, 0.4151753215255505e-01, 0.2323556867603191e+00, 0.8346423132579881e+00, 0.4193569213002444e+00, 0.5070048505400149e+00, 0.9880568155195324e-01, 0.6350898210108906e+00, 0.2370382383359541e+00, 0.3299228061150288e+00, 0.4349239764921071e+00, 0.5262500692361848e+00, 0.5263549923651836e+00, 0.8161052184432745e+00, 0.5568267671289201e+00, 0.7852669932205985e+00, 0.4481607592551184e-01, 0.6290146498792036e+00, 0.9736787804665343e+00, 0.5369941621471508e+00, 0.6750777504837204e+00, 0.9452511173220645e+00, 0.3252402467696922e+00, 0.8149000487070703e+00, 0.8087485664240622e+00, 0.8643101435258340e+00, 0.4683483752594740e+00, 0.5054760923588362e+00, 0.9646412930917844e+00, 0.1611953384510251e+00, 0.6426851941066625e-01, 0.3732266796993424e+00, 0.6384029126799168e+00, 0.3844644469584264e+00, 0.6635273633462982e+00, 0.2553934870916856e+00, 0.7037835913610216e+00, 0.7783327991001192e+00, 0.7381327592761330e+00, 0.2765915485450557e+00, 0.9846188872724937e-01, 0.1992562101416243e+00, 0.2440052895970430e+00, 0.3301477999473963e+00, 0.5338134839029927e+00, 0.5498293871712301e+00, 0.1974401363114953e+00, 0.7407533997655510e+00, 0.6821638940995054e+00, 0.9814777780110351e+00, 0.4891456709117443e+00, 0.6868139367703679e+00, 0.4181531426430654e+00, 0.6623070699739451e+00, 0.9202771282456093e+00, 0.2973080397025486e+00, 0.4403051603022431e+00, 0.1603780540263299e+00, 0.4910562348998989e+00, 0.7838834745165658e+00, 0.1931809149066596e+00},
			n:    21,
			zOut: []float64{0.2163023058341640e+01, 0.1906105992609667e+01, 0.1885682431608721e+01, 0.1629672232824707e+01, 0.1530256904983391e+01, 0.1307296833197682e+01, 0.1220997281181084e+01, 0.9674288853031870e+00, 0.8179252376135864e+00, 0.6735586841320580e+00, 0.6320597056120635e+00, 0.5126868322939253e+00, 0.4828683228163860e+00, 0.4745823594217251e+00, 0.3685025909289511e+00, 0.2947763441663148e+00, 0.8912041525268403e-01, 0.6456597576955210e-01, 0.6259461554584676e-01, 0.1853533814873023e-01, 0.2023108580404890e-03, 0.8671199725506423e-13, 0.1722304485286045e-09, 0.1666294529992108e-09, 0.1220997281181084e+01, 0.1774302413926244e-21, 0.1370178691819834e-09, 0.1399340123848638e-09, 0.9674288853031870e+00, 0.4276620702127187e-22, 0.3173068333548179e-09, 0.2274343348931791e-09, 0.8179252376135864e+00, 0.7888609052210118e-30, 0.5644994943584269e-09, 0.8117631577107350e-09, 0.6735586841320580e+00, 0.2962158340816585e-16, 0.9564613436107998e-08, 0.6775718950029066e-08, 0.6320597056120635e+00, 0.4964115825120276e-16, 0.1710244235260994e+02, 0.1710244235260994e+02, 0.1120000000000000e+03, 0.3265306122448980e+01, 0.6250000000000000e+01, 0.5299633299932952e-07, 0.4828683228163860e+00, 0.2680954403830288e-19, 0.1311205982337445e-05, 0.9790063018353157e-06, 0.4745823594217251e+00, 0.5160286279848935e-24, 0.1890785247579508e-05, 0.1394729630636152e-05, 0.3685025909289511e+00, 0.4693722386065020e-28, 0.6031882082954460e-05, 0.2415384446316590e-05, 0.2947763441663148e+00, 0.1882813323774143e-21, 0.7814092950831008e-04, 0.8799876387761915e-04, 0.8912041525268403e-01, 0.1005797654156790e-28, 0.3017726164846613e-06, 0.1278833151586840e-06, 0.6456597576955210e-01, 0.4635851024992188e-19, 0.2982580055136947e-07, 0.4759731822531787e-07, 0.6259461554584676e-01, 0.4490807166402986e-22, 0.9170799331172770e-06, 0.3025792024297308e-06, 0.1853533814873023e-01, 0.4985600920996795e-27, 0.9814614537388964e-04, 0.3116488226164178e-03, 0.2023108580404890e-03, 0.0000000000000000e+00, 0.9878507322060364e-03, 0.9125548412055403e-03},
			info: 0,
		},
		{
			z:    []float64{0.4943635351465083e+00, 0.1295760812930432e+00, 0.7491711827268356e+00, 0.8583007697336751e+00, 0.8597988452333791e+00, 0.9020619105716389e+00, 0.2915077046929524e+00, 0.2667690273281318e+00, 0.2640508133196529e-01, 0.6592981688042527e+00, 0.5004017812459820e+00, 0.7056211227388411e+00, 0.2254860841977465e+00, 0.8570117342825790e+00, 0.6863131135194549e+00, 0.6029337708611949e+00, 0.6631267757314846e+00, 0.5332167960220966e+00, 0.3047497072553197e+00, 0.3019927731659721e+00, 0.6244714872283929e+00, 0.3204566006732108e+00, 0.9437738905138152e+00, 0.4462876020196550e+00, 0.6693016494515551e+00, 0.5687239407477801e+00, 0.5130358959101252e+00, 0.1634623710980615e+00, 0.6356065394794165e+00, 0.9187737904031696e+00, 0.8612817405760382e+00, 0.1602125514297330e+00, 0.5297426144233635e+00, 0.3699984812614213e+00, 0.3455622529703073e+00, 0.9962779456692529e+00, 0.3564481259772661e+00, 0.2073475342531472e-01, 0.5839325153640170e+00, 0.6856088547326490e+00, 0.1606020836922776e+00, 0.2475573692197572e+00, 0.8169030104559274e+00, 0.1727021335202554e+00, 0.8794153095346040e+00, 0.1052546999951709e-01, 0.4332903851899277e+00, 0.4302943891085251e+00, 0.6360909664440160e+00, 0.1924830187263967e+00, 0.6792019925733094e+00, 0.7328300669715262e+00, 0.8282900281692296e+00, 0.1586820387506954e+00, 0.5999731076117462e+00, 0.4873494502000296e+00, 0.4446262392098149e+00, 0.6430807880043099e-01, 0.4949024386471711e+00, 0.7324156973660632e+00, 0.4547667791304575e+00, 0.7542985454738853e+00, 0.3755946849636591e+00, 0.7135765429513076e+00, 0.7888131931751277e+00, 0.6384460726826821e+00, 0.4685063200928931e+00, 0.1343379262734882e+00, 0.8857603211032115e+00, 0.5798608180272281e+00, 0.8765662925729737e+00, 0.2107645345329755e+00, 0.7032612644639602e-01, 0.2672549085142431e+00, 0.5322118352980683e+00, 0.4697779217142906e-01, 0.1959612687762308e+00, 0.7346288295330818e+00, 0.4373432557584073e+00, 0.2514178299506654e+00, 0.7845430350805238e+00, 0.7823648156079857e+00, 0.6098686444303012e+00, 0.2687771415020881e+00},
			n:    21,
			zOut: []float64{0.2638973127799410e+01, 0.2299941431214051e+01, 0.2280019421450360e+01, 0.2103910562764817e+01, 0.1753406216295798e+01, 0.1675269066003242e+01, 0.1419658234290438e+01, 0.1365959224584141e+01, 0.1187696680558471e+01, 0.1126073702945326e+01, 0.8926412592525027e+00, 0.7715773240739381e+00, 0.5428647577479762e+00, 0.4329885186763779e+00, 0.3589145864992100e+00, 0.2461266947628802e+00, 0.1802035742271595e+00, 0.1083562693061272e+00, 0.7909884021715781e-01, 0.2853616071040153e-01, 0.1859995500862765e-03, 0.5169878828456423e-25, 0.7099412452593561e-13, 0.1129160994681896e-12, 0.1419658234290438e+01, 0.7523863534922662e-17, 0.1645873945742296e-11, 0.1162641799741134e-11, 0.1365959224584141e+01, 0.3171195595439500e-23, 0.6168272840237550e-11, 0.8582511762369885e-11, 0.1187696680558471e+01, 0.8519789992808060e-16, 0.1934844654065141e-10, 0.2568816579510325e-10, 0.1126073702945326e+01, 0.8564308176121538e-22, 0.5690968370753065e-10, 0.7456610067231580e-10, 0.8926412592525027e+00, 0.4500097668303756e-13, 0.2149240165292988e+02, 0.2149240165292987e+02, 0.1030000000000000e+03, 0.2950113378684807e+01, 0.4854368932038835e+01, 0.1146501122497315e-08, 0.5428647577479762e+00, 0.2966117003631004e-27, 0.2887486457338918e-08, 0.3484100742696835e-08, 0.4329885186763779e+00, 0.9478952637135678e-26, 0.8311221870318503e-08, 0.6953572191921854e-08, 0.3589145864992100e+00, 0.2375820297137957e-16, 0.1629548458200623e-07, 0.1368379043565619e-07, 0.2461266947628802e+00, 0.1012567897168591e-16, 0.3168042687452701e-07, 0.3807114536183845e-07, 0.1802035742271595e+00, 0.8688408444571280e-16, 0.7017919615597339e-07, 0.9241973720004823e-07, 0.1083562693061272e+00, 0.3747089299799806e-28, 0.3320058009883818e-06, 0.6266972429387508e-06, 0.7909884021715781e-01, 0.9269115636346889e-29, 0.6452918276099027e-05, 0.1508443954686445e-04, 0.2853616071040153e-01, 0.2499111347740165e-24, 0.3360677328899296e-03, 0.1546059694047028e-03, 0.1859995500862765e-03, 0.1972152263052530e-28, 0.5448826922428762e-03, 0.1724316467181159e-02},
			info: 0,
		},
		{
			z:    []float64{0.5641760158025995e+00, 0.2227931624220032e+00, 0.1040729150876691e+00, 0.7228520011975436e+00, 0.2490915552877161e+00, 0.4463107957028827e+00, 0.6882435259034004e+00, 0.2173580561342540e+00, 0.8146865135594571e+00, 0.3630549548424193e+00, 0.2813912484076728e+00, 0.7560034280086618e+00, 0.7800835186160227e-01, 0.9984316063995433e+00, 0.2057310185040729e+00, 0.4428139729521006e+00, 0.1946530136132755e+00, 0.8953866467149748e+00, 0.5910093704925600e+00, 0.9736056943014803e+00, 0.7342559355497796e+00, 0.5630651274913434e+00, 0.3586901690989497e+00, 0.1859950386974873e-01, 0.4012559193852120e+00, 0.5767984160414075e-01, 0.3379735138652440e+00, 0.6337834884220164e+00, 0.8830566914548466e-01, 0.2985048049064926e+00, 0.9820684138520455e+00, 0.1144932752938616e+00, 0.5905532956519596e+00, 0.4757653978696945e+00, 0.1997612463043563e-01, 0.6356348606671800e+00, 0.8478495728965164e+00, 0.4997410778119411e+00, 0.6861774168985810e+00, 0.6912919871514965e+00, 0.1276032443467088e+00, 0.2695430969348002e+00, 0.9446688115451145e+00, 0.5104603739010690e+00, 0.8863068797328323e+00, 0.3347723354379355e+00, 0.2977917172311930e+00, 0.9464871506253476e+00, 0.4279639349386329e+00, 0.6378279454376028e+00, 0.2778621310060349e+00, 0.6971058794896068e-01, 0.6309805553452471e+00, 0.3447281367207256e-01, 0.8480855191050544e+00, 0.6700112511333863e+00, 0.1616260465745026e+00, 0.7744510887458210e+00, 0.8408544531170117e+00, 0.8444763755373497e-01, 0.3286971575766536e+00, 0.8244867044222992e+00, 0.8036107174439190e+00, 0.4654902702844697e+00, 0.8300602115903392e+00, 0.5911511948645825e+00, 0.5734805041784674e+00, 0.6046912223886362e+00, 0.4400460907177143e+00, 0.4069795298963303e+00, 0.2413714645456244e+00, 0.7838758775665940e+00, 0.1371288591293247e+00, 0.8435169936355635e-01, 0.4867567406491472e+00, 0.2914527168883616e+00, 0.7540094160758792e+00, 0.6792178828893637e+00, 0.8924641252729165e+00, 0.2188072300242774e+00, 0.1899571860965691e+00, 0.7806965639263314e+00, 0.8706240780095753e-01, 0.7170634647198669e+00},
			n:    21,
			zOut: []float64{0.2553381332090116e+01, 0.2166390759033473e+01, 0.1737362408504838e+01, 0.1528419999405254e+01, 0.1507484707859341e+01, 0.1334735890703865e+01, 0.1222641790395049e+01, 0.1186722126839839e+01, 0.1073660052771764e+01, 0.9488489095119069e+00, 0.9141860725715962e+00, 0.7726247731371300e+00, 0.6075856291402457e+00, 0.5409244297850613e+00, 0.4230641503671385e+00, 0.2356839373494127e+00, 0.1583434617720056e+00, 0.3187519618749328e-01, 0.2747107248999187e-01, 0.1480095837386152e-02, 0.5569185181399456e-04, 0.3215260779752862e-17, 0.2284971140234050e-13, 0.6915955898032105e-13, 0.1222641790395049e+01, 0.9860761315262648e-31, 0.9831000121605633e-10, 0.3802561298586086e-10, 0.1186722126839839e+01, 0.2066815571679051e-27, 0.6142487424703478e-09, 0.1524532000885779e-08, 0.1073660052771764e+01, 0.3155443620884047e-29, 0.1933670198753400e-07, 0.4311109498288097e-07, 0.9488489095119069e+00, 0.6613963504021090e-17, 0.7125410558010122e-07, 0.6137903750393884e-07, 0.9141860725715962e+00, 0.2713620929442760e-22, 0.1897294248760472e+02, 0.1897294248760472e+02, 0.1070000000000000e+03, 0.3233560090702948e+01, 0.4672897196261682e+01, 0.2313496546014482e-05, 0.6075856291402457e+00, 0.2008094738046662e-26, 0.6354651604911740e-04, 0.4633045629030685e-04, 0.5409244297850613e+00, 0.2465190328815662e-30, 0.8082164119218066e-03, 0.5151838483992701e-03, 0.4230641503671385e+00, 0.3299331849996360e-25, 0.9881103085087938e-03, 0.1036550305023510e-02, 0.2356839373494127e+00, 0.2111622871095604e-25, 0.1203513003711909e-02, 0.1300398061184096e-02, 0.1583434617720056e+00, 0.1737664936697394e-23, 0.8260928074113359e-03, 0.9484297179228856e-03, 0.3187519618749328e-01, 0.0000000000000000e+00, 0.1785275423606748e-03, 0.4485462762371783e-03, 0.2747107248999187e-01, 0.8804949879714635e-25, 0.5200632154389635e-08, 0.3234666977722410e-08, 0.1480095837386152e-02, 0.5399595124056782e-24, 0.5661901580915454e-02, 0.5860254353052324e-02, 0.5569185181399456e-04, 0.0000000000000000e+00, 0.1190560893471898e-01, 0.9131404251707014e-02},
			info: 0,
		},
		{
			z:    []float64{0.7290518158132969e+00, 0.5196659176766039e-03, 0.6083923160589086e+00, 0.3459319602615154e+00, 0.3277021139736517e+00, 0.6020592719504614e-01, 0.4608784204502808e+00, 0.8430478676154098e+00, 0.3856212930081542e+00, 0.6602823264317716e+00, 0.4451601253283373e+00, 0.4130313322309198e-01, 0.7414750648249350e+00, 0.1630235543068703e-01, 0.6004925370465983e+00, 0.6953287054569501e+00, 0.7809046354974044e+00, 0.7892746585788727e+00, 0.8348704064067982e+00, 0.5999411739898802e+00, 0.5502575606089829e+00, 0.4392206918044746e+00, 0.7984758158185817e+00, 0.9084185319619590e+00, 0.1786837110145214e+00, 0.2414921532322530e+00, 0.5630913108462098e+00, 0.7820323704275389e+00, 0.5523326888742272e+00, 0.9015512185421630e+00, 0.6659841765947113e+00, 0.2845604319423287e+00, 0.3491764093356571e+00, 0.9628247275953050e-01, 0.6976897044251388e+00, 0.2960314575787897e+00, 0.5554565606281359e-01, 0.3558757052257084e+00, 0.9341850945519581e+00, 0.6665938574616496e+00, 0.9614102712845256e+00, 0.8607214960584614e+00, 0.4038740733880183e+00, 0.8195623831081322e+00, 0.6129183541897144e+00, 0.7215327066311872e-01, 0.5031970254369666e-01, 0.1396452298202989e+00, 0.9662061833347503e+00, 0.1270920838674061e+00, 0.4311106155362532e+00, 0.8052749852152479e+00, 0.4015794631208873e+00, 0.9090852867289334e-01, 0.6606775367976888e+00, 0.1481179966505828e+00, 0.9638580924515038e+00, 0.9709844996525795e+00, 0.1150673643193312e-01, 0.1429293205078334e+00, 0.2900042758147406e+00, 0.9072348766684573e+00, 0.5781273728635536e+00, 0.5436950045999074e+00, 0.1857622262967125e+00, 0.9164714647876027e+00, 0.1370982861482903e+00, 0.4992756875010576e+00, 0.1554652709586491e+00, 0.3126479857720679e+00, 0.4107075699045055e+00, 0.2895846225773548e+00, 0.6273125726015746e+00, 0.2838905730735005e+00, 0.5809021463377108e+00, 0.7647718231557297e+00, 0.2069816461533343e+00, 0.3140055757156704e+00, 0.1235726138058405e+00, 0.4368723232060789e-01, 0.7675622822273089e+00, 0.9898297711455730e+00, 0.5845844813092991e+00, 0.3724928805727834e+00},
			n:    21,
			zOut: []float64{0.2656606156715505e+01, 0.2361088067613179e+01, 0.2270460825471249e+01, 0.2122871466419806e+01, 0.1864972075992565e+01, 0.1483766917780093e+01, 0.1313593002893207e+01, 0.1121546276208008e+01, 0.1100149868182268e+01, 0.9659264313206680e+00, 0.7624972132947719e+00, 0.7289080649182198e+00, 0.6752163576811447e+00, 0.6611764720975886e+00, 0.5004340625462813e+00, 0.2440674815088013e+00, 0.2171809202435314e+00, 0.1235497606154792e+00, 0.3737121186344041e-01, 0.2834840406904476e-01, 0.5846755428136478e-02, 0.7443896006885402e-16, 0.5016677150024004e-07, 0.3707799983650212e-07, 0.1313593002893207e+01, 0.2958228394578794e-29, 0.1581957950672119e-06, 0.2298083114979900e-06, 0.1121546276208008e+01, 0.1915993820645148e-16, 0.4749045639556273e-06, 0.4145613346073258e-06, 0.1100149868182268e+01, 0.9026685982667287e-18, 0.2821283116535875e-06, 0.3158894087015811e-06, 0.9659264313206680e+00, 0.2037048580685469e-20, 0.2246093425341190e-06, 0.2076005360058006e-06, 0.7624972132947719e+00, 0.1234566306928925e-21, 0.2124557779286299e+02, 0.2124557779286298e+02, 0.1010000000000000e+03, 0.3002267573696145e+01, 0.2970297029702970e+01, 0.4897987464996691e-04, 0.6752163576811447e+00, 0.7461812581271996e-18, 0.7761124913531668e-04, 0.7483244468930916e-04, 0.6611764720975886e+00, 0.1038114253693458e-19, 0.1582094808245393e-03, 0.1235322388201987e-03, 0.5004340625462813e+00, 0.1010613662291919e-16, 0.2610534203903574e-03, 0.4450812691210283e-03, 0.2440674815088013e+00, 0.1110435042389887e-20, 0.5668320309921573e-04, 0.5311544381771655e-04, 0.2171809202435314e+00, 0.4733165431326071e-27, 0.6599010146996593e-04, 0.7896621041843452e-04, 0.1235497606154792e+00, 0.3155385912111625e-18, 0.1746230697391169e-01, 0.1331928313395539e-01, 0.3737121186344041e-01, 0.7395570986446986e-31, 0.1364698415102933e-02, 0.9400709901879650e-03, 0.2834840406904476e-01, 0.1581043861486090e-22, 0.9826835453665868e-03, 0.1307033404080312e-02, 0.5846755428136478e-02, 0.5593364605928029e-23, 0.6675960788692170e-02, 0.6853670175678725e-02},
			info: 0,
		},
		{
			z:    []float64{0.4059771234161322e+00, 0.9607215623246773e+00, 0.2458768097786967e-01, 0.2349999394464353e-01, 0.5195584553698999e+00, 0.4267565401751807e+00, 0.5653505726897825e+00, 0.1993982232067181e+00, 0.7123509462851438e+00, 0.7322091078130321e+00, 0.6397865818692764e+00, 0.7795245712118530e+00, 0.9924677795119503e+00, 0.9446777193650685e+00, 0.9161440637569995e+00, 0.4356325057978820e+00, 0.6968317286853016e+00, 0.8623411680463823e+00, 0.8789901991689911e+00, 0.3964880116495537e+00, 0.3452640838632506e+00, 0.2457053330236874e+00, 0.3088650440709877e+00, 0.3819576657222301e+00, 0.3428589632713919e-01, 0.5108638376023356e-01, 0.1646491337519898e+00, 0.4402398340108123e+00, 0.6741907107293844e+00, 0.8399512836956711e+00, 0.5689849385173426e+00, 0.5208530271268840e+00, 0.1800472375386978e+00, 0.7659410457647264e+00, 0.2320702606914905e+00, 0.1105800266048680e+00, 0.9555355666039600e-02, 0.1859192596152662e+00, 0.7825627400019242e-02, 0.5286418396370182e+00, 0.2569279638014341e+00, 0.5549104171411018e+00, 0.7406835450501670e+00, 0.8319568443444482e+00, 0.1523286720686696e+00, 0.6741822248281739e+00, 0.7686049385045001e+00, 0.7046090895325479e+00, 0.5187887151753254e+00, 0.1082987708911324e+00, 0.1267838014025160e+00, 0.1115865001832446e+00, 0.9799554493413084e+00, 0.2112830058825086e+00, 0.1414836977037485e+00, 0.6416690587599562e+00, 0.7257743058080667e+00, 0.5941533679852271e+00, 0.2889670048515264e+00, 0.2352763591376699e+00, 0.4250778140801711e+00, 0.1017573282390372e-01, 0.7592168685641336e+00, 0.1236630021486789e+00, 0.3794258137484455e+00, 0.3894249799986492e+00, 0.3467719607731882e+00, 0.5651244918121399e+00, 0.7941305753999619e+00, 0.7160004674307343e+00, 0.6194397297375442e+00, 0.1453439098617406e+00, 0.3817157280391337e-03, 0.9381666563671931e+00, 0.8543357988390619e+00, 0.4675645670082479e+00, 0.8133557631373793e+00, 0.1160066189475003e+00, 0.6551894700973839e+00, 0.6157191087148942e+00, 0.4651705119652626e+00, 0.2935400080400429e+00, 0.4317648586994048e+00, 0.8522331953027451e-01},
			n:    21,
			zOut: []float64{0.3090946182879871e+01, 0.2427731098824810e+01, 0.2178358732328273e+01, 0.1944657148720347e+01, 0.1388313875420951e+01, 0.1366009016577358e+01, 0.1254539813854158e+01, 0.1096743376160052e+01, 0.8976114875705017e+00, 0.7925182471468240e+00, 0.7633415317747125e+00, 0.5940562138773567e+00, 0.3406702196684873e+00, 0.2541261981011808e+00, 0.2081990106830526e+00, 0.1967266434974441e+00, 0.8212572008660911e-01, 0.6231598248219099e-01, 0.2121611392818772e-01, 0.6627852439078377e-02, 0.2020564063973749e-05, 0.5156511541642804e-17, 0.3563959083802911e-09, 0.2193664839500192e-09, 0.1254539813854158e+01, 0.5107396913706617e-17, 0.2245013704648247e-08, 0.3514139913607919e-08, 0.1096743376160052e+01, 0.1770166510860316e-21, 0.1137128600534393e-07, 0.7589514151253922e-08, 0.8976114875705017e+00, 0.3631375006656261e-16, 0.1027683956127825e-09, 0.1703981899256795e-09, 0.7925182471468240e+00, 0.0000000000000000e+00, 0.2692111438183261e-08, 0.1491447500870467e-08, 0.7633415317747125e+00, 0.3655079503368504e-18, 0.1896683648658551e+02, 0.1896683648658551e+02, 0.9800000000000000e+02, 0.2941043083900227e+01, 0.3061224489795918e+01, 0.1473194721443778e-07, 0.3406702196684873e+00, 0.8487843002529896e-19, 0.1107295924775035e-05, 0.6897939613884539e-06, 0.2541261981011808e+00, 0.8699892772041781e-19, 0.4010969039165872e-05, 0.5564243023106877e-05, 0.2081990106830526e+00, 0.1258849778182470e-17, 0.2192657884798391e-05, 0.1921286515322920e-05, 0.1967266434974441e+00, 0.1498835719919922e-28, 0.3181611995531238e-06, 0.2328457188758453e-05, 0.8212572008660911e-01, 0.9162137014768053e-16, 0.4655593170815260e-04, 0.2906458517213153e-04, 0.6231598248219099e-01, 0.7111877894410489e-19, 0.1084455050840344e-03, 0.4940544045928809e-04, 0.2121611392818772e-01, 0.1296887328183343e-24, 0.1059468102013224e-02, 0.9250155264543548e-03, 0.6627852439078377e-02, 0.2014616961124939e-22, 0.1082171160654875e-03, 0.1415095048461561e-03, 0.2020564063973749e-05, 0.0000000000000000e+00, 0.6572087989553770e-04, 0.6490933881766271e-04},
			info: 0,
		},
		{
			z:    []float64{0.3515867373170093e+00, 0.6900024519663606e-01, 0.2562617636787797e+00, 0.4448144612254716e+00, 0.1882829208975508e+00, 0.5473764707914393e+00, 0.2385465140832452e+00, 0.1069888484826550e+00, 0.6650479699038470e+00, 0.5577458834974632e+00, 0.3827016483759793e-01, 0.2430476485682098e-01, 0.4730689827049095e-01, 0.7028807845337007e+00, 0.2183345614483843e+00, 0.4504620631941809e+00, 0.4989666416356805e+00, 0.3524179690092576e+00, 0.2245980971932510e+00, 0.4235501615333767e+00, 0.5262756658101302e+00, 0.9543218824354944e+00, 0.7932768470961233e+00, 0.6115989702914060e-01, 0.5090745247561921e+00, 0.6231416396947956e+00, 0.5848645563028462e+00, 0.4261243979220120e-01, 0.7488050001299923e+00, 0.1371419590365067e+00, 0.2276573560104874e+00, 0.3333510148295767e+00, 0.8801005866377587e+00, 0.1783788042757909e+00, 0.7167665061134074e+00, 0.8328416386285911e+00, 0.3239179761297745e+00, 0.3396256067647054e+00, 0.3124818554244235e-01, 0.3558253686357221e+00, 0.3434895378785284e+00, 0.5623459587025392e-01, 0.2638452014850361e+00, 0.8291744484791080e+00, 0.1974847584119865e+00, 0.7319795333910079e+00, 0.2610606436005336e+00, 0.5710127034520260e+00, 0.1439281438102160e+00, 0.9086761346521347e+00, 0.8079247890865698e+00, 0.9018578929430654e+00, 0.4900454364738460e+00, 0.8138610668520670e+00, 0.6711229676830790e+00, 0.5055629692453131e+00, 0.7743169767236130e+00, 0.4534345901632091e+00, 0.9064089158232752e+00, 0.4827899713942460e+00, 0.7358703780808750e+00, 0.9369729458478693e-02, 0.8597279323370060e+00, 0.7253136203352766e+00, 0.9571052925069841e+00, 0.2164676786228616e+00, 0.5574107740713430e-05, 0.7154716238932374e+00, 0.1141135068223202e+00, 0.6209438587853456e+00, 0.7851847140042024e+00, 0.4215540797089058e+00, 0.9132020640887749e+00, 0.1510454496157759e+00, 0.4172986352579149e+00, 0.5037450189692434e+00, 0.4485087403424131e+00, 0.5740668076879305e+00, 0.2978796610784261e+00, 0.6581099078338648e-01, 0.8930273171304047e+00, 0.8152477689300778e+00, 0.1069556485483132e+00, 0.6932157626522990e+00},
			n:    21,
			zOut: []float64{0.2205969711876647e+01, 0.1949821527895322e+01, 0.1555841877124327e+01, 0.1396030697440207e+01, 0.1339074151471965e+01, 0.1304928492707023e+01, 0.1082549957806062e+01, 0.9163938874206132e+00, 0.8887122139609793e+00, 0.7285328536138327e+00, 0.6136134334950112e+00, 0.5962413214799335e+00, 0.4250735423840991e+00, 0.3604939377250871e+00, 0.2295486599338548e+00, 0.1443459206161604e+00, 0.1184188267216090e+00, 0.7075746168152415e-01, 0.2006576919306350e-01, 0.3756273174580983e-02, 0.4503973951446061e-03, 0.2214970142320351e-19, 0.1258326061053631e-10, 0.9003557400980630e-11, 0.1082549957806062e+01, 0.1308878013942703e-25, 0.3368372626791194e-10, 0.2572022209508410e-10, 0.9163938874206132e+00, 0.7614331624527915e-16, 0.1049500061330476e-08, 0.2928487158224854e-08, 0.8887122139609793e+00, 0.1203547791264655e-21, 0.7079518778653070e-07, 0.2057204985301009e-06, 0.7285328536138327e+00, 0.1084769825180869e-21, 0.6504687546036223e-07, 0.2011551828019932e-07, 0.6136134334950112e+00, 0.1678480469223895e-16, 0.1595062091511705e+02, 0.1595062091511705e+02, 0.1020000000000000e+03, 0.3156462585034014e+01, 0.2941176470588236e+01, 0.1336032003974816e-08, 0.4250735423840991e+00, 0.1033222323690773e-11, 0.1230907576002882e-05, 0.1377773980605759e-05, 0.3604939377250871e+00, 0.5435093612356237e-21, 0.5904438579725169e-05, 0.1247176535660427e-04, 0.2295486599338548e+00, 0.3794809073678436e-22, 0.8654598111714590e-03, 0.7607233838455693e-03, 0.1443459206161604e+00, 0.2335329392584174e-18, 0.9486551585203510e-04, 0.5848864829043123e-04, 0.1184188267216090e+00, 0.2421802979028506e-27, 0.4440929582264130e-04, 0.3669242548994226e-04, 0.7075746168152415e-01, 0.1891688450719986e-26, 0.5237855019006247e-04, 0.2468105637453731e-04, 0.2006576919306350e-01, 0.1180787648903814e-27, 0.8134126440113327e-03, 0.8865157422262554e-03, 0.3756273174580983e-02, 0.2581152881883151e-26, 0.5819544721666204e-02, 0.3152501351079271e-02, 0.4503973951446061e-03, 0.0000000000000000e+00, 0.5880303886253363e-04, 0.1585743718762697e-03},
			info: 0,
		},
		{
			z:    []float64{0.5610544265871693e+00, 0.3865731523333528e+00, 0.8792353035941016e+00, 0.8523516652694250e+00, 0.1444314813898040e-01, 0.8826603847895033e+00, 0.6868176286586357e+00, 0.1885799699020001e+00, 0.8756021324147915e+00, 0.9000748303342890e+00, 0.6057362077051875e+00, 0.7116572505145777e+00, 0.4651180401287146e+00, 0.2732222796277717e-01, 0.4226433912686857e+00, 0.2137613193249559e+00, 0.8364980217841237e+00, 0.7760779616422476e+00, 0.4204777290352190e+00, 0.6224635225380051e+00, 0.2764474942580449e+00, 0.3783169947455127e+00, 0.5180995217194659e+00, 0.3065081068358929e+00, 0.7865831746887696e+00, 0.1478697978858728e+00, 0.9439830303880672e+00, 0.8253272390565236e-01, 0.3067445628749279e+00, 0.7129198739696287e+00, 0.1655397340526111e+00, 0.1744497852178990e+00, 0.8123176080018479e-01, 0.7468152661528306e+00, 0.8702942511684653e-01, 0.2665897748347560e+00, 0.9213749341388631e+00, 0.1523897006311256e+00, 0.9645030292913545e+00, 0.9612618327522493e-01, 0.4802656873918926e+00, 0.9400504426452867e+00, 0.4625574582408437e+00, 0.7387695442245192e+00, 0.7695082904503155e+00, 0.1394488941705607e+00, 0.4086909510206329e+00, 0.8420266381115991e+00, 0.9564685490270811e+00, 0.7091850518571272e+00, 0.5799173993611073e+00, 0.1780919033120022e+00, 0.4196947685163247e+00, 0.9892865772966106e+00, 0.5502405891255741e+00, 0.7805747931381838e+00, 0.8443114386354278e-01, 0.3818004737775779e+00, 0.5006744900666209e+00, 0.8330036981446504e+00, 0.6972526215524509e+00, 0.5997317354738497e+00, 0.6745658299087246e+00, 0.2369187863883299e+00, 0.4162907871251388e+00, 0.3987550165661536e+00, 0.3127240434721901e+00, 0.9587921004875174e+00, 0.9734600215022703e+00, 0.9507333301136496e+00, 0.3445022862066339e+00, 0.6240012410156072e+00, 0.3036264632031094e-01, 0.6999839547669153e+00, 0.4672138296892350e+00, 0.4669965382645248e+00, 0.7561275857160221e-01, 0.9250454776547237e+00, 0.8620177110728862e+00, 0.5266728805873626e+00, 0.6377535363121675e+00, 0.6098026285363055e+00, 0.8075088997828720e+00, 0.8486821693163010e+00},
			n:    21,
			zOut: []float64{0.2468092632282687e+01, 0.2093427673862981e+01, 0.2055392019191517e+01, 0.1647867143267642e+01, 0.1615849876515927e+01, 0.1589328968238607e+01, 0.1271214388518367e+01, 0.1121382381594411e+01, 0.1069461264755295e+01, 0.1002295596318809e+01, 0.8815764834354939e+00, 0.8051252546404786e+00, 0.6374343329196452e+00, 0.5469723867709347e+00, 0.4015178604185544e+00, 0.3707602923265790e+00, 0.1795105850679969e+00, 0.1400595531050117e+00, 0.2548349514639473e-01, 0.1561366451013735e-02, 0.1155321273825210e-02, 0.3137752941767923e-22, 0.1448569525610494e-09, 0.1682297708320945e-10, 0.1271214388518367e+01, 0.3747089299799806e-29, 0.3127891707973917e-07, 0.1432924100618686e-07, 0.1121382381594411e+01, 0.1829309924661015e-19, 0.1414348814679943e-06, 0.1338467241717145e-06, 0.1069461264755295e+01, 0.2814655709828570e-26, 0.1421706541245010e-06, 0.1494636997707923e-06, 0.1002295596318809e+01, 0.2555408239179452e-16, 0.1745096068338993e-06, 0.1639215185470380e-06, 0.8815764834354939e+00, 0.5195421261043265e-21, 0.1992546887610216e+02, 0.1992546887610217e+02, 0.1040000000000000e+03, 0.3263038548752835e+01, 0.3846153846153846e+01, 0.5490557769901970e-06, 0.6374343329196452e+00, 0.1696050946225175e-28, 0.4391959465127471e-06, 0.4441570425561870e-06, 0.5469723867709347e+00, 0.4122920742224575e-16, 0.4792107785920969e-06, 0.5244422797027597e-06, 0.4015178604185544e+00, 0.4437342591868191e-30, 0.1189624719598388e-05, 0.1545254324980458e-05, 0.3707602923265790e+00, 0.0000000000000000e+00, 0.3830671301160838e-05, 0.5436396467135237e-05, 0.1795105850679969e+00, 0.1558228073167050e-14, 0.5423210427919237e-04, 0.1228583690085413e-03, 0.1400595531050117e+00, 0.8832743148296122e-19, 0.1395228897382397e-04, 0.1537585917376726e-04, 0.2548349514639473e-01, 0.0000000000000000e+00, 0.8277350694369951e-03, 0.9528953717632958e-03, 0.1561366451013735e-02, 0.2465190328815662e-31, 0.7095009159403311e-03, 0.6952609237045969e-03, 0.1155321273825210e-02, 0.1309509102666880e-27, 0.9046555260897671e-03, 0.1226313636978587e-02},
			info: 0,
		},
		{
			z:    []float64{0.1242522906682814e+00, 0.6390609620209767e+00, 0.4481822861180138e+00, 0.4850355616354706e+00, 0.4018997557884576e+00, 0.7075817751574985e+00, 0.3076496509717662e+00, 0.9845619472128853e+00, 0.4421050939804582e+00, 0.3472138710835164e+00, 0.7694977672753175e+00, 0.4786944038481857e+00, 0.6974278852457209e+00, 0.5004517708240744e+00, 0.7658513494839985e+00, 0.7508522623862479e-01, 0.8645614962374704e-02, 0.2513671118506070e+00, 0.9355687048630774e+00, 0.2836435232395548e+00, 0.8653687008315966e+00, 0.3043727382738487e+00, 0.8973655399535756e+00, 0.7135517870607274e+00, 0.4956863425678929e+00, 0.8362506849216409e+00, 0.9829450079482006e+00, 0.9413718361369072e+00, 0.2542921002853715e-01, 0.5951158954104814e+00, 0.9317747763384295e+00, 0.9085271053958202e+00, 0.9993466426511500e+00, 0.2497009223772528e-01, 0.8726460080387569e+00, 0.3868463012727243e+00, 0.5820757557404177e-01, 0.7974577837432384e-01, 0.8242609714155934e+00, 0.1887913799920241e+00, 0.3268225406700475e+00, 0.2264596835393817e+00, 0.4020160264981738e+00, 0.8597685353848280e+00, 0.5429598215579996e+00, 0.4134507486351844e+00, 0.8781720494573462e+00, 0.3655827200857601e+00, 0.2876763179222336e+00, 0.9193112804533413e+00, 0.3958413207165046e-01, 0.7980401670837647e+00, 0.5101422689515223e-01, 0.2194050847732673e+00, 0.6115669547384739e-02, 0.9036470382476453e+00, 0.6696681747677364e+00, 0.1689100553906829e+00, 0.9284887836539969e+00, 0.8748192507086142e+00, 0.7181738874264668e+00, 0.8990747837549429e-01, 0.7166811698029575e+00, 0.8712804839027968e+00, 0.2571898936326318e+00, 0.2264160460654618e+00, 0.6364724288557849e+00, 0.9550337894922362e+00, 0.8995892333665090e+00, 0.3728949594240628e+00, 0.6496276036419958e+00, 0.7218580021816370e+00, 0.2620053580399828e+00, 0.8492826434932900e+00, 0.1090756692624820e+00, 0.1230806328807070e+00, 0.5326342483791896e+00, 0.3425391815117244e+00, 0.4714474424671373e+00, 0.9044440560476328e+00, 0.2298320061743346e+00, 0.7771450381369926e+00, 0.4623588499686800e+00, 0.8996441805847293e+00},
			n:    21,
			zOut: []float64{0.2749036998648791e+01, 0.2657147938200912e+01, 0.2156780738808330e+01, 0.2057290308234013e+01, 0.1910481192038085e+01, 0.1550197528741744e+01, 0.1531663815430223e+01, 0.1307411568456853e+01, 0.1180055477995477e+01, 0.1145160315453326e+01, 0.8462599780670359e+00, 0.8196023258344234e+00, 0.7505058481318937e+00, 0.3802378569170251e+00, 0.3493658644293456e+00, 0.2733852715996189e+00, 0.1920071438579977e+00, 0.3602127141457542e-01, 0.1503353393017178e-01, 0.5047074669949666e-02, 0.4814167031169938e-03, 0.1615587133892632e-26, 0.3701495167228324e-14, 0.8166164251788096e-14, 0.1531663815430223e+01, 0.5400326177233737e-21, 0.8345943934662111e-13, 0.3845954584916692e-13, 0.1307411568456853e+01, 0.2270604060615577e-17, 0.1098158814550649e-11, 0.6011774632275279e-12, 0.1180055477995477e+01, 0.6617444900424221e-23, 0.1039362721731823e-10, 0.1780733867152956e-10, 0.1145160315453326e+01, 0.3193234771665464e-17, 0.8582967920523009e-10, 0.1451582492090454e-09, 0.8462599780670359e+00, 0.4268426102972081e-16, 0.2191317346756291e+02, 0.2191317346756291e+02, 0.1100000000000000e+03, 0.3303854875283447e+01, 0.6363636363636363e+01, 0.9730504757902699e-07, 0.7505058481318937e+00, 0.1718125187449148e-14, 0.1758342086676891e-05, 0.2642640058729254e-05, 0.3802378569170251e+00, 0.9047523974206004e-14, 0.3754566725544834e-04, 0.5673943129185001e-04, 0.3493658644293456e+00, 0.2726303288443817e-26, 0.2006689219611138e-03, 0.1777791597062721e-03, 0.2733852715996189e+00, 0.1852245405458936e-26, 0.4273484804153724e-06, 0.2289494629295668e-05, 0.1920071438579977e+00, 0.7673537210642012e-17, 0.3631464818839244e-02, 0.3052322482564877e-02, 0.3602127141457542e-01, 0.1402889840288648e-17, 0.2065820498676818e-02, 0.7907462477517949e-03, 0.1503353393017178e-01, 0.3588277137446075e-19, 0.9842420036565967e-03, 0.4295497278130008e-03, 0.5047074669949666e-02, 0.1100487290795119e-18, 0.2169791314757801e-02, 0.2746294001732303e-02, 0.4814167031169938e-03, 0.9926167349879280e-23, 0.2524480967032520e-02, 0.1685323473608901e-02},
			info: 0,
		},
		{
			z:    []float64{0.4452569544189223e+00, 0.3712416527218666e+00, 0.4540009468556715e+00, 0.9149505778015055e-01, 0.9825460881415958e+00, 0.8144387623295611e+00, 0.4257438310114644e+00, 0.4055209729019219e+00, 0.1358301754544711e-01, 0.6660365069195171e+00, 0.4791028777450690e-01, 0.3657122138300755e+00, 0.9244230955293443e+00, 0.2570776992045346e+00, 0.5718524872194095e+00, 0.6959027703199671e+00, 0.8022093165313721e+00, 0.4495167304148069e+00, 0.4533829141769831e+00, 0.4399857636092745e+00, 0.8343772053001745e+00, 0.5841091089133705e+00, 0.9225724116024000e+00, 0.3646730358635919e+00, 0.4342402996301750e+00, 0.1979669913465428e+00, 0.2710080252534286e+00, 0.4064878156937679e+00, 0.1329017000110605e+00, 0.3577863781516848e+00, 0.7737638611946227e+00, 0.3628629851888825e+00, 0.6191378950237536e+00, 0.9181610808896479e-01, 0.3164967052049129e+00, 0.1800354212961807e+00, 0.5261304159866632e+00, 0.2095779664090124e+00, 0.4734470785970568e+00, 0.8233700542774806e+00, 0.1405149000531442e+00, 0.5739228235076023e+00, 0.4484023224779539e+00, 0.7953415183519565e+00, 0.5683020712105596e+00, 0.3828815589637512e+00, 0.9478512018818865e+00, 0.9398576744287683e+00, 0.6799918835962694e+00, 0.9795978835389229e+00, 0.9415167003494995e+00, 0.9804631144197878e+00, 0.4529282233852833e+00, 0.6987441800620822e+00, 0.1071819561656476e+00, 0.6287695276127018e+00, 0.2573913359217608e+00, 0.5283299892256954e-01, 0.1312057002484920e+00, 0.2566822109082798e+00, 0.5022596747022927e+00, 0.3755634037822867e+00, 0.7687685406410046e+00, 0.9286257388850563e+00, 0.2460950353669618e+00, 0.4615682752011302e+00, 0.2668978314403934e+00, 0.9526334220011422e+00, 0.7113266960956600e+00, 0.8033610237680390e+00, 0.2049912091857929e+00, 0.9104136201043411e+00, 0.9905140155095393e+00, 0.5284892163797259e+00, 0.4617116226676921e-01, 0.7638541825257228e+00, 0.8058860223281950e-01, 0.6562405799504624e+00, 0.4217948031372573e+00, 0.8444441663937204e+00, 0.4135300708069887e+00, 0.3992676961789670e+00, 0.9443059736340937e+00, 0.4697907769676380e+00},
			n:    21,
			zOut: []float64{0.2339652518244840e+01, 0.2127240999798084e+01, 0.2122097736373912e+01, 0.1640409192349226e+01, 0.1522936721900112e+01, 0.1513900850773956e+01, 0.1190428480890818e+01, 0.1098839257111754e+01, 0.9060499827420537e+00, 0.8483833633939547e+00, 0.7423625631343986e+00, 0.6441873762582893e+00, 0.5561064852023264e+00, 0.5332801819376103e+00, 0.3323799891685881e+00, 0.2356658160961767e+00, 0.1910001776595044e+00, 0.1052937995306845e+00, 0.3493206724380617e-01, 0.1582963148293342e-01, 0.1362410302334481e-03, 0.2902907641693595e-19, 0.1235088787739235e-18, 0.5938156039510254e-18, 0.1190428480890818e+01, 0.2366582715663035e-29, 0.1736669751992271e-12, 0.4479686287025860e-12, 0.1098839257111754e+01, 0.4930380657631324e-31, 0.6845104399398104e-11, 0.1631652613834293e-10, 0.9060499827420537e+00, 0.3623410171350593e-13, 0.4397239464622790e-08, 0.8783797026819435e-08, 0.8483833633939547e+00, 0.1680831004707752e-20, 0.6292544501750849e-07, 0.1207646260809090e-06, 0.7423625631343986e+00, 0.1044536639065691e-17, 0.1870111343232326e+02, 0.1870111343232326e+02, 0.1030000000000000e+03, 0.3002267573696145e+01, 0.3883495145631068e+01, 0.8796985420666049e-06, 0.5561064852023264e+00, 0.6232001151245993e-28, 0.4765675419124029e-06, 0.5281771988202168e-06, 0.5332801819376103e+00, 0.5995342879679690e-28, 0.3211676631039122e-06, 0.3513325837054497e-06, 0.3323799891685881e+00, 0.2177256098409993e-25, 0.1826919937583927e-06, 0.1747056528730059e-06, 0.2356658160961767e+00, 0.1311326116357865e-17, 0.1784537561302489e-06, 0.2175474607275274e-06, 0.1910001776595044e+00, 0.3862262991962074e-24, 0.9127544222909640e-06, 0.4962248419758332e-06, 0.1052937995306845e+00, 0.1072850831100576e-26, 0.2872845137271569e-03, 0.1413151427897814e-03, 0.3493206724380617e-01, 0.8028654523313936e-17, 0.5779626756191757e-04, 0.4784930511330926e-04, 0.1582963148293342e-01, 0.9359045779542084e-26, 0.9339546012110267e-04, 0.6206712854083305e-04, 0.1362410302334481e-03, 0.4535950205020818e-29, 0.1799327870469576e-03, 0.1621654177500190e-02},
			info: 0,
		},
		{
			z:    []float64{0.6327743759434090e-01, 0.8874473183212240e+00, 0.1587228549487630e+00, 0.1869853022948459e+00, 0.1852952724381735e+00, 0.2190328352455733e+00, 0.7936705141328082e+00, 0.1573124860628259e+00, 0.2446308768539528e+00, 0.2441044501798444e+00, 0.4435884001235265e+00, 0.1783624365771731e+00, 0.6874700271616803e+00, 0.4796486989431838e+00, 0.2471425348644392e+00, 0.4391077480264999e+00, 0.9973269002312380e+00, 0.4770343529783838e+00, 0.3451566405930041e+00, 0.5011834710046675e+00, 0.8072188861666797e+00, 0.5748577907200507e+00, 0.2986003422862493e+00, 0.4760318526964945e+00, 0.5838957192070238e-01, 0.1760621214885278e+00, 0.7926085978219721e+00, 0.3379849703418611e+00, 0.6012719579585296e+00, 0.1130783704430351e+00, 0.5590675745254436e-01, 0.8616930340961984e+00, 0.9917564091315376e+00, 0.4219575368674273e+00, 0.1044382974319413e+00, 0.1886528536659160e-01, 0.9337085742573710e+00, 0.6447952074628677e+00, 0.7158824182577913e+00, 0.3883216757352448e+00, 0.2116568255149501e+00, 0.9819105316462939e+00, 0.8471687522868032e-01, 0.2785996362910685e+00, 0.8775405646426044e+00, 0.5671584270354416e+00, 0.4912868754910720e+00, 0.5767395813214109e+00, 0.7327323379759062e+00, 0.8184048812627022e+00, 0.3325270745990432e+00, 0.3026500963479654e+00, 0.7228591188664935e+00, 0.1094677581261434e+00, 0.5280482398171430e+00, 0.9974727380694979e+00, 0.5087630734843742e+00, 0.6953603638889684e+00, 0.9103967979863506e+00, 0.2584730811693510e+00, 0.5498750728524477e+00, 0.9758543899455846e+00, 0.8309066632970131e+00, 0.4311646442586020e+00, 0.4732818688792167e+00, 0.4096051739313099e+00, 0.7479697576436509e+00, 0.9369473766445329e+00, 0.7380538090811954e+00, 0.1199951604231148e+00, 0.5672128274092054e+00, 0.7150763241893471e+00, 0.6134240647991112e+00, 0.7095501841622953e-01, 0.1510080432732774e-01, 0.2912426074708729e+00, 0.1611746025295486e+00, 0.3946903662654964e+00, 0.7408818971566200e-01, 0.3978128748254128e+00, 0.3580504147215868e+00, 0.2603799996304434e-01, 0.8990784999913340e+00, 0.1985602490013758e+00},
			n:    21,
			zOut: []float64{0.2139906798692060e+01, 0.2104052265745164e+01, 0.2080474521860817e+01, 0.1784368581097839e+01, 0.1501145759273247e+01, 0.1395549430074518e+01, 0.1236063891814824e+01, 0.1123736402991315e+01, 0.8211474325690415e+00, 0.7411912844479571e+00, 0.7336682913123216e+00, 0.5274366057326734e+00, 0.3715621485686555e+00, 0.3171624722931449e+00, 0.2809062056956198e+00, 0.1668884413843382e+00, 0.9744092162322979e-01, 0.7613224124753509e-01, 0.1665106039654164e-01, 0.4700366860927376e-02, 0.1401918312945850e-02, 0.1084683744678891e-29, 0.4021673570416241e-08, 0.3931910860092540e-08, 0.1236063891814824e+01, 0.1439956010332256e-19, 0.3626387646080745e-08, 0.3543555828237728e-08, 0.1123736402991315e+01, 0.3006506681978320e-24, 0.3371387381785702e-08, 0.3448854867663729e-08, 0.8211474325690415e+00, 0.6197291271416269e-26, 0.2831422041774839e-08, 0.2886341750444270e-08, 0.7411912844479571e+00, 0.0000000000000000e+00, 0.2880306038760265e-08, 0.2936321908308739e-08, 0.7336682913123216e+00, 0.2406371799476454e-20, 0.1752158704199472e+02, 0.1752158704199472e+02, 0.1050000000000000e+03, 0.3111111111111111e+01, 0.7619047619047619e+01, 0.3652487989473621e-08, 0.3715621485686555e+00, 0.1451701975030564e-21, 0.4296538801302010e-08, 0.3992227758256396e-08, 0.3171624722931449e+00, 0.6162975822039155e-31, 0.5151357395533313e-08, 0.5974032086218965e-08, 0.2809062056956198e+00, 0.2039658356197591e-16, 0.1134165265363298e-07, 0.1513934018844884e-07, 0.1668884413843382e+00, 0.2400943524542200e-24, 0.1781584505522029e-06, 0.1218300442987205e-06, 0.9744092162322979e-01, 0.6162975822039155e-31, 0.9557876398738343e-06, 0.5872690889712619e-06, 0.7613224124753509e-01, 0.5074727415413836e-17, 0.2309052370169074e-04, 0.5924498773169641e-04, 0.1665106039654164e-01, 0.1400228106767296e-28, 0.5116509326795711e-04, 0.5154480234673206e-04, 0.4700366860927376e-02, 0.1956855682120442e-20, 0.1061141342429108e-03, 0.9755516139369119e-04, 0.1401918312945850e-02, 0.0000000000000000e+00, 0.1229085807195224e-02, 0.8009097549502667e-02},
			info: 0,
		},
		{
			z:    []float64{0.7225493913848618e+00, 0.3268825038779278e+00, 0.2580957584922012e+00, 0.4222896610254372e+00, 0.8243187922209068e+00, 0.3973391932422579e+00, 0.3700698452657641e+00, 0.2073422474585492e+00, 0.3071746797249822e+00, 0.2370168833330754e+00, 0.7779583030993944e+00, 0.4350651843345934e+00, 0.7867543754352369e+00, 0.3439673345654075e+00, 0.3736479358698906e+00, 0.8450641599616520e+00, 0.3880138308747846e+00, 0.2479782080134303e+00, 0.3220864609073060e+00, 0.8460149388339712e+00, 0.6082870752479030e+00, 0.3396261004129498e+00, 0.1231305479976765e+00, 0.6514493424347317e+00, 0.4987273421671501e-01, 0.2199653339283912e+00, 0.5626339029868159e+00, 0.6755574738917364e+00, 0.7016850168758351e+00, 0.7957347983005405e+00, 0.3394687111682781e-01, 0.1490794513934588e+00, 0.9837873705641814e+00, 0.5038299137267350e+00, 0.8394265886439850e+00, 0.1438624872728633e-01, 0.4444190467253504e+00, 0.1489248400425094e+00, 0.6410535698206483e+00, 0.2739267916732867e+00, 0.6522097474411971e+00, 0.9086492388715542e+00, 0.1949071722314141e+00, 0.7176724675018002e+00, 0.9247176136838016e+00, 0.4929580602677628e+00, 0.3824418236647242e+00, 0.2425574232540663e+00, 0.7085287909144397e+00, 0.2574302555167504e+00, 0.6093160252921369e+00, 0.2417561032032995e+00, 0.4795262420397247e+00, 0.4658200993830509e+00, 0.9033699918994937e+00, 0.1295021719769064e+00, 0.6219991535812029e+00, 0.7040800557612208e+00, 0.8644629036591867e-01, 0.5129394685202899e+00, 0.7988983132437487e+00, 0.8322467913624354e+00, 0.9905487172695071e-01, 0.7093832076693246e+00, 0.1992462440538834e-01, 0.6378263512182120e+00, 0.5823949252324323e-02, 0.7826279628890456e+00, 0.5919828344466795e+00, 0.3815528510244244e+00, 0.6537355893096064e-01, 0.6459825623191251e-01, 0.4466210788758083e+00, 0.6620124961190184e+00, 0.6318809895815939e+00, 0.8145861476435108e+00, 0.5938169865214039e+00, 0.8092284578140500e+00, 0.5378701643349270e+00, 0.1648044346935199e+00, 0.9874291179811017e+00, 0.3210667070512012e+00, 0.6809988920516613e+00, 0.6410080963943865e+00},
			n:    21,
			zOut: []float64{0.2108032684782400e+01, 0.1971253922847928e+01, 0.1952878543820764e+01, 0.1839982524288513e+01, 0.1730503177731785e+01, 0.1509287569836986e+01, 0.1316544049683026e+01, 0.1117715356834192e+01, 0.9058841126016787e+00, 0.8452427684353671e+00, 0.7582219581243467e+00, 0.6392786878439012e+00, 0.5749674919024932e+00, 0.5111167278611346e+00, 0.3859485208834002e+00, 0.2295319067023061e+00, 0.2262113890497363e+00, 0.1122658486667464e+00, 0.7234153262002044e-01, 0.4527048542558528e-01, 0.8319414807792288e-04, 0.2547288901631409e-13, 0.9052716547280089e-04, 0.7818093042872877e-04, 0.1316544049683026e+01, 0.0000000000000000e+00, 0.2537869832861900e-04, 0.2058109204222074e-04, 0.1117715356834192e+01, 0.0000000000000000e+00, 0.1154377886785102e-04, 0.1215574110748570e-04, 0.9058841126016787e+00, 0.8843729980912407e-17, 0.1391102994778336e-08, 0.5248885105740662e-08, 0.8452427684353671e+00, 0.2445468806185137e-28, 0.2192473584471834e-06, 0.3792344586441173e-06, 0.7582219581243467e+00, 0.6480317173221865e-17, 0.1885256245409039e+02, 0.1885256245409039e+02, 0.1090000000000000e+03, 0.3303854875283447e+01, 0.3669724770642202e+01, 0.3715603111278862e-08, 0.5749674919024932e+00, 0.4928548884551519e-18, 0.2515491583997865e-07, 0.9138389520203617e-08, 0.5111167278611346e+00, 0.6587911462745685e-21, 0.7857712130631377e-05, 0.2298529163657988e-05, 0.3859485208834002e+00, 0.6162975822039155e-32, 0.1022329296046741e-03, 0.8680151473672892e-04, 0.2295319067023061e+00, 0.0000000000000000e+00, 0.4616464530663241e-04, 0.3877924919889212e-04, 0.2262113890497363e+00, 0.5522026336547083e-29, 0.2365660345682357e-03, 0.1990352708186489e-03, 0.1122658486667464e+00, 0.9015102222917331e-20, 0.6825572508936817e-03, 0.6443371781720532e-03, 0.7234153262002044e-01, 0.1168271446196110e-24, 0.1518594231951364e-05, 0.3554999772664582e-05, 0.4527048542558528e-01, 0.1023373475125114e-24, 0.5037539097182833e-02, 0.4761613742874584e-02, 0.8319414807792288e-04, 0.1479114197289397e-30, 0.9105689905446296e-02, 0.4825995872157137e-02},
			info: 0,
		},
		{
			z:    []float64{0.3256696334281521e+00, 0.7530856859911462e+00, 0.8853989127988440e+00, 0.4058112879440673e+00, 0.4296659748141172e+00, 0.5414662225246791e-01, 0.7727106089758434e+00, 0.8219533891865206e+00, 0.8804898818241804e+00, 0.2701661776792790e+00, 0.3062100113053869e+00, 0.7616550925245322e+00, 0.7441861437457686e+00, 0.1208643477044503e+00, 0.1894387751691341e+00, 0.5502297763300060e+00, 0.7033314588214317e+00, 0.9049640752657374e+00, 0.7642474001418834e+00, 0.6461873559439021e+00, 0.7323555348666727e+00, 0.6137344390498661e+00, 0.4469041036073067e+00, 0.6120974783290001e+00, 0.3695185251601272e+00, 0.9332999793731735e+00, 0.3981597952563400e+00, 0.5926926060619504e+00, 0.2218776156238759e+00, 0.5556127447847847e-01, 0.2642041684868913e+00, 0.3100431241269808e-01, 0.7617158805289858e+00, 0.5576464290832684e+00, 0.1499291988028689e+00, 0.6071627267784354e+00, 0.2903444085740193e+00, 0.8982141027722228e+00, 0.3437740722392461e+00, 0.5257340992985249e-02, 0.9772197173932363e+00, 0.2743313505008338e-01, 0.5939995532981283e+00, 0.8125099455585232e+00, 0.8394050677385213e+00, 0.2410326841076476e+00, 0.6066214991817382e+00, 0.1828025638429278e+00, 0.5406675263078469e+00, 0.1108622672142920e+00, 0.2412280709874803e+00, 0.5713495470758625e+00, 0.6315445401927943e+00, 0.2035563756883153e+00, 0.8696814083183412e+00, 0.2313134367709406e+00, 0.4474447231170641e+00, 0.6238312383525342e+00, 0.4961806049996582e+00, 0.8121574484576765e+00, 0.8702938949430352e+00, 0.4187164629520156e+00, 0.9204090241956668e+00, 0.8587176140225750e-02, 0.5171057705188283e+00, 0.5195595031109387e+00, 0.1704600468811621e+00, 0.3205951022793651e+00, 0.5643494948239225e-01, 0.9895063272544601e+00, 0.5554681247022339e+00, 0.6098243547887802e+00, 0.4730289261335907e+00, 0.1850396134174421e+00, 0.9997475966289492e+00, 0.6812817712215841e+00, 0.8515433432907883e+00, 0.7377047940023331e+00, 0.2280516830782600e+00, 0.7204549837953970e+00, 0.6096260713045146e+00, 0.9381128478076188e+00, 0.8446173293023428e+00, 0.6255387686605924e+00},
			n:    21,
			zOut: []float64{0.2655567567233002e+01, 0.2307219644143945e+01, 0.2068899090176533e+01, 0.2040332602619700e+01, 0.1742115921396612e+01, 0.1681856112031183e+01, 0.1680515965675234e+01, 0.1379580829731510e+01, 0.1074312270120212e+01, 0.9816962393879983e+00, 0.7217300335876665e+00, 0.6931029681515243e+00, 0.5680300928567689e+00, 0.5098239945058581e+00, 0.4280073603039590e+00, 0.2868581986899945e+00, 0.1650069923584772e+00, 0.9685136354131559e-01, 0.5953175886938158e-01, 0.1010639579838301e-01, 0.1940920539235378e-02, 0.0000000000000000e+00, 0.6683246174917478e-10, 0.4267275295503314e-10, 0.1680515965675234e+01, 0.3510431028233503e-28, 0.1660414363340253e-09, 0.1048896312812690e-09, 0.1379580829731510e+01, 0.1462018530271286e-15, 0.3421106985799154e-06, 0.1788748688597153e-06, 0.1074312270120212e+01, 0.6708481546523056e-14, 0.2133146148219686e-04, 0.4629406838917969e-04, 0.9816962393879983e+00, 0.1747667198202037e-19, 0.5113942524303023e-03, 0.6287577911928029e-03, 0.7217300335876665e+00, 0.3213860550383917e-17, 0.2115308632171850e+02, 0.2115308632171849e+02, 0.1090000000000000e+03, 0.3217687074829932e+01, 0.4587155963302752e+01, 0.4202140396420195e-04, 0.5680300928567689e+00, 0.4489765112701642e-21, 0.3523329095202763e-09, 0.1167512512099963e-08, 0.5098239945058581e+00, 0.1652551767516096e-21, 0.2872281528375301e-06, 0.1040291448945555e-06, 0.4280073603039590e+00, 0.3262461795428160e-16, 0.6249760558326680e-04, 0.5475920486023645e-04, 0.2868581986899945e+00, 0.1568256893218114e-20, 0.1414968677825121e-03, 0.1814031293335571e-03, 0.1650069923584772e+00, 0.8677469957431130e-28, 0.6720068122740940e-03, 0.4975589868187537e-03, 0.9685136354131559e-01, 0.5127595883936577e-29, 0.9260272777115331e-03, 0.1730290225790419e-02, 0.5953175886938158e-01, 0.7227833914448031e-23, 0.5968169418548035e-04, 0.1670361992731526e-04, 0.1010639579838301e-01, 0.1295106991551472e-19, 0.1213518994781533e-03, 0.2534883535393275e-03, 0.1940920539235378e-02, 0.0000000000000000e+00, 0.1005634135102979e-02, 0.2318000467665894e-02},
			info: 0,
		},
		{
			z:    []float64{0.1944566011999833e+00, 0.2181910996636003e+00, 0.3150099357031281e+00, 0.2091722274105127e+00, 0.6593858021341683e+00, 0.9426815336068497e+00, 0.5288011644568881e+00, 0.7784091503839152e+00, 0.1751638954124558e+00, 0.6164772800195180e+00, 0.6225196859922683e+00, 0.2786175486675760e+00, 0.1945914328217568e+00, 0.5779085626077999e+00, 0.5568218939451671e+00, 0.5170566254269496e+00, 0.5742861029155353e+00, 0.6923056150383281e+00, 0.5876692361605969e+00, 0.4295639666640205e+00, 0.8612105519992207e+00, 0.9963836880013683e+00, 0.2692934182903581e+00, 0.6181020857251435e+00, 0.1251902537663725e+00, 0.2838625146280274e+00, 0.2999174442521634e+00, 0.7258439193048929e+00, 0.2048904835805728e+00, 0.1174987481558037e+00, 0.6887427109309550e+00, 0.7409107864964065e+00, 0.5306892441542433e+00, 0.6214581212205206e-01, 0.1234501953117273e-01, 0.2038789370511589e+00, 0.6180963044354376e+00, 0.2820746320472540e+00, 0.4643004781082601e+00, 0.2642696366861867e+00, 0.4102198269957935e-01, 0.1495900075045802e+00, 0.2060405615637534e+00, 0.3399183002928583e+00, 0.7918555051917199e+00, 0.9699614514519834e+00, 0.2922678000248715e+00, 0.1376674512222148e+00, 0.2496993200590716e+00, 0.3432539555227123e+00, 0.5835005548357151e+00, 0.4094205671447549e+00, 0.4876670000237163e+00, 0.7973523995663028e+00, 0.8851386592921134e+00, 0.1523554028450115e+00, 0.7049520832902729e+00, 0.4689643982524618e+00, 0.4347705823917429e+00, 0.8152140283905303e+00, 0.9259201525760803e+00, 0.3175789772106010e+00, 0.3950010049586650e+00, 0.7107346574834400e+00, 0.8109922991383097e+00, 0.6889072760433030e+00, 0.7861311781839575e-02, 0.7751963519893628e+00, 0.7382950156197396e+00, 0.9729693586970557e+00, 0.7283826758396350e+00, 0.1269651196972527e+00, 0.3530947260257147e+00, 0.2061448083515627e-01, 0.5931058475369740e+00, 0.9689379021952851e+00, 0.3254887771415839e+00, 0.2808523639662175e-01, 0.6211468186653845e+00, 0.6037463613076512e+00, 0.8746435454108382e+00, 0.5300695323546331e+00, 0.5276801989236730e+00, 0.2128951663669798e-01},
			n:    21,
			zOut: []float64{0.2424279264689105e+01, 0.2270324150901105e+01, 0.1935128898096250e+01, 0.1814500336114905e+01, 0.1538233405195664e+01, 0.1297421320254112e+01, 0.1267282488999314e+01, 0.1249158514677014e+01, 0.9155076211264116e+00, 0.8242950831432680e+00, 0.5949232544739386e+00, 0.5630079212779312e+00, 0.4530053256303904e+00, 0.3458193312768856e+00, 0.2146623096050983e+00, 0.1686454472303586e+00, 0.1155214217797334e+00, 0.5180948935108346e-01, 0.2789744048901975e-01, 0.8268186305777192e-02, 0.6680158028153346e-04, 0.0000000000000000e+00, 0.3986945646893946e-08, 0.7023072825762287e-08, 0.1267282488999314e+01, 0.0000000000000000e+00, 0.2355158572444734e-07, 0.1877711213087836e-07, 0.1249158514677014e+01, 0.3384506411488746e-20, 0.2648360770785381e-11, 0.1741394977127960e-10, 0.9155076211264116e+00, 0.3299177303725225e-12, 0.2487861145673591e-05, 0.4028407591944186e-05, 0.8242950831432680e+00, 0.2441420600100056e-16, 0.1869515956083083e-05, 0.1842279235203962e-05, 0.5949232544739386e+00, 0.2024938139529812e-20, 0.1807975801219765e+02, 0.1807975801219765e+02, 0.9300000000000000e+02, 0.2836734693877551e+01, 0.1075268817204301e+01, 0.9521662195121400e-05, 0.4530053256303904e+00, 0.1259501632163109e-23, 0.1468124594065504e-05, 0.4670591953230941e-05, 0.3458193312768856e+00, 0.7975064420889996e-22, 0.1308295990671250e-06, 0.6690001990389955e-07, 0.2146623096050983e+00, 0.1927976052360153e-26, 0.3812574759844485e-05, 0.3580340604691736e-05, 0.1686454472303586e+00, 0.3739200690747596e-27, 0.6844384387279355e-05, 0.5557405125771409e-05, 0.1155214217797334e+00, 0.7506098603421864e-21, 0.2139339757899294e-04, 0.3026575107963825e-04, 0.5180948935108346e-01, 0.3886068668379982e-20, 0.1983680974127293e-03, 0.1336830557694390e-03, 0.2789744048901975e-01, 0.1991084924777834e-26, 0.4443273484877918e-03, 0.6821022035819646e-03, 0.8268186305777192e-02, 0.1921747358113822e-20, 0.1778955160858100e-05, 0.3048292256883238e-06, 0.6680158028153346e-04, 0.7915528772828604e-21, 0.4499131616906370e-02, 0.1547765737453591e-02},
			info: 0,
		},
		{
			z:    []float64{0.3849545441014558e+00, 0.9201984078647510e+00, 0.4611893507512446e+00, 0.9426704990067624e+00, 0.4513996483425642e+00, 0.3546824995764211e+00, 0.2673932938327498e+00, 0.2250322436282782e+00, 0.3003875396759296e+00, 0.8637078150569484e+00, 0.2463125523040188e-01, 0.6381498516846796e+00, 0.9054059378226459e+00, 0.9211368359293561e+00, 0.6899055937753183e+00, 0.6469675067081565e+00, 0.2524386712411693e+00, 0.4816130873217878e+00, 0.8437265349008123e+00, 0.8843672501609817e+00, 0.3917808232071412e+00, 0.4109107468556734e+00, 0.6294664210780233e+00, 0.5231064834871043e+00, 0.6006449315760222e+00, 0.2321330919598237e+00, 0.9857216391074231e-01, 0.5246894901632879e+00, 0.2429723268334654e+00, 0.3385437000243748e+00, 0.1784805178646218e+00, 0.8008314854810928e+00, 0.3388059467143087e+00, 0.4957241256656372e+00, 0.6108069077503485e+00, 0.8350989375447051e+00, 0.1087391555565628e+00, 0.5847974581188020e+00, 0.8316850716702742e+00, 0.8921011586703487e+00, 0.1947070853010260e+00, 0.1394928972102342e+00, 0.7929088249487071e+00, 0.4078510186300481e+00, 0.6849316610029904e+00, 0.1436003498260631e+00, 0.5673845335495399e+00, 0.7998164012861997e-01, 0.6667892725157643e+00, 0.7612117828169483e+00, 0.8195812739875934e+00, 0.1507277483884870e+00, 0.6683936863561600e+00, 0.1889659251016290e+00, 0.7979363461805246e+00, 0.9481151487908223e+00, 0.3017570036067704e+00, 0.2677628114499495e+00, 0.4479716416042271e+00, 0.3773062273227097e+00, 0.1756267436681674e+00, 0.6251009447636316e+00, 0.3010990216120933e+00, 0.6044916624270988e+00, 0.3012752666004853e+00, 0.7113839134297870e+00, 0.7355680503868338e+00, 0.1830572310895536e+00, 0.2065116792960049e+00, 0.2432895898830973e+00, 0.9615089470298006e+00, 0.3137165160357294e+00, 0.5335631824040450e+00, 0.8324142824771229e+00, 0.9749825715503555e+00, 0.3108065469391890e+00, 0.2584483510692804e+00, 0.5679665509067858e+00, 0.9084113884935793e+00, 0.3061643504016712e+00, 0.1856860631474661e+00, 0.4086356189591027e+00, 0.2761517708925276e+00, 0.9203155374121844e+00},
			n:    21,
			zOut: []float64{0.2651616639993538e+01, 0.2278042999629812e+01, 0.2198077975076974e+01, 0.2185608493228643e+01, 0.1866238785250178e+01, 0.1694317687161655e+01, 0.1314780820581521e+01, 0.1267230115137409e+01, 0.1165102366599241e+01, 0.1094106513205574e+01, 0.9375639320451692e+00, 0.7228525903025090e+00, 0.5063995930233953e+00, 0.4433597755655621e+00, 0.3782868675227946e+00, 0.2710178897462153e+00, 0.1447594933682902e+00, 0.1060101050946834e+00, 0.8449698031014202e-01, 0.1468593250969371e-01, 0.8406928034951682e-06, 0.3591986573571771e-20, 0.3864386735331416e-10, 0.3252595492372177e-10, 0.1314780820581521e+01, 0.1447952803137974e-21, 0.5374081097788385e-10, 0.4849713226733927e-10, 0.1267230115137409e+01, 0.2958228394578794e-29, 0.8017357434413146e-10, 0.7063808972759981e-10, 0.1165102366599241e+01, 0.1139115147139141e-26, 0.1961041727821929e-09, 0.1374026522849091e-09, 0.1094106513205574e+01, 0.3915388343685865e-19, 0.1756082891845291e-08, 0.3338672481507716e-08, 0.9375639320451692e+00, 0.4870203640784043e-16, 0.2132455639604580e+02, 0.2132455639604580e+02, 0.9900000000000000e+02, 0.2972789115646258e+01, 0.5050505050505050e+01, 0.1300256830939750e-04, 0.5063995930233953e+00, 0.4338734978715565e-29, 0.3205424704587136e-04, 0.3879495859639124e-04, 0.4433597755655621e+00, 0.9466330862652142e-25, 0.6653798840082298e-04, 0.6266988927201412e-04, 0.3782868675227946e+00, 0.2274100449667269e-22, 0.3059080514749475e-04, 0.2450157710629126e-04, 0.2710178897462153e+00, 0.2739429933906694e-24, 0.4451877567018950e-05, 0.2739439060253447e-05, 0.1447594933682902e+00, 0.1897052704875489e-25, 0.3311215248867682e-05, 0.2822048496282046e-05, 0.1060101050946834e+00, 0.7924889161564951e-19, 0.2588042287900903e-05, 0.3305698843095029e-05, 0.8449698031014202e-01, 0.2523395625690618e-18, 0.1425730854724969e-04, 0.2612802087529877e-04, 0.1468593250969371e-01, 0.7520086356827106e-20, 0.6278930605158724e-03, 0.1193309863751681e-02, 0.8406928034951682e-06, 0.0000000000000000e+00, 0.8769018979884540e-02, 0.6748334476252631e-02},
			info: 0,
		},
		{
			z:    []float64{0.5539290016733247e+00, 0.6936589108803458e+00, 0.1862325586311042e-01, 0.3903593359823143e+00, 0.1387446022374186e+00, 0.9230050933369500e+00, 0.1655735508788293e+00, 0.5464594833562775e+00, 0.4407574799078734e+00, 0.7597516703282015e+00, 0.3240675365298944e+00, 0.8527429657828770e+00, 0.6134024974884296e+00, 0.1359668624923763e+00, 0.8589771621484943e+00, 0.8334002673394481e+00, 0.3811010712979018e+00, 0.4518439634289880e+00, 0.4121953913957921e-01, 0.1499929777106017e+00, 0.7537932319194001e+00, 0.1137770685080763e+00, 0.9362285670837264e+00, 0.2284833451474525e+00, 0.4661006612092690e+00, 0.3461611111488332e+00, 0.1608705680575839e-01, 0.9250298701911358e+00, 0.5983544857783111e+00, 0.9400090024445320e+00, 0.6595514287179831e+00, 0.2725099566160494e+00, 0.6509556024164401e+00, 0.8851211780351773e+00, 0.5925872091724521e+00, 0.5318402341230010e+00, 0.3225952236300995e+00, 0.6233031538827258e+00, 0.1806586091116282e+00, 0.9476369741031940e+00, 0.6784219735316235e+00, 0.6934023884718178e+00, 0.5000312772557033e+00, 0.6725383579734943e+00, 0.6771923299216058e+00, 0.9125469473100194e+00, 0.9862018367238429e+00, 0.7259311136907298e+00, 0.9021849324334038e+00, 0.6032549715715884e+00, 0.9017706724408630e+00, 0.8975979926873651e+00, 0.5949035726420406e+00, 0.6903449880442312e+00, 0.7574844360343417e+00, 0.2889632382233942e-01, 0.9428474184445177e+00, 0.5555118914598791e+00, 0.8663544108664935e+00, 0.6853450780608091e+00, 0.1464483859238053e+00, 0.6491672315887742e+00, 0.2994712877436206e+00, 0.3101752077576794e+00, 0.4920466664329196e+00, 0.2135103260181662e+00, 0.3809190441316870e+00, 0.8437350743416491e+00, 0.5443983884818225e+00, 0.7426189539459086e+00, 0.1055227287563778e+00, 0.3059118205598027e+00, 0.8189910523272392e+00, 0.9773505795713493e+00, 0.7305661438576656e+00, 0.6062516615534109e+00, 0.4660033490547544e+00, 0.5413353206637471e+00, 0.2388208915142312e+00, 0.6428463909118429e+00, 0.2982699820336984e+00, 0.2856298024316706e-01, 0.5487207914459959e+00, 0.4464180688275057e+00},
			n:    21,
			zOut: []float64{0.2448593467642387e+01, 0.2141330855004301e+01, 0.2126430153924173e+01, 0.1907639051889359e+01, 0.1812500110390200e+01, 0.1510069013602020e+01, 0.1441897976798092e+01, 0.1263171337642769e+01, 0.1138107892295268e+01, 0.1088450894719558e+01, 0.9398511957468885e+00, 0.8468227425873083e+00, 0.6124964740092524e+00, 0.5436278175488470e+00, 0.4066256939946141e+00, 0.3972422051503922e+00, 0.2390787026361968e+00, 0.4848296808782174e-01, 0.2905803980602126e-01, 0.1302961362478940e-02, 0.3616542154739030e-05, 0.3187755557432356e-18, 0.8836722322771784e-05, 0.6065531139006927e-05, 0.1441897976798092e+01, 0.7730421460348975e-16, 0.3574427637238573e-07, 0.1935178018527985e-07, 0.1263171337642769e+01, 0.8058524406205312e-18, 0.1418632976070110e-05, 0.4069026170914173e-05, 0.1138107892295268e+01, 0.1371633476674845e-23, 0.4721385684453755e-04, 0.3623692584267971e-04, 0.1088450894719558e+01, 0.1790018845564752e-20, 0.4055026077404345e-04, 0.2446096735206838e-04, 0.9398511957468885e+00, 0.3178330447896772e-25, 0.2094278317138010e+02, 0.2094278317138010e+02, 0.1040000000000000e+03, 0.3031746031746032e+01, 0.4807692307692307e+01, 0.8138465546053722e-05, 0.6124964740092524e+00, 0.1019617235038814e-22, 0.4736809894069740e-03, 0.1915264360566004e-03, 0.5436278175488470e+00, 0.2004410489406572e-21, 0.1259085689345719e-03, 0.2338760337361548e-03, 0.4066256939946141e+00, 0.1231392444306089e-15, 0.6226403214695822e-04, 0.7644683470656998e-04, 0.3972422051503922e+00, 0.5914962592392188e-17, 0.1887068321547840e-03, 0.1345267246510674e-03, 0.2390787026361968e+00, 0.1936824017247308e-20, 0.7333182971373067e-03, 0.1062997506116107e-02, 0.4848296808782174e-01, 0.3799154119544393e-26, 0.1269933545612727e-03, 0.1297822869175177e-03, 0.2905803980602126e-01, 0.8465676581597317e-24, 0.5642309440908588e-03, 0.2732446225021143e-03, 0.1302961362478940e-02, 0.2467375723434259e-18, 0.9177892725485271e-03, 0.8034707353774361e-03, 0.3616542154739030e-05, 0.2761013168273541e-28, 0.6531167933367503e-03, 0.6727959129527735e-03},
			info: 0,
		},
		{
			z:    []float64{0.7455810318731756e+00, 0.1881309445499726e+00, 0.6290241532486281e+00, 0.9426231091333456e+00, 0.1402641401457146e+00, 0.5545071521563689e+00, 0.3467970399972181e+00, 0.6378935846273492e-01, 0.8187242451993508e+00, 0.8444039716090014e+00, 0.8807815832461214e+00, 0.6871811349512845e+00, 0.6041217734774926e+00, 0.2285364565760638e+00, 0.6287288909172152e+00, 0.5441550558534458e+00, 0.6062929607453951e+00, 0.1528830611582420e+00, 0.7289323622783690e+00, 0.8693274251763169e+00, 0.1210720262902459e+00, 0.4752572018677603e+00, 0.8160358228459934e+00, 0.5003926181135285e+00, 0.2800920281530351e+00, 0.3817159580569316e+00, 0.1419563352692587e+00, 0.9738793587569783e+00, 0.2402077997739175e+00, 0.5021080238100061e+00, 0.5325521311583831e+00, 0.7002793445871702e+00, 0.2004913666518293e-01, 0.6858750037076770e+00, 0.5705320248969311e+00, 0.6000416876176061e+00, 0.9254592880635680e+00, 0.2987366812581649e+00, 0.8838368946481180e+00, 0.7495294261248863e+00, 0.2516401660161148e+00, 0.5770724542103510e+00, 0.8689432882806168e+00, 0.3108844333247283e+00, 0.7611598373381380e+00, 0.9533545884676758e+00, 0.6146629999183371e+00, 0.7337933880625785e+00, 0.1335018938357140e+00, 0.4054745880121539e+00, 0.9816031767048012e+00, 0.5190257866591561e+00, 0.9457220484783406e+00, 0.2563725588490263e+00, 0.8953616129834293e+00, 0.1343673038869742e+00, 0.1198867110907023e+00, 0.7765966504091196e+00, 0.1685346783514826e+00, 0.9322265874533907e+00, 0.4968937019786546e+00, 0.3933065437909874e+00, 0.7046190939244956e-01, 0.5772052710604483e+00, 0.1220174671595003e+00, 0.3586914192309758e+00, 0.4743117898783903e+00, 0.1205436116155321e+00, 0.2068106627971966e-01, 0.5035688415619853e+00, 0.2656792568844590e-01, 0.4951625786650252e+00, 0.1600264513881963e+00, 0.4218870487180432e+00, 0.4847863747130776e-01, 0.9478135093620923e+00, 0.1811694594266104e+00, 0.5311488460048615e+00, 0.2296061187775216e+00, 0.9932681440344262e+00, 0.2007921586496573e+00, 0.9684478357621775e+00, 0.9322927111902295e+00, 0.6234102172880590e+00},
			n:    21,
			zOut: []float64{0.2617080551859897e+01, 0.2305187568066598e+01, 0.1942735806791493e+01, 0.1834812272403632e+01, 0.1760060933961288e+01, 0.1720372963502770e+01, 0.1676173250234917e+01, 0.1530300579122039e+01, 0.1214313538868506e+01, 0.1146550683576815e+01, 0.9555081169996513e+00, 0.8235353939361046e+00, 0.7281321711646775e+00, 0.5964515885643180e+00, 0.4105953672636856e+00, 0.2154672622243388e+00, 0.1469861329023305e+00, 0.1350631498164301e+00, 0.8120368457133148e-01, 0.1548517647968005e-01, 0.1861612641330624e-04, 0.6389970231972139e-21, 0.6259229043737505e-18, 0.1065975851985723e-17, 0.1676173250234917e+01, 0.1398996249446581e-17, 0.1485385179581525e-16, 0.5390878377546583e-16, 0.1530300579122039e+01, 0.1934286939601921e-26, 0.4995235293268131e-13, 0.1480097232724468e-13, 0.1214313538868506e+01, 0.1393981231259665e-11, 0.4031414073293921e-10, 0.9065983401759033e-10, 0.1146550683576815e+01, 0.7754818242684634e-25, 0.9068533370171412e-09, 0.1932480064220526e-08, 0.9555081169996513e+00, 0.5937598997073433e-12, 0.2185603480843691e+02, 0.2185603480843692e+02, 0.1010000000000000e+03, 0.3183673469387755e+01, 0.2970297029702970e+01, 0.4106209724092858e-05, 0.7281321711646775e+00, 0.2221432309102369e-26, 0.2012416535197213e-04, 0.1179395510019159e-04, 0.5964515885643180e+00, 0.6492360922940637e-17, 0.6055695087058826e-05, 0.1676327789285107e-04, 0.4105953672636856e+00, 0.1174834768927548e-24, 0.8526024871833974e-05, 0.9608606324255413e-05, 0.2154672622243388e+00, 0.1186446801452402e-26, 0.9887524717240814e-05, 0.1072842889916555e-04, 0.1469861329023305e+00, 0.0000000000000000e+00, 0.1666923382276663e-04, 0.2139923074663986e-04, 0.1350631498164301e+00, 0.4860988391194038e-18, 0.4968094868839375e-08, 0.1325007303324406e-08, 0.8120368457133148e-01, 0.5476819983069490e-18, 0.5501364953991437e-04, 0.1068949633822309e-04, 0.1548517647968005e-01, 0.6621786174648700e-13, 0.4934492558904634e-03, 0.7815729865935395e-03, 0.1861612641330624e-04, 0.0000000000000000e+00, 0.2047621357235247e-01, 0.1218736604744046e-01},
			info: 0,
		},
		{
			z:    []float64{0.1913768959569714e+00, 0.7347223265181069e+00, 0.9087350122086006e+00, 0.8876448886267929e-02, 0.1794058368310475e+00, 0.3375830657319635e+00, 0.4153249303964603e+00, 0.3079259326141542e+00, 0.5095638682609168e+00, 0.6300946130469818e+00, 0.5367870439046851e-02, 0.6082577193589970e+00, 0.6096872650047096e+00, 0.8098482943683755e+00, 0.4018780482667224e+00, 0.1612961166583111e+00, 0.4165836183710623e+00, 0.6711774659609234e+00, 0.1871437048914691e+00, 0.2043648411447756e+00, 0.4603921058522200e+00, 0.5138380788557162e-01, 0.3299651081607302e+00, 0.5178779891127856e+00, 0.8488474652006981e+00, 0.4000528745818374e+00, 0.7038372186752184e-01, 0.1091866126814279e+00, 0.4601541814804277e-01, 0.2814862519917873e+00, 0.5969661695911915e+00, 0.1111325580672384e+00, 0.1377964900539917e+00, 0.4488655138014651e+00, 0.5793089439934721e+00, 0.4068390675279384e+00, 0.3141858292757815e-01, 0.3803058398371814e+00, 0.6865263808463873e-01, 0.6565571596516916e+00, 0.4227763797508006e+00, 0.9281332433572439e+00, 0.2549706813172651e+00, 0.1472316879439791e+00, 0.8389980826186394e+00, 0.4949206978487660e+00, 0.8778524239605636e+00, 0.8125876339222501e+00, 0.6384442407604712e+00, 0.6297206683503800e+00, 0.1074594814776241e+00, 0.4635106216187717e+00, 0.2149027083261391e+00, 0.2926633791426133e+00, 0.8718806127632718e+00, 0.9358351753143842e+00, 0.5812389276262170e+00, 0.8361764419241092e+00, 0.1334582685582402e+00, 0.6700349085889619e+00, 0.1370175035793201e+00, 0.2605729802823288e+00, 0.7055670307426516e+00, 0.1974656950667419e-01, 0.9516894704106690e+00, 0.7509460514650641e+00, 0.9770872584819335e-01, 0.1679929405438133e+00, 0.2605432695744189e+00, 0.7255281751885829e+00, 0.2060091110826470e+00, 0.5123028703888126e+00, 0.5392241233948379e+00, 0.3215743887975069e+00, 0.4306560982435532e+00, 0.9326432909148183e+00, 0.1891146429259456e+00, 0.5585690444839775e+00, 0.8103752159402208e+00, 0.3850798219907741e+00, 0.6027394925107610e-01, 0.6960376568363590e+00, 0.6132631218829975e+00, 0.5859904896405407e+00},
			n:    21,
			zOut: []float64{0.1948845699194504e+01, 0.1757754653408917e+01, 0.1739740003489702e+01, 0.1427189755042558e+01, 0.1354312190162734e+01, 0.1300861212306309e+01, 0.1144086431430055e+01, 0.9728683413380136e+00, 0.7924660843958955e+00, 0.7190108168957414e+00, 0.5334843173898538e+00, 0.4234595469843583e+00, 0.3840099969172215e+00, 0.2310167712576353e+00, 0.1706582281146164e+00, 0.1481114719972525e+00, 0.9844570800522298e-01, 0.8643465546688130e-01, 0.2116850399857309e-01, 0.1490679637756442e-02, 0.9506251872618699e-05, 0.0000000000000000e+00, 0.3139247800552953e-11, 0.1114956279480716e-10, 0.1144086431430055e+01, 0.2611946364919696e-16, 0.1910542599593587e-05, 0.1867627468366115e-05, 0.9728683413380136e+00, 0.3548331964791209e-13, 0.9422609200466975e-06, 0.3861796048985941e-05, 0.7924660843958955e+00, 0.4543838814073028e-27, 0.1545075054473009e-06, 0.2314398440634665e-06, 0.7190108168957414e+00, 0.1033975765689929e-24, 0.9239329035775786e-07, 0.1002180390059042e-06, 0.5334843173898538e+00, 0.0000000000000000e+00, 0.1525542457368567e+02, 0.1525542457368567e+02, 0.1120000000000000e+03, 0.3308390022675737e+01, 0.5357142857142857e+01, 0.1175851227353848e-05, 0.3840099969172215e+00, 0.1009741958682895e-27, 0.6285056589702800e-05, 0.7777675415159152e-05, 0.2310167712576353e+00, 0.4642054879586336e-19, 0.2707022647265855e-04, 0.1462219920606713e-04, 0.1706582281146164e+00, 0.0000000000000000e+00, 0.1442808298792521e-03, 0.1499267314065209e-03, 0.1481114719972525e+00, 0.1784642171583157e-21, 0.9060208842449760e-06, 0.2604899516157433e-05, 0.9844570800522298e-01, 0.3285517707584525e-17, 0.4009882974094740e-08, 0.9813054658490708e-08, 0.8643465546688130e-01, 0.1441804758967627e-17, 0.1005867330167897e-05, 0.3548177460196556e-06, 0.2116850399857309e-01, 0.1161145423573808e-16, 0.4264476922944564e-04, 0.6523019800084465e-04, 0.1490679637756442e-02, 0.5282426871388410e-23, 0.4120097727637851e-03, 0.5256559484655862e-03, 0.9506251872618699e-05, 0.7858215819253763e-23, 0.4122429400478702e-03, 0.4195795458123320e-03},
			info: 0,
		},
		{
			z:    []float64{0.8707866955342474e+00, 0.7817747598907121e-01, 0.3395074897480099e-01, 0.4299950297507704e+00, 0.8088090483755312e+00, 0.8059665333356031e+00, 0.8715468955879169e-01, 0.7726332912417854e+00, 0.8113440783694211e+00, 0.3884398998833286e+00, 0.2376236729669906e+00, 0.1163685212573567e+00, 0.9165435854442110e+00, 0.6267984817484763e+00, 0.9624876615834328e+00, 0.3034180669232360e+00, 0.2061796047621195e+00, 0.5768621491208505e+00, 0.9925917678590149e+00, 0.6443157924817072e+00, 0.5967958277754652e+00, 0.6529032825080359e+00, 0.2463148548215904e+00, 0.3434579503978075e+00, 0.9563439871443443e+00, 0.2121028540747388e+00, 0.7195547303990343e+00, 0.2508441805243417e+00, 0.1006260142041168e+00, 0.8893309100691102e+00, 0.5028754540045453e+00, 0.8271029410166070e+00, 0.1744058577319449e+00, 0.1497291367846938e+00, 0.9760674104574272e+00, 0.1217237467828105e+00, 0.5182991497218187e+00, 0.9638099804000833e-01, 0.3920754919448237e+00, 0.5405769677178972e+00, 0.2099631365577761e+00, 0.7305783624173950e+00, 0.5230361650507421e+00, 0.3249231142539155e+00, 0.6943021078034506e+00, 0.8039502137729849e+00, 0.5430939865698426e+00, 0.5802454813209417e+00, 0.4521353811622019e+00, 0.6022646695138674e+00, 0.5085578584349214e+00, 0.2084880202460930e+00, 0.7893764544854661e+00, 0.8499187425465698e+00, 0.6507732600828079e+00, 0.6030189256946239e+00, 0.2748834677816949e+00, 0.7991444375081345e+00, 0.3953548021356785e+00, 0.8063610109500122e+00, 0.9322494176165492e+00, 0.2295756744488692e+00, 0.9380170922026533e+00, 0.6349173046266506e+00, 0.8138238710526029e+00, 0.1723717901392456e+00, 0.7436778507578146e+00, 0.9748296910217159e+00, 0.9719004334667378e+00, 0.1097372829332979e+00, 0.6782829454939702e+00, 0.5097404868982898e+00, 0.9485025358778609e+00, 0.7313319469015522e-01, 0.6156780373408383e+00, 0.8764490831370680e+00, 0.2737186188000360e+00, 0.3500606337779347e+00, 0.7029831161979777e+00, 0.1747477098480691e+00, 0.8887436470613648e+00, 0.8441051781034927e+00, 0.3534934185139980e+00, 0.9278581973721739e+00},
			n:    21,
			zOut: []float64{0.2399590582568773e+01, 0.2223029747781157e+01, 0.2028465681071774e+01, 0.1928601105940012e+01, 0.1734632538043284e+01, 0.1705587556623443e+01, 0.1335586809119933e+01, 0.1201248979220471e+01, 0.1103973785860777e+01, 0.9516762968928613e+00, 0.8501027578146555e+00, 0.7334218559333251e+00, 0.5694462065979470e+00, 0.4616704914677346e+00, 0.3869349008726894e+00, 0.2370229392379994e+00, 0.1895639186923105e+00, 0.6837204156138489e-01, 0.3336561258733767e-01, 0.4796873010443177e-02, 0.8309969413683648e-03, 0.1974829088984998e-20, 0.1178494024155901e-08, 0.8095343015842228e-09, 0.1335586809119933e+01, 0.2220446049250313e-15, 0.1020576889344504e-06, 0.1802026823995536e-06, 0.1201248979220471e+01, 0.0000000000000000e+00, 0.6243063866214613e-06, 0.3862955415798747e-06, 0.1103973785860777e+01, 0.6981113564263453e-22, 0.3544097463307990e-10, 0.1357364267827356e-10, 0.9516762968928613e+00, 0.9072531498765813e-25, 0.7979152396248553e-09, 0.3946058728874109e-09, 0.8501027578146555e+00, 0.3130538746543124e-14, 0.2014792167783968e+02, 0.2014792167783968e+02, 0.1020000000000000e+03, 0.3129251700680272e+01, 0.5882352941176471e+01, 0.5818593578868780e-05, 0.5694462065979470e+00, 0.1588643710672175e-16, 0.4031382989989102e-05, 0.4054337523014161e-05, 0.4616704914677346e+00, 0.4437342591868191e-30, 0.3922192530785082e-05, 0.4209867122852495e-05, 0.3869349008726894e+00, 0.1490631566505624e-25, 0.4185837521330395e-06, 0.8450171931154743e-06, 0.2370229392379994e+00, 0.9509832565716118e-20, 0.2486232683772935e-03, 0.1020105306066691e-03, 0.1895639186923105e+00, 0.4152563805083406e-26, 0.4122104994398350e-03, 0.2603466333914367e-03, 0.6837204156138489e-01, 0.2482703040911568e-24, 0.1451022916339763e-02, 0.8718585538326752e-03, 0.3336561258733767e-01, 0.7888609052210118e-30, 0.3124612722893384e-02, 0.2800832977531187e-02, 0.4796873010443177e-02, 0.5945360652724886e-23, 0.1330631700030230e-06, 0.1700369028500688e-07, 0.8309969413683648e-03, 0.0000000000000000e+00, 0.1665788949608015e-02, 0.1927727749653244e-02},
			info: 0,
		},
		{
			z:    []float64{0.5557143146495949e+00, 0.5989968816903124e+00, 0.2940668070493605e+00, 0.6865591553636113e+00, 0.2538362881116337e+00, 0.8562930572057048e-01, 0.8935270452331215e+00, 0.2427258280604294e-01, 0.8872146530392093e+00, 0.8383596820641517e+00, 0.7298494135137694e+00, 0.5456106579379609e+00, 0.5284508278981448e+00, 0.4551015440756836e+00, 0.8310297436868961e+00, 0.2533050402685522e+00, 0.8204820449217906e+00, 0.3961557150240700e+00, 0.9768702237400030e+00, 0.2370292739142171e+00, 0.2559318188841475e-02, 0.9750517337670606e-01, 0.4332045279801727e+00, 0.9393777930670477e+00, 0.1401411199977110e-01, 0.6412771440674316e+00, 0.3992540155849937e+00, 0.6710909099058460e+00, 0.7286319488310415e+00, 0.2796269361967505e+00, 0.7441899058930037e+00, 0.6626826587793098e+00, 0.9214594500725222e+00, 0.3161807402236700e+00, 0.5522479249937589e+00, 0.8328895958825197e+00, 0.3188368475648113e+00, 0.7495883496609020e+00, 0.7525354981370723e+00, 0.4819658709067065e-01, 0.8655147680740739e+00, 0.9438778396406793e+00, 0.2488475915004000e+00, 0.6823535213934872e+00, 0.1937525404626215e+00, 0.5005903164733780e+00, 0.8863816362271992e-01, 0.8279647895785316e+00, 0.7576076924432378e+00, 0.7026499641621615e+00, 0.4430381204856144e+00, 0.7250369983032848e+00, 0.6848785243425647e+00, 0.4460397323983623e+00, 0.4708648248777212e+00, 0.2715064773162834e+00, 0.3027887490966231e+00, 0.7428389405348396e+00, 0.4450045451144832e+00, 0.4683793136386452e+00, 0.2207684946192057e+00, 0.2427030064074031e+00, 0.9554840783351275e+00, 0.4627007756736100e+00, 0.2498171419709001e+00, 0.6457883819068694e+00, 0.9852852516910280e+00, 0.1549706320399313e-01, 0.5863698153730145e+00, 0.8639664213849532e+00, 0.3535803218615043e+00, 0.5300900503312423e+00, 0.6390014357018353e+00, 0.7961857807391770e+00, 0.2691637990251612e+00, 0.5302467438659471e+00, 0.3050412123368091e+00, 0.6256487701731338e+00, 0.3334512397543001e+00, 0.4160884062357342e+00, 0.1255438899512573e+00, 0.1645317210625844e+00, 0.3238197294562333e+00, 0.8765325383521217e+00},
			n:    21,
			zOut: []float64{0.2427342887003738e+01, 0.2350720031505609e+01, 0.2023223336640000e+01, 0.1998000565720247e+01, 0.1766392007527096e+01, 0.1571733737439641e+01, 0.1560415852981921e+01, 0.1417948678305651e+01, 0.1258973312710257e+01, 0.1056655934390461e+01, 0.8835393928887348e+00, 0.8723851974669989e+00, 0.7820423442972025e+00, 0.7419840433476598e+00, 0.4998280519834369e+00, 0.4045332331022956e+00, 0.9795545665796999e-01, 0.6816976350034422e-01, 0.5040003524610775e-01, 0.3066447317814986e-01, 0.1707038609221110e-04, 0.0000000000000000e+00, 0.7788285758834551e-10, 0.4967814553062384e-10, 0.1560415852981921e+01, 0.2584939414223288e-25, 0.2197123097785426e-09, 0.1273532664170934e-09, 0.1417948678305651e+01, 0.1666074231826777e-26, 0.1275101209751314e-13, 0.3690153580151478e-13, 0.1258973312710257e+01, 0.3205217501722064e-17, 0.9823174457391989e-10, 0.2521354047663225e-09, 0.1056655934390461e+01, 0.7258232531419596e-14, 0.3712131332588551e-06, 0.9470624651125662e-06, 0.8835393928887348e+00, 0.1013078509970455e-14, 0.2186292540627962e+02, 0.2186292540627962e+02, 0.1200000000000000e+03, 0.3639455782312925e+01, 0.8333333333333334e+01, 0.8044662327724623e-11, 0.7820423442972025e+00, 0.1609007833004439e-16, 0.1435272653873044e-05, 0.2709954877521105e-06, 0.7419840433476598e+00, 0.7040565915227063e-17, 0.3841536774665796e-05, 0.2871402006582635e-05, 0.4998280519834369e+00, 0.2100142105025380e-22, 0.2079096157763567e-04, 0.3245111531694086e-04, 0.4045332331022956e+00, 0.1391829401523057e-17, 0.6737785910120600e-04, 0.9764140649613147e-04, 0.9795545665796999e-01, 0.0000000000000000e+00, 0.6779864796214124e-03, 0.3130513090482015e-03, 0.6816976350034422e-01, 0.2326445472805242e-24, 0.3547436816484852e-02, 0.2240007954218211e-02, 0.5040003524610775e-01, 0.1487168151587342e-22, 0.3710761871988309e-06, 0.2590766399899962e-07, 0.3066447317814986e-01, 0.7395570986446986e-31, 0.2806228106658499e-03, 0.1300265719680322e-02, 0.1707038609221110e-04, 0.5107323294632100e-21, 0.5584126334680110e-02, 0.2863707539154273e-03},
			info: 0,
		},
		{
			z:    []float64{0.8091224458140820e+00, 0.8008972278830493e+00, 0.2578176398184776e+00, 0.1256248089219657e+00, 0.6581359192692769e+00, 0.2304361454551056e+00, 0.7582551898966047e+00, 0.1757884371165862e+00, 0.3186400696162690e+00, 0.7479165768101737e+00, 0.3726824047207358e+00, 0.8797387692666324e+00, 0.5750380180879821e+00, 0.5660816893967801e+00, 0.4438713938433396e+00, 0.1758644078670230e+00, 0.5631776732547016e+00, 0.3584358301396408e+00, 0.9436101806256550e+00, 0.5626737326978751e+00, 0.6424649400467324e+00, 0.8481112336470399e+00, 0.5655978701315552e+00, 0.8705727446437589e+00, 0.2336253154027212e-01, 0.5296192722406249e+00, 0.2416414888807097e+00, 0.8901032371078870e+00, 0.6213866196646989e+00, 0.7130035148592161e+00, 0.7957781913464947e+00, 0.8476792055481847e+00, 0.9456821938668590e+00, 0.9119251640705257e+00, 0.6793496748825844e+00, 0.1959111178023272e+00, 0.1712010119398182e+00, 0.1826738976471362e+00, 0.2526159624017289e+00, 0.1636253698204285e+00, 0.5186411299293459e+00, 0.9760895321955887e-01, 0.6893646610477002e+00, 0.9919227436537437e-01, 0.3790249398464486e+00, 0.9060767708258410e+00, 0.1329141344440885e+00, 0.8748053076614649e+00, 0.3613991427617499e+00, 0.1317424493721138e+00, 0.1676759484196766e-01, 0.7195688942861501e+00, 0.7508197149131951e+00, 0.1084062966598669e+00, 0.7799888885196686e+00, 0.8731271868042269e+00, 0.7281667421703623e+00, 0.5506096404209293e+00, 0.2344365678709384e+00, 0.8253982006984435e+00, 0.6925029659724733e+00, 0.2137159289768908e+00, 0.6304949708624844e+00, 0.2263541700174215e+00, 0.3731966957484361e+00, 0.4056710443895672e-01, 0.1518575777839423e+00, 0.7000182640684851e+00, 0.5207195688041865e+00, 0.3546513077756396e+00, 0.5223091585165126e+00, 0.1312526670646489e+00, 0.6075518716591177e+00, 0.7815133160786110e+00, 0.1466092133856621e+00, 0.6908403097208041e+00, 0.5289383481177163e+00, 0.3060973141340825e+00, 0.2719751878023001e+00, 0.4367099092855503e+00, 0.1678669773704986e+00, 0.2384522838837465e+00, 0.7578135131488263e+00, 0.2260651503779529e+00},
			n:    21,
			zOut: []float64{0.3001718609114466e+01, 0.2461580649918252e+01, 0.2139800724137603e+01, 0.2110198913058292e+01, 0.1773285881581521e+01, 0.1731854037718996e+01, 0.1434868444776101e+01, 0.1292334845681036e+01, 0.1202898865215905e+01, 0.9074690574580867e+00, 0.8145281637318031e+00, 0.6833997197084192e+00, 0.6303801796404412e+00, 0.5115989792032534e+00, 0.4365028215455523e+00, 0.3347059637856380e+00, 0.2704761845997483e+00, 0.1007567293499011e+00, 0.7404356572986662e-01, 0.2225224658576705e-01, 0.1003499792335912e-03, 0.1163915500165708e-19, 0.5702083577853931e-06, 0.6235005324467382e-06, 0.1434868444776101e+01, 0.7352183636659830e-27, 0.3221030091629643e-04, 0.6932009635225803e-04, 0.1292334845681036e+01, 0.1547654926086715e-20, 0.2859096049606110e-03, 0.1689403705904475e-03, 0.1202898865215905e+01, 0.5187952724823741e-19, 0.3457740509453089e-04, 0.1548723988845099e-04, 0.9074690574580867e+00, 0.1883346903736854e-18, 0.4924250006037621e-06, 0.3064183963004930e-06, 0.8145281637318031e+00, 0.7436271633171000e-16, 0.2193475493251988e+02, 0.2193475493251988e+02, 0.9400000000000000e+02, 0.2913832199546485e+01, 0.3191489361702128e+01, 0.6928384721316203e-06, 0.6303801796404412e+00, 0.4344707644406440e-17, 0.8243004381261832e-06, 0.1242944619712133e-05, 0.5115989792032534e+00, 0.1321342016245195e-28, 0.6830327722875946e-05, 0.4610227507804771e-05, 0.4365028215455523e+00, 0.7922538128148438e-16, 0.5185315150043044e-04, 0.9574812800552337e-04, 0.3347059637856380e+00, 0.4042912139257686e-29, 0.6190562688247171e-03, 0.1111720505727282e-02, 0.2704761845997483e+00, 0.1659171176970474e-17, 0.3895087988050344e-03, 0.1929449868539905e-03, 0.1007567293499011e+00, 0.2919870821923328e-23, 0.2088900906409071e-03, 0.1786849109209585e-03, 0.7404356572986662e-01, 0.6310887241768094e-28, 0.1266951898160773e-03, 0.1217799054154213e-03, 0.2225224658576705e-01, 0.3697785493223493e-31, 0.1828672832615040e-03, 0.2196698921083079e-03, 0.1003499792335912e-03, 0.0000000000000000e+00, 0.1256202468809242e-02, 0.9158129895566399e-03},
			info: 0,
		},
		{
			z:    []float64{0.7473165994595915e+00, 0.9383144044637891e+00, 0.9705766820209645e+00, 0.3093136353727625e+00, 0.7158638842786125e+00, 0.8927477442087851e+00, 0.5846837014009232e-02, 0.2823227977987499e+00, 0.4656399961948744e+00, 0.1783480145929806e-01, 0.8517241739910267e+00, 0.2968236367983081e+00, 0.3828703962512207e+00, 0.1618551942929359e+00, 0.8422815023410849e+00, 0.9667213467147939e+00, 0.1872774458350456e+00, 0.9673785609387944e+00, 0.1668393270938346e+00, 0.1793363294699661e+00, 0.3882509817575230e+00, 0.9322497724034516e+00, 0.1678634979113953e+00, 0.4000730353927472e+00, 0.1370854040387474e+00, 0.1383728853360747e+00, 0.8728086099072359e+00, 0.7142605368034651e+00, 0.1298497843576266e+00, 0.4890695005398649e+00, 0.6561831309305255e+00, 0.2551502145039813e+00, 0.3137603328050955e+00, 0.4725573834392259e+00, 0.1864194559915778e+00, 0.2476903242862807e+00, 0.3016682498100809e-01, 0.6028908129301045e+00, 0.9699532639034771e+00, 0.6804787742392342e+00, 0.5085224418131328e+00, 0.8160312832212636e+00, 0.4537844508149779e+00, 0.7698002058576117e+00, 0.9102837491884663e-01, 0.7957963901796614e+00, 0.4129806172970929e+00, 0.1183929303892000e+00, 0.7231979382284242e+00, 0.6908050716155305e+00, 0.9129211251463741e+00, 0.8407226028232299e+00, 0.7511043357392063e+00, 0.6614037096770188e+00, 0.5816383867961591e+00, 0.9508597818692400e+00, 0.6671834981267941e+00, 0.5353731249555691e+00, 0.8270836995328532e+00, 0.4748649574057349e+00, 0.7310961648034807e+00, 0.5329698192925740e+00, 0.5537463638220994e+00, 0.8202244086458278e+00, 0.3089778794417031e-01, 0.8347421543151022e+00, 0.7341688304615497e+00, 0.6293051268804178e+00, 0.9685330131023850e-01, 0.2218306173097127e+00, 0.4005982063157664e+00, 0.8004209862846372e+00, 0.9322471342744776e-02, 0.5252230232113276e+00, 0.8078446044346348e+00, 0.8663675770089085e-02, 0.4174348230921995e+00, 0.5316704871571061e+00, 0.8080873214304936e+00, 0.3191328650837369e+00, 0.2265528787275422e+00, 0.1299687059183835e+00, 0.5606006275148095e+00, 0.2716326223111132e+00},
			n:    21,
			zOut: []float64{0.2557856639061994e+01, 0.2221569823028635e+01, 0.2113175506056959e+01, 0.1791034203034858e+01, 0.1626866366713385e+01, 0.1553853827140545e+01, 0.1378038738404736e+01, 0.1313711478279983e+01, 0.1103021858691488e+01, 0.8924372074434684e+00, 0.7465727975101516e+00, 0.5658880123536838e+00, 0.5427616116084486e+00, 0.3934406603234193e+00, 0.3139076008306549e+00, 0.2922956394268174e+00, 0.1324157743308766e+00, 0.8492495339426936e-01, 0.1726019296564064e-01, 0.1478900502211735e-02, 0.3047316799617542e-04, 0.0000000000000000e+00, 0.2209423397088722e-18, 0.1230851536694111e-18, 0.1378038738404736e+01, 0.4653537180464338e-22, 0.4724643033910209e-17, 0.2943893377745906e-17, 0.1313711478279983e+01, 0.2193156221990361e-16, 0.1119006893822957e-16, 0.1741431705208298e-16, 0.1103021858691488e+01, 0.2831227495869906e-12, 0.2860829853751284e-15, 0.4129571258389818e-15, 0.8924372074434684e+00, 0.1711229892219076e-22, 0.2249832739645017e-14, 0.3254005212313171e-14, 0.7465727975101516e+00, 0.5231751935973153e-19, 0.1964254226427022e+02, 0.1964254226427022e+02, 0.1110000000000000e+03, 0.3285714285714286e+01, 0.8108108108108109e+01, 0.1373504512438542e-12, 0.5427616116084486e+00, 0.0000000000000000e+00, 0.2663664432452481e-11, 0.1451131289962852e-11, 0.3934406603234193e+00, 0.3021704178298789e-14, 0.8991877456137337e-10, 0.1576401145017008e-09, 0.3139076008306549e+00, 0.2274554202459952e-24, 0.7202855721215940e-08, 0.4214854283718170e-08, 0.2922956394268174e+00, 0.3975858962313899e-27, 0.6024353396932969e-07, 0.1022672043122323e-06, 0.1324157743308766e+00, 0.6969933701296916e-21, 0.1098572510799548e-05, 0.1404105143401733e-05, 0.8492495339426936e-01, 0.6511321020566649e-24, 0.5604469692434887e-06, 0.6104659421650381e-06, 0.1726019296564064e-01, 0.9860761315262648e-31, 0.7747760678878360e-06, 0.1026260244983655e-05, 0.1478900502211735e-02, 0.1334496117765360e-28, 0.5102973157105362e-05, 0.1258609455819239e-04, 0.3047316799617542e-04, 0.0000000000000000e+00, 0.4753862314998056e-03, 0.1863315015192667e-02},
			info: 0,
		},
		{
			z:    []float64{0.2548672807470598e-01, 0.1495421137012933e+00, 0.6637277283391485e-01, 0.6233200993667132e-01, 0.4958061473131723e+00, 0.9804571594196556e+00, 0.5741929121502225e+00, 0.9319350517768583e+00, 0.5154498317218548e-01, 0.4165283024496188e+00, 0.9821854184367015e+00, 0.7510033276340442e+00, 0.2648151368970406e+00, 0.9698327797382789e-01, 0.1433813304773114e+00, 0.1687421369042604e+00, 0.6230509959313209e+00, 0.8106153730403777e+00, 0.5719487513699320e+00, 0.5761793262532514e+00, 0.4983158430615118e+00, 0.8860830618783583e+00, 0.7136722399358287e+00, 0.6310350013419201e+00, 0.5509601824732582e+00, 0.5806336147624858e-01, 0.1818648959869262e+00, 0.2346005241916651e+00, 0.5346851174706903e+00, 0.4444466121668155e+00, 0.8418378722202517e+00, 0.6164532479196363e+00, 0.4298553148227666e+00, 0.2296849794057675e+00, 0.3620982682376475e+00, 0.3645096805309075e+00, 0.9980228064491022e+00, 0.7106112428543574e+00, 0.6651697421094976e+00, 0.9830151455980368e+00, 0.3483046917749841e+00, 0.3668589729383628e+00, 0.5655614401040915e+00, 0.8891526728148491e+00, 0.9164649086142118e+00, 0.4065736846475075e+00, 0.1470436468080603e+00, 0.5802557267939668e+00, 0.8272597245263099e+00, 0.7163617706554632e+00, 0.3488790096540706e+00, 0.6404227109073246e+00, 0.1472796557920839e+00, 0.8764536863696604e-02, 0.5020645462548778e-01, 0.6906675094909457e+00, 0.4668434450047272e+00, 0.7886435616506033e+00, 0.8775817704321313e+00, 0.4246952359284411e+00, 0.1646609549881360e+00, 0.6697681588293083e+00, 0.7914692739599321e+00, 0.1396674072195736e-01, 0.5167232939537366e+00, 0.9374340964740459e+00, 0.4228531365925756e+00, 0.9727064517088465e+00, 0.5765501520393458e+00, 0.5648508784605104e+00, 0.3049637657479770e+00, 0.3063190445223624e+00, 0.5204528216889723e+00, 0.8180178920731451e+00, 0.4938773599630263e+00, 0.8975563911549045e-01, 0.5983688838646251e+00, 0.3698925131813697e+00, 0.2992530973981942e+00, 0.7463539101278609e+00, 0.8005980511688328e+00, 0.7944834752945218e-01, 0.3132925544163919e+00, 0.8866951016419623e+00},
			n:    21,
			zOut: []float64{0.2548744028948828e+01, 0.2439584200134194e+01, 0.2276898263864132e+01, 0.2083856178014486e+01, 0.2039342663979975e+01, 0.1949418505713503e+01, 0.1425638870874207e+01, 0.1087729835971203e+01, 0.9667696375323971e+00, 0.8022433247488755e+00, 0.5846254874613492e+00, 0.5355025899480791e+00, 0.3940805795947064e+00, 0.2663780336330241e+00, 0.2413081251593111e+00, 0.1817740817404329e+00, 0.1073144197241267e+00, 0.4327520903789019e-01, 0.3141654146246065e-01, 0.2017433957164045e-01, 0.3181705377209195e-03, 0.2778526523712400e-12, 0.1839247883012458e-10, 0.2248848580297948e-10, 0.1425638870874207e+01, 0.4660214926831417e-16, 0.5684400877302917e-10, 0.6387162891082867e-10, 0.1087729835971203e+01, 0.6729715978266918e-13, 0.1741585337702246e-09, 0.2256709509706825e-09, 0.9667696375323971e+00, 0.8583633357857125e-17, 0.5122248377584688e-09, 0.7095864943104643e-09, 0.8022433247488755e+00, 0.1479114197289397e-30, 0.3545102092401686e-08, 0.2547082836533246e-08, 0.5846254874613492e+00, 0.1684748530176799e-17, 0.2002639308765254e+02, 0.2002639308765254e+02, 0.9900000000000000e+02, 0.2829931972789116e+01, 0.2020202020202020e+01, 0.1450844904575395e-08, 0.3940805795947064e+00, 0.8431345355002174e-26, 0.2834737824482197e-07, 0.1323731756234205e-07, 0.2663780336330241e+00, 0.1205202152489745e-19, 0.2553256673132341e-06, 0.1983391491151363e-06, 0.2413081251593111e+00, 0.5052150994009667e-21, 0.4174664103103917e-06, 0.5361294231508838e-06, 0.1817740817404329e+00, 0.4777592300362271e-19, 0.1103166278762495e-05, 0.1411552925870387e-05, 0.1073144197241267e+00, 0.1369462531463676e-25, 0.4868310689385493e-05, 0.3749559753873054e-05, 0.4327520903789019e-01, 0.6517810995642209e-18, 0.3308901711413922e-04, 0.5021439046381042e-04, 0.3141654146246065e-01, 0.3980258690751375e-19, 0.2091355707842741e-03, 0.1263714817696205e-03, 0.2017433957164045e-01, 0.9608259223012324e-19, 0.2907989762339014e-03, 0.3270695437261140e-04, 0.3181705377209195e-03, 0.2078018254813854e-20, 0.5046681645946245e-02, 0.5971971384809690e-02},
			info: 0,
		},
		{
			z:    []float64{0.7556699778432414e+00, 0.9264145479756024e+00, 0.2732571748073197e+00, 0.6411921070422063e+00, 0.9885815056247259e+00, 0.3723918099656685e+00, 0.8347397513753253e+00, 0.1449609854320509e+00, 0.7237973270068254e+00, 0.7937879503069174e+00, 0.1580314367294657e+00, 0.6183123759968190e-01, 0.8526562423479169e+00, 0.4515796037285547e+00, 0.4273610709781387e+00, 0.9916977661043136e-01, 0.6106106148524300e+00, 0.8331371477897991e+00, 0.3226596180395027e+00, 0.9411873968460633e+00, 0.4713107215814810e+00, 0.2408559451890391e-01, 0.8159283865403915e+00, 0.2484819964854622e+00, 0.9053457264816500e-01, 0.7765873412573800e+00, 0.6903393919382517e-01, 0.4737861094943234e+00, 0.2953181034757211e+00, 0.1455852388723616e+00, 0.3595484224812872e+00, 0.7546693773577520e+00, 0.2176499262400425e+00, 0.2180463762362660e+00, 0.4619066216292487e+00, 0.9195384505328125e+00, 0.7817199663072332e+00, 0.7252058550809327e+00, 0.7231496659924924e+00, 0.4997135548263443e+00, 0.7857040551775740e-01, 0.9096915224111880e+00, 0.5048166916632318e+00, 0.9737715836828489e+00, 0.6597329703373522e+00, 0.1262659377119990e-01, 0.3525536903406803e+00, 0.6167857219758321e+00, 0.7408468432196044e+00, 0.7867196986400673e+00, 0.2313289425583596e+00, 0.8524506859175643e+00, 0.1392061690435467e+00, 0.4799927308227143e+00, 0.5988201473645773e+00, 0.8148615971957359e+00, 0.9991849415795491e+00, 0.4062458282025133e+00, 0.8021012361110058e-01, 0.7041715844729369e+00, 0.9319571339573152e-01, 0.9303801779957639e+00, 0.6825635414858495e+00, 0.2651022334979941e+00, 0.2966022780498856e-01, 0.2756350604456157e+00, 0.3996117016882065e+00, 0.4084803031952775e+00, 0.7693093419592285e+00, 0.5425247266459055e+00, 0.7317803042615013e+00, 0.9398853135227920e+00, 0.5923013015298344e+00, 0.6749323490534131e-01, 0.2677632908794654e+00, 0.1468441510303787e+00, 0.1088650133568649e+00, 0.2179701521651023e+00, 0.8673093102161434e+00, 0.5667697713082290e+00, 0.5061732946886482e+00, 0.4707781162637231e+00, 0.4572852405093119e+00, 0.2817384205908577e+00},
			n:    21,
			zOut: []float64{0.2549866140639053e+01, 0.2289692363298807e+01, 0.2042824675422646e+01, 0.1807091836494319e+01, 0.1608583829794490e+01, 0.1518595775727545e+01, 0.1431682400757094e+01, 0.1278141352537376e+01, 0.1218077200015686e+01, 0.1115746594519385e+01, 0.9489661418934578e+00, 0.7950491688285616e+00, 0.6705557578092293e+00, 0.4223587911677574e+00, 0.3211641543143434e+00, 0.1395786344205778e+00, 0.9463219952875908e-01, 0.4563841563552573e-01, 0.3718976888616738e-01, 0.2766660139323238e-01, 0.2861060880329385e-03, 0.1489369389057270e-25, 0.4009456273839574e-11, 0.5913965075566282e-11, 0.1431682400757094e+01, 0.9808641529204798e-18, 0.1658075466067222e-10, 0.2375182467377229e-10, 0.1278141352537376e+01, 0.1561716996498461e-20, 0.5563872147228742e-10, 0.4003249215027731e-10, 0.1218077200015686e+01, 0.8198555146385643e-18, 0.1047304739332037e-09, 0.1509388346471184e-09, 0.1115746594519385e+01, 0.6804594514014344e-21, 0.1132515351796206e-08, 0.7305070647547987e-09, 0.9489661418934578e+00, 0.6324297261214257e-19, 0.2036338790917205e+02, 0.2036338790917205e+02, 0.1050000000000000e+03, 0.3147392290249433e+01, 0.2857142857142857e+01, 0.1188392639382130e-06, 0.6705557578092293e+00, 0.2287929633195320e-19, 0.1283168907741916e-05, 0.1048613283787068e-05, 0.4223587911677574e+00, 0.1189072130544977e-23, 0.1798535669474704e-05, 0.2477367923932174e-05, 0.3211641543143434e+00, 0.8187795594567816e-23, 0.5813187502570164e-05, 0.4274698539216166e-05, 0.1395786344205778e+00, 0.4387796946624011e-14, 0.1512969274311187e-04, 0.4383088766938462e-04, 0.9463219952875908e-01, 0.2802074325023381e-22, 0.1713816584845726e-03, 0.1715892923902221e-03, 0.4563841563552573e-01, 0.8331483361628599e-18, 0.1806157324013782e-06, 0.1193242630586832e-05, 0.3718976888616738e-01, 0.1396316416906445e-20, 0.3480495660704905e-03, 0.4402048010276426e-03, 0.2766660139323238e-01, 0.1504073756330590e-24, 0.6353091961964474e-03, 0.1317363577877200e-03, 0.2861060880329385e-03, 0.4589883047388968e-23, 0.3883745848947868e-02, 0.5358907697572753e-02},
			info: 0,
		},
		{
			z:    []float64{0.8456683250225028e-01, 0.3058015837199188e+00, 0.4317902183064470e-01, 0.3671979469896071e+00, 0.3595487854233631e+00, 0.8791524368561765e+00, 0.1256853689373587e+00, 0.3494000337993621e+00, 0.2131672180923028e+00, 0.4488324680055523e+00, 0.9452502344450827e+00, 0.7197446096567599e+00, 0.8918071015747069e+00, 0.5551949519753436e+00, 0.5627883747367335e+00, 0.6930858113379974e+00, 0.7026202611250518e+00, 0.3856840921343180e+00, 0.4453224782611998e+00, 0.9627551339340559e+00, 0.6255469039377685e+00, 0.6093210861157965e+00, 0.1297243619022892e-01, 0.2745425059216151e+00, 0.5902573930769862e+00, 0.3658719785914519e+00, 0.8179101736936932e+00, 0.4251816548465440e+00, 0.3319568042078717e+00, 0.8294675656951641e+00, 0.1906193594777114e+00, 0.6862699206380130e+00, 0.6832362040809048e+00, 0.8968537122850908e+00, 0.3753636108676777e+00, 0.9926770667520373e+00, 0.7473790867948205e-01, 0.3653635034012445e+00, 0.6553100464759598e-01, 0.1045856631958887e+00, 0.2265961870023414e+00, 0.3205967446655300e-01, 0.1845082884370131e+00, 0.2250092498055734e+00, 0.5876866067653603e+00, 0.2146512915654570e+00, 0.3937206506545190e+00, 0.4110924225795073e+00, 0.3228836105384403e+00, 0.3531543308069818e-01, 0.9472843614842231e+00, 0.2348052564069324e+00, 0.1315455878409307e+00, 0.2246729768637089e+00, 0.8445861542610614e+00, 0.9528962673528295e+00, 0.4907499555456967e+00, 0.9420172239501383e+00, 0.4896535244350841e-01, 0.5088622551995382e+00, 0.1789066474661110e+00, 0.3566730680524121e+00, 0.4279936896698711e-01, 0.5241586506933626e-02, 0.9191361995860731e+00, 0.7666474814508011e+00, 0.8302596584023555e+00, 0.4738482866502817e+00, 0.2085735695866969e+00, 0.3976808123484415e+00, 0.4364253757027529e+00, 0.2407360056333678e+00, 0.4002772343201099e+00, 0.7717338386755229e+00, 0.5023514841326452e+00, 0.8988902694804358e+00, 0.4259852339101408e+00, 0.2067093317010162e+00, 0.8456128611542781e+00, 0.1290027044731792e+00, 0.2598901078235174e+00, 0.5836006873243835e+00, 0.9777637054617091e+00, 0.4896357840832322e+00},
			n:    21,
			zOut: []float64{0.2625813250937807e+01, 0.2316814186652984e+01, 0.2239421861777163e+01, 0.1853308891402210e+01, 0.1827033391537137e+01, 0.1481398580531558e+01, 0.1385699025778513e+01, 0.1101045596651604e+01, 0.1077490875294086e+01, 0.6790358065311344e+00, 0.6361857378195889e+00, 0.5095707085100130e+00, 0.4739579905788711e+00, 0.4209497658895578e+00, 0.3012142858983374e+00, 0.2432742659092710e+00, 0.2088674538811938e+00, 0.1664409530858738e+00, 0.2606656040091844e-01, 0.1205810631995507e-01, 0.9325512290661112e-07, 0.1292469707114106e-24, 0.1719039818430849e-04, 0.1845080438772963e-04, 0.1385699025778513e+01, 0.2224123229986848e-19, 0.6819642419856913e-05, 0.9079053758699490e-05, 0.1101045596651604e+01, 0.6054420446023063e-14, 0.5392861994521241e-06, 0.8422816920290823e-06, 0.1077490875294086e+01, 0.7330726620037819e-25, 0.4290042344637304e-06, 0.3471234449409943e-06, 0.6790358065311344e+00, 0.1969273319091676e-18, 0.8635424232755040e-07, 0.8294124288205587e-07, 0.6361857378195889e+00, 0.4466775307674635e-22, 0.1958564738864290e+02, 0.1958564738864290e+02, 0.9700000000000000e+02, 0.2927437641723356e+01, 0.2061855670103093e+01, 0.4284285350948107e-09, 0.4739579905788711e+00, 0.4844247598072386e-17, 0.1501956526577450e-07, 0.4599310764980267e-07, 0.4209497658895578e+00, 0.5697556379758414e-13, 0.1468884761079027e-05, 0.1935721204206928e-05, 0.3012142858983374e+00, 0.3414189997796539e-26, 0.7129394687116921e-05, 0.9262068822983554e-05, 0.2432742659092710e+00, 0.2465190328815662e-31, 0.3320148255009530e-04, 0.2563439348523979e-04, 0.2088674538811938e+00, 0.8874685183736383e-29, 0.9377671959730327e-04, 0.7166613903737452e-04, 0.1664409530858738e+00, 0.1175218723307338e-19, 0.5883141447281398e-03, 0.4140832085136277e-03, 0.2606656040091844e-01, 0.2634795423438179e-27, 0.4797918408258123e-02, 0.4833036765864798e-02, 0.1205810631995507e-01, 0.6530371216255944e-19, 0.7811358433513861e-02, 0.1392611548870054e-01, 0.9325512290661112e-07, 0.1190420191421466e-12, 0.6274023555637595e-01, 0.1241063086176471e-01},
			info: 0,
		},
		{
			z:    []float64{0.6345014789904082e+00, 0.9828013844807603e+00, 0.5292373213190880e+00, 0.3345627500234365e+00, 0.9345959669050260e+00, 0.5873733336147081e+00, 0.8888462128293518e+00, 0.1044020545901757e+00, 0.9375280272458696e+00, 0.1233411973951652e+00, 0.3330283913003940e+00, 0.2432208394685875e-01, 0.6106490172620382e+00, 0.9136489477065954e+00, 0.8250850334045809e+00, 0.7866479843500775e+00, 0.1629924039124250e+00, 0.2349971551206693e+00, 0.1024684774133804e+00, 0.7852727749024598e+00, 0.1427984573527686e+00, 0.4315181982028241e+00, 0.6564047419338838e+00, 0.5555590030741936e+00, 0.8205569452257172e+00, 0.4368031922318115e+00, 0.2897575605849890e+00, 0.6163386054890784e+00, 0.7016794584968877e+00, 0.7265422079630662e+00, 0.6393115149906525e+00, 0.7552597718778955e+00, 0.5029129969011173e+00, 0.4874732646810529e+00, 0.5094624150009296e+00, 0.4766369914172104e+00, 0.9003818070855094e+00, 0.6813014247956244e+00, 0.2988296447575162e+00, 0.6597790416796705e+00, 0.5646192150605487e+00, 0.3275024686793908e+00, 0.2725061169408716e-01, 0.8098076662320616e+00, 0.3345276190659139e+00, 0.1812780354242056e+00, 0.1997802571871318e+00, 0.5440765633504190e+00, 0.9774510594766147e+00, 0.1863483792916641e+00, 0.5910097330685633e+00, 0.2734489448477760e+00, 0.4395818444416557e+00, 0.9022786559097939e+00, 0.1029995124123335e+00, 0.9437933734967648e+00, 0.1226746192846208e+00, 0.1122541771291640e+00, 0.1756516455268763e+00, 0.9109529416432673e+00, 0.3050345572159197e-01, 0.1746678233544180e+00, 0.3750993781268622e+00, 0.4476224977918100e+00, 0.5437446694835568e+00, 0.3729575052852190e+00, 0.3542378566336749e+00, 0.7449948633772266e+00, 0.3157555204945224e+00, 0.7753984984296890e+00, 0.3284038993889418e+00, 0.5159361949232710e-01, 0.4529339074400007e-01, 0.5693581512511349e+00, 0.2213156237057901e+00, 0.9395831428155459e+00, 0.3843047271617620e+00, 0.2968653921042081e+00, 0.6156090467550724e+00, 0.3170556612853570e-01, 0.8265655132088339e+00, 0.3165750369023552e+00, 0.7055602587055542e+00, 0.8365600455841105e+00},
			n:    21,
			zOut: []float64{0.2473859321883559e+01, 0.2422834159796051e+01, 0.2342770718537345e+01, 0.2190393245355733e+01, 0.2048748874742353e+01, 0.1880327408972340e+01, 0.1569014269461521e+01, 0.1302496773444914e+01, 0.1193514796722045e+01, 0.1150651682141956e+01, 0.8969794111087930e+00, 0.8226581338194529e+00, 0.6917042463755413e+00, 0.5972139808466055e+00, 0.3480375932216391e+00, 0.2932789944580818e+00, 0.2090971942801221e+00, 0.1221616224423389e+00, 0.1000478595340333e+00, 0.3412207557320287e-01, 0.3160927987836547e-03, 0.2386777554836693e-25, 0.1787847554062023e-08, 0.1930667016131883e-08, 0.1569014269461521e+01, 0.5039432348921582e-18, 0.3047636948536222e-07, 0.1223330280849913e-07, 0.1302496773444914e+01, 0.0000000000000000e+00, 0.7408697490815375e-06, 0.1498788057488595e-05, 0.1193514796722045e+01, 0.6273021918317486e-26, 0.8867922405978688e-08, 0.3710956866722375e-08, 0.1150651682141956e+01, 0.1462883977034466e-17, 0.2681277488017278e-06, 0.6188429230273071e-06, 0.8969794111087930e+00, 0.4930380657631324e-31, 0.2269022845551642e+02, 0.2269022845551641e+02, 0.1110000000000000e+03, 0.3215419501133787e+01, 0.4504504504504505e+01, 0.4606971433208740e-07, 0.6917042463755413e+00, 0.7123549154360197e-18, 0.1197511514786163e-07, 0.1386949181837661e-07, 0.5972139808466055e+00, 0.4652335183636722e-20, 0.1082703076660205e-07, 0.1003906263016573e-07, 0.3480375932216391e+00, 0.3573127201018094e-16, 0.2388210318135170e-06, 0.5694782222336181e-06, 0.2932789944580818e+00, 0.7601337464964823e-24, 0.7791755144624492e-05, 0.3242296896441773e-05, 0.2090971942801221e+00, 0.1663630656286764e-22, 0.9916420504550920e-06, 0.3916254919710180e-05, 0.1221616224423389e+00, 0.1406614427791829e-18, 0.1356917845842480e-04, 0.1379449187478977e-04, 0.1000478595340333e+00, 0.2800456213534592e-28, 0.3213444959426999e-04, 0.2790230345312924e-04, 0.3412207557320287e-01, 0.1075846717720530e-17, 0.8531299184272352e-03, 0.1419520770966347e-02, 0.3160927987836547e-03, 0.1848892746611746e-31, 0.7376316906583622e-02, 0.6546496744187779e-02},
			info: 0,
		},
		{
			z:    []float64{0.7781991870339823e+00, 0.3799726947828087e+00, 0.8225406656461727e+00, 0.4322410258771066e+00, 0.4965834581857734e+00, 0.1642548945285990e+00, 0.4773703559733889e+00, 0.7580746486223987e+00, 0.9220155673403277e+00, 0.1624062232083541e+00, 0.5664470130564820e+00, 0.7367013127895712e+00, 0.4720756942873998e+00, 0.8695958825780965e+00, 0.6045812450089678e+00, 0.8503421320137978e+00, 0.7291568033584502e+00, 0.9563602214514951e+00, 0.1245752972610169e+00, 0.2621011209284912e+00, 0.8642265339858576e+00, 0.9528021036615359e+00, 0.9515862784178062e+00, 0.2632504807072628e+00, 0.8585529108378003e+00, 0.3025834513231899e+00, 0.7125877353473579e+00, 0.9221275053156179e+00, 0.9787992369955746e+00, 0.8789465750552752e+00, 0.9318675804398889e+00, 0.8827440896392771e+00, 0.3124337393280541e+00, 0.3124656972853590e-01, 0.5860766227864144e+00, 0.9812515177915087e+00, 0.4480000254042209e+00, 0.5121903109069345e+00, 0.1837298514334742e+00, 0.2680410429176278e+00, 0.5529294510898309e+00, 0.3627557909974974e+00, 0.9282219359125773e+00, 0.2173723036967439e+00, 0.7593665130145739e+00, 0.6245533364325671e+00, 0.1767680905493787e+00, 0.3823356866170564e+00, 0.5171578058722374e+00, 0.5376155024979712e+00, 0.5280269975706942e-01, 0.7947058949878518e+00, 0.1214716509814368e+00, 0.6840110600217510e+00, 0.7041243787921371e+00, 0.6945194787199669e+00, 0.8184844284849127e+00, 0.3440855810599863e+00, 0.1619958520504678e+00, 0.6990470088709023e+00, 0.3532225659848430e+00, 0.8229090368317697e+00, 0.8349696782028652e+00, 0.6932180266797183e-01, 0.6880274120498576e+00, 0.7566065163897195e+00, 0.2981422921509080e-01, 0.8636153925759227e-02, 0.7928093803058089e-01, 0.3096144259285381e+00, 0.6829886066202427e+00, 0.8246144682759358e+00, 0.2007259621364732e+00, 0.8710862365466575e+00, 0.6898755422119236e+00, 0.9105030198433044e+00, 0.8974818523094739e+00, 0.8909991694059211e+00, 0.8084341913662618e-01, 0.6072211485624930e+00, 0.9680950749726419e+00, 0.3932275197607872e+00, 0.8781208296025552e+00, 0.5527043030768199e+00},
			n:    21,
			zOut: []float64{0.3158195987040060e+01, 0.2674397094280163e+01, 0.2611167345511754e+01, 0.2117792657212280e+01, 0.2026395303611904e+01, 0.1943316400161173e+01, 0.1877195462180231e+01, 0.1801757023901512e+01, 0.1273111869927408e+01, 0.1088705127135854e+01, 0.9559661931705944e+00, 0.9013607966201198e+00, 0.6445742634037150e+00, 0.5446175180211763e+00, 0.4486902226406620e+00, 0.3854842198344371e+00, 0.2741584921777233e+00, 0.1210105287753585e+00, 0.6940868812740908e-01, 0.2059029850850508e-01, 0.3673564803695461e-02, 0.8077935669463161e-25, 0.2195808487174199e-10, 0.8449955132778232e-10, 0.1877195462180231e+01, 0.1633831636278255e-19, 0.3497961877980620e-08, 0.7650955873935623e-08, 0.1801757023901512e+01, 0.4072241987713803e-24, 0.2126842707416970e-06, 0.1130018646589442e-06, 0.1273111869927408e+01, 0.1740941751301712e-20, 0.3388039768888790e-03, 0.2863573324805332e-03, 0.1088705127135854e+01, 0.2858227958137079e-16, 0.2905468688553340e-03, 0.2787380120464217e-03, 0.9559661931705944e+00, 0.1315475667900737e-21, 0.2494156905704573e+02, 0.2494156905704573e+02, 0.1010000000000000e+03, 0.3074829931972789e+01, 0.1980198019801980e+01, 0.2647338938369395e-05, 0.6445742634037150e+00, 0.2341846562030201e-22, 0.2667503836084351e-03, 0.1510510164587073e-03, 0.5446175180211763e+00, 0.2242636890234710e-24, 0.5495840118850513e-05, 0.4810476319752364e-05, 0.4486902226406620e+00, 0.3552400222016096e-16, 0.2740332708518062e-05, 0.2562621073339741e-05, 0.3854842198344371e+00, 0.1036641484462207e-22, 0.2248949759797675e-05, 0.2492010005917310e-05, 0.2741584921777233e+00, 0.6071156729318302e-20, 0.4130294607665845e-03, 0.1141603094124651e-03, 0.1210105287753585e+00, 0.2468629837810345e-16, 0.7230132634032991e-02, 0.6618694284253705e-02, 0.6940868812740908e-01, 0.5487745597049798e-23, 0.4151017130761703e-02, 0.4525758324635968e-02, 0.2059029850850508e-01, 0.6083695301064443e-26, 0.3095027772299370e-02, 0.3218440937246976e-02, 0.3673564803695461e-02, 0.6106717417722413e-22, 0.6443405665983428e-02, 0.1622720765285148e-02},
			info: 0,
		},
		{
			z:    []float64{0.1985414918961074e+00, 0.4386372064833388e-01, 0.9272679997999695e+00, 0.9276564660006267e+00, 0.9509373652834754e+00, 0.6247274629358034e+00, 0.4838472987754385e+00, 0.6222976111124274e+00, 0.4248800474043221e+00, 0.9755567793536163e+00, 0.8912410754920851e+00, 0.2313492132656925e-01, 0.1563015923526294e+00, 0.7905214363406186e+00, 0.4927898149840948e+00, 0.5370919594301193e+00, 0.4644799081842790e+00, 0.9960946978984735e+00, 0.2493095982292803e+00, 0.3814725574373231e+00, 0.2940816225707878e+00, 0.6535898793246008e+00, 0.2084037192546966e+00, 0.5132959253232522e+00, 0.4251209694264122e+00, 0.9378579353373483e+00, 0.9786807729708236e+00, 0.7781441628535176e+00, 0.3951333405907657e+00, 0.3920742203259495e+00, 0.1500883321285087e+00, 0.9121803806188729e+00, 0.8516361075320147e+00, 0.6824417668614835e-01, 0.1390781730568696e+00, 0.3112124735573820e+00, 0.2535919561468232e+00, 0.7644352497405866e+00, 0.9218252733114815e+00, 0.9901458352558505e+00, 0.3279511469507669e+00, 0.8365860218696105e+00, 0.3198677553816687e+00, 0.6133665370063144e+00, 0.2580491557527305e+00, 0.6586714927743139e+00, 0.3933698458458805e+00, 0.7194383869235133e-01, 0.2213903901048253e-02, 0.1496106697134404e+00, 0.6766438239122079e+00, 0.7197724372602395e+00, 0.4797148651858846e+00, 0.5998154083597278e+00, 0.5590962662612089e+00, 0.6336887670809047e+00, 0.1072453740736307e+00, 0.7505878626477551e+00, 0.1427362910235584e+00, 0.6016163404777434e+00, 0.9488238354107547e+00, 0.4206748665323531e+00, 0.4473756257202280e+00, 0.2658295729210566e+00, 0.2340075619597504e+00, 0.9617320234413022e+00, 0.5962761690405762e+00, 0.8519099982463364e+00, 0.5571747680961063e+00, 0.1982921114470393e-01, 0.2779304562486129e-01, 0.7924439389736488e+00, 0.9672136382383878e-01, 0.8417095886148459e+00, 0.7473598542298929e+00, 0.9746159458640847e+00, 0.5038819873190444e+00, 0.9169242656654556e+00, 0.5252626769192049e+00, 0.5100979601115528e+00, 0.2790149607164115e+00, 0.4554892778467722e+00, 0.9762420639370454e+00, 0.9832519154713986e+00},
			n:    21,
			zOut: []float64{0.2773247209998878e+01, 0.2684570079202721e+01, 0.2535407140485478e+01, 0.2251849067745984e+01, 0.2036533658934636e+01, 0.1904369707782458e+01, 0.1446171755624304e+01, 0.1299751831669502e+01, 0.1269533063987320e+01, 0.1007664938847636e+01, 0.8251797533661941e+00, 0.5818474246453018e+00, 0.5486201168977923e+00, 0.3772420051451134e+00, 0.3380340897724159e+00, 0.2023767003460897e+00, 0.1840234711841343e+00, 0.9225167203340243e-01, 0.4232592978815707e-01, 0.2771262583534443e-01, 0.7321455618534900e-04, 0.2977163580658995e-22, 0.3752078887401622e-10, 0.4179376226086309e-10, 0.1446171755624304e+01, 0.2236367039610280e-12, 0.1299965948722022e-10, 0.1595321259646779e-10, 0.1299751831669502e+01, 0.9926167350636332e-23, 0.8604643146515003e-10, 0.1864418628665438e-09, 0.1269533063987320e+01, 0.1306353659045996e-24, 0.1991121186029356e-08, 0.8983175437487846e-09, 0.1007664938847636e+01, 0.1213184768518325e-22, 0.3444432736029153e-07, 0.6437805745712839e-07, 0.8251797533661941e+00, 0.7772022937681214e-16, 0.2242878545784906e+02, 0.2242878545784905e+02, 0.1000000000000000e+03, 0.3024943310657596e+01, 0.3000000000000000e+01, 0.1709907812443784e-06, 0.5486201168977923e+00, 0.2972680326362443e-23, 0.1350081364219064e-06, 0.1255933647347444e-06, 0.3772420051451134e+00, 0.8046381233254320e-28, 0.7022416754922569e-08, 0.2768431074740138e-07, 0.3380340897724159e+00, 0.1719716773381806e-27, 0.7079953450794871e-07, 0.6648251035830866e-07, 0.2023767003460897e+00, 0.1940724044588524e-24, 0.4764709677102475e-07, 0.4597049352360845e-07, 0.1840234711841343e+00, 0.3459682911934042e-21, 0.4438757837191216e-07, 0.4402069254647340e-07, 0.9225167203340243e-01, 0.2596900924099520e-17, 0.8125925620122143e-07, 0.6770863964779914e-07, 0.4232592978815707e-01, 0.1280067631025782e-15, 0.3584234088053806e-06, 0.4822962810861105e-06, 0.2771262583534443e-01, 0.1012934875847350e-18, 0.1556396151524861e-05, 0.2916279508701459e-05, 0.7321455618534900e-04, 0.1540743955509789e-32, 0.4190434742116713e-02, 0.1362677535010021e-02},
			info: 0,
		},
		{
			z:    []float64{0.6703344952482478e+00, 0.3577116069291627e+00, 0.8432263018920840e+00, 0.9584130033491455e+00, 0.2861685401430958e+00, 0.9668570623620085e+00, 0.6957796127156375e+00, 0.7603022900919236e+00, 0.7677099838587329e+00, 0.2702658617954818e+00, 0.9607212720070546e-01, 0.3973573502375266e+00, 0.8455511814548733e+00, 0.6862903202354254e-01, 0.1789860417278408e+00, 0.1112926642215833e+00, 0.8320621955974556e+00, 0.2637090148175287e+00, 0.7109403389502641e+00, 0.6494529820495407e+00, 0.8828985636942858e+00, 0.9098303542318168e+00, 0.1777489928582532e+00, 0.8977650882542143e-01, 0.7154356412046237e-01, 0.2693107133832517e-01, 0.7956794415059365e-02, 0.8852594037406200e+00, 0.2162284138438216e+00, 0.9216829080200467e+00, 0.9874282279096652e+00, 0.6087745498247736e+00, 0.7694028142872190e+00, 0.5918349824988693e+00, 0.1915561020798640e+00, 0.1135335598824661e-01, 0.6670902390154858e+00, 0.6601113994758983e+00, 0.7779237148990382e+00, 0.9023999203058302e+00, 0.1848842750705929e+00, 0.6120355291150891e+00, 0.7209828136916797e+00, 0.3124354740483853e+00, 0.9520753904353354e+00, 0.9694533109968584e+00, 0.6869212426966154e+00, 0.3510392134251344e+00, 0.7696249856763533e+00, 0.9203915575905880e+00, 0.4616165896255233e+00, 0.6796681592119477e+00, 0.9261196886552191e+00, 0.4511929507295228e+00, 0.5162093776469306e+00, 0.3093101029632945e+00, 0.1162653262260159e+00, 0.1951366360676013e+00, 0.7945099687545387e+00, 0.9707806531485516e+00, 0.4161172559602336e+00, 0.5813461713597171e+00, 0.8442974147064882e+00, 0.9135490286183924e+00, 0.3261094875410662e+00, 0.7080588637364270e+00, 0.3833080381406134e+00, 0.9095051028453779e+00, 0.6909172438403831e+00, 0.5187928284332267e+00, 0.5783778059525303e+00, 0.9615928062069277e+00, 0.2127624116065856e+00, 0.7254035177440832e+00, 0.2587362009618467e+00, 0.7570212065831576e+00, 0.2401238007004233e+00, 0.2744934873404078e+00, 0.9420426561890750e+00, 0.7098059460475599e+00, 0.2691363643891775e+00, 0.4536316870833661e+00, 0.6839445343274818e+00, 0.9336823543434762e+00},
			n:    21,
			zOut: []float64{0.2634467575625416e+01, 0.2594635276829839e+01, 0.2463070407655619e+01, 0.2293127065694029e+01, 0.2072673505384417e+01, 0.1459936798288049e+01, 0.1378497856793854e+01, 0.1333239471494390e+01, 0.1102542278566289e+01, 0.8930469384549427e+00, 0.8520787727779708e+00, 0.6953298055150813e+00, 0.6847648677749263e+00, 0.2185116940651705e+00, 0.2008337976176149e+00, 0.1757551361146239e+00, 0.1271484717317025e+00, 0.5049064582277562e-01, 0.3515271339576785e-01, 0.7966224551514293e-02, 0.1685389559757755e-03, 0.1635721388548769e-22, 0.3551032227485648e-11, 0.3089325791958004e-11, 0.1378497856793854e+01, 0.6463020148652660e-17, 0.4439365591922079e-11, 0.5247211770454983e-11, 0.1333239471494390e+01, 0.8616343098037817e-25, 0.4343651596456738e-10, 0.2452653826174233e-10, 0.1102542278566289e+01, 0.9876517884224792e-16, 0.1549700306365952e-08, 0.1012434062383302e-08, 0.8930469384549427e+00, 0.4606319384654338e-19, 0.1115188986357716e-08, 0.1237013304894199e-08, 0.8520787727779708e+00, 0.1447566071967798e-23, 0.2127343784310997e+02, 0.2127343784310997e+02, 0.1270000000000000e+03, 0.3675736961451247e+01, 0.9448818897637794e+01, 0.2782364756724027e-11, 0.6847648677749263e+00, 0.6100457017578579e-23, 0.1535488071926541e-09, 0.5733950833190260e-09, 0.2185116940651705e+00, 0.6436934318898579e-12, 0.5207333190112868e-06, 0.4302907440331721e-06, 0.2008337976176149e+00, 0.1972152263052530e-30, 0.2421587237749379e-06, 0.2920528765870267e-06, 0.1757551361146239e+00, 0.8101601496619791e-27, 0.1434500575863456e-11, 0.9045851929625007e-13, 0.1271484717317025e+00, 0.1560107593833772e-17, 0.1620731838764033e-06, 0.1206839073349329e-06, 0.5049064582277562e-01, 0.3155443620884047e-29, 0.7221494757183839e-05, 0.4279630756550309e-05, 0.3515271339576785e-01, 0.0000000000000000e+00, 0.2092539651031481e-04, 0.3641666945464839e-04, 0.7966224551514293e-02, 0.1018583906139065e-23, 0.4276808715243643e-05, 0.6380573645111411e-06, 0.1685389559757755e-03, 0.0000000000000000e+00, 0.2828744147621957e-02, 0.2649548441232189e-02},
			info: 0,
		},
		{
			z:    []float64{0.7647749557731113e+00, 0.1439440544256733e+00, 0.7500486933916728e+00, 0.6414832886183546e+00, 0.8122572937542115e+00, 0.9529234674941257e+00, 0.3352470221802866e+00, 0.6936226545338587e+00, 0.9931729574752579e-01, 0.3187856199297912e-01, 0.1671537284832858e+00, 0.6799274061730229e+00, 0.8177446911383146e+00, 0.5321042755342652e+00, 0.1611635863708949e+00, 0.1556305029012977e+00, 0.8064987994430675e+00, 0.6086425937675999e-01, 0.9539387771768112e+00, 0.5483826791143146e+00, 0.1624656726546599e+00, 0.8604262189788422e+00, 0.3348601405085260e+00, 0.6512790728986305e+00, 0.3357978279833680e+00, 0.7104789129438286e+00, 0.4161092786312924e+00, 0.8790797243791704e+00, 0.1469866181354552e+00, 0.7277776395098579e-01, 0.4317417806367096e-01, 0.8731968970940820e+00, 0.4856440529809196e+00, 0.2315660100703048e+00, 0.2955911335168215e+00, 0.4119299294790694e+00, 0.4565975215473586e+00, 0.8915397570999283e+00, 0.1639735129501203e+00, 0.5055946074157277e+00, 0.9530743160791620e+00, 0.9836351183281582e+00, 0.2913429799811792e+00, 0.9289008959282562e+00, 0.5996674664338365e+00, 0.6609996028219567e+00, 0.7317528474660470e+00, 0.9903648665746729e+00, 0.3982305249886617e+00, 0.3563571411857538e+00, 0.8851540908639802e+00, 0.8253410045707608e+00, 0.8085297859672740e+00, 0.3918185712378985e+00, 0.7922842941861202e+00, 0.7487878715845681e+00, 0.3117744132747395e+00, 0.7135433852632408e+00, 0.8969220948763633e+00, 0.5840741001028926e+00, 0.3261515465187765e+00, 0.5368137772779212e+00, 0.6371156779602756e+00, 0.5575308178040213e+00, 0.1319054467962255e+00, 0.6274305773732142e+00, 0.9230553933157697e+00, 0.3589158151867838e+00, 0.1429059780004028e+00, 0.6433214640390554e+00, 0.9159704795536289e+00, 0.8998439182451070e+00, 0.5665265465351250e+00, 0.4290736239553739e+00, 0.4923118412965143e+00, 0.3733795042755129e+00, 0.4008465132226234e+00, 0.8630963810875470e+00, 0.2835407905147469e+00, 0.8431337072866282e+00, 0.7450545748778526e+00, 0.2948369760127234e+00, 0.6896108484670013e+00, 0.1073791710993542e+00},
			n:    21,
			zOut: []float64{0.2492440020550355e+01, 0.1951232214041701e+01, 0.1924880150346451e+01, 0.1800337146913507e+01, 0.1731259644349016e+01, 0.1515025732591961e+01, 0.1392131403063820e+01, 0.1337121843604905e+01, 0.1335944341051471e+01, 0.9695979508906304e+00, 0.8602855628657217e+00, 0.7598736716193354e+00, 0.5663438341779535e+00, 0.4991982558529897e+00, 0.3900147250500268e+00, 0.2612685334229408e+00, 0.1040703325016068e+00, 0.5095987426708586e-01, 0.4270469018444285e-01, 0.6313950453962554e-02, 0.4526318588114864e-04, 0.1623810366421630e-16, 0.5786287269352850e-19, 0.2440824886543042e-19, 0.1392131403063820e+01, 0.1972152263052530e-30, 0.1305662432930495e-17, 0.2687054746502436e-17, 0.1337121843604905e+01, 0.6807276388656606e-23, 0.2125450465832215e-16, 0.4166517172824519e-16, 0.1335944341051471e+01, 0.3971441341244662e-25, 0.3131313418539388e-15, 0.1598422579069803e-15, 0.9695979508906304e+00, 0.2612707318091991e-24, 0.3703169268599905e-13, 0.5897085302084673e-13, 0.8602855628657217e+00, 0.3944304526105059e-30, 0.1999104914098575e+02, 0.1999104914098576e+02, 0.1050000000000000e+03, 0.3185941043083900e+01, 0.4761904761904762e+01, 0.3815530651290367e-11, 0.5663438341779535e+00, 0.4442864618204738e-26, 0.2314135864505331e-10, 0.3280099800133426e-10, 0.4991982558529897e+00, 0.2524354896707238e-28, 0.8792792965154248e-10, 0.6284153656438893e-10, 0.3900147250500268e+00, 0.2716911819042188e-17, 0.6371083763388994e-09, 0.9063994743331422e-09, 0.2612685334229408e+00, 0.7682295356186976e-14, 0.2615630848198228e-07, 0.4934681155229334e-07, 0.1040703325016068e+00, 0.3134368286722418e-21, 0.1338644268367565e-05, 0.2571041054206291e-05, 0.5095987426708586e-01, 0.6829203137199389e-20, 0.1734538060272007e-04, 0.9160403953625247e-05, 0.4270469018444285e-01, 0.5620633949699709e-29, 0.1970936154220316e-07, 0.7875060113736862e-07, 0.6313950453962554e-02, 0.1074365444038600e-22, 0.1950342748319920e-02, 0.1960570817697857e-02, 0.4526318588114864e-04, 0.5045801736573469e-22, 0.2719687288170513e-02, 0.7229547441545878e-02},
			info: 0,
		},
		{
			z:    []float64{0.6830594937182136e+00, 0.2806903427821971e+00, 0.3842580501524426e+00, 0.7113293946876786e+00, 0.6120481717427926e+00, 0.4087522624125913e+00, 0.2683959794755927e+00, 0.1982680719912362e+00, 0.5503761648124630e+00, 0.9947639519912582e+00, 0.4817851020535202e-01, 0.1841410650783141e+00, 0.7364485595361944e+00, 0.2204876163244174e+00, 0.8785531172976315e+00, 0.3900984044089164e+00, 0.9898399332392419e+00, 0.4652324815840650e+00, 0.6574114801218780e+00, 0.6641201888165797e+00, 0.5371222655177923e+00, 0.2241732155112269e+00, 0.6726821610374456e-01, 0.2361953456198683e+00, 0.5602287124209401e+00, 0.7836687903368798e+00, 0.2633616918428828e+00, 0.9669553142730151e+00, 0.3692579170869618e+00, 0.5395054150347700e-01, 0.4776047537674022e-01, 0.5691410116455676e+00, 0.1304181940038973e+00, 0.2107738642513712e+00, 0.5434523182333528e-01, 0.4019375962886824e+00, 0.7125153526157032e+00, 0.5305712841430671e+00, 0.1164846974244694e+00, 0.2037577925839799e+00, 0.2045617511271008e+00, 0.6122054957321945e+00, 0.5114531146106966e+00, 0.2882478115017332e+00, 0.5304873168821023e+00, 0.1796380649103803e+00, 0.9949092378946756e-01, 0.5448372048900864e+00, 0.1971761401002603e+00, 0.9487011110186927e+00, 0.8526405089060669e+00, 0.4533764477818548e+00, 0.8410819408489681e+00, 0.7592016862970913e+00, 0.8131142938762341e+00, 0.1445496351282255e+00, 0.6614974180377753e+00, 0.3394841721225367e+00, 0.4878107636737276e+00, 0.5499609605641745e+00, 0.1789798083030991e+00, 0.2008397619575961e+00, 0.5247567247775218e+00, 0.7394478154071386e+00, 0.6713478520024037e+00, 0.3894295843862983e+00, 0.9099978858741096e+00, 0.9872931573704680e-01, 0.5370559804237263e+00, 0.8492599321050560e+00, 0.4671993983916084e+00, 0.6397070306147701e+00, 0.2884512208824136e+00, 0.5002881141083532e+00, 0.3356213102526899e+00, 0.4102730364689955e+00, 0.8172446941984062e+00, 0.1463391840674726e+00, 0.6374760821459636e+00, 0.2986995049911381e+00, 0.6932950085590186e+00, 0.8475246239619540e+00, 0.3550038162879430e+00, 0.9747731646460122e+00},
			n:    21,
			zOut: []float64{0.2302037253547019e+01, 0.1904432769194928e+01, 0.1838629010091233e+01, 0.1679631021950300e+01, 0.1638588328999231e+01, 0.1556939208705248e+01, 0.1154775776884602e+01, 0.1123025367728587e+01, 0.1005963210310813e+01, 0.7892040895461047e+00, 0.6171474041407791e+00, 0.5107385873854940e+00, 0.4745466376069911e+00, 0.2987072329916590e+00, 0.2518976395749056e+00, 0.1562926917018191e+00, 0.1208002043434920e+00, 0.1117929648180551e+00, 0.3025264937486962e-01, 0.5461437067005637e-02, 0.3901591662109151e-04, 0.2028008633963607e-17, 0.1197298285059118e-05, 0.1163352934520667e-05, 0.1154775776884602e+01, 0.2502221352972909e-22, 0.7438201518083012e-06, 0.8784873727121712e-06, 0.1123025367728587e+01, 0.2368236188315088e-19, 0.3676741199329849e-11, 0.3039307086935086e-11, 0.1005963210310813e+01, 0.2065553394230697e-24, 0.6686270095375819e-11, 0.8550052948388210e-11, 0.7892040895461047e+00, 0.1239058396467615e-20, 0.2067843352216928e-06, 0.6687212675645054e-07, 0.6171474041407791e+00, 0.2620094590259913e-21, 0.1757090250187976e+02, 0.1757090250187975e+02, 0.9500000000000000e+02, 0.2798185941043084e+01, 0.1052631578947368e+01, 0.6997553999213272e-08, 0.4745466376069911e+00, 0.5994838008700348e-24, 0.1052996378646298e-04, 0.6312169743031761e-04, 0.2987072329916590e+00, 0.9860761315262648e-30, 0.1466875513591431e-03, 0.1024839055944592e-03, 0.2518976395749056e+00, 0.7516632826070909e-17, 0.5524256710713082e-08, 0.1151290676113078e-07, 0.1562926917018191e+00, 0.1348952147927930e-27, 0.1222984298841777e-06, 0.3042067118561864e-06, 0.1208002043434920e+00, 0.4760428464210509e-23, 0.9492582004636672e-05, 0.3402727365723720e-04, 0.1117929648180551e+00, 0.7018716354804804e-22, 0.2431231599464108e-03, 0.1315672135071063e-03, 0.3025264937486962e-01, 0.4377821080194221e-21, 0.1333575496853992e-02, 0.6988231628130009e-03, 0.5461437067005637e-02, 0.2193916840728260e-24, 0.1845211740639344e-02, 0.8881725247692956e-03, 0.3901591662109151e-04, 0.2067951531382569e-23, 0.6469870796864946e-02, 0.1027022788337614e-01},
			info: 0,
		},
		{
			z:    []float64{0.3270384527330652e+00, 0.4068565573647237e+00, 0.5445258191923925e+00, 0.9571802305689175e+00, 0.1840747434207346e-01, 0.8759261930487382e+00, 0.3018283835341639e+00, 0.4754674889513868e+00, 0.9620709342523251e+00, 0.3596342978433105e+00, 0.1820397917245298e+00, 0.9562190617892271e+00, 0.8004073710305811e+00, 0.9503584926946729e+00, 0.1634422399578277e+00, 0.3053134184266467e+00, 0.3871438286377282e+00, 0.1225252880124038e+00, 0.5955538022158587e+00, 0.9831372787057571e+00, 0.2130378801829313e+00, 0.1502296787149880e+00, 0.5120891507658710e+00, 0.2769158347903287e+00, 0.6740978559159013e+00, 0.6671510755324899e+00, 0.5252429573239310e+00, 0.7312875683466940e+00, 0.6359518339326128e+00, 0.3635213813950466e-01, 0.7509704695539338e+00, 0.2796455326190517e+00, 0.3929933998036216e+00, 0.2441550702438192e+00, 0.5181887041338979e+00, 0.2149631476903890e-01, 0.1922157358558590e+00, 0.8352708839021310e+00, 0.2143118694803675e+00, 0.8099207585987700e+00, 0.4013415573783083e+00, 0.8990854368069680e+00, 0.2274365144039603e-01, 0.1574221848735892e+00, 0.5021896402707141e+00, 0.6811320055313477e+00, 0.4008235511220088e+00, 0.2607115591315068e+00, 0.8396807341169598e+00, 0.3393611517447541e+00, 0.3095200315094494e+00, 0.8849677836408325e+00, 0.3309280721041051e+00, 0.2177939175852258e+00, 0.7396054658394657e-02, 0.5387313739545944e+00, 0.5592732178453728e+00, 0.4422242955757759e+00, 0.6010599951901635e+00, 0.2729214395073326e+00, 0.8861412244709392e+00, 0.3303460134234409e+00, 0.3230906405176623e+00, 0.5979281304041633e+00, 0.1747480524852378e+00, 0.5019201598522602e+00, 0.3041629115671023e+00, 0.7344252813174572e+00, 0.5145018596354390e+00, 0.7032668540687012e+00, 0.4542245378490408e+00, 0.6883469599293311e+00, 0.7365046871633816e+00, 0.6892200550994384e+00, 0.9612097768268251e+00, 0.3687306903044788e+00, 0.4902139977664450e+00, 0.4004732628576805e+00, 0.2121831569752699e+00, 0.3483027732062219e+00, 0.4140454821962284e+00, 0.5590476647584736e+00, 0.1191647727722587e+00, 0.7126611758207085e+00},
			n:    21,
			zOut: []float64{0.2451808019367049e+01, 0.2128252216129400e+01, 0.1884015768998821e+01, 0.1793403527862625e+01, 0.1731698897481475e+01, 0.1605979322764491e+01, 0.1346135733575790e+01, 0.1222183737737495e+01, 0.1054517637460965e+01, 0.8541696661721662e+00, 0.8335553929836017e+00, 0.7643627371640215e+00, 0.6493538020248019e+00, 0.5117674171078171e+00, 0.4801902398267136e+00, 0.1830481952424627e+00, 0.1336751260833609e+00, 0.7548204386515513e-01, 0.4276641694583261e-01, 0.1152852330305948e-01, 0.4825291327526518e-04, 0.1046558665109411e-21, 0.5148060409033168e-10, 0.1742874021361670e-10, 0.1346135733575790e+01, 0.3944304526105059e-30, 0.1177385120842054e-08, 0.5739784145251797e-09, 0.1222183737737495e+01, 0.5228352308675503e-16, 0.2817945521555932e-11, 0.1891433639936970e-11, 0.1054517637460965e+01, 0.1775328283321620e-16, 0.9506244555381454e-11, 0.6780149876052939e-11, 0.8541696661721662e+00, 0.4289572169691169e-11, 0.3560204585171959e-10, 0.5552509181949502e-10, 0.8335553929836017e+00, 0.8925278809447169e-21, 0.1975794267501038e+02, 0.1975794267501038e+02, 0.9900000000000000e+02, 0.3029478458049887e+01, 0.2020202020202020e+01, 0.4057773458399609e-09, 0.6493538020248019e+00, 0.3196600703119960e-17, 0.4936238682205707e-06, 0.2172173957444601e-06, 0.5117674171078171e+00, 0.2475160268480207e-20, 0.1127880382626992e-04, 0.1031817493791828e-04, 0.4801902398267136e+00, 0.2327139670401985e-28, 0.8359957951383868e-05, 0.8045173068477057e-05, 0.1830481952424627e+00, 0.2763487036072314e-23, 0.1049648761427048e-04, 0.8607871065540926e-05, 0.1336751260833609e+00, 0.6392286336815055e-18, 0.5478798785309189e-04, 0.3178071334079827e-04, 0.7548204386515513e-01, 0.1886324196564483e-25, 0.4604839715747456e-05, 0.3206078546562559e-04, 0.4276641694583261e-01, 0.2130807968310579e-24, 0.2160396768988393e-02, 0.2401524776593251e-02, 0.1152852330305948e-01, 0.4597153189491485e-23, 0.1577121702032026e-02, 0.1548884890976950e-02, 0.4825291327526518e-04, 0.0000000000000000e+00, 0.1144436370534878e-01, 0.4805213857273134e-02},
			info: 0,
		},
		{
			z:    []float64{0.1204296195656108e+00, 0.2644532009534747e+00, 0.2981945708465356e+00, 0.3379851046426219e-01, 0.5872157683391875e+00, 0.3661495849020495e+00, 0.2783310889774259e+00, 0.8471009797765506e+00, 0.6087926721027869e+00, 0.7227374835926721e+00, 0.1157491272838733e+00, 0.5565144099551486e+00, 0.8949691253636921e+00, 0.4271566773654165e+00, 0.6387841251831913e+00, 0.3125047228733280e-01, 0.2092736013478247e-01, 0.9188324418961444e+00, 0.2275180796997284e+00, 0.4519662960462961e+00, 0.1155877493866434e+00, 0.3574940735178409e+00, 0.1489772525039219e+00, 0.3847908470095970e+00, 0.4780110568180811e+00, 0.3489586723075616e+00, 0.2559758537216194e+00, 0.7015930720521131e+00, 0.6512181083044030e+00, 0.1619173394771481e+00, 0.5094142654629881e+00, 0.9037662631759247e+00, 0.8829671919577533e+00, 0.5739200804260277e+00, 0.6584413366512047e+00, 0.7174965427568090e+00, 0.2705911606916550e+00, 0.6519148889036277e+00, 0.8284421625450799e+00, 0.1417109678591228e+00, 0.8472265674806589e-01, 0.3638971914001953e+00, 0.8802661885427909e-01, 0.5730596475071164e+00, 0.6156880168413905e+00, 0.3769532609458294e+00, 0.6302840487845919e+00, 0.8652028697564479e+00, 0.5886269364510055e-01, 0.9901772381773877e-01, 0.9679961670969370e+00, 0.9025028618130532e+00, 0.2858592550914936e+00, 0.4523960301168739e-01, 0.9961073801516410e+00, 0.5404176614913075e+00, 0.2161216977420954e+00, 0.6517643909270738e+00, 0.1408193879882935e-01, 0.2323315528058443e+00, 0.4040647670376405e+00, 0.5063393857452272e+00, 0.2149141182072994e+00, 0.8405813011731451e+00, 0.8044826512704509e+00, 0.5364711289689184e+00, 0.2922476360373625e+00, 0.9518940562302493e+00, 0.1980063292193738e+00, 0.6243757640305569e-01, 0.3641799792979717e+00, 0.4781083740929701e+00, 0.3488268687432241e+00, 0.8284107864073382e+00, 0.5855039308834178e+00, 0.2593007284605915e+00, 0.9998845891621441e+00, 0.5048349794999019e+00, 0.5828201446782487e+00, 0.8921301000231074e+00, 0.2672402237051515e+00, 0.3352853759971494e+00, 0.5977528180071631e+00, 0.1151194604047919e+00},
			n:    21,
			zOut: []float64{0.2557315455353142e+01, 0.2083235685800309e+01, 0.1875944452104717e+01, 0.1852714062580819e+01, 0.1643995946812320e+01, 0.1314218355716144e+01, 0.1293789198407225e+01, 0.1108624029644386e+01, 0.1060866236520538e+01, 0.6893140608436983e+00, 0.6265017468006577e+00, 0.6169803536834606e+00, 0.4472638275156540e+00, 0.3909824987049766e+00, 0.2990543016857948e+00, 0.1644755201813542e+00, 0.1145158516686298e+00, 0.6240413476421734e-01, 0.2540980528285533e-01, 0.1114684260186565e-01, 0.3077034058364147e-04, 0.1514612938024343e-27, 0.1068838715614041e-10, 0.1816953714834036e-10, 0.1293789198407225e+01, 0.1634858280249726e-16, 0.1546152805249061e-09, 0.7139985980503436e-10, 0.1108624029644386e+01, 0.1852295205932338e-17, 0.2931760870340471e-08, 0.5717158524063146e-08, 0.1060866236520538e+01, 0.1656986554198631e-24, 0.3897819821179759e-07, 0.2050786216968739e-07, 0.6893140608436983e+00, 0.1054613376713825e-21, 0.1114796130440952e-06, 0.1754727955273104e-06, 0.6265017468006577e+00, 0.0000000000000000e+00, 0.1823878313701335e+02, 0.1823878313701335e+02, 0.1020000000000000e+03, 0.3172335600907029e+01, 0.3921568627450980e+01, 0.4944285541926277e-08, 0.4472638275156540e+00, 0.7188113412291574e-19, 0.5360570179815202e-09, 0.5055207828345367e-09, 0.3909824987049766e+00, 0.7011182861561962e-17, 0.5424596001050196e-08, 0.1149191082508281e-07, 0.2990543016857948e+00, 0.1397964075969312e-24, 0.2148922978775398e-05, 0.1039924072201963e-05, 0.1644755201813542e+00, 0.1424715207546021e-19, 0.1286356681922344e-03, 0.6709371685087021e-04, 0.1145158516686298e+00, 0.1923558431290915e-24, 0.1040681698574874e-03, 0.9690504707177204e-04, 0.6240413476421734e-01, 0.3021337266996475e-27, 0.3740811812377314e-06, 0.8725750691496404e-06, 0.2540980528285533e-01, 0.6622271595176623e-28, 0.2937391640354928e-03, 0.1148070460566611e-03, 0.1114684260186565e-01, 0.3845696912952433e-29, 0.9582588059437623e-03, 0.1284334386264186e-02, 0.3077034058364147e-04, 0.1610185014413679e-23, 0.1713355521956925e-01, 0.1560043716021792e-01},
			info: 0,
		},
		{
			z:    []float64{0.6550531738065145e+00, 0.9006303757888976e+00, 0.9204609778380736e+00, 0.4658341923257867e+00, 0.5436644687290008e+00, 0.2908556755036122e+00, 0.1209811495723523e+00, 0.6227378829506942e+00, 0.3843019792679099e+00, 0.4431829610151485e+00, 0.4903210952175132e+00, 0.4210264133022774e+00, 0.3200750839632230e+00, 0.2252794777033682e+00, 0.5128630958928748e-01, 0.5136099492114639e+00, 0.4906286803922755e+00, 0.1747340137845675e+00, 0.6367238354860758e+00, 0.7515407582129341e+00, 0.4275500565044866e+00, 0.2014278558825494e+00, 0.7137603260056157e+00, 0.8686875192977085e+00, 0.7169591605793454e+00, 0.9050302202813415e+00, 0.9899533400874350e-01, 0.9969046692447475e+00, 0.6121260436585757e+00, 0.6518683331715626e+00, 0.5607404970528083e+00, 0.4000687152662108e-03, 0.1221862374388689e+00, 0.4484923714483016e+00, 0.4074629631039243e+00, 0.2322333097032281e-01, 0.4732551077558298e+00, 0.7342979327909982e+00, 0.3458749207719752e+00, 0.7729568887054280e+00, 0.2047858477824707e+00, 0.7991048365911296e+00, 0.5370237056253078e+00, 0.2233168849800682e-01, 0.5135288197983614e+00, 0.9129664417816657e+00, 0.9407166870649586e+00, 0.4321958081203685e+00, 0.5563774896630214e-01, 0.9494906037764664e+00, 0.9674063566885983e+00, 0.2302624676329315e+00, 0.1506811640032519e+00, 0.4445855507514112e+00, 0.6038782945102905e-01, 0.4499742519338505e+00, 0.1940991655604004e+00, 0.6493052482066142e-01, 0.2902929850901331e+00, 0.2876163569026815e+00, 0.9140475482126542e+00, 0.7006705577674063e+00, 0.2819386709216735e+00, 0.9125816873773608e+00, 0.9433170896498744e-01, 0.1632821600392161e+00, 0.3150556796654537e+00, 0.4051206017420083e+00, 0.2687493393481850e+00, 0.5768057516945001e+00, 0.4306858988251338e+00, 0.3827211690774009e+00, 0.3593520435650671e+00, 0.5610694397953092e+00, 0.7151178950427733e+00, 0.7138830385076256e-01, 0.5281951083990266e-01, 0.4880677722651299e+00, 0.9334247489037772e+00, 0.6764242544288053e+00, 0.8551178347453967e+00, 0.1876923010834376e+00, 0.7408118952029578e+00, 0.7967219280126981e+00},
			n:    21,
			zOut: []float64{0.2493394334366539e+01, 0.2460477172068809e+01, 0.2083159145455880e+01, 0.1779228959307519e+01, 0.1724095029781113e+01, 0.1560812668654360e+01, 0.1120482054552078e+01, 0.1088744749350826e+01, 0.9613598859526896e+00, 0.9456175481954929e+00, 0.9119552728538921e+00, 0.7767939725863090e+00, 0.6893167116879397e+00, 0.4408717103973492e+00, 0.2706330695081346e+00, 0.2343515836463984e+00, 0.6795401725828518e-01, 0.6600498464192311e-01, 0.2004873521291848e-01, 0.1421097732958122e-01, 0.4015560238058376e-03, 0.2226481018895784e-25, 0.2564499425740104e-12, 0.2658329977886514e-12, 0.1120482054552078e+01, 0.4315813836253463e-23, 0.1335501643346940e-12, 0.1368307862991113e-12, 0.1088744749350826e+01, 0.3926080427856238e-18, 0.1185565636052547e-12, 0.1214013431095007e-12, 0.9613598859526896e+00, 0.2719992401202049e-26, 0.9906513980413714e-13, 0.1012398200894782e-12, 0.9456175481954929e+00, 0.3937993638863291e-26, 0.9389079133840153e-13, 0.9189527811944332e-13, 0.9119552728538921e+00, 0.1580250204306566e-21, 0.1970991413883184e+02, 0.1970991413883184e+02, 0.1080000000000000e+03, 0.3401360544217687e+01, 0.8333333333333334e+01, 0.6496201881821226e-13, 0.6893167116879397e+00, 0.2040319821514109e-19, 0.6886568712781699e-13, 0.6342531805840532e-13, 0.4408717103973492e+00, 0.9116462109840799e-16, 0.6575726430673280e-12, 0.1146915438924438e-11, 0.2706330695081346e+00, 0.6100742213737356e-14, 0.5548784478521690e-10, 0.9411844035493366e-10, 0.2343515836463984e+00, 0.2464780121144947e-24, 0.1267470050081896e-08, 0.7549586177318002e-09, 0.6795401725828518e-01, 0.6058451752097371e-27, 0.2494439518946058e-07, 0.4030144639288306e-07, 0.6600498464192311e-01, 0.1931466728993349e-20, 0.1742718098912744e-06, 0.1063050845015484e-06, 0.2004873521291848e-01, 0.2078695185822955e-22, 0.1365398583912438e-05, 0.6056776980192580e-06, 0.1421097732958122e-01, 0.3944304526105059e-30, 0.8436693551699617e-06, 0.8772483974218289e-06, 0.4015560238058376e-03, 0.5424456586666688e-20, 0.1372662528173007e-02, 0.2323379424199439e-03},
			info: 0,
		},
		{
			z:    []float64{0.5955158028190186e+00, 0.9601282390728449e-01, 0.2944266054820921e+00, 0.3924250378682403e+00, 0.4462256781176310e-01, 0.5690757232665151e+00, 0.7855339329019884e-01, 0.8681545225840360e+00, 0.7288006955920397e+00, 0.2963229749240101e+00, 0.2728029241156386e+00, 0.4237536504971896e+00, 0.9003368942847588e+00, 0.7357957067011865e+00, 0.4901129309073038e-01, 0.4716171621744532e+00, 0.4890099406800087e+00, 0.3570165784460172e+00, 0.7160897857565474e-01, 0.6109824519907316e+00, 0.1221913259189585e+00, 0.1847302629402909e+00, 0.4009366884871202e-01, 0.9703472440233364e+00, 0.6369052870991366e+00, 0.5580400433416689e+00, 0.9024815855223057e+00, 0.1227045689636573e+00, 0.9668897094172768e+00, 0.5989409973498315e+00, 0.9709139844325040e+00, 0.9532564522700616e+00, 0.6239574483953332e+00, 0.7056703505253071e+00, 0.9506691238320553e-01, 0.7998586962365666e+00, 0.6556519136464994e+00, 0.4181856697120553e+00, 0.9478314170226607e+00, 0.3832663309972284e-01, 0.4257218688507075e+00, 0.2049443327129853e+00, 0.9168853967010917e+00, 0.5423988966788751e-02, 0.3002879152136316e+00, 0.7743151012217909e+00, 0.5154350917338609e+00, 0.9658560730800175e+00, 0.6314599250676610e+00, 0.5584937104443134e+00, 0.4741445335250092e+00, 0.3157508358915576e+00, 0.4349529093465575e+00, 0.7860003715032308e+00, 0.8695087804152180e-02, 0.3920445280700808e+00, 0.3765485221235618e+00, 0.4334604109656897e+00, 0.1636117026415890e+00, 0.3006093728066712e+00, 0.2129148932459900e+00, 0.3092975700919495e+00, 0.1986761620686717e-01, 0.9075541031649185e+00, 0.9781911433730479e+00, 0.6945955075731600e+00, 0.9959584477684137e+00, 0.7505944883792014e+00, 0.7627854018352902e+00, 0.1267035334952504e+00, 0.8056780234474171e+00, 0.4507060196586773e+00, 0.1799397964631048e+00, 0.7554691089798000e+00, 0.3012534688386570e+00, 0.9612172487804931e+00, 0.4056708256608614e+00, 0.6648221051068217e+00, 0.1966051207552482e+00, 0.6661008672469584e+00, 0.6875730168317549e+00, 0.9591133633640367e+00, 0.8995318056907078e+00, 0.1435403265941513e+00},
			n:    21,
			zOut: []float64{0.2826953331143077e+01, 0.2184946373215138e+01, 0.2011432985807930e+01, 0.1983939371799505e+01, 0.1783403572375380e+01, 0.1527291743334771e+01, 0.1218359458122959e+01, 0.1031625203087357e+01, 0.9141017580644314e+00, 0.8935891545682184e+00, 0.7995009441187808e+00, 0.6760292370761557e+00, 0.5375939998175604e+00, 0.5031272745805969e+00, 0.4136950654384539e+00, 0.4077555149266558e+00, 0.1977184484214556e+00, 0.1172157517521918e+00, 0.4197144528461013e-01, 0.1336078057720851e-01, 0.3545889291755145e-06, 0.2544082546855865e-16, 0.2583036406189355e-05, 0.3206950450307273e-05, 0.1218359458122959e+01, 0.6512835633504673e-25, 0.5241867641497365e-06, 0.5644106780290774e-06, 0.1031625203087357e+01, 0.0000000000000000e+00, 0.4305359229680111e-07, 0.8271940560154021e-07, 0.9141017580644314e+00, 0.0000000000000000e+00, 0.3911233321019150e-06, 0.2241550799736728e-06, 0.8935891545682184e+00, 0.2385737520619250e-22, 0.4206034452718400e-11, 0.9899461762377114e-11, 0.7995009441187808e+00, 0.9331359083488707e-16, 0.2008361176810137e+02, 0.2008361176810137e+02, 0.1130000000000000e+03, 0.3444444444444445e+01, 0.6194690265486726e+01, 0.1651034509856568e-04, 0.5375939998175604e+00, 0.1421716677825516e-23, 0.1528222338287967e-04, 0.2433829126418849e-04, 0.5031272745805969e+00, 0.6632003279205155e-19, 0.4757657931696731e-05, 0.2751276356124619e-05, 0.4136950654384539e+00, 0.1680371209901916e-16, 0.1318069984220972e-06, 0.1659409431372840e-06, 0.4077555149266558e+00, 0.2554525986432683e-22, 0.5944173486813264e-06, 0.1429245864393051e-05, 0.1977184484214556e+00, 0.1494891415393817e-27, 0.8465737022131437e-03, 0.4160790916597339e-03, 0.1172157517521918e+00, 0.6780166836549887e-20, 0.3655496023749206e-04, 0.5562211290196199e-04, 0.4197144528461013e-01, 0.1803076020780872e-22, 0.2337686132102848e-02, 0.5241185550833300e-02, 0.1336078057720851e-01, 0.3234329711406148e-27, 0.6369386968947829e-02, 0.2727172056719641e-02, 0.3545889291755145e-06, 0.0000000000000000e+00, 0.6135698199493001e-02, 0.4453215245051444e-02},
			info: 0,
		},
		{
			z:    []float64{0.5256994769563229e+00, 0.4751970954150097e+00, 0.5550798296722601e+00, 0.3565746805143110e+00, 0.3971523551134015e-01, 0.7088262788771817e+00, 0.6369448567868402e+00, 0.6372974860231623e+00, 0.5963599363854956e+00, 0.7996832763398349e+00, 0.7266271077857295e+00, 0.3813393981851099e+00, 0.8884393254050771e-01, 0.8685934834267716e+00, 0.3797066359545181e+00, 0.4541910736411658e+00, 0.6285354031372432e+00, 0.5295256159852654e+00, 0.8908170817260027e+00, 0.5793264969983637e+00, 0.6056871126785802e+00, 0.6045507124525907e+00, 0.9734230310286895e-01, 0.9482358811543057e+00, 0.8804820874856442e+00, 0.4472263119108183e+00, 0.5843860139759072e+00, 0.5372922689923049e+00, 0.5799351791336661e+00, 0.5116182356749631e+00, 0.3816355069915517e+00, 0.6475808331559241e+00, 0.8746013736579017e-02, 0.8411650861050215e+00, 0.2992737481736951e-01, 0.6246901010621124e+00, 0.3123078564554325e+00, 0.7411164234121632e+00, 0.5960432842954070e+00, 0.8102594121204127e+00, 0.3552957522458323e+00, 0.6235337320803771e+00, 0.5962195242077326e+00, 0.4845790335098474e+00, 0.4045953379382708e+00, 0.2452522545656534e+00, 0.2743318280596185e+00, 0.2023077390748810e+00, 0.4206973723636350e+00, 0.4039949452544738e+00, 0.9114680949622015e+00, 0.7779482986378454e+00, 0.6812819304160683e+00, 0.5580962104642140e-01, 0.4287711156774094e+00, 0.9817581515129085e+00, 0.4431516884275333e+00, 0.6237912221093689e-01, 0.8364254576520689e+00, 0.8759899245723605e+00, 0.8787473635901457e+00, 0.4702211826699049e+00, 0.8284521047414805e+00, 0.2041855966018515e+00, 0.8060130309143703e+00, 0.5501723689038956e+00, 0.2188842792675016e-01, 0.5672241818001204e+00, 0.6450459353754012e+00, 0.2431902854401001e+00, 0.5193698672717711e+00, 0.1962120228423043e+00, 0.6361488000943472e-01, 0.5342761659162559e+00, 0.1758994081846166e-01, 0.7459187699999678e+00, 0.5953394313659540e+00, 0.9379918174958790e+00, 0.4032831592210619e+00, 0.7704238877222783e+00, 0.9410981148731864e+00, 0.8355998775704846e+00, 0.7904637124553195e+00, 0.4324126135288506e+00},
			n:    21,
			zOut: []float64{0.2360605380677641e+01, 0.2349812365456326e+01, 0.2259072978559188e+01, 0.2074669549844899e+01, 0.1734529465065066e+01, 0.1635336277856887e+01, 0.1535245292061319e+01, 0.1453596007978833e+01, 0.1156130097482147e+01, 0.1084746900314168e+01, 0.1034520391904089e+01, 0.8201881896309742e+00, 0.6191782955028541e+00, 0.5267433243518804e+00, 0.4355086546946592e+00, 0.4155619327338856e+00, 0.3396295360604018e+00, 0.9317218785919168e-01, 0.5773665727788814e-01, 0.1842462732727454e-01, 0.2018232369069597e-07, 0.4517585523147273e-24, 0.2167663685989129e-10, 0.1523023581682742e-10, 0.1535245292061319e+01, 0.1340432450151543e-25, 0.8069500146896454e-10, 0.1279555645700667e-09, 0.1453596007978833e+01, 0.6673291668607916e-16, 0.1337219714947293e-07, 0.4537550993048733e-08, 0.1156130097482147e+01, 0.2388039732285047e-25, 0.6170125127971897e-05, 0.1353762614268721e-04, 0.1084746900314168e+01, 0.3862470231561244e-18, 0.2338329578753200e-04, 0.2318969913953679e-04, 0.1034520391904089e+01, 0.6409494854920721e-30, 0.2200440813282189e+02, 0.2200440813282190e+02, 0.9400000000000000e+02, 0.2857142857142857e+01, 0.0000000000000000e+00, 0.2156912239043876e-04, 0.6191782955028541e+00, 0.4311940021813490e-19, 0.2098085305628239e-04, 0.2047701462113856e-04, 0.5267433243518804e+00, 0.7292491488692259e-18, 0.2483658471590867e-04, 0.2278231239253353e-04, 0.4355086546946592e+00, 0.6335396825622830e-16, 0.5560492899787359e-04, 0.8068766398366361e-04, 0.4155619327338856e+00, 0.7336406418555410e-28, 0.3220618566867426e-03, 0.1956958209756436e-03, 0.3396295360604018e+00, 0.1297676189088564e-27, 0.4121897802138222e-02, 0.2587205720417875e-02, 0.9317218785919168e-01, 0.7651950780643815e-28, 0.2826512581544380e-02, 0.2814359488753958e-02, 0.5773665727788814e-01, 0.4923437167052441e-21, 0.1821079370679042e-04, 0.2969056479875054e-04, 0.1842462732727454e-01, 0.4842851680813876e-21, 0.4217326450522519e-02, 0.5738852897157041e-02, 0.2018232369069597e-07, 0.2018232369069597e-07, 0.3702854219118327e-02, 0.7091388009151286e-02},
			info: 0,
		},
		{
			z:    []float64{0.5196081465664333e+00, 0.1450873999446859e+00, 0.1902498073897446e+00, 0.9459083122514356e-01, 0.6971960507146802e-01, 0.2126447963850432e+00, 0.1693138701795316e-01, 0.6615449591751588e-01, 0.6671851537110856e+00, 0.7116922416084167e+00, 0.7735840319989629e+00, 0.8593705742984634e+00, 0.6060015557980403e+00, 0.1071620560007485e+00, 0.6056582690297291e+00, 0.2064801115284116e+00, 0.4992099275808294e+00, 0.9056413330105546e+00, 0.6051575714193578e+00, 0.1395071235858423e+00, 0.3386142237730057e+00, 0.2065895571122717e+00, 0.2774285740015214e-01, 0.4649920904615868e+00, 0.9062834189437683e+00, 0.3022801499192989e+00, 0.2125071125007829e+00, 0.2366624610094756e+00, 0.3232052419625674e+00, 0.5879745505340277e+00, 0.3212955913685801e+00, 0.9110441809347004e+00, 0.2144184579231917e+00, 0.5617161155671668e-01, 0.9725855527757206e+00, 0.9798782363582490e+00, 0.9666637418953663e+00, 0.1842420381101751e+00, 0.9810363127661145e+00, 0.4061071205983764e+00, 0.4929437214282740e+00, 0.3087398230344144e+00, 0.8692083335886002e+00, 0.7401065606674918e+00, 0.6829833274325647e+00, 0.9797795404622264e+00, 0.6101846761247042e+00, 0.7796537878703258e+00, 0.4568641605073986e-01, 0.1752946998854403e+00, 0.7558659638245062e+00, 0.4081915619381170e+00, 0.5344465439920063e+00, 0.3199118710728778e+00, 0.3613720388280138e+00, 0.1409062127217534e+00, 0.2811520826338662e+00, 0.9462348779627723e+00, 0.4911007415294493e+00, 0.2488061765237712e+00, 0.1661818317483100e+00, 0.1203910774987040e+00, 0.4544163143251944e+00, 0.8767242907488393e+00, 0.6688224142601292e+00, 0.2497550267789769e+00, 0.2658845545571695e+00, 0.4072601097670568e+00, 0.5517137496493807e+00, 0.4206663434956901e+00, 0.3655154771809294e+00, 0.6509199815744504e-01, 0.6480911975614350e+00, 0.1271267318339999e+00, 0.8874102966539428e+00, 0.9607257650026479e+00, 0.8902777005093224e+00, 0.7074702462790970e+00, 0.5045879991550398e+00, 0.3357826713243962e+00, 0.9303671751018463e+00, 0.3667268693834904e+00, 0.7709773173263057e+00, 0.1274123220614284e+00},
			n:    21,
			zOut: []float64{0.2669666350125788e+01, 0.2458073928301015e+01, 0.1960648799383531e+01, 0.1674511937828339e+01, 0.1629444569956276e+01, 0.1605168511500282e+01, 0.1169506153633164e+01, 0.8378899015114316e+00, 0.8214470290652885e+00, 0.7290022816300561e+00, 0.5956393798847858e+00, 0.5147057006185641e+00, 0.3415078453317431e+00, 0.3316800239182673e+00, 0.2245847640277715e+00, 0.1981086483360383e+00, 0.1729134064527326e+00, 0.1081292683666610e+00, 0.4856623040087921e-01, 0.2705373732739987e-02, 0.9745444154776346e-03, 0.7298616502634276e-19, 0.4601636530183765e-14, 0.3001385089664378e-13, 0.1169506153633164e+01, 0.5522026336547083e-28, 0.9439817212889541e-12, 0.1534086358433234e-11, 0.8378899015114316e+00, 0.7174347921301560e-11, 0.1095021852663145e-10, 0.1580074435702411e-10, 0.8214470290652885e+00, 0.3131705799125763e-20, 0.2247124942125492e-10, 0.3228253356903498e-10, 0.7290022816300561e+00, 0.6063934311659721e-18, 0.3794147471177530e-13, 0.3318524623173314e-12, 0.5956393798847858e+00, 0.3765687391680646e-13, 0.1809487464842083e+02, 0.1809487464842083e+02, 0.1050000000000000e+03, 0.3215419501133787e+01, 0.3809523809523809e+01, 0.2077325438490571e-07, 0.3415078453317431e+00, 0.1645262238367970e-20, 0.7912752069326401e-07, 0.6053132028546428e-07, 0.3316800239182673e+00, 0.4184118241292247e-26, 0.1343987419983520e-06, 0.1754397466154310e-06, 0.2245847640277715e+00, 0.3340387958036406e-22, 0.6150231447604006e-06, 0.4780001259056422e-06, 0.1981086483360383e+00, 0.1070733604161610e-20, 0.1003229473607176e-05, 0.1287291281183416e-05, 0.1729134064527326e+00, 0.1298955481663118e-20, 0.1611116216931278e-05, 0.3807035614938680e-06, 0.1081292683666610e+00, 0.4901279578837721e-23, 0.3565927169328702e-03, 0.2312261838015572e-03, 0.4856623040087921e-01, 0.3325573155747123e-13, 0.4389787505319806e-03, 0.9715248586459800e-03, 0.2705373732739987e-02, 0.1348432029845013e-20, 0.7371401467208961e-04, 0.2975668602269332e-04, 0.9745444154776346e-03, 0.4135903062570168e-24, 0.7273370201297700e-04, 0.3602244970122637e-04},
			info: 0,
		},
		{
			z:    []float64{0.3441168249350421e+00, 0.7064572711980356e+00, 0.6589781374655990e+00, 0.8687841598825752e+00, 0.7811747459515918e+00, 0.7470750719206745e+00, 0.7591454338662803e+00, 0.7956934079357347e-01, 0.3416983743372617e+00, 0.3365394610260509e+00, 0.8635365040683821e+00, 0.4793906833418223e+00, 0.9394584418725971e+00, 0.7354375226759881e+00, 0.3959912256877245e+00, 0.4088558388864650e+00, 0.6899389113237457e+00, 0.6329180251180861e+00, 0.6760812245614891e+00, 0.1743991444245150e+00, 0.3881927716152306e+00, 0.2103848624556167e+00, 0.5309784956583943e+00, 0.5758384660199964e+00, 0.6542468116269122e+00, 0.2141275453071043e+00, 0.4917208240158387e+00, 0.6996271491850585e+00, 0.7074081025976766e+00, 0.1896811377589238e+00, 0.4872676647104760e+00, 0.8987347035581787e+00, 0.1651654313203957e+00, 0.7969431328829826e+00, 0.8979672856778195e+00, 0.2621352236509209e+00, 0.2994430908669790e+00, 0.8412000921937168e+00, 0.6910228781616401e-01, 0.9849719192333963e+00, 0.2835195798153757e+00, 0.6126955834278749e+00, 0.9957514180764350e+00, 0.4133859474205875e+00, 0.2126790023013077e+00, 0.9230696247878700e-01, 0.5165813646587417e+00, 0.9232126059423650e+00, 0.6160486127374017e+00, 0.5529065454633691e+00, 0.6862175458708878e+00, 0.2677675454795836e-01, 0.6729513992144193e+00, 0.8612993184178528e+00, 0.5455358966165057e+00, 0.2518716982253303e+00, 0.5312849864208884e+00, 0.7551228210749875e+00, 0.1095520838657484e+00, 0.8767983608629261e+00, 0.9578119141004069e+00, 0.6878921114137557e+00, 0.2890574974795965e+00, 0.9851780344062913e+00, 0.9392886821673129e+00, 0.4125982690623264e+00, 0.6817912182549608e+00, 0.7805388095263401e+00, 0.1567802887930034e+00, 0.1901362770322003e+00, 0.5155717108920093e+00, 0.5470641908290981e+00, 0.9496522047623522e+00, 0.4367251554723609e+00, 0.7016249991347693e+00, 0.6652534390290816e+00, 0.3892590437090979e+00, 0.1464591367640408e+00, 0.9179795596761495e+00, 0.1527313083317114e+00, 0.8661173016992874e+00, 0.8044520995141484e+00, 0.4399581275677011e+00, 0.6590078920726725e+00},
			n:    21,
			zOut: []float64{0.2654671686308588e+01, 0.2409077903364995e+01, 0.2090082174851482e+01, 0.1940605709090538e+01, 0.1909776782011245e+01, 0.1629990303184568e+01, 0.1566043989638403e+01, 0.1473218419964192e+01, 0.1321200673977518e+01, 0.1275771383585936e+01, 0.9826159405158056e+00, 0.8184731809209335e+00, 0.6715230178331039e+00, 0.5669018352800101e+00, 0.3641671875118937e+00, 0.2699135892930649e+00, 0.1446269090001342e+00, 0.8937359624216266e-01, 0.5860931848549461e-01, 0.3094108728329079e-01, 0.6182329613034853e-03, 0.9233403587623171e-22, 0.2552413399194827e-12, 0.1309204970235786e-12, 0.1566043989638403e+01, 0.2003762315848454e-18, 0.9222125989396634e-12, 0.1831545934750265e-11, 0.1473218419964192e+01, 0.1899959664962892e-11, 0.5682525570474145e-10, 0.1113952503246639e-09, 0.1321200673977518e+01, 0.5311918312726658e-16, 0.2115322762070563e-08, 0.1208354514490733e-08, 0.1275771383585936e+01, 0.1424261829852281e-16, 0.6059411460508015e-08, 0.1032477558436010e-07, 0.9826159405158056e+00, 0.5397353496908506e-22, 0.2226820292130466e+02, 0.2226820292130466e+02, 0.9500000000000000e+02, 0.2927437641723356e+01, 0.1052631578947368e+01, 0.2555943178691847e-06, 0.6715230178331039e+00, 0.1218922493912602e-18, 0.4789189736721279e-06, 0.5466231772857101e-06, 0.5669018352800101e+00, 0.1893266172530428e-27, 0.9952011217509916e-06, 0.1128113648538554e-05, 0.3641671875118937e+00, 0.1993230626440035e-24, 0.2427606592544749e-05, 0.2156829827516489e-05, 0.2699135892930649e+00, 0.2761013168273541e-29, 0.3900010091178873e-05, 0.3393761026267541e-05, 0.1446269090001342e+00, 0.2150416673893778e-15, 0.8242388755752233e-05, 0.1105863902029246e-04, 0.8937359624216266e-01, 0.2934562567422164e-27, 0.3455780897358792e-04, 0.2250328346305790e-04, 0.5860931848549461e-01, 0.8147325136863849e-21, 0.2346512312055719e-04, 0.8962661529405173e-05, 0.3094108728329079e-01, 0.2460979885389816e-17, 0.5238375287742530e-03, 0.1080192509223090e-02, 0.6182329613034853e-03, 0.8383483586160605e-23, 0.7400552677463828e-02, 0.1581674273722876e-01},
			info: 0,
		},
		{
			z:    []float64{0.7940281584071446e+00, 0.8540600349699839e+00, 0.8158431165852809e-01, 0.5431841788581357e+00, 0.3696613346727944e+00, 0.2343742079469738e+00, 0.4891909888056500e-01, 0.6769876984160987e+00, 0.4777767465052760e+00, 0.1867381312399053e+00, 0.2018744873845245e+00, 0.5511201479607295e+00, 0.6938788283912793e+00, 0.8167542438070282e+00, 0.7904606414789531e+00, 0.9443564310071292e+00, 0.7287247677237652e-01, 0.8645122013586991e+00, 0.1884651475116826e+00, 0.3844755283611681e+00, 0.9959264361467982e+00, 0.6424370932833342e+00, 0.1972122925077952e+00, 0.2842024247377670e+00, 0.9819646913482807e+00, 0.9118347224008859e+00, 0.8184691845197246e+00, 0.7051587281589254e+00, 0.7604703230109544e+00, 0.6312964755149379e+00, 0.5240863862347888e+00, 0.3442050916384676e-01, 0.2415614308212055e+00, 0.2814868323669945e+00, 0.6529284673126197e+00, 0.3727305084153835e+00, 0.5033733868757848e+00, 0.2317122058804952e+00, 0.7555584130128312e+00, 0.5854566742645219e+00, 0.5481204696337160e+00, 0.8479425268049923e+00, 0.2310874615764000e+00, 0.1250993726775007e-01, 0.6243285982203539e-01, 0.8533587246073391e+00, 0.9203815588639257e+00, 0.9256849509751471e+00, 0.6691405057262187e+00, 0.8847091531299658e+00, 0.6783572983386376e+00, 0.4701257141291857e+00, 0.8976078424378102e+00, 0.8575018884445876e+00, 0.4119363561363949e+00, 0.2824477027676924e+00, 0.2787507690368071e+00, 0.7994878185780909e+00, 0.6141832897278305e+00, 0.6772728066124333e+00, 0.1568652581579784e+00, 0.8025492691231176e+00, 0.2609459151100056e+00, 0.4956700691019098e+00, 0.1008839464621498e+00, 0.6129709499983976e+00, 0.4551038858718992e-02, 0.8382785474023564e+00, 0.9327452694814308e+00, 0.9710431593941808e+00, 0.3785578217695214e+00, 0.9620839159000718e+00, 0.3183561960196257e-01, 0.9167635157854341e+00, 0.8989971039988554e+00, 0.2723769512210017e-01, 0.4176537489735596e+00, 0.9619881273217982e+00, 0.8761769579995293e+00, 0.6385245520487358e+00, 0.6821739872929905e+00, 0.3927943300877799e+00, 0.3299501391296433e-01, 0.6026481165267817e+00},
			n:    21,
			zOut: []float64{0.2841529467847260e+01, 0.2556740368064117e+01, 0.2029069210305357e+01, 0.1816492749229813e+01, 0.1807397947918166e+01, 0.1724050761249482e+01, 0.1394683696862690e+01, 0.1363282300837870e+01, 0.1235317529564993e+01, 0.1007821728138393e+01, 0.9728972767837659e+00, 0.8110838192609224e+00, 0.6190683239156912e+00, 0.3314348466978195e+00, 0.2969816495631171e+00, 0.2506421226611442e+00, 0.1551624233480766e+00, 0.1141049603637759e+00, 0.6806098279643324e-01, 0.4065442795375918e-01, 0.1509783792061944e-04, 0.2745446733802996e-19, 0.2334033424405603e-14, 0.7232047343832039e-15, 0.1394683696862690e+01, 0.7652022750203868e-17, 0.1502464319787860e-09, 0.5990324067027169e-10, 0.1363282300837870e+01, 0.5511008105786353e-19, 0.1215529748555577e-07, 0.3576186028975079e-08, 0.1235317529564993e+01, 0.6271335389136957e-17, 0.2589527811059762e-05, 0.3563684464760320e-05, 0.1007821728138393e+01, 0.6928731028484552e-15, 0.2286607739649456e-04, 0.2916328836696281e-04, 0.9728972767837659e+00, 0.4013724285764508e-25, 0.2143649169120057e+02, 0.2143649169120056e+02, 0.1080000000000000e+03, 0.3174603174603174e+01, 0.5555555555555555e+01, 0.5006788488237952e-04, 0.6190683239156912e+00, 0.1272779738919789e-24, 0.5392273695714327e-05, 0.6586751690782050e-05, 0.3314348466978195e+00, 0.9952016744778614e-24, 0.4588114650322476e-05, 0.4104030919118472e-05, 0.2969816495631171e+00, 0.1254277227268884e-20, 0.1579210272170037e-05, 0.2523508055026877e-05, 0.2506421226611442e+00, 0.1514835081255253e-22, 0.9117975197333799e-05, 0.1415311657319248e-04, 0.1551624233480766e+00, 0.3272195034856757e-26, 0.3128414626489953e-04, 0.4770966856082524e-04, 0.1141049603637759e+00, 0.7575261888623807e-20, 0.6277758538102519e-05, 0.4644422903178713e-05, 0.6806098279643324e-01, 0.3715389965619895e-21, 0.5956965112658162e-05, 0.4447744978930882e-05, 0.4065442795375918e-01, 0.1972152263052530e-30, 0.2965586553650948e-04, 0.1900611263569203e-04, 0.1509783792061944e-04, 0.1009741958682895e-26, 0.1608958133772104e-02, 0.9583670521235791e-03},
			info: 0,
		},
	} {
		z := make([]float64, len(test.z))
		copy(z, test.z)

		info := impl.Dlasq2(test.n, z)
		if !floats.EqualApprox(test.zOut, z, dTol) {
			diff := make([]float64, len(z))
			floats.SubTo(diff, z, test.zOut)
			for i := range diff {
				diff[i] = math.Abs(diff[i])
			}
			t.Errorf("Case %v, Z Mismatch", c)
		}
		if test.info != info {
			t.Errorf("Info mismatch. Want %v, got %v", test.info, info)
		}
	}

	rnd := rand.New(rand.NewSource(1))
	// Perform a bunch of random tests to check for access out of bounds or
	// infinite loops.
	// TODO(btracey): Implement direct tests.
	// bi := blas64.Implementation()
	for _, n := range []int{5, 8, 20, 25} {
		for k := 0; k < 10; k++ {
			z := make([]float64, 4*n)
			for i := range z {
				z[i] = rnd.Float64()
			}
			zCopy := make([]float64, len(z))
			copy(zCopy, z)

			// Compute the eigenvalues
			impl.Dlasq2(n, z)

			// Below is the code to test the eigenvalues. Eventually implement
			// real tests.
			// The code below is missing the transformation from L and U into
			// the symmetric tridiagonal matrix.
			// See discussion http://icl.cs.utk.edu/lapack-forum/viewtopic.php?f=5&t=4839
			// for format.

			/*
				ldl := n
				ldu := n
				u := make([]float64, n*n)
				for i := 0; i < n; i++ {
					u[i*ldu+i] = zCopy[2*i]
					if i != n-1 {
						u[i*ldu+i+1] = 1
					}
				}
				l := make([]float64, n*n)
				for i := 0; i < n; i++ {
					l[i*ldl+i] = 1
					if i != n-1 {
						l[(i+1)*ldl+i] = zCopy[2*i+1]
					}
				}

				ldTriDi := n
				triDi := make([]float64, n*n)
				bi.Dgemm(blas.NoTrans, blas.NoTrans, n, n, n, 1, l, ldl, u, ldu, 0, triDi, ldTriDi)

				tridi2 := make([]float64, n*n)
				bi.Dgemm(blas.Trans, blas.NoTrans, n, n, n, 1, triDi, n, triDi, n, 0, tridi2, n)

				// Eigenvalues have the property that det(A - lambda I ) = 0
				triDiCopy := make([]float64, len(triDi))
				copy(triDiCopy, triDi)
				for _, lambda := range z[1:n] {
					copy(triDi, triDiCopy)
					for i := 0; i < n; i++ {
						triDi[i*n+i] -= lambda
					}

					// Compute LU
					//ipiv := make([]int, n)
					//impl.Dgetrf(n, n, triDi, n, ipiv)
					ok := impl.Dpotrf(blas.Upper, n, triDi, n)
					fmt.Println(ok)

					var det float64
					for i := 0; i < n; i++ {
						det += math.Log(math.Abs(triDi[i*n+i]))
					}
					fmt.Println("det = ", math.Exp(det))
				}
			*/
		}
	}
}
