/*
Copyright 2021 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package e2e

import (
	"context"
	"fmt"
	"os"
	"strconv"
	"time"

	"github.com/onsi/ginkgo"
	"github.com/onsi/gomega"
	v1 "k8s.io/api/core/v1"
	storagev1 "k8s.io/api/storage/v1"
	"k8s.io/apimachinery/pkg/api/resource"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	clientset "k8s.io/client-go/kubernetes"
	"k8s.io/kubernetes/test/e2e/framework"
	fnodes "k8s.io/kubernetes/test/e2e/framework/node"
	fpod "k8s.io/kubernetes/test/e2e/framework/pod"
	fpv "k8s.io/kubernetes/test/e2e/framework/pv"
)

var _ = ginkgo.Describe("[csi-block-vanilla] [csi-file-vanilla] "+
	"[csi-guest] [csi-supervisor] Improved CSI Idempotency Tests", func() {
	f := framework.NewDefaultFramework("idempotency-csi")
	const defaultVolumeOpsScale = 30
	const defaultVolumeOpsScaleWCP = 29
	var (
		client            clientset.Interface
		fullSyncWaitTime  int
		namespace         string
		scParameters      map[string]string
		storagePolicyName string
		volumeOpsScale    int
		isServiceStopped  bool
		serviceName       string
	)

	ginkgo.BeforeEach(func() {
		bootstrap()
		client = f.ClientSet
		namespace = getNamespaceToRunTests(f)
		scParameters = make(map[string]string)
		isServiceStopped = false
		storagePolicyName = GetAndExpectStringEnvVar(envStoragePolicyNameForSharedDatastores)
		nodeList, err := fnodes.GetReadySchedulableNodes(f.ClientSet)

		framework.ExpectNoError(err, "Unable to find ready and schedulable Node")

		if !(len(nodeList.Items) > 0) {
			framework.Failf("Unable to find ready and schedulable Node")
		}

		if guestCluster {
			svcClient, svNamespace := getSvcClientAndNamespace()
			setResourceQuota(svcClient, svNamespace, rqLimit)
		}

		if os.Getenv("VOLUME_OPS_SCALE") != "" {
			volumeOpsScale, err = strconv.Atoi(os.Getenv(envVolumeOperationsScale))
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
		} else {
			if vanillaCluster {
				volumeOpsScale = defaultVolumeOpsScale
			} else {
				volumeOpsScale = defaultVolumeOpsScaleWCP
			}
		}
		framework.Logf("VOLUME_OPS_SCALE is set to %v", volumeOpsScale)

		if os.Getenv(envFullSyncWaitTime) != "" {
			fullSyncWaitTime, err = strconv.Atoi(os.Getenv(envFullSyncWaitTime))
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
			// Full sync interval can be 1 min at minimum so full sync wait time has to be more than 120s
			if fullSyncWaitTime < 120 || fullSyncWaitTime > defaultFullSyncWaitTime {
				framework.Failf("The FullSync Wait time %v is not set correctly", fullSyncWaitTime)
			}
		} else {
			fullSyncWaitTime = defaultFullSyncWaitTime
		}
	})

	ginkgo.AfterEach(func() {
		if supervisorCluster {
			deleteResourceQuota(client, namespace)
		}
		if guestCluster {
			svcClient, svNamespace := getSvcClientAndNamespace()
			setResourceQuota(svcClient, svNamespace, defaultrqLimit)
		}
		if isServiceStopped {
			if serviceName == "CSI" {
				framework.Logf("Starting CSI driver")
				ignoreLabels := make(map[string]string)
				err := updateDeploymentReplicawithWait(client, 1, vSphereCSIControllerPodNamePrefix,
					csiSystemNamespace)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())

				// Wait for the CSI Pods to be up and Running
				list_of_pods, err := fpod.GetPodsInNamespace(client, csiSystemNamespace, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				num_csi_pods := len(list_of_pods)
				err = fpod.WaitForPodsRunningReady(client, csiSystemNamespace, int32(num_csi_pods), 0,
					pollTimeout, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
			} else {
				vcAddress := e2eVSphere.Config.Global.VCenterHostname + ":" + sshdPort
				ginkgo.By(fmt.Sprintf("Starting %v on the vCenter host", serviceName))
				err := invokeVCenterServiceControl(startOperation, serviceName, vcAddress)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcRunningMessage)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
			}
		}
	})

	/*
		Create volume when CNS goes down
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC
		3. Bring down vsan-health service and wait for 5mins (default provisioner timeout)
		4. Bring up vsan-health
		5. Wait for pvcs to be bound
		6. Delete pvcs and SC
		7. Verify no orphan volumes are left
	*/
	ginkgo.It("create volume when CNS goes down - idempotency", func() {
		serviceName = vsanhealthServiceName
		createVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName,
			scParameters, volumeOpsScale, isServiceStopped)
	})

	/*
		Create volume when vpxd goes down
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC
		3. Bring down vpxd service and wait for 5mins (default provisioner timeout)
		4. Bring up vpxd
		5. Wait for pvcs to be bound
		6. Delete pvcs and SC
		7. Verify no orphan volumes are left
	*/
	ginkgo.It("create volume when VPXD goes down - idempotency", func() {
		serviceName = vpxdServiceName
		createVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName,
			scParameters, volumeOpsScale, isServiceStopped)
	})

	/*
		Create volume when SPS goes down
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC
		3. Bring down sps service and wait for 5mins (default provisioner timeout)
		4. Bring up sps
		5. Wait for pvcs to be bound
		6. Delete pvcs and SC
		7. Verify no orphan volumes are left
	*/
	ginkgo.It("create volume when SPS goes down - idempotency", func() {
		serviceName = spsServiceName
		createVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName,
			scParameters, volumeOpsScale, isServiceStopped)
	})

	/*
		Create volume when csi restarts
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC
		3. Restart CSI driver when VC task has started
		4. Wait for PVCs to be bound
		5. Delete PVCs and SC
		6. Verify no orphan volumes are left
	*/
	ginkgo.It("create volume when CSI restarts - idempotency", func() {
		serviceName = "CSI"
		createVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName,
			scParameters, volumeOpsScale, isServiceStopped)
	})

	/*
		Extend volume when csi restarts
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC and wait for PVCs to be bound
		3. Extend pvcs size
		4. Restart CSI when VC task is ongoing
		5. Verify resize was successful
		6. Delete pvcs and SC
		7. Verify no orphan volumes are left
	*/
	ginkgo.It("extend volume when csi restarts - idempotency", func() {
		serviceName = "CSI"
		extendVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName, scParameters,
			volumeOpsScale, true, isServiceStopped)
	})

	/*
		Extend volume when CNS goes down
		1. Create a SC using a thick provisioned policy
		2. Create a PVCs using SC and wait for PVC to be bound
		3. Extend pvcs and wait for vc task to start
		4. Bring down vsan-health service for 6mins
		5. Restart vsan-health service
		6. Verify resize was successful
		7. Delete pvcs and SC
		8. Verify no orphan volumes are left
	*/
	ginkgo.It("extend volume when CNS goes down - idempotency", func() {
		serviceName = vsanhealthServiceName
		extendVolumeWithServiceDown(serviceName, namespace, client, storagePolicyName, scParameters,
			volumeOpsScale, true, isServiceStopped)
	})
})

// createVolumeWithServiceDown creates the volumes and immediately stops the services and wait for
// the service to be up again and validates the volumes are bound
func createVolumeWithServiceDown(serviceName string, namespace string, client clientset.Interface,
	storagePolicyName string, scParameters map[string]string, volumeOpsScale int, isServiceStopped bool) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ginkgo.By(fmt.Sprintf("Invoking Test for create volume when %v goes down", serviceName))
	var storageclass *storagev1.StorageClass
	var persistentvolumes []*v1.PersistentVolume
	var pvclaims []*v1.PersistentVolumeClaim
	var err error
	var fullSyncWaitTime int
	ignoreLabels := make(map[string]string)
	pvclaims = make([]*v1.PersistentVolumeClaim, volumeOpsScale)

	// Decide which test setup is available to run
	if vanillaCluster {
		ginkgo.By("CNS_TEST: Running for vanilla k8s setup")
		// TODO: Create Thick Storage Policy from Pre-setup to support 6.7 Setups
		scParameters[scParamStoragePolicyName] = "Management Storage Policy - Regular"
		// Check if it is file volumes setups
		if rwxAccessMode {
			scParameters[scParamFsType] = nfs4FSType
		}
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", false, "idempotency")
	} else if supervisorCluster {
		ginkgo.By("CNS_TEST: Running for WCP setup")
		thickProvPolicy := os.Getenv(envStoragePolicyNameWithThickProvision)
		if thickProvPolicy == "" {
			ginkgo.Skip(envStoragePolicyNameWithThickProvision + " env variable not set")
		}
		profileID := e2eVSphere.GetSpbmPolicyID(thickProvPolicy)
		scParameters[scParamStoragePolicyID] = profileID
		// create resource quota
		createResourceQuota(client, namespace, rqLimit, thickProvPolicy)
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", false, thickProvPolicy)
	} else {
		ginkgo.By("CNS_TEST: Running for GC setup")
		thickProvPolicy := os.Getenv(envStoragePolicyNameWithThickProvision)
		if thickProvPolicy == "" {
			ginkgo.Skip(envStoragePolicyNameWithThickProvision + " env variable not set")
		}
		createResourceQuota(client, namespace, rqLimit, thickProvPolicy)
		scParameters[svStorageClassName] = thickProvPolicy
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", false, thickProvPolicy)
	}
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	defer func() {
		err := client.StorageV1().StorageClasses().Delete(ctx, storageclass.Name,
			*metav1.NewDeleteOptions(0))
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	}()

	ginkgo.By("Creating PVCs using the Storage Class")
	framework.Logf("VOLUME_OPS_SCALE is set to %v", volumeOpsScale)
	for i := 0; i < volumeOpsScale; i++ {
		fmt.Printf("Creating %v pvc ", i)
		accessMode := v1.ReadWriteOnce

		// Check if it is file volumes setups
		if rwxAccessMode {
			accessMode = v1.ReadWriteMany
		}
		pvclaims[i], err = createPVC(client, namespace, nil, "", storageclass, accessMode)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	}

	if serviceName == "CSI" {
		ginkgo.By("Stopping CSI driver")
		// TODO: Stop printing csi logs on the console
		collectPodLogs(ctx, client, csiSystemNamespace)
		err = updateDeploymentReplicawithWait(client, 0, vSphereCSIControllerPodNamePrefix,
			csiSystemNamespace)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = true

		defer func() {
			if isServiceStopped {
				framework.Logf("Starting CSI driver")
				err = updateDeploymentReplicawithWait(client, 1, vSphereCSIControllerPodNamePrefix,
					csiSystemNamespace)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				// Wait for the CSI Pods to be up and Running
				list_of_pods, err := fpod.GetPodsInNamespace(client, csiSystemNamespace, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				num_csi_pods := len(list_of_pods)
				err = fpod.WaitForPodsRunningReady(client, csiSystemNamespace, int32(num_csi_pods), 0,
					pollTimeout, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				isServiceStopped = false
			}
		}()
		framework.Logf("Starting CSI driver")
		err = updateDeploymentReplicawithWait(client, 1, vSphereCSIControllerPodNamePrefix,
			csiSystemNamespace)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		// Wait for the CSI Pods to be up and Running
		list_of_pods, err := fpod.GetPodsInNamespace(client, csiSystemNamespace, ignoreLabels)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		num_csi_pods := len(list_of_pods)
		err = fpod.WaitForPodsRunningReady(client, csiSystemNamespace, int32(num_csi_pods), 0,
			pollTimeout, ignoreLabels)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = false

		ginkgo.By(fmt.Sprintf("Sleeping for %v seconds to allow full sync finish", fullSyncWaitTime))
		time.Sleep(time.Duration(fullSyncWaitTime) * time.Second)
	} else {
		ginkgo.By(fmt.Sprintf("Stopping %v on the vCenter host", serviceName))
		vcAddress := e2eVSphere.Config.Global.VCenterHostname + ":" + sshdPort
		err = invokeVCenterServiceControl(stopOperation, serviceName, vcAddress)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = true
		err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcStoppedMessage)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())

		defer func() {
			if isServiceStopped {
				ginkgo.By(fmt.Sprintf("Starting %v on the vCenter host", serviceName))
				err = invokeVCenterServiceControl(startOperation, serviceName, vcAddress)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcRunningMessage)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				isServiceStopped = false
			}
		}()

		ginkgo.By("Sleeping for 5+1 min for default provisioner timeout")
		time.Sleep(pollTimeoutSixMin)

		ginkgo.By(fmt.Sprintf("Starting %v on the vCenter host", serviceName))
		err = invokeVCenterServiceControl(startOperation, serviceName, vcAddress)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = false
		err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcRunningMessage)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())

		ginkgo.By("Sleeping for full sync interval")
		time.Sleep(time.Duration(fullSyncWaitTime) * time.Second)
	}

	//After service restart
	bootstrap()

	ginkgo.By("Waiting for all claims to be in bound state")
	persistentvolumes, err = fpv.WaitForPVClaimBoundPhase(client, pvclaims,
		framework.ClaimProvisionTimeout)
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	defer func() {
		for _, claim := range pvclaims {
			err := fpv.DeletePersistentVolumeClaim(client, claim.Name, namespace)
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
		}
		ginkgo.By("Verify PVs, volumes are deleted from CNS")
		for _, pv := range persistentvolumes {
			err := fpv.WaitForPersistentVolumeDeleted(client, pv.Name, framework.Poll,
				framework.PodDeleteTimeout)
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
			volumeID := pv.Spec.CSI.VolumeHandle
			err = e2eVSphere.waitForCNSVolumeToBeDeleted(volumeID)
			gomega.Expect(err).NotTo(gomega.HaveOccurred(),
				fmt.Sprintf("Volume: %s should not be present in the CNS after it is deleted from "+
					"kubernetes", volumeID))
		}
	}()
}

// extendVolumeWithServiceDown extends the volumes and immediately stops the service and wait for
// the service to be up again and validates the volumes are bound
func extendVolumeWithServiceDown(serviceName string, namespace string, client clientset.Interface,
	storagePolicyName string, scParameters map[string]string, volumeOpsScale int, extendVolume bool,
	isServiceStopped bool) {
	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	ginkgo.By(fmt.Sprintf("Invoking Test for create volume when %v goes down", serviceName))
	var storageclass *storagev1.StorageClass
	var persistentvolumes []*v1.PersistentVolume
	var pvclaims []*v1.PersistentVolumeClaim
	var err error
	var fullSyncWaitTime int
	ignoreLabels := make(map[string]string)
	pvclaims = make([]*v1.PersistentVolumeClaim, volumeOpsScale)

	// Decide which test setup is available to run
	if vanillaCluster {
		ginkgo.By("CNS_TEST: Running for vanilla k8s setup")
		// TODO: Create Thick Storage Policy from Pre-setup to support 6.7 Setups
		scParameters[scParamStoragePolicyName] = "Management Storage Policy - Regular"
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", true, "idempotency")
		// Check nfs4FSType for file volumes
		if rwxAccessMode {
			ginkgo.Skip("File Volume extend is not supported, skipping the test")
		}
	} else if supervisorCluster {
		ginkgo.By("CNS_TEST: Running for WCP setup")
		thickProvPolicy := os.Getenv(envStoragePolicyNameWithThickProvision)
		if thickProvPolicy == "" {
			ginkgo.Skip(envStoragePolicyNameWithThickProvision + " env variable not set")
		}
		profileID := e2eVSphere.GetSpbmPolicyID(thickProvPolicy)
		scParameters[scParamStoragePolicyID] = profileID
		// create resource quota
		createResourceQuota(client, namespace, rqLimit, thickProvPolicy)
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", true, thickProvPolicy)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	} else {
		ginkgo.By("CNS_TEST: Running for GC setup")
		thickProvPolicy := os.Getenv(envStoragePolicyNameWithThickProvision)
		if thickProvPolicy == "" {
			ginkgo.Skip(envStoragePolicyNameWithThickProvision + " env variable not set")
		}
		createResourceQuota(client, namespace, rqLimit, thickProvPolicy)
		scParameters[svStorageClassName] = thickProvPolicy
		scParameters[scParamFsType] = ext4FSType
		storageclass, err = createStorageClass(client, scParameters, nil, "", "", true, thickProvPolicy)
	}
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	defer func() {
		err := client.StorageV1().StorageClasses().Delete(ctx, storageclass.Name,
			*metav1.NewDeleteOptions(0))
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	}()

	ginkgo.By("Creating PVCs using the Storage Class")
	framework.Logf("VOLUME_OPS_SCALE is set to %v", volumeOpsScale)
	for i := 0; i < volumeOpsScale; i++ {
		fmt.Printf("Creating %v pvc ", i)
		pvclaims[i], err = fpv.CreatePVC(client, namespace,
			getPersistentVolumeClaimSpecWithStorageClass(namespace, diskSize, storageclass, nil, ""))
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	}

	ginkgo.By("Waiting for all claims to be in bound state")
	persistentvolumes, err = fpv.WaitForPVClaimBoundPhase(client, pvclaims,
		framework.ClaimProvisionTimeout)
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	defer func() {
		for _, claim := range pvclaims {
			err := fpv.DeletePersistentVolumeClaim(client, claim.Name, namespace)
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
		}
		ginkgo.By("Verify PVs, volumes are deleted from CNS")
		for _, pv := range persistentvolumes {
			err := fpv.WaitForPersistentVolumeDeleted(client, pv.Name, framework.Poll,
				framework.PodDeleteTimeout)
			gomega.Expect(err).NotTo(gomega.HaveOccurred())
			volumeID := pv.Spec.CSI.VolumeHandle
			err = e2eVSphere.waitForCNSVolumeToBeDeleted(volumeID)
			gomega.Expect(err).NotTo(gomega.HaveOccurred(),
				fmt.Sprintf("Volume: %s should not be present in the "+
					"CNS after it is deleted from kubernetes", volumeID))
		}
	}()

	ginkgo.By("Create POD")
	pod, err := createPod(client, namespace, nil, pvclaims, false, "")
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	defer func() {
		ginkgo.By("Deleting the pod")
		err = fpod.DeletePodWithWait(client, pod)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
	}()

	err = fpod.WaitForPodNameRunningInNamespace(client, pod.Name, namespace)
	gomega.Expect(err).NotTo(gomega.HaveOccurred())

	// Modify PVC spec to trigger volume expansion
	// We expand the PVC while no pod is using it to ensure offline expansion
	ginkgo.By("Expanding current pvc")
	for _, claim := range pvclaims {
		currentPvcSize := claim.Spec.Resources.Requests[v1.ResourceStorage]
		newSize := currentPvcSize.DeepCopy()
		newSize.Add(resource.MustParse("4Gi"))
		framework.Logf("currentPvcSize %v, newSize %v", currentPvcSize, newSize)
		claims, err := expandPVCSize(claim, newSize, client)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		gomega.Expect(claims).NotTo(gomega.BeNil())
	}

	if serviceName == "CSI" {
		ginkgo.By("Stopping CSI driver")
		// TODO: Stop printing csi logs on the console
		collectPodLogs(ctx, client, csiSystemNamespace)
		err = updateDeploymentReplicawithWait(client, 0, vSphereCSIControllerPodNamePrefix,
			csiSystemNamespace)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = true

		defer func() {
			if isServiceStopped {
				framework.Logf("Starting CSI driver")
				err = updateDeploymentReplicawithWait(client, 1, vSphereCSIControllerPodNamePrefix,
					csiSystemNamespace)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				// Wait for the CSI Pods to be up and Running
				list_of_pods, err := fpod.GetPodsInNamespace(client, csiSystemNamespace, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				num_csi_pods := len(list_of_pods)
				err = fpod.WaitForPodsRunningReady(client, csiSystemNamespace, int32(num_csi_pods), 0,
					pollTimeout, ignoreLabels)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				isServiceStopped = false
			}
		}()

		framework.Logf("Starting CSI driver")
		err = updateDeploymentReplicawithWait(client, 1, vSphereCSIControllerPodNamePrefix,
			csiSystemNamespace)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())

		// Wait for the CSI Pods to be up and Running
		list_of_pods, err := fpod.GetPodsInNamespace(client, csiSystemNamespace, ignoreLabels)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		num_csi_pods := len(list_of_pods)
		err = fpod.WaitForPodsRunningReady(client, csiSystemNamespace, int32(num_csi_pods), 0,
			pollTimeout, ignoreLabels)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = false

		ginkgo.By(fmt.Sprintf("Sleeping for %v seconds to allow full sync finish", fullSyncWaitTime))
		time.Sleep(time.Duration(fullSyncWaitTime) * time.Second)
	} else {
		ginkgo.By(fmt.Sprintf("Stopping %v on the vCenter host", serviceName))
		vcAddress := e2eVSphere.Config.Global.VCenterHostname + ":" + sshdPort
		err = invokeVCenterServiceControl(stopOperation, serviceName, vcAddress)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = true
		err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcStoppedMessage)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		defer func() {
			if isServiceStopped {
				ginkgo.By(fmt.Sprintf("Starting %v on the vCenter host", serviceName))
				err = invokeVCenterServiceControl(startOperation, serviceName, vcAddress)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcRunningMessage)
				gomega.Expect(err).NotTo(gomega.HaveOccurred())
				isServiceStopped = false
			}
		}()

		ginkgo.By("Sleeping for 5+1 min for default provisioner timeout")
		time.Sleep(pollTimeoutSixMin)

		ginkgo.By(fmt.Sprintf("Starting %v on the vCenter host", serviceName))
		err = invokeVCenterServiceControl(startOperation, serviceName, vcAddress)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		isServiceStopped = false
		err = waitVCenterServiceToBeInState(serviceName, vcAddress, svcRunningMessage)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())
		time.Sleep(totalResizeWaitPeriod)
	}
	//After service restart
	bootstrap()

	ginkgo.By("Waiting for file system resize to finish")
	for _, claim := range pvclaims {
		claims, err := waitForFSResize(claim, client)
		gomega.Expect(err).NotTo(gomega.HaveOccurred())

		pvcConditions := claims.Status.Conditions
		expectEqual(len(pvcConditions), 0, "pvc should not have conditions")
	}
}
