/*
Copyright 2020 The Knative Authors

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1

import (
	"testing"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"knative.dev/pkg/apis"

	eventingduck "knative.dev/eventing/pkg/apis/duck/v1"
	"knative.dev/eventing/pkg/apis/eventing"
)

func TestInMemoryChannelValidation(t *testing.T) {
	tests := []CRDTest{{
		name: "empty",
		cr: &InMemoryChannel{
			Spec: InMemoryChannelSpec{},
		},
		want: nil,
	}, {
		name: "valid subscribers array",
		cr: &InMemoryChannel{
			Spec: InMemoryChannelSpec{
				ChannelableSpec: eventingduck.ChannelableSpec{
					SubscribableSpec: eventingduck.SubscribableSpec{
						Subscribers: []eventingduck.SubscriberSpec{{
							SubscriberURI: apis.HTTP("subscriberendpoint"),
							ReplyURI:      apis.HTTP("resultendpoint"),
						}},
					}},
			},
		},
		want: nil,
	}, {
		name: "empty subscriber at index 1",
		cr: &InMemoryChannel{
			Spec: InMemoryChannelSpec{
				ChannelableSpec: eventingduck.ChannelableSpec{
					SubscribableSpec: eventingduck.SubscribableSpec{
						Subscribers: []eventingduck.SubscriberSpec{{
							SubscriberURI: apis.HTTP("subscriberendpoint"),
							ReplyURI:      apis.HTTP("replyendpoint"),
						}, {}},
					}},
			},
		},
		want: func() *apis.FieldError {
			fe := apis.ErrMissingField("spec.subscribable.subscriber[1].replyURI", "spec.subscribable.subscriber[1].subscriberURI")
			fe.Details = "expected at least one of, got none"
			return fe
		}(),
	}, {
		name: "2 empty subscribers",
		cr: &InMemoryChannel{
			Spec: InMemoryChannelSpec{
				ChannelableSpec: eventingduck.ChannelableSpec{
					SubscribableSpec: eventingduck.SubscribableSpec{
						Subscribers: []eventingduck.SubscriberSpec{{}, {}},
					},
				},
			},
		},
		want: func() *apis.FieldError {
			var errs *apis.FieldError
			fe := apis.ErrMissingField("spec.subscribable.subscriber[0].replyURI", "spec.subscribable.subscriber[0].subscriberURI")
			fe.Details = "expected at least one of, got none"
			errs = errs.Also(fe)
			fe = apis.ErrMissingField("spec.subscribable.subscriber[1].replyURI", "spec.subscribable.subscriber[1].subscriberURI")
			fe.Details = "expected at least one of, got none"
			errs = errs.Also(fe)
			return errs
		}(),
	}, {
		name: "invalid scope annotation",
		cr: &InMemoryChannel{
			ObjectMeta: metav1.ObjectMeta{
				Annotations: map[string]string{
					eventing.ScopeAnnotationKey: "notvalid",
				},
			},
			Spec: InMemoryChannelSpec{},
		},
		want: func() *apis.FieldError {
			fe := apis.ErrInvalidValue("notvalid", "metadata.annotations.[eventing.knative.dev/scope]")
			fe.Details = "expected either 'cluster' or 'namespace'"
			return fe
		}(),
	}}

	doValidateTest(t, tests)
}
