/*
Copyright 2019 The Knative Authors
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package handler

import (
	"bytes"
	"context"
	"errors"
	"net/http"
	"net/http/httptest"
	"strconv"
	"testing"

	"k8s.io/apimachinery/pkg/types"
	"knative.dev/pkg/metrics/metricskey"
	"knative.dev/pkg/metrics/metricstest"
	"knative.dev/serving/pkg/activator"
	"knative.dev/serving/pkg/activator/util"
	"knative.dev/serving/pkg/apis/serving"
)

func TestRequestMetricHandler(t *testing.T) {
	testNamespace := "real-namespace"
	testRevName := "real-name"
	testPod := "testPod"

	tests := []struct {
		label       string
		baseHandler http.HandlerFunc
		newHeader   map[string]string
		wantCode    int
		wantPanic   bool
	}{
		{
			label: "normal response",
			baseHandler: http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusOK)
			}),
			wantCode: http.StatusOK,
		},
		{
			label: "panic response",
			baseHandler: http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusBadRequest)
				panic(errors.New("handler error"))
			}),
			wantCode:  http.StatusBadRequest,
			wantPanic: true,
		},
	}

	for _, test := range tests {
		t.Run(test.label, func(t *testing.T) {
			handler := NewMetricHandler(testPod, test.baseHandler)

			resp := httptest.NewRecorder()
			req := httptest.NewRequest(http.MethodPost, "http://example.com", bytes.NewBufferString(""))
			if test.newHeader != nil && len(test.newHeader) != 0 {
				for k, v := range test.newHeader {
					req.Header.Add(k, v)
				}
			}

			rev := revision(testNamespace, testRevName)

			defer reset()
			defer func() {
				err := recover()
				if test.wantPanic && err == nil {
					t.Error("Want ServeHTTP to panic, got nothing.")
				}

				if resp.Code != test.wantCode {
					t.Errorf("Response Status = %d,  want: %d", resp.Code, test.wantCode)
				}

				labelCode := test.wantCode
				if test.wantPanic {
					labelCode = http.StatusInternalServerError
				}

				wantTags := map[string]string{
					"pod_name":                        testPod,
					"container_name":                  activator.Name,
					metricskey.LabelNamespaceName:     rev.Namespace,
					metricskey.LabelServiceName:       rev.Labels[serving.ServiceLabelKey],
					metricskey.LabelConfigurationName: rev.Labels[serving.ConfigurationLabelKey],
					metricskey.LabelRevisionName:      rev.Name,
					metricskey.LabelResponseCode:      strconv.Itoa(labelCode),
					metricskey.LabelResponseCodeClass: strconv.Itoa(labelCode/100) + "xx",
				}
				metricstest.CheckCountData(t, requestCountM.Name(), wantTags, 1)
				metricstest.CheckStatsReported(t, responseTimeInMsecM.Name())
			}()

			reqCtx := util.WithRevision(context.Background(), rev)
			reqCtx = util.WithRevID(reqCtx, types.NamespacedName{Namespace: testNamespace, Name: testRevName})
			handler.ServeHTTP(resp, req.WithContext(reqCtx))
		})
	}
}

func reset() {
	metricstest.Unregister(requestConcurrencyM.Name(), requestCountM.Name(), responseTimeInMsecM.Name())
	register()
}

func BenchmarkMetricHandler(b *testing.B) {
	baseHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {})
	reqCtx := util.WithRevision(context.Background(), revision(testNamespace, testRevName))

	handler := NewMetricHandler("benchPod", baseHandler)

	resp := httptest.NewRecorder()
	b.Run("sequential", func(b *testing.B) {
		req := httptest.NewRequest(http.MethodGet, "http://example.com", nil).WithContext(reqCtx)
		for j := 0; j < b.N; j++ {
			handler.ServeHTTP(resp, req)
		}
	})

	b.Run("parallel", func(b *testing.B) {
		b.RunParallel(func(pb *testing.PB) {
			req := httptest.NewRequest(http.MethodGet, "http://example.com", nil).WithContext(reqCtx)
			for pb.Next() {
				handler.ServeHTTP(resp, req)
			}
		})
	})
}
