#!/bin/bash -e

KUBEVIRT_VERSION="2.0.0"
KUBEVIRT_CONVERSION_RELEASE="3"
KUBEVIRT_VMWARE_RELEASE="9"

QUAY_NS=quay.io/kubevirt

CONTAINER_MGMT=docker

if ! type "${CONTAINER_MGMT}" &>/dev/null; then
    if type "podman" &>/dev/null; then
        CONTAINER_MGMT=podman
    fi
fi

RPM_VERSION="$(
    git describe --always --tags --match "v[0-9]*" |
    sed -e '/^v[0-9.]\+\(-\|$\)/!{q1}; s/^v\([0-9.]\+\).*/\1/'
)"
if git describe --exact-match --tags --match "v[0-9]*" > /dev/null 2>&1 ; then
    RPM_RELEASE="1"
else
    GIT="$(
        git describe --always --tags --match "v[0-9]*" --dirty=.dr |
        sed -r 's/^/git/; s/^[^-]*-//; s/-g/.git/; s/-/_/g'
    )"
    RPM_RELEASE="0.$GIT.$(date -u +%Y%m%d%H%M%S)"
fi

ROLE_NAME="manageiq.v2v-conversion-host"
PACKAGE_NAME="v2v-conversion-host"
ROLE_RPM_NAME="${PACKAGE_NAME}-ansible"

PREFIX="${PREFIX:-/usr/local}"
DATA_DIR="${DATA_DIR:-${PREFIX}/share}"
BIN_DIR="${BIN_DIR:-${PREFIX}/bin}"

ROLES_DIR="$DATA_DIR/ansible/roles"
AUX_DATA_DIR="$DATA_DIR/$ROLE_RPM_NAME"

TARBALL="$PACKAGE_NAME-$RPM_VERSION.tar.gz"

_replace_vars() {
  sed \
   -e "s|@RPM_VERSION@|$RPM_VERSION|g" \
   -e "s|@RPM_RELEASE@|$RPM_RELEASE|g" \
   -e "s|@PACKAGE_NAME@|$PACKAGE_NAME|g" \
   < "$1" > "$2"
  echo "Wrote $2"
}


do_prep() {
  _replace_vars "$PACKAGE_NAME.spec.in" "$PACKAGE_NAME.spec"
  _replace_vars "wrapper/meta.py.in" "wrapper/meta.py"
}

do_dist() {
  do_prep
  echo "Creating spec file and tar archive '$TARBALL' ... "
  (
  git ls-files ; ls \
    wrapper/meta.py \
  ) | tar --files-from /proc/self/fd/0 -czf "$TARBALL" "$PACKAGE_NAME.spec"
  echo "tar archive '$TARBALL' created."
}

do_install() {
  echo "Installing data..."

  mkdir -p $ROLES_DIR
  cp -pR "ansible/$ROLE_NAME" "$ROLES_DIR"

  if [[ -z "${RPM_BUILD_ROOT}" ]]; then
      PYTHON_PREFIX="${PREFIX:-$(python-config --prefix)}"
      python setup.py install --prefix "${PYTHON_PREFIX}"
  fi

  mkdir -p "$AUX_DATA_DIR/playbooks"
  install -t "$AUX_DATA_DIR/playbooks" ansible/examples/*.yml

  echo "Installation done."
}

do_build_conversion() {
    TAG="v$KUBEVIRT_VERSION-$KUBEVIRT_CONVERSION_RELEASE"
    IMAGE="$QUAY_NS/kubevirt-v2v-conversion"

    # TODO: make sure the TAG is not used yet to avoid overwrite

    pushd kubevirt-conversion
    # TODO: use RPM with wrapper
    ${CONTAINER_MGMT} build -v "$(dirname $PWD)":/source -t "$IMAGE:$TAG" .
    popd

    # TODO: When to tag as 'latest'? Do it manualy for now.
    #${CONTAINER_MGMT} push quay.io/nyoxi/kubevirt-conversion:latest
}

do_build_vmware() {
    TAG="v$KUBEVIRT_VERSION-$KUBEVIRT_VMWARE_RELEASE"
    IMAGE="$QUAY_NS/kubevirt-vmware"

    # Prepare golang environment
    pushd kubevirt-vmware > /dev/null
    export GOPATH="$(pwd)/build/GOPATH"
    if [ -e "$GOPATH" ] ; then
        echo "GOPATH exists ($GOPATH)" >&2
        echo "Remove it and try again" >&2
        exit 1
    fi
    IPATH="$GOPATH/src/github.com/ManageIQ/manageiq-v2v-conversion_host/"
    mkdir -p "$IPATH"
    pushd $IPATH > /dev/null
    ln -s $(dirs -l +2)/kubevirt-vmware
    cd kubevirt-vmware

    # Build operator
    operator-sdk build "$IMAGE:$TAG"

    # Drop out and clean
    popd > /dev/null # $IPATH/kubevirt-vmware
    popd > /dev/null # /kubevirt-vmware
    rm -frv "$GOPATH"
}

do_images() {
    do_build_conversion
    do_build_vmware
}

do_$1
