/**
 * Copyright (C) 2014 JBoss Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.dashbuilder.dataset.backend;

import java.io.*;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.enterprise.context.ApplicationScoped;
import javax.inject.Inject;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.dashbuilder.config.Config;
import org.dashbuilder.dataprovider.DataSetProviderRegistry;
import org.dashbuilder.dataset.def.CSVDataSetDef;
import org.dashbuilder.dataset.def.DataSetDef;
import org.dashbuilder.dataset.def.DataSetDefRegistry;
import org.slf4j.Logger;

/**
 * This class looks for Data set definition files within an specific server directory and deploys them.
 */
@ApplicationScoped
public class DataSetDefDeployer {

    @Inject @Config("")
    protected String directory;

    @Inject @Config("3000")
    protected int pollingTime;

    @Inject
    protected DataSetDefRegistry dataSetDefRegistry;

    @Inject
    protected DataSetProviderRegistry dataSetProviderRegistry;

    @Inject
    protected DataSetDefJSONMarshaller jsonMarshaller;

    @Inject
    protected Logger log;

    protected Thread watcherThread;

    public String getDirectory() {
        return directory;
    }

    public boolean isRunning() {
        return !StringUtils.isBlank(directory);
    }

    @PostConstruct
    protected void init() {
        if (!StringUtils.isBlank(directory)) {
            deploy(directory);
        }
    }

    @PreDestroy
    public synchronized void stop() {
        directory = null;
    }

    public synchronized void deploy(String dir) {
        if (validateDirectory(dir)) {
            log.info("Data sets deployment directory = " + dir);
            directory = dir;
            doDeploy();

            if (pollingTime > 0) {
                watcherThread = new Thread(new Runnable() {
                    public void run() {
                        // TODO: replace by NIO WatcherService (requires upgrade to Java 7)
                        while (directory != null) {
                            try {
                                Thread.sleep(pollingTime);
                                doDeploy();
                            } catch (InterruptedException e) {
                                log.error("Data set watcher thread error.", e);
                            }
                        }
                    }
                });
                watcherThread.start();
            }
        }
        else {
            log.warn("Data sets deployment directory invalid: " + dir);
            directory = null;
        }
    }

    protected boolean validateDirectory(String dir) {
        if (StringUtils.isBlank(dir)) {
            return false;
        }
        File rootDir = new File(dir);
        if (!rootDir.exists()) {
            return false;
        }
        if (!rootDir.isDirectory()) {
            return false;
        }
        return true;
    }

    /**
     * Look into the deployment directory and processes any data set definition file found.
     */
    protected synchronized void doDeploy() {
        if (StringUtils.isBlank(directory)) return;

        // Look for data sets deploy
        File[] files = new File(directory).listFiles(_deployFilter);
        if (files != null) {
            for (File f : files) {
                try {
                    // Avoid repetitions
                    f.delete();

                    // Get the .dset file
                    File dsetFile = new File(f.getAbsolutePath().replace(".deploy", ""));
                    if (!dsetFile.exists()) continue;

                    // Read & parse the data set
                    FileReader fileReader = new FileReader(dsetFile);
                    String json = IOUtils.toString(fileReader);
                    DataSetDef def = jsonMarshaller.fromJson(json);
                    if (StringUtils.isBlank(def.getUUID())) def.setUUID(dsetFile.getName());

                    // CSV specific ...
                    if (def instanceof CSVDataSetDef) {
                        CSVDataSetDef csvDef = (CSVDataSetDef) def;
                        File csvFile = getCSVFile(csvDef);
                        if (csvFile != null) {
                            csvDef.setFilePath(csvFile.getAbsolutePath());
                        } else {
                            log.error("Data set CSV file not found: " + f.getName());
                            continue;
                        }
                    }

                    // Check if the data set really needs to be registered.
                    DataSetDef existingDef = dataSetDefRegistry.getDataSetDef(def.getUUID());
                    if (existingDef != null && jsonMarshaller.toJsonString(existingDef).equals(jsonMarshaller.toJsonString(def))) {
                        // Avoid redundant deployments
                        log.info("Data set already deployed: " + def.getUUID());
                    }
                    else {
                        // Register the data set
                        dataSetDefRegistry.registerDataSetDef(def, "system", "deploy(" + def.getUUID() + ")");
                        log.info("Data set deployed: " + def.getUUID());
                    }
                } catch (Exception e) {
                    log.error("Data set deployment error: " + f.getName(), e);
                }
            }
        }

        // Look for data sets undeploy
        files = new File(directory).listFiles(_undeployFilter);
        if (files != null) {
            for (File f : files) {
                try {
                    // Avoid repetitions
                    f.delete();

                    // Un-deploy the given uuid
                    String uuid = f.getName().replace(".undeploy", "");
                    DataSetDef def = dataSetDefRegistry.getDataSetDef(uuid);
                    if (def != null) {
                        dataSetDefRegistry.removeDataSetDef(uuid, "system", "undeploy(" + uuid + ")");
                        log.info("Data set deleted: " + def.getName());
                    } else {
                        log.error("Data set not found: " + uuid);
                    }
                } catch (Exception e) {
                    log.error("Data set un-deploy error: " + f.getName(), e);
                }
            }
        }
    }

    FilenameFilter _deployFilter = new FilenameFilter() {
        public boolean accept(File dir, String name) {
            return name.endsWith(".deploy");
        }
    };

    FilenameFilter _undeployFilter = new FilenameFilter() {
        public boolean accept(File dir, String name) {
            return name.endsWith(".undeploy");
        }
    };

    public File getCSVFile(CSVDataSetDef def) throws Exception {
        String path = def.getFilePath();
        if (StringUtils.isBlank(path)) return null;

        File f = new File(path);
        if (f.exists()) return f;

        f = new File(directory, path);
        if (f.exists()) return f;
        return null;
    }
}
