
#Marshalling Test Framework Readme

This readme describes the marshalling test framework. 

The idea is that we take the tests that we already have, and use their outputs to 
test the marshalling and unmarshalling mechanisms that Drools and jBPM relies on.

*   [Status](#status)
*   [What is the marshalling test framework](#what)
*   [Using the marshalling test framework](#using)
    *   [An example test](#example)
    *   [Changes that invalidate saved marshalled data](#changes)
*   [Parameters](#parameters)
*   [How the framework actually works](#how)
    *   [Saving snapshots](#snapshots)
    *   [Comparing to previous marshalled data](#comparing)
    *   [The "base" database](#base)  
*   [Conclusions about marshalling in Drools/jBPM](#comparing)

<h2 id="status">Status</h2>

This framework currently works, although it has not been turned on in drools. 

<h2 id="what">What is the marshalling test framework?</h2>

The basic idea for this test framework is that we take existing tests, 
and use the marshalled data they create to 1. make sure that the 
un/marshalling logic works and 2. check backwards compatibility.

 > We use a binary persistence format (using a byte[]) to transform a process instance or the session state into a binary to put it in the   
 >  database. It is possible however that this binary serialization algorithm might change over time (when new features are added etc.).   
...   
 > we are gonna have to make sure that, if a user updates from one jBPM version to a newer, that the old binary array can still be read in   
 > (or transformed).   
...   
 > The idea is that we should set up some infrastructure to keep track of this. And when we do encounter issues, we can then see how we can   
 >  solve these.   
...   
 > if we want to be able to detect issues, we should at least have some binary session and process instance info objects for   
 >  each version (starting jBPM 5.1) so that we can run them against future versions.   

The easiest way that I could think of to implement this -- without affecting 
the actual implementation code -- turned out to be based on creating proxy 
classes for `EntityManagerFactory`, `EntityManager` and `UserTransaction` 
instances. In short, every time that marshalled data is persisted, we 
save a copy of the marshalled data for later comparison. 

<h2 id="using">Using the marshalling test framework</h2>

In order to use this framework in a test, you need to do the all of the 
things described below. Please also see the example below where this is 
done, posted after the descriptions. 

1.   Convert your test to use JUnit 4 annotations, instead of extending the
     `TestCase` class.   
     *    This is primarily because we need to be able to run a method after 
          all methods in the test have run, which is, as far as I know, 
          only possible using the `@AfterClass` annotation. If you're using 
          TestNG (or whatever), the same applies -- make sure that you 
          have a method that has an `@AfterClass` or equivalent method.    
2.   Modify the setup and tearDown (`@After` and `@Before`) methods in your 
     jUnit test so that methods from the marshalling test framework are 
     called. 
3.   Make sure you use the objects created by the marshalling test framework 
     in the actual test methods.    
     *   This usually involves declaring a global variable, filling it 
         in your setup/`@Before` method and using it your test/`@Test` method. 
4.   Make sure to add an `@AfterClass` method that will call the static
     `MarshallingTestUtil.compareMarshallingDataFromTest(String persistenceUnitName)`
     method to compare the marshalled data generated by the test to the 
     snapshots taken and stored in the base database(s). 
     *   If the test class that you're modifying extends another test class,
         please make sure to call the 
         `MarshallingTestUtil.compareMarshallingDataFromTest(Class<?> testClass, String persistenceUnitName)`
         method. For the `testClass` parameter, please give the Class in 
         which the actual test methods reside. 
5.   Lastly, the following files should also exist in your project: 
     *   The `datasource.properties`, while not necessary, is helpful in order to 
         create a new base database. See 
         [jbpm 5 (and Drools) database testing](http://community.jboss.org/wiki/JBPM5DatabaseTesting) 
         as well.
     *   The `baseData-*.h2.db` files are (flat file) H2 databases that 
         store marshalled data snaphots from the current and previous versions 
         of the project. At the very least, there should be a 
         `baseData-current.h2.db` file. 

```
    <project>   
        src/   
            test/   
                resources/   
                    datasource.properties   
                    marshalling/   
                        baseData-*.h2.db   
```

<h3 id="example">An example test</h3>

As an example, we're going to take the following test and modify it so 
that it uses the marshalling test framework. 

 *   [TimerAndCalendarTest](https://github.com/kiegroup/drools/blob/b869611e377e9fc5e036c64c296eeaba75a5cd0e/drools-persistence-jpa/src/test/java/org/drools/timer/integrationtests/TimerAndCalendarTest.java)

<h4>Before using the framework</h4>

The following is the code in the original test that we will change in order 
to use the marshalling test framework:

```java
    public class TimerAndCalendarTest {
        private PoolingDataSource    ds1;
        private EntityManagerFactory emf;
```
and also [here](https://github.com/kiegroup/drools/blob/b869611e377e9fc5e036c64c296eeaba75a5cd0e/drools-persistence-jpa/src/test/java/org/drools/timer/integrationtests/TimerAndCalendarTest.java#L245): 

```java
        @Before
        public void setUp() throws Exception {
            ds1 = new PoolingDataSource();
            ds1.setUniqueName( "jdbc/testDS1" );
            ds1.setClassName( "org.h2.jdbcx.JdbcDataSource" );
            ds1.setMaxPoolSize( 3 );
            ds1.setAllowLocalTransactions( true );
            ds1.getDriverProperties().put( "user",
                                           "sa" );
            ds1.getDriverProperties().put( "password",
                                           "sasa" );
            ds1.getDriverProperties().put( "URL",
                                           "jdbc:h2:mem:mydb" );
            ds1.init();
            emf = Persistence.createEntityManagerFactory( "org.drools.persistence.jpa" );
        }
    
        @After
        public void tearDown() throws Exception {
            try {
                emf.close();
                ds1.close();
            } catch ( Exception e ) {
                e.printStackTrace();
            }
        }
    
    
        private StatefulKnowledgeSession createSession(KnowledgeBase kbase) {
            final KnowledgeSessionConfiguration conf = KnowledgeBaseFactory.newKnowledgeSessionConfiguration();
            conf.setOption( ClockTypeOption.get( ClockType.PSEUDO_CLOCK.getId() ) );
            StatefulKnowledgeSession ksession = JPAKnowledgeService.newStatefulKnowledgeSession( kbase,
                                                                                                 conf,
                                                                                                 createEnvironment() );
    
            return ksession;
        }
```
and lastly, the following method ([here](https://github.com/kiegroup/drools/blob/b869611e377e9fc5e036c64c296eeaba75a5cd0e/drools-persistence-jpa/src/test/java/org/drools/timer/integrationtests/TimerAndCalendarTest.java#L296)) 
will be removed.  

```java
       private Environment createEnvironment() {
           Environment env = KnowledgeBaseFactory.newEnvironment();
           env.set( EnvironmentName.ENTITY_MANAGER_FACTORY,
                    emf );
           env.set( EnvironmentName.GLOBALS,
                    new MapGlobalResolver() );
           return env;
       }
```

<h4>After using the framework</h4>

The method the framework makes available to initialize the persistence 
environment returns a `HashMap<String, Object>` instance containing a number 
of objects that will be used by the test. These include a proxy object 
for the `EntityManagerFactory` and the `UserTransaction` instances.

The first section becomes the following: 

```java
    public class TimerAndCalendarTest {
        private static Logger logger = LoggerFactory.getLogger(TimerAndCalendarTest.class);
        private HashMap<String, Object> context;
```

Notice that instead of having a global `PoolingDataSource` variable, we now have a `HashMap` context variable. 

The `@Before` and `@After` methods change to become the following:   

```java
        @Before
        public void setUp() throws Exception {
            context = PersistenceUtil.setupWithPoolingDataSource(PersistenceUtil.DROOLS_PERSISTENCE_UNIT_NAME);
        }
    
        @After
        public void tearDown() throws Exception {
            PersistenceUtil.tearDown(context);
        }
        
        private StatefulKnowledgeSession createSession(KnowledgeBase kbase) {
            final KnowledgeSessionConfiguration conf = KnowledgeBaseFactory.newKnowledgeSessionConfiguration();
            conf.setOption( ClockTypeOption.get( ClockType.PSEUDO_CLOCK.getId() ) );
            StatefulKnowledgeSession ksession = JPAKnowledgeService.newStatefulKnowledgeSession( kbase,
                                                                                                 conf,
                                                                                                 PersistenceUtil.createEnvironment(context) );
            return ksession;
        }
```
Notice that when we create a new `StatefulKnowledgeSession`, we use 
the `PersistenceUtil.createEnvironment(context)` method to create the `Environment` 
object used. 

Everything that was done before is still done, except that it's now done 
in the `PersistenceUtil.setupWithPoolingDataSource(String persistenceUnitName)` 
method.  The `PersistenceUtil` class is the `org.jbpm.persistence.util.PersistenceUtil` 
class, by the way. 

Lastly, we also add the following method to the class:   

```java
        @AfterClass
        public static void compareMarshallingData() { 
            MarshallingTestUtil.compareMarshallingDataFromTest(DROOLS_PERSISTENCE_UNIT_NAME);
        }
```
This method retrieves all of the data that was saved in the test and then 
compares it to the marshalled data saved.

<h3 id="changes">Changes that invalidate the saved marshalled data</h3>

What's important to remember about how the test framework is that it saves 
a copy -- a snapshot -- of all marshalled data persisted. 

In short, any modification to a test class method or any of the logic 
it uses, invalidates all of the marshalled data saved during that test method. 

<h4>Why does a change invalidated the marshalled test data?</h4>

At the end of the test, the test framework has saved a number of (marshalled 
data) snapshots. Each snapshot can uniquely be identified by the name of 
the test class and method that it was made during in combination with a unique id 
corresponding to the order in which it was made. 

When the data is retrieved later, the framework retrieves the same set 
of data that was made during previous run(s) of the same test, if the same
test existed in a previous verion of Drools or jBPM.

However, if the test logic is changed in any way, there's no guarantee anymore
that the marshalled data saved in previous verions of the tests, corresponds
to the data saved now. 

<h2 id="parameters">Parameters</h2>

There are a few parameters with which we can change the behaviour of the 
framework: 

 *   In the `src/test/resources/datasource.properties` file, we can add a 
     `makeBaseDb` property with a value of `no`  or `yes`. If this property is
     set to `yes`, then a new base database is created based on the current code.

 *    The `PersistenceUtil` class contains a private static boolean `TEST_MARSHALLING`. 
      This boolean is currently set to `false`, which ensures that the marhsalling 
      framework is not used. 

 *    Lastly, if your test (class) uses persistence, but your test does not persist
      any (worthwhile) marshalling data, you can use the   
      `PersistenceUtil.setupWithPoolingDataSource(String persistenceUnitName, boolean useFramework)`
      method in order to ensure that when persistence is setup for the test, no use
      of the marshalling test framework's objects are used. To do this, simply make
      sure that the second (useFramework) parameter is `false`.
      *   For example: 

```java
    @Before
    public void setup() {
        // This test does only plays with tx's, it doesn't actually persist
        // any interersting (wrt marshalling) SessionInfo objects
        boolean testMarshalling = false;

        context = PersistenceUtil.setupWithPoolingDataSource(DROOLS_PERSISTENCE_UNIT_NAME, testMarshalling);
        emf = (EntityManagerFactory) context.get(ENTITY_MANAGER_FACTORY);
    }
```

<h2 id="how">How the marshalling test framework actually works</h2>

The marshalling test framework does 3 things: 

 *   It saves snapshots of the marshalled data persisted.
 *   It compares the snapshots of the marshalled data persisted in the 
     most recent test run to snapshots made in other versions of Drools 
     or jBPM.
 *   It can be used to create a new "base" database consisting of snapshots
     of marshalled data, when a new version of Drools or jBPM is released. 

<h3 id="snapshots">Saving snapshots</h3>

The `PersistenceUtil` class was initially developed to implement the functionality described
in the [jbpm 5 (and Drools) database testing](http://community.jboss.org/wiki/JBPM5DatabaseTesting)
document. 

The `PersistenceUtil.setupWithPool.setupWithPoolingDataSource(...)` method 
populates a `HashMap<String, Object>` object with at least 3 objects:   

1.  A proxy object for the `EntityManagerFactory`.   
    *   This object returns a proxy for an `EntityManager` object when the
        `EntityManagerFactory.createEntityManager()` method is called.   
2.  A proxy object for a `UserTransaction` object.   
3.  A `PoolingDataSource` object.   

When the test runs, a `SessionInfo`, `WorkItemInfo` or `ProcessInstanceInfo` object
will eventually be created, updated, filled with marshalled information (see the `.update()` 
methods in all 3 classes) and persisted. These objects are used to store the rules or process
state.

The `EntityManager` and `UserTransaction` proxy objects then ensure that when this happens, 
a copy of the marshalled data (see next paragraph), is stored. 

The marshalled data for these 3 objects is stored in the following fields:   

 *   `SessionInfo.rulesByteArray`
 *   `WorkItemInfo.workItemByteArray`
 *   `ProcessInstanceInfo.processInstanceByteArray`

Every time that the proxy objects see that one of the above mentioned objects is being 
saved, they will check to see whether or not the marshalled data associated with the object
has changed. If so, it creates a new `MarshalledData` object. 

<h4>Snapshot content</h4>

A `MarshalledData` object contains the following fields: 

```java
public class MarshalledData { 

    @Id
    @GeneratedValue(strategy=GenerationType.AUTO, generator="marshalledDataIdSeq")
    public Integer id;
    
    @Lob
    public byte[] byteArray;
    
    public String testMethodName;
    public Integer snapshotNumber;
  
    public String marshalledObjectClassName;
    public Long marshalledObjectId;
    
```

 *   The `byteArray` field is used to store the marshalled data from the 
     `SessionInfo`, `WorkItemInfo` or `ProcessInstanceInfo` object. 
 *   The `testMethodName` field contains the (full) name of the test class 
     and test method being run.
     *   We can find this by looking at the stack trace, however in some 
         cases this is not possible. For example, if a test initalizes 
         Timer jobs, which run in a different thread than the test class 
         method, then it's (practically) impossible to figure out which 
         test the Timer has been started for. In this case we do **not** 
         save a `MarshalledData` object. 
 *   The `snapshotNumber` identifies this `MarshalledData` object with 
     regard to other `MarshalledData` objects created for this specific 
     test class method. 
     *   The first `MarshalledData` object created in a test class method 
         will have a `snapshotNumber` of 0, the next 1, and so forth.
 *   The `marshalledObjectClassName` contains the (full) class name of 
     the class containing the marshalled data (`SessionInfo`, etc.) and 
     the `marshalledObjectId` contains the id of that object. 

Besides the above mentioned fields, there are also other private transient 
fields used by the `MarshalledData` that are used in the internal logic of the class.

<h3 id="comparing">Comparing to previous marshalled data</h3>

Once all test methods have run, the `@AfterClass` method is run. In tests 
modified to use the marshalling test framework (as the `TimerAndCalendarTest` 
above is), we then compare the marshalled data generated during the test 
to marshalled data generated in previous versions of Drools and/or jBPM. 

 *   First, we reopen the (flat file h2) database in which the data from 
     the current test run is saved, and retrieve the marshalled data snapshots. 
 *   We then do the same for the marshalled data in the base databases.
 *   For each snapshot, we then check that we can unmarshall the snapshot.
 *   Using the `CompareViaReflectionUtil` class, we then compare the 
     object unmarshalled using the test data and the object unmarshalled
     using the base data. 
     *   The `CompareViaReflectionUtil` recursively compares all objects 
         in the object tree of the unmarshalled object. See the javadoc
         of that class for more info. 


