/*
 * Copyright 2016 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.drools.workbench.services.verifier.api.client.relations;

import org.drools.workbench.services.verifier.api.client.AnalyzerConfigurationMock;
import org.drools.workbench.services.verifier.api.client.configuration.AnalyzerConfiguration;
import org.drools.workbench.services.verifier.api.client.index.keys.Key;
import org.drools.workbench.services.verifier.api.client.index.keys.UUIDKey;
import org.drools.workbench.services.verifier.api.client.maps.InspectorList;
import org.drools.workbench.services.verifier.api.client.maps.util.HasKeys;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;

public class RelationResolverConflictsTest {

    private AnalyzerConfiguration configuration;

    private RelationResolver relationResolver;
    private InspectorList a;
    private InspectorList b;
    private Person isConflicting;
    private Person firstItemInA;


    @Before
    public void setUp() throws
                        Exception {
        configuration = new AnalyzerConfigurationMock();

        a = new InspectorList( configuration );
        b = new InspectorList( configuration );

        firstItemInA = spy( new Person( 10 ) );
        isConflicting = spy( new Person( 15 ) );
        a.add( firstItemInA );
        a.add( isConflicting );

        b.add( new Person( 10 ) );

        relationResolver = new RelationResolver( a,
                                                 true );
    }

    @Test
    public void empty() throws
                        Exception {

        relationResolver = new RelationResolver( new InspectorList( configuration ) );
        assertFalse( relationResolver.isConflicting( new InspectorList( configuration ) ) );
    }

    @Test
    public void recheck() throws
                          Exception {

        assertTrue( relationResolver.isConflicting( b ) );

        verify( firstItemInA ).conflicts( any() );

        reset( firstItemInA );

        assertTrue( relationResolver.isConflicting( b ) );

        verify( firstItemInA,
                never() ).conflicts( any() );
    }

    @Test
    public void recheckWithUpdate() throws
                                    Exception {

        assertTrue( relationResolver.isConflicting( b ) );

        reset( firstItemInA );

        // UPDATE
        isConflicting.setAge( 10 );

        assertFalse( relationResolver.isConflicting( b ) );

        verify( firstItemInA ).conflicts( any() );
    }

    @Test
    public void recheckConflictingItemRemoved() throws
                                                Exception {

        assertTrue( relationResolver.isConflicting( b ) );

        reset( firstItemInA );

        // UPDATE
        a.remove( isConflicting );

        assertFalse( relationResolver.isConflicting( b ) );

        verify( firstItemInA ).conflicts( any() );
    }

    @Test
    public void recheckOtherListBecomesEmpty() throws
                                               Exception {

        assertTrue( relationResolver.isConflicting( b ) );

        reset( firstItemInA,
               isConflicting );

        // UPDATE
        b.clear();

        assertFalse( relationResolver.isConflicting( b ) );

        verify( firstItemInA,
                never() ).conflicts( any() );
        verify( isConflicting,
                never() ).conflicts( any() );
    }

    public class Person
            implements IsConflicting,
                       HasKeys {

        int age;

        private UUIDKey uuidKey = configuration.getUUID( this );

        public Person( final int age ) {
            this.age = age;
        }

        @Override
        public UUIDKey getUuidKey() {
            return uuidKey;
        }

        @Override
        public Key[] keys() {
            return new Key[]{
                    uuidKey
            };
        }

        public void setAge( final int age ) {
            this.age = age;
        }

        @Override
        public boolean conflicts( final Object other ) {
            if ( other instanceof Person ) {
                return age != ( (Person) other ).age;
            } else {
                return false;
            }
        }

    }
}