/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Copyright 2002 Elmar Grom
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.util;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.UninstallData;
import com.izforge.izpack.installer.Unpacker;

import java.io.File;
import java.util.Collections;
import java.util.Vector;

/*---------------------------------------------------------------------------*/
/**
 * This class performs housekeeping and cleanup tasks. There can only be one instance of
 * <code>Housekeeper</code> per Java runtime, therefore this class is implemented as a
 * 'Singleton'. <br>
 * <br>
 * It is VERY important to perform pre-shutdown cleanup operations through this class. Do NOT rely
 * on operations like <code>deleteOnExit()</code> shutdown hooks or <code>finalize()</code>for
 * cleanup. Because <code>shutDown()</code> uses <code>System.exit()</code> to terminate, these
 * methods will not work at all or will not work reliably.
 *
 * @author Elmar Grom
 * @version 0.0.1 / 2/9/02
 */
/*---------------------------------------------------------------------------*/
public class Housekeeper
{

    // ------------------------------------------------------------------------
    // Variable Declarations
    // ------------------------------------------------------------------------
    private static Housekeeper me = null;

    /**
     * The cleanup jobs must run before file tree deletion.
     */
    private Vector<CleanupClient> cleanupClients = new Vector<CleanupClient>();

    /**
     * We hold a separate queue for Librarian because it must run as the very last cleanup job right
     * before a shutdown.
     */
    private Vector<CleanupClient> librarianClients = new Vector<CleanupClient>();

    /*--------------------------------------------------------------------------*/
    /**
     * This class is implemented as a 'Singleton'. Therefore the constructor is private to prevent
     * instantiation of this class. Use <code>getInstance()</code> to obtain an instance for use.
     * <br>
     * <br>
     * For more information about the 'Singleton' pattern I highly recommend the book Design
     * Patterns by Gamma, Helm, Johnson and Vlissides ISBN 0-201-63361-2.
     */
    /*--------------------------------------------------------------------------*/
    private Housekeeper()
    {
    }

    /*--------------------------------------------------------------------------*/
    /**
     * Returns an instance of <code>Housekeeper</code> to use.
     *
     * @return an instance of <code>Housekeeper</code>.
     */
    /*--------------------------------------------------------------------------*/
    public static Housekeeper getInstance()
    {
        if (me == null)
        {
            me = new Housekeeper();
        }

        return (me);
    }

    /*--------------------------------------------------------------------------*/
    /**
     * Use to register objects that need to perform cleanup operations before the application shuts
     * down.
     *
     * @param client reference of to an object that needs to perform cleanup operations.
     */
    /*--------------------------------------------------------------------------*/
    public void registerForCleanup(CleanupClient client)
    {
        // IZPACK-276:
        // if the client is an instance of Librarian hold it at a special place to call it at the
        // very last time
        if (client instanceof Librarian)
        {
            librarianClients.add(0, client);
        }
        else
        {
            cleanupClients.add(client);
        }
    }

    /**
     * Runs all of the cleanupClients registered through the regsterForCleanup method.
     */
    public void runCleanup()
    {
        // IZPACK-276
        // Do the cleanup of the last registered client at the fist time (first in last out)
        // Do not run the librarian cleanup yet. Save this for right before shutdown.
        // System.out.println("Running Custom Cleanup Jobs");
        runClients(cleanupClients);
    }

    private void runClients(Vector<CleanupClient> clients)
    {
        for (int i = clients.size() - 1; i >= 0; i--)
        {
            try
            {
                (clients.elementAt(i)).cleanUp();
            }
            catch (Throwable exception)
            {
                // At this point we can not afford to treat exceptions. Cleanup
                // that
                // can not be completed might unfortunately leave some garbage
                // behind.
                // If we have a logging module, any exceptions received here
                // should
                // be written to the log.
            }
        }
    }

    /**
     * Wipes files written during installation
     */
    public void wipeFiles() {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        UninstallData udata = UninstallData.getInstance();
        // We set interrupt to all running Unpacker and wait 40 sec for maximum.
        // If interrupt is discarded (return value false), return immediately:
        if (!Unpacker.interruptAll(40000)) {
            return;
        }

        // Uninstaller dir gets created somewhere else, so manually add to the files list:
        udata.getInstalledFilesList().add(idata.getInstallPath() + "/uninstaller");

        //Order of deletion matters
        Collections.sort(udata.getInstalledFilesList(), Collections.reverseOrder());

        // Wipe the files that had been installed
        for (String p : udata.getInstalledFilesList()) {
            try {
                File f = new File(p);
                if (!f.delete()) {
                    Debug.log("Cannot delete " + p);
                }
            } catch (SecurityException e) {
                Debug.log(p + " not deleted: " + e.getMessage());
            }
        }
    }
    /*--------------------------------------------------------------------------*/
    /**
     * This methods shuts the application down. First, it will call all clients that have registered
     * for cleanup operations. Once this has been accomplished, the application will be forceably
     * terminated. <br>
     * <br>
     * <b>THIS METHOD DOES NOT RETURN!</b>
     * 
     * @param exitCode the exit code that should be returned to the calling process.
     */
    /*--------------------------------------------------------------------------*/
    public void shutDown(int exitCode)
    {
        // Run cleanUp jobs:
        shutDownNoConsole(exitCode);
    }

    public void shutDownNoConsole(int exitCode) {
        Debug.log("Running cleanup Jobs.");
        runClients(this.cleanupClients);
        Debug.log("Running Librarian Cleanup");
        runClients(librarianClients);
        Debug.log("Shutting Down");
        System.exit(exitCode);
    }

    /**
     * Same as shutDown, but deliberately skips running all cleanUp jobs except for the librarian.
     * This can be called if the client cleanUp jobs are invoked separately to avoid running them
     * twice.
     * 
     * @param exitCode
     */
    public void shutDownNoCleanUp(int exitCode)
    {
        Debug.log("Running Librarian Cleanup");
        runClients(librarianClients);
        Debug.log("Shutting Down");
        System.exit(exitCode);
    }
}
/*---------------------------------------------------------------------------*/
