/*************************************************************************************
 * Copyright (c) 2011-2014 Red Hat, Inc. and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     JBoss by Red Hat - Initial implementation.
 ************************************************************************************/
package org.jboss.tools.project.examples.tests;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Properties;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.datatools.connectivity.ConnectionProfileConstants;
import org.eclipse.datatools.connectivity.ConnectionProfileException;
import org.eclipse.datatools.connectivity.ProfileManager;
import org.eclipse.datatools.connectivity.db.generic.IDBConnectionProfileConstants;
import org.eclipse.datatools.connectivity.db.generic.IDBDriverDefinitionConstants;
import org.eclipse.datatools.connectivity.drivers.DriverInstance;
import org.eclipse.datatools.connectivity.drivers.DriverManager;
import org.eclipse.datatools.connectivity.drivers.IDriverMgmtConstants;
import org.eclipse.datatools.connectivity.drivers.IPropertySet;
import org.eclipse.datatools.connectivity.drivers.PropertySetImpl;
import org.eclipse.datatools.connectivity.drivers.models.TemplateDescriptor;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.wst.server.core.IRuntime;
import org.eclipse.wst.server.core.IRuntimeType;
import org.eclipse.wst.server.core.IRuntimeWorkingCopy;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerType;
import org.eclipse.wst.server.core.IServerWorkingCopy;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.ServerUtil;
import org.eclipse.wst.server.core.internal.RuntimeWorkingCopy;
import org.eclipse.wst.server.core.internal.ServerWorkingCopy;
import org.jboss.tools.seam.core.project.facet.SeamRuntime;
import org.jboss.tools.seam.core.project.facet.SeamRuntimeManager;
import org.jboss.tools.seam.core.project.facet.SeamVersion;

/**
 * @author snjeza
 */
public class ProjectExamplesTestUtil {

	public static final int IDLE_TIME = 60 * 1000;

	public static final String CONNECTION_PROFILE_NAME = "DefaultDS";

	public static final String SEAM_RUNTIME_NAME = "Seam 2.0";

	public static final String JBOSS_AS_RUNTIME_NAME = "JBoss AS 4.2 Runtime";

	public static final String JBOSS_AS_SERVER_NAME = "JBoss AS 4.2.3 Server";

	public static final String JBOSS_AS_HOST = "localhost"; //$NON-NLS-1$

	public static final String JBOSS_AS_DEFAULT_CONFIGURATION_NAME = "default"; //$NON-NLS-1$

	public static final String HSQL_DRIVER_DEFINITION_ID = "DriverDefn.Hypersonic DB"; //$NON-NLS-1$

	public static final String HSQL_DRIVER_NAME = "Hypersonic DB"; //$NON-NLS-1$

	public static final String HSQL_DRIVER_TEMPLATE_ID = "org.eclipse.datatools.enablement.hsqldb.1_8.driver"; //$NON-NLS-1$

	public static final String DTP_DB_URL_PROPERTY_ID = "org.eclipse.datatools.connectivity.db.URL"; //$NON-NLS-1$

	public static final String HSQL_PROFILE_ID = "org.eclipse.datatools.enablement.hsqldb.connectionProfile";

	public static final String JBOSS_AS_HOME = System.getProperty(
			"jbosstools.test.jboss.home.4.2",
			"target/requirements/jboss-4.2.3.GA".replace('/',
					File.separatorChar));

	public static final String JBOSS_AS_RUNTIME_ID = "org.jboss.ide.eclipse.as.runtime.42";

	public static final String JBOSS_AS_SERVER_ID = "org.jboss.ide.eclipse.as.42";

	public static final String SEAM_HOME_PROPERTY = System.getProperty(
			"jbosstools.test.seam.2.0.1.GA.home",
			"target/requirements/jboss-seam-2.0.1.GA".replace('/',
					File.separatorChar));

	public static final String HSQLDB_DRIVER_JAR_NAME = "hsqldb.jar"; //$NON-NLS-1$

	public static final String HSQLDB_DRIVER_LOCATION = "/common/lib/" + HSQLDB_DRIVER_JAR_NAME; //$NON-NLS-1$

	public static void initRuntimes() throws Exception {
		switchPerspective("org.jboss.tools.seam.ui.SeamPerspective");

		String asLocation = JBOSS_AS_HOME;

		String runtimeType = JBOSS_AS_RUNTIME_ID;
		String serverType = JBOSS_AS_SERVER_ID;

		createJBossServer(new File(asLocation), serverType, runtimeType,
				JBOSS_AS_SERVER_NAME, JBOSS_AS_RUNTIME_NAME);

		String seamPath = SEAM_HOME_PROPERTY;
		createSeamRuntime(SEAM_RUNTIME_NAME, seamPath, SeamVersion.SEAM_2_0);

		createDriver(asLocation, HSQLDB_DRIVER_LOCATION);
	}

	public static void removeProjects() throws CoreException {
		final IWorkspace workspace = ResourcesPlugin.getWorkspace();
		workspace.run(new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				IProject[] projects = workspace.getRoot().getProjects();
				for (int i = 0; i < projects.length; i++) {
					projects[i].delete(true, true, monitor);
				}
			}
		}, new NullProgressMonitor());
	}

	protected static void switchPerspective(final String pid) {
		Display.getDefault().syncExec(new Runnable() {

			public void run() {
				IWorkbench workbench = PlatformUI.getWorkbench();
				IPerspectiveRegistry perspectiveRegistry = workbench
						.getPerspectiveRegistry();
				IPerspectiveDescriptor perspective = perspectiveRegistry
						.findPerspectiveWithId(pid);
				workbench.getActiveWorkbenchWindow().getActivePage()
						.setPerspective(perspective);
			}
		});
	}

	public static void createJBossServer(File asLocation, String serverType,
			String runtimeType, String name, String runtimeName)
			throws CoreException {
		if (!asLocation.isDirectory()) {
			return;
		}
		IPath jbossAsLocationPath = new Path(asLocation.getAbsolutePath());

		IServer[] servers = ServerCore.getServers();
		for (int i = 0; i < servers.length; i++) {
			IRuntime runtime = servers[i].getRuntime();
			if (runtime != null
					&& runtime.getLocation().equals(jbossAsLocationPath)) {
				return;
			}
		}

		IRuntime runtime = null;
		IRuntime[] runtimes = ServerCore.getRuntimes();
		for (int i = 0; i < runtimes.length; i++) {
			if (runtimes[0].getLocation().equals(jbossAsLocationPath)) {
				runtime = runtimes[0].createWorkingCopy();
				break;
			}
		}

		IProgressMonitor progressMonitor = new NullProgressMonitor();
		if (runtime == null) {
			runtime = createRuntime(runtimeName, asLocation.getAbsolutePath(),
					progressMonitor, runtimeType);
		}
		if (runtime != null) {
			System.out.println("Creating "+ runtime.getName());
			createServer(runtime, serverType, name, progressMonitor);
		}
	}

	protected static IRuntime createRuntime(String runtimeName,
			String jbossASLocation, IProgressMonitor progressMonitor,
			String runtimeType) throws CoreException {
		IRuntimeWorkingCopy runtime = null;
		String type = null;
		String version = null;
		String runtimeId = null;
		IPath jbossAsLocationPath = new Path(jbossASLocation);
		IRuntimeType[] runtimeTypes = ServerUtil.getRuntimeTypes(type, version,
				runtimeType);
		if (runtimeTypes.length > 0) {
			runtime = runtimeTypes[0].createRuntime(runtimeId, progressMonitor);
			runtime.setLocation(jbossAsLocationPath);
			if (runtimeName != null) {
				runtime.setName(runtimeName);
			}
			((RuntimeWorkingCopy) runtime)
					.setAttribute(
							"org.jboss.ide.eclipse.as.core.runtime.configurationName", JBOSS_AS_DEFAULT_CONFIGURATION_NAME); //$NON-NLS-1$

			return runtime.save(false, progressMonitor);
		}
		return runtime;
	}

	protected static void createDriver(String jbossASLocation,
			String driverLocation) throws ConnectionProfileException,
			IOException {
		if (ProfileManager.getInstance().getProfileByName(
				CONNECTION_PROFILE_NAME) != null) {
			return;
		}
		String driverPath = new File(jbossASLocation + driverLocation)
				.getCanonicalPath(); //$NON-NLS-1$

		DriverInstance driver = DriverManager.getInstance()
				.getDriverInstanceByName(HSQL_DRIVER_NAME);
		if (driver == null) {
			TemplateDescriptor descr = TemplateDescriptor
					.getDriverTemplateDescriptor(HSQL_DRIVER_TEMPLATE_ID);
			IPropertySet instance = new PropertySetImpl(HSQL_DRIVER_NAME,
					HSQL_DRIVER_DEFINITION_ID);
			instance.setName(HSQL_DRIVER_NAME);
			instance.setID(HSQL_DRIVER_DEFINITION_ID);
			Properties props = new Properties();

			IConfigurationElement[] template = descr.getProperties();
			for (int i = 0; i < template.length; i++) {
				IConfigurationElement prop = template[i];
				String id = prop.getAttribute("id"); //$NON-NLS-1$

				String value = prop.getAttribute("value"); //$NON-NLS-1$
				props.setProperty(id, value == null ? "" : value); //$NON-NLS-1$
			}
			props.setProperty(DTP_DB_URL_PROPERTY_ID, "jdbc:hsqldb:."); //$NON-NLS-1$
			props.setProperty(IDriverMgmtConstants.PROP_DEFN_TYPE,
					descr.getId());
			props.setProperty(IDriverMgmtConstants.PROP_DEFN_JARLIST,
					driverPath);

			instance.setBaseProperties(props);
			DriverManager.getInstance().removeDriverInstance(instance.getID());
			System.gc();
			DriverManager.getInstance().addDriverInstance(instance);
		}

		driver = DriverManager.getInstance().getDriverInstanceByName(
				HSQL_DRIVER_NAME);
		if (driver != null
				&& ProfileManager.getInstance().getProfileByName(
						CONNECTION_PROFILE_NAME) == null) { //$NON-NLS-1$
			// create profile
			Properties props = new Properties();
			props.setProperty(
					ConnectionProfileConstants.PROP_DRIVER_DEFINITION_ID,
					HSQL_DRIVER_DEFINITION_ID);
			props.setProperty(
					IDBConnectionProfileConstants.CONNECTION_PROPERTIES_PROP_ID,
					""); //$NON-NLS-1$
			props.setProperty(
					IDBDriverDefinitionConstants.DRIVER_CLASS_PROP_ID,
					driver.getProperty(IDBDriverDefinitionConstants.DRIVER_CLASS_PROP_ID));
			props.setProperty(
					IDBDriverDefinitionConstants.DATABASE_VENDOR_PROP_ID,
					driver.getProperty(IDBDriverDefinitionConstants.DATABASE_VENDOR_PROP_ID));
			props.setProperty(
					IDBDriverDefinitionConstants.DATABASE_VERSION_PROP_ID,
					driver.getProperty(IDBDriverDefinitionConstants.DATABASE_VERSION_PROP_ID));
			props.setProperty(
					IDBDriverDefinitionConstants.DATABASE_NAME_PROP_ID,
					"Default"); //$NON-NLS-1$
			props.setProperty(IDBDriverDefinitionConstants.PASSWORD_PROP_ID, ""); //$NON-NLS-1$
			props.setProperty(
					IDBConnectionProfileConstants.SAVE_PASSWORD_PROP_ID,
					"false"); //$NON-NLS-1$
			props.setProperty(
					IDBDriverDefinitionConstants.USERNAME_PROP_ID,
					driver.getProperty(IDBDriverDefinitionConstants.USERNAME_PROP_ID));
			props.setProperty(IDBDriverDefinitionConstants.URL_PROP_ID, driver
					.getProperty(IDBDriverDefinitionConstants.URL_PROP_ID));

			ProfileManager
					.getInstance()
					.createProfile(
							CONNECTION_PROFILE_NAME,
							"The JBoss AS Hypersonic embedded database", HSQL_PROFILE_ID, props, "", false); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}

	}

	protected static IServerWorkingCopy createServer(IRuntime runtime,
			String runtimeType, String name, IProgressMonitor progressMonitor)
			throws CoreException {
		IServerType serverType = ServerCore.findServerType(runtimeType);
		IServerWorkingCopy server = serverType.createServer(null, null,
				runtime, progressMonitor);

		server.setHost(JBOSS_AS_HOST);
		server.setName(name);

		// JBossServer.DEPLOY_DIRECTORY
		String deployVal = runtime
				.getLocation()
				.append("server").append(JBOSS_AS_DEFAULT_CONFIGURATION_NAME).append("deploy").toOSString(); //$NON-NLS-1$ //$NON-NLS-2$
		((ServerWorkingCopy) server)
				.setAttribute(
						"org.jboss.ide.eclipse.as.core.server.deployDirectory", deployVal); //$NON-NLS-1$

		// IDeployableServer.TEMP_DEPLOY_DIRECTORY
		String deployTmpFolderVal = runtime
				.getLocation()
				.append("server").append(JBOSS_AS_DEFAULT_CONFIGURATION_NAME).append("tmp").append("jbosstoolsTemp").toOSString(); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		((ServerWorkingCopy) server)
				.setAttribute(
						"org.jboss.ide.eclipse.as.core.server.tempDeployDirectory", deployTmpFolderVal); //$NON-NLS-1$

		server.save(false, progressMonitor);
		return server;
	}

	public static void createSeamRuntime(String name, String seamPath,
			SeamVersion seamVersion) {
		SeamRuntime seamRuntime = SeamRuntimeManager.getInstance()
				.findRuntimeByName(name);
		if (seamRuntime != null) {
			return;
		}
		File seamFolder = new File(seamPath);
		if (seamFolder.exists() && seamFolder.isDirectory()) {
			SeamRuntime rt = new SeamRuntime();
			rt.setHomeDir(seamPath);
			rt.setName(name);
			rt.setDefault(true);
			rt.setVersion(seamVersion);
			SeamRuntimeManager.getInstance().addRuntime(rt);
		}
	}

	public static String toString(Collection<IMarker> markers) {
		String sep = "";
		StringBuilder sb = new StringBuilder();
		if (markers != null) {
			for (IMarker marker : markers) {
				try {
					sb.append(sep).append(marker.getAttribute(IMarker.MESSAGE));
				} catch (CoreException ex) {
					// ignore
				}
				sep = ", ";
			}
		}
		return sb.toString();
	}

}
