/*
 * ModeShape (http://www.modeshape.org)
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * See the AUTHORS.txt file in the distribution for a full listing of 
 * individual contributors.
 *
 * ModeShape is free software. Unless otherwise indicated, all code in ModeShape
 * is licensed to you under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 * 
 * ModeShape is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.modeshape.jcr.security;

import org.modeshape.common.util.CheckArg;
import org.modeshape.jcr.Environment;

/**
 * Base class for {@link org.modeshape.jcr.security.AuthenticationProvider} implementations which leverage the active
 * {@link org.modeshape.jcr.Environment} instance to perform custom authentication.
 * 
 * @author Horia Chiorean (hchiorea@redhat.com)
 */
public abstract class EnvironmentAuthenticationProvider implements AuthenticationProvider {
    
    private Environment environment;
    private String securityDomain;
    
    /**
     * No-arg constructor, required in order for these providers to be created via reflection at repository start.
     */                                                               
    public EnvironmentAuthenticationProvider() {
    }

    /**
     * Sets the active repository environment for this provider.
     * 
     * @param environment a {@link org.modeshape.jcr.Environment} instance, never {@code null}
     */
    public void setEnvironment( Environment environment ) {
        CheckArg.isNotNull(environment, "environment");
        this.environment = environment;
    }

    /**
     * Sets the name of the security domain in which authentication will be attempted.
     * 
     * @param securityDomain the name of a valid security domain, never {@code null}
     */
    public void setSecurityDomain( String securityDomain ) {
        CheckArg.isNotNull(securityDomain, "securityDomain");
        this.securityDomain = securityDomain;
    }
    
    protected Environment environment() {
        return environment;
    }

    protected String securityDomain() {
        return securityDomain;
    }

    /**
     * Initializes this provider instance, after the {@link org.modeshape.jcr.Environment) and the security domain have been set
     */
    public abstract void initialize();
}
