/*
 * JBoss, Home of Professional Open Source.
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */
package org.komodo.relational.commands.parameter;

import java.util.List;
import org.komodo.relational.RelationalConstants.Nullable;
import org.komodo.relational.commands.workspace.WorkspaceCommandsI18n;
import org.komodo.relational.model.Parameter;
import org.komodo.relational.model.Parameter.Direction;
import org.komodo.shell.CommandResultImpl;
import org.komodo.shell.api.Arguments;
import org.komodo.shell.api.CommandResult;
import org.komodo.shell.api.TabCompletionModifier;
import org.komodo.shell.api.WorkspaceStatus;
import org.komodo.shell.commands.SetPropertyCommand;
import org.komodo.spi.repository.Repository.UnitOfWork;
import org.komodo.utils.StringUtils;
import org.komodo.utils.i18n.I18n;

/**
 * A shell command to set {@link Parameter parameter} properties.
 */
public final class SetParameterPropertyCommand extends ParameterShellCommand {

    static final String NAME = SetPropertyCommand.NAME;

    /**
     * @param status
     *        the shell's workspace status (cannot be <code>null</code>)
     */
    public SetParameterPropertyCommand( final WorkspaceStatus status ) {
        super( NAME, status );
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#doExecute()
     */
    @Override
    protected CommandResult doExecute() {
        CommandResult result = null;

        try {
            final String name = requiredArgument( 0, I18n.bind( WorkspaceCommandsI18n.missingPropertyNameValue ) );
            final String value = requiredArgument( 1, I18n.bind( WorkspaceCommandsI18n.missingPropertyNameValue ) );

            final Parameter parameter = getParameter();
            final UnitOfWork transaction = getTransaction();
            String errorMsg = null;

            if ( DATATYPE_NAME.equals( name ) ) {
                parameter.setDatatypeName( transaction, value );
            } else if ( DEFAULT_VALUE.equals( name ) ) {
                parameter.setDefaultValue( transaction, value );
            } else if ( DIRECTION.equals( name ) ) {
                if ( Direction.IN.name().equals( value ) ) {
                    parameter.setDirection( transaction, Direction.IN );
                } else if ( Direction.IN_OUT.name().equals( value ) ) {
                    parameter.setDirection( transaction, Direction.IN_OUT );
                } else if ( Direction.OUT.name().equals( value ) ) {
                    parameter.setDirection( transaction, Direction.OUT );
                } else if ( Direction.VARIADIC.name().equals( value ) ) {
                    parameter.setDirection( transaction, Direction.VARIADIC );
                } else {
                    errorMsg = I18n.bind( ParameterCommandsI18n.invalidDirectionPropertyValue, NULLABLE );
                }
            } else if ( LENGTH.equals( name ) ) {
                try {
                    final long length = Long.parseLong( value );
                    parameter.setLength( transaction, length );
                } catch ( final NumberFormatException e ) {
                    errorMsg = I18n.bind( WorkspaceCommandsI18n.invalidIntegerPropertyValue, LENGTH );
                }
            } else if ( NULLABLE.equals( name ) ) {
                if ( Nullable.NO_NULLS.name().equals( value ) ) {
                    parameter.setNullable( transaction, Nullable.NO_NULLS );
                } else if ( Nullable.NULLABLE.name().equals( value ) ) {
                    parameter.setNullable( transaction, Nullable.NULLABLE );
                } else if ( Nullable.NULLABLE_UNKNOWN.name().equals( value ) ) {
                    parameter.setNullable( transaction, Nullable.NULLABLE_UNKNOWN );
                } else {
                    errorMsg = I18n.bind( ParameterCommandsI18n.invalidNullablePropertyValue, NULLABLE );
                }
            } else if ( PRECISION.equals( name ) ) {
                try {
                    final long precision = Long.parseLong( value );
                    parameter.setPrecision( transaction, precision );
                } catch ( final NumberFormatException e ) {
                    errorMsg = I18n.bind( WorkspaceCommandsI18n.invalidIntegerPropertyValue, PRECISION );
                }
            } else if ( RESULT.equals( name ) ) {
                if ( Boolean.TRUE.toString().equals( value ) || Boolean.FALSE.toString().equals( value ) ) {
                    parameter.setResult( transaction, Boolean.parseBoolean( value ) );
                } else {
                    errorMsg = I18n.bind( WorkspaceCommandsI18n.invalidBooleanPropertyValue, RESULT );
                }
            } else if ( SCALE.equals( name ) ) {
                try {
                    final long scale = Long.parseLong( value );
                    parameter.setScale( transaction, scale );
                } catch ( final NumberFormatException e ) {
                    errorMsg = I18n.bind( WorkspaceCommandsI18n.invalidIntegerPropertyValue, SCALE );
                }
            } else {
                errorMsg = I18n.bind( WorkspaceCommandsI18n.invalidPropertyName, name, Parameter.class.getSimpleName() );
            }

            if ( StringUtils.isBlank( errorMsg ) ) {
                result = new CommandResultImpl( I18n.bind( WorkspaceCommandsI18n.setPropertySuccess, name ) );
            } else {
                result = new CommandResultImpl( false, errorMsg, null );
            }
        } catch ( final Exception e ) {
            result = new CommandResultImpl( e );
        }

        return result;
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#getMaxArgCount()
     */
    @Override
    protected int getMaxArgCount() {
        return 2;
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#printHelpDescription(int)
     */
    @Override
    protected void printHelpDescription( final int indent ) {
        print( indent, I18n.bind( ParameterCommandsI18n.setParameterPropertyHelp, getName() ) );
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#printHelpExamples(int)
     */
    @Override
    protected void printHelpExamples( final int indent ) {
        print( indent, I18n.bind( ParameterCommandsI18n.setParameterPropertyExamples ) );
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#printHelpUsage(int)
     */
    @Override
    protected void printHelpUsage( final int indent ) {
        print( indent, I18n.bind( ParameterCommandsI18n.setParameterPropertyUsage ) );
    }

    /**
     * {@inheritDoc}
     *
     * @see org.komodo.shell.BuiltInShellCommand#tabCompletion(java.lang.String, java.util.List)
     */
    @Override
    public TabCompletionModifier tabCompletion( final String lastArgument,
                              final List< CharSequence > candidates ) throws Exception {
        final Arguments args = getArguments();

        if ( args.isEmpty() ) {
            if ( lastArgument == null ) {
                candidates.addAll( ALL_PROPS );
            } else {
                for ( final String item : ALL_PROPS ) {
                    if ( item.toUpperCase().startsWith( lastArgument.toUpperCase() ) ) {
                        candidates.add( item );
                    }
                }
            }
        }

        if ( ( args.size() == 1 ) ) {
            String theArg = getArguments().get(0);
            if( RESULT.equals(theArg) ) {
                updateCandidatesForBooleanProperty( lastArgument, candidates );
            } else if( NULLABLE.equals(theArg) ) {
                candidates.add( Nullable.NO_NULLS.name() );
                candidates.add( Nullable.NULLABLE.name() );
                candidates.add( Nullable.NULLABLE_UNKNOWN.name() );
            } else if( DIRECTION.equals(theArg)) {
                candidates.add( Direction.IN.name() );
                candidates.add( Direction.IN_OUT.name() );
                candidates.add( Direction.OUT.name() );
                candidates.add( Direction.VARIADIC.name() );
            }
        }
        return TabCompletionModifier.AUTO;
    }

}
