/*
 * ModeShape (http://www.modeshape.org)
 * See the COPYRIGHT.txt file distributed with this work for information
 * regarding copyright ownership.  Some portions may be licensed
 * to Red Hat, Inc. under one or more contributor license agreements.
 * See the AUTHORS.txt file in the distribution for a full listing of
 * individual contributors.
 *
 * ModeShape is free software. Unless otherwise indicated, all code in ModeShape
 * is licensed to you under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * ModeShape is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org
 */
package org.modeshape.sequencer.msoffice.powerpoint;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import org.apache.poi.hslf.usermodel.HSLFSlide;
import org.apache.poi.hslf.usermodel.HSLFSlideShow;
import org.apache.poi.hslf.usermodel.HSLFSlideShowImpl;
import org.apache.poi.hslf.usermodel.HSLFTextParagraph;
import org.apache.poi.hslf.usermodel.HSLFTextRun;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;

/**
 * Utility for extracting metadata from PowerPoint files
 */
public class PowerPointMetadataReader {
    
    public static PowerpointMetadata instance( InputStream stream ) throws IOException {
        POIFSFileSystem fs = new POIFSFileSystem(stream);
        HSLFSlideShow rawSlideShow = new HSLFSlideShow(fs);
        List<SlideMetadata> slidesMetadata = new ArrayList<SlideMetadata>();
        for (HSLFSlide slide : rawSlideShow.getSlides()) {
            slidesMetadata.add(processSlide(rawSlideShow, slide));
        }
        PowerpointMetadata deck = new PowerpointMetadata();
        deck.setSlides(slidesMetadata);
        deck.setMetadata(new HSLFSlideShowImpl(fs).getSummaryInformation());
        return deck;
    }
    
    private static SlideMetadata processSlide(HSLFSlideShow rawSlideShow, HSLFSlide slide) {
        SlideMetadata slideMetadata = new SlideMetadata();
        // process title
        String title = slide.getTitle();
        slideMetadata.setTitle(title);
        
        // process notes
        slideMetadata.setNotes(collectText(slide.getNotes().getTextParagraphs(), title));
        // process text
        slideMetadata.setText(collectText(slide.getTextParagraphs(), title));
        
        // process thumbnail
        Dimension pgsize = rawSlideShow.getPageSize();
        
        BufferedImage img = new BufferedImage(pgsize.width, pgsize.height, BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics = img.createGraphics();
        // clear the drawing area
        graphics.setPaint(Color.white);
        graphics.fill(new Rectangle2D.Float(0, 0, pgsize.width, pgsize.height));
        
        // render
        slide.draw(graphics);
        
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            javax.imageio.ImageIO.write(img, "png", out);
            slideMetadata.setThumbnail(out.toByteArray());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        
        return slideMetadata;
    }
    
    private static String collectText(List<List<HSLFTextParagraph>> paragraphs, String title) {
        StringBuilder stringBuilder = new StringBuilder();
        for (List<HSLFTextParagraph> paragraphList : paragraphs) {
            for (HSLFTextParagraph paragraph : paragraphList) {
                for (HSLFTextRun textRun : paragraph.getTextRuns()) {
                    String rawText = textRun.getRawText();
                    if (!title.equals(rawText)) {
                        stringBuilder.append(rawText);   
                    }
                }
            }
        }
        return stringBuilder.toString();
    }
}
