/**
 * JBoss, a Division of Red Hat
 * Copyright 2006, Red Hat Middleware, LLC, and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
* This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ide.eclipse.as.ui.views.server;

import java.util.ArrayList;
import java.util.Properties;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.PageBook;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerLifecycleListener;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.ui.ServerUICore;
import org.jboss.ide.eclipse.as.core.server.UnitedServerListener;
import org.jboss.ide.eclipse.as.core.server.UnitedServerListenerManager;
import org.jboss.ide.eclipse.as.core.server.internal.JBossServer;
import org.jboss.ide.eclipse.as.ui.JBossServerUIPlugin;
import org.jboss.ide.eclipse.as.ui.Messages;
import org.jboss.ide.eclipse.as.ui.views.server.extensions.PropertySheetFactory;
import org.jboss.ide.eclipse.as.ui.views.server.extensions.ServerViewProvider;
import org.jboss.ide.eclipse.as.ui.views.server.extensions.PropertySheetFactory.ISimplePropertiesHolder;
import org.jboss.ide.eclipse.as.ui.views.server.extensions.PropertySheetFactory.SimplePropertiesPropertySheetPage;

/**
 * 
 * @author Rob Stryker <rob.stryker@redhat.com>
 *
 */
public class ExtensionTableViewer extends TreeViewer {

	protected TableViewerPropertySheet propertySheet;
	protected Action disableCategoryAction, refreshAction;
	protected boolean suppressingRefresh = false;
	public ExtensionTableViewer(Tree tree) {
		super(tree);
		setContentProvider(new ContentProviderDelegator());
		setLabelProvider(new LabelProviderDelegator());
		propertySheet = new TableViewerPropertySheet();
		createActions();
		
		UnitedServerListenerManager.getDefault().addListener(new UnitedServerListener() { 
			public void serverRemoved(IServer server) {
				final IServer server2 = server;
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						Object o = getInput();
						if( server2.equals(o)) {
							setInput(null);
						}
					}
				});
			} 
		});
	}

	protected void createActions() {
		disableCategoryAction = new Action() {
			public void run() {
				Display.getDefault().asyncExec(new Runnable() {
					public void run() {
						Object selected = getSelectedElement();
						if( selected instanceof ServerViewProvider) {
							((ServerViewProvider)selected).setEnabled(false);
							refresh();
						}
					} 
				} );
			}
		};
		disableCategoryAction.setText(Messages.DisableCategoryAction);
		refreshAction = new Action() { 
			public void run() {
				Object o1 = ((IStructuredSelection)getSelection()).getFirstElement();
				if( o1 != null ) {
					// tell the delegate to refresh its model for the selected item, before we refresh the viewer
					ServerViewProvider provider = o1 instanceof ServerViewProvider ? (ServerViewProvider)o1 : ((ContentWrapper)o1).getProvider();
					if( provider != null )
						provider.getDelegate().refreshModel(o1);
				}
				Object el = getSelectedElement();
				if( el instanceof ServerViewProvider ) 
					refresh(el);
				else 
					refresh(((IStructuredSelection)getSelection()).getFirstElement());
			}
		};
		refreshAction.setText("Refresh Item");
	}
	
	public static class ContentWrapper {
		private Object o;
		private ServerViewProvider provider;
		
		public ContentWrapper(Object o, ServerViewProvider provider) {
			this.o = o;
			this.provider = provider;
		}

		public Object getElement() {
			return o;
		}

		public ServerViewProvider getProvider() {
			return provider;
		}
		
		public boolean equals(Object other) {
			if( other == null ) return false;
			
			if( other instanceof ContentWrapper ) {
				if( ((ContentWrapper)other).getElement() == null )
					return o == null;
				return ((ContentWrapper)other).getElement().equals(o); 
			}	
			return false;
		}
		
		public int hashCode() {
			if( o == null ) return super.hashCode();
			return o.hashCode();
		}
	}
	
	protected class LabelProviderDelegator extends LabelProvider {
		public String getText(Object obj) {
			if( obj instanceof JBossServer) {
				JBossServer server = (JBossServer)obj;
				String ret = server.getServer().getName(); 
				return ret;
			}
			if( obj instanceof ServerViewProvider) {
				return ((ServerViewProvider)obj).getName();
			}
			
			if( obj instanceof ContentWrapper ) {
				ContentWrapper wrapper = (ContentWrapper)obj;
				return wrapper.getProvider().getDelegate().getLabelProvider().getText(wrapper.getElement());
			}
			return obj.toString();
		}
		public Image getImage(Object obj) {
			if( obj instanceof JBossServer ) {
				return ServerUICore.getLabelProvider().getImage(((JBossServer)obj).getServer());				
			}
			if( obj instanceof ServerViewProvider ) {
				return ((ServerViewProvider)obj).getImage();
			}
			
			if( obj instanceof ContentWrapper ) {
				Object object2 = ((ContentWrapper)obj).getElement();
				return ((ContentWrapper)obj).getProvider().getDelegate().getLabelProvider().getImage(object2);
			}
			
			return null;
		}
	}
	protected class ContentProviderDelegator implements ITreeContentProvider {
		
		public ContentProviderDelegator() {
		}

		public ContentWrapper[] wrap( Object[] elements, ServerViewProvider provider ) {
			if( elements == null ) return new ContentWrapper[0];
			
			ContentWrapper[] wrappers = new ContentWrapper[elements.length];
			for( int i = 0; i < wrappers.length; i++ ) {
				wrappers[i] = new ContentWrapper(elements[i], provider);
			}
			return wrappers;
		}
		
		public Object[] getElements(Object inputElement) {
			if( inputElement == null ) return new Object[0];
			IServer tmp = (IServer)getInput();
			if( tmp == null )
				return new Object[0];
			return JBossServerUIPlugin.getDefault().getEnabledViewProviders(tmp);
		}

		public Object[] getChildren(Object parentElement) {
			if( parentElement == null ) return new Object[0];

			if( parentElement instanceof ServerViewProvider) {
				Object[] ret = ((ServerViewProvider)parentElement).getDelegate().getContentProvider().getChildren(parentElement);
				return wrap(ret, ((ServerViewProvider)parentElement));
			}
			
			if( parentElement instanceof ContentWrapper ) {
				ContentWrapper parentWrapper = (ContentWrapper)parentElement;
				Object[] o = null;
				try {
					o = parentWrapper.getProvider().getDelegate().getContentProvider().getChildren(parentWrapper.getElement());
				} catch( Exception e) {
					JBossServerUIPlugin.log("Error finding children of " + parentElement, e);
				}
				if( o == null ) 
					return new Object[0];
				return wrap(o, parentWrapper.getProvider());
			}
			return new Object[0];
		}

		public Object getParent(Object element) {
			return null;
		}

		public boolean hasChildren(Object element) {
			if( element instanceof ServerViewProvider ) 
				return ((ServerViewProvider)element).getDelegate().getContentProvider().hasChildren(element);
			if( element instanceof ContentWrapper ) {
				ContentWrapper parentWrapper = (ContentWrapper)element;
				return parentWrapper.getProvider().getDelegate().getContentProvider().hasChildren(parentWrapper.getElement());
			}
			return getChildren(element).length > 0 ? true : false;
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			ServerViewProvider[] providers = JBossServerUIPlugin.getDefault().
				getEnabledViewProviders(newInput instanceof IServer ? (IServer)newInput : null);
			for( int i = 0; i < providers.length; i++ ) {
				try {
					providers[i].getDelegate().getContentProvider().inputChanged(viewer, oldInput, newInput);
				} catch( Exception e) {
					JBossServerUIPlugin.log("Error changing input to  " + newInput, e);
				}
			}
		}
		
	}

	public Object getRawElement(Object o) {
		if( o instanceof ContentWrapper )
			return ((ContentWrapper)o).getElement();
		return o;
	}

	public Object[] getRawElements(IStructuredSelection selection) {
		ArrayList<Object> tmp = new ArrayList<Object>();
		Object[] obj = selection.toArray();
		Object tmp2;
		for( int i = 0; i < obj.length; i++ ) {
			tmp2 = getRawElement(obj[i]);
			if( tmp2 != null ) tmp.add(tmp2);
		}
		return (Object[]) tmp.toArray(new Object[tmp.size()]);
	}
	
	public ServerViewProvider getParentViewProvider(Object o) {
		if( o instanceof ContentWrapper ) 
			return ((ContentWrapper)o).getProvider();
		return null;
	}
	
	public Object getSelectedElement() {
		ISelection sel = getSelection();
		if( sel instanceof IStructuredSelection ) {
			Object o = ((IStructuredSelection)sel).getFirstElement(); 
			if( o != null && o instanceof ContentWrapper ) {
				o = ((ContentWrapper)o).getElement();
			}
			return o;
		}
		return null;
	}

	protected void fillSelectedContextMenu(Shell shell, IMenuManager mgr) {
		ISelection sel = getSelection();
		if (sel instanceof IStructuredSelection) {
			ServerViewProvider provider = findProvider((IStructuredSelection)sel);
			if (provider != null) {
				Object[] selected = getRawElements((IStructuredSelection)sel);
				provider.getDelegate().fillContextMenu(shell, mgr, selected);
			}
		}
	}

	protected ServerViewProvider findProvider(IStructuredSelection sel) {
		Object[] selected = sel.toArray();
		ServerViewProvider provider = null;
		ServerViewProvider tmpProvider;
		for( int i = 0; i < selected.length; i++ ) {
			if( selected[i] instanceof ServerViewProvider ) {
				tmpProvider = (ServerViewProvider)selected[i];
			} else {
				tmpProvider = getParentViewProvider(selected[i]);
			}
			if( provider == null ) 
				provider = tmpProvider;
			else if( tmpProvider != provider )
				return null;
		}
		return provider;
	}
	
	protected void fillJBContextMenu(Shell shell, IMenuManager menu) {
		Object selected = getSelectedElement();
		menu.add(refreshAction);
		if( selected instanceof ServerViewProvider ) {
			menu.add(disableCategoryAction);
		}
		
		menu.add(new Separator());
	}

	public IPropertySheetPage getPropertySheet() {
			return propertySheet;
	}
	
	public class TableViewerPropertySheet implements IPropertySheetPage {

		private PageBook book;
		private ArrayList<ServerViewProvider> addedControls = new ArrayList<ServerViewProvider>();
		private SimplePropertiesPropertySheetPage topLevelPropertiesPage;
		
		public void createControl(Composite parent) {
			topLevelPropertiesPage = PropertySheetFactory.createSimplePropertiesSheet(new TopLevelProperties());
			book = new PageBook(parent, SWT.NONE);
			addedControls.clear();
			topLevelPropertiesPage.createControl(book);
			book.showPage(topLevelPropertiesPage.getControl());
		}

		public void dispose() {
		}

		public Control getControl() {
			return book;
		}
		
		public void setActionBars(IActionBars actionBars) {
		}

		public void setFocus() {
		}

		public void selectionChanged(IWorkbenchPart part, ISelection sel) {
			Object selected = getSelectedObject(sel);
			if( selected != null ) {
				
				IPropertySheetPage page = null;
				if( selected instanceof ContentWrapper ) {
					page = getDelegatePage(((ContentWrapper)selected).getProvider());
				} else /*if( selected instanceof ServerViewProvider ) {
					page = getDelegatePage((ServerViewProvider)selected);
				} else if( page == null ) */ { 
					page = topLevelPropertiesPage;
				}
				if( page != null ) {
					page.selectionChanged(part, sel);
					book.showPage(page.getControl());
				}
			}
		}
		private IPropertySheetPage getDelegatePage(ServerViewProvider provider) {
			IPropertySheetPage returnSheet = null;
			returnSheet = provider.getDelegate().getPropertySheetPage();
			if( !addedControls.contains(provider) && returnSheet != null) {
				returnSheet.createControl(book);
				addedControls.add(provider);
			}
			return returnSheet;
		}
		
		public Object getSelectedObject(ISelection sel) {
			if( sel instanceof IStructuredSelection ) {
				IStructuredSelection selection = (IStructuredSelection)sel;
				Object selected = selection.getFirstElement();
				return selected;
			}
			return null;
		}
		
	}
	/**
	 * Properties for the top level elements 
	 *    (a server or a category / extension point 
	 * @author rstryker
	 *
	 */
	class TopLevelProperties implements ISimplePropertiesHolder {
		public Properties getProperties(Object selected) {
			Properties ret = new Properties();
			if( selected instanceof ServerViewProvider ) {
				ServerViewProvider provider = (ServerViewProvider)selected;
				ret.setProperty(Messages.ExtensionID, provider.getId());
				ret.setProperty(Messages.ExtensionName, provider.getName());
				ret.setProperty(Messages.ExtensionDescription, provider.getDescription());
				ret.setProperty(Messages.ExtensionProviderClass, provider.getDelegateName());
			}
			return ret;
		}


		public String[] getPropertyKeys(Object selected) {
			if( selected instanceof ServerViewProvider) {
				return new String[] { 
						Messages.ExtensionName, Messages.ExtensionDescription, 
						Messages.ExtensionID, Messages.ExtensionProviderClass
				};
				
			}
			if( selected instanceof JBossServer ) {
				return new String[] {
						Messages.ServerRuntimeVersion, Messages.ServerHome, 
						Messages.ServerConfigurationName, Messages.ServerDeployDir,
				};
			}
			return new String[] {};
		}
	}

	public void dispose() {
		// The Loner
		propertySheet.dispose();
		
		ServerViewProvider[] providers = JBossServerUIPlugin.getDefault().getAllServerViewProviders();
		for( int i = 0; i < providers.length; i++ ) {
			providers[i].dispose();
		}
	}

	
	public void suppressingRefresh(Runnable runnable) {
		boolean currentVal = suppressingRefresh;
		suppressingRefresh = true;
		runnable.run();
		suppressingRefresh = currentVal;
	}

	public void refresh() {
		refresh(null);
	}
	public void refresh(final Object element) {
		if(!suppressingRefresh) {
			try {
				super.refresh(element);
			} catch( Exception e ) {
				JBossServerUIPlugin.log("Error refreshing view. ", e);
			}
		} 
	}
	
	public boolean elementInTree(Object element) {
		Widget[] w = super.findItems(element);
		return w != null && w.length > 0;
	}


}
