package org.jbpm.gd.jpdl.ui.properties;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Assignment;
import org.jbpm.gd.jpdl.model.JpdlElementFactory;
import org.jbpm.gd.jpdl.model.Task;

public class TaskConfigurationComposite implements SelectionListener, FocusListener {
	
	public static TaskConfigurationComposite create(TabbedPropertySheetWidgetFactory widgetFactory, Composite parent) {
		TaskConfigurationComposite result = new TaskConfigurationComposite();
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}
	
	private TabbedPropertySheetWidgetFactory widgetFactory;
	private Composite parent;
	
	private Group taskInfoGroup;
	private Group assignmentInfoGroup;
	
	private Label taskNameLabel;
	private Text taskNameText;
	private Label taskDescriptionLabel;
	private Text taskDescriptionText;
	private Button taskBlockingButton;
	private Button taskSignallingButton;
	private Button taskNotifyButton;
	private Label taskPriorityLabel;
	private CCombo taskPriorityCombo;
	private Label assignmentTypeLabel;
	private CCombo assignmentTypeCombo;
	
	
	private AssignmentInformationPageBook assignmentInfoPageBook;	
	private Composite swimlaneInfoComposite;
	private Text swimlaneNameText;
	private Label swimlaneNameLabel;
	
    private Task task;
    private String assignmentTypeComboSelection = "";    
	
	private TaskConfigurationComposite() {}
	
	public void setTask(Task task) {
		if (this.task == task) return;
		unhookListeners();
		this.task = task;
		clearControls();
		if (task != null) {
			updateControls();
			hookListeners();
		}
	}
	
	private void hookListeners() {
		taskNameText.addSelectionListener(this);
		taskNameText.addFocusListener(this);
		assignmentTypeCombo.addSelectionListener(this);
		taskPriorityCombo.addSelectionListener(this);
		taskBlockingButton.addSelectionListener(this);
		taskSignallingButton.addSelectionListener(this);
		taskNotifyButton.addSelectionListener(this);
		taskDescriptionText.addSelectionListener(this);
		taskDescriptionText.addFocusListener(this);
		swimlaneNameText.addSelectionListener(this);
		swimlaneNameText.addFocusListener(this);
	}
	
	private void unhookListeners() {
		taskNameText.removeSelectionListener(this);
		taskNameText.removeFocusListener(this);
		assignmentTypeCombo.removeSelectionListener(this);
		taskPriorityCombo.removeSelectionListener(this);
		taskBlockingButton.removeSelectionListener(this);
		taskSignallingButton.removeSelectionListener(this);
		taskNotifyButton.removeSelectionListener(this);
		taskDescriptionText.removeSelectionListener(this);
		taskDescriptionText.removeFocusListener(this);
		swimlaneNameText.removeSelectionListener(this);
		swimlaneNameText.removeFocusListener(this);
	}
	
	private void clearControls() {
 		taskNameText.setText("");
 		taskNameText.setEnabled(false);
 		assignmentTypeCombo.setText(""); 
 		assignmentTypeCombo.setEnabled(false);
 		taskPriorityCombo.setText("Normal");
 		taskPriorityCombo.setEnabled(false);
 		taskBlockingButton.setSelection(false);
 		taskBlockingButton.setEnabled(false);
 		taskSignallingButton.setSelection(true);
 		taskSignallingButton.setEnabled(false);
 		taskNotifyButton.setSelection(false);
 		taskNotifyButton.setEnabled(false);
 		taskDescriptionText.setText("");
 		taskDescriptionText.setEnabled(false);
 		assignmentInfoPageBook.clear();
 		assignmentInfoPageBook.setVisible(false);
 		swimlaneNameText.setText("");
 		swimlaneInfoComposite.setVisible(false);
	}
	
	private void updateControls() {
		taskNameText.setText(task.getName() == null ? "" : task.getName());
		taskNameText.setEnabled(true);
		assignmentTypeCombo.setEnabled(true);
		if (task.getSwimlane() != null) {
			swimlaneNameText.setText(task.getSwimlane());
			assignmentTypeCombo.setText("Swimlane");
			swimlaneInfoComposite.setVisible(true);
		} else if (task.getAssignment() != null) {
			Assignment assignment = task.getAssignment();
	 		if (assignment.getExpression() != null) {
	 			assignmentTypeCombo.setText("Expression");
	 		} else if (assignment.getActorId() != null) {
	 			assignmentTypeCombo.setText("Actor");
	 		} else if (assignment.getPooledActors() != null) {
	 			assignmentTypeCombo.setText("Pooled Actors");
	 		} else if (assignment.getClassName() != null) {
	 			assignmentTypeCombo.setText("Handler");
	 		}
	 		assignmentInfoPageBook.setAssignment(assignment);
	 		assignmentInfoPageBook.setVisible(true);
		}
		taskPriorityCombo.setText(replaceFirstWithUpperCase(task.getPriority()));
		taskPriorityCombo.setEnabled(true);
		taskBlockingButton.setSelection("true".equals(task.getBlocking()));
		taskBlockingButton.setEnabled(true);
		taskSignallingButton.setSelection("true".equals(task.getSignalling()));
		taskSignallingButton.setEnabled(true);
		taskNotifyButton.setSelection("true".equals(task.getNotify()));
		taskNotifyButton.setEnabled(true);
		taskDescriptionText.setText(task.getDescription() == null ? "" : task.getDescription());
		taskDescriptionText.setEnabled(true);
	}
	
	private String replaceFirstWithUpperCase(String str) {
		if (str == null) return null;
		StringBuffer result = new StringBuffer();
		result.append(str.substring(0, 1).toUpperCase());
		result.append(str.substring(1));
		return result.toString();
	}
	
	private void create() {
		createTaskInfoArea();
		createAssignmentInfoArea();
		clearControls();
	}
	
	private void createAssignmentInfoArea() {
		assignmentInfoGroup = widgetFactory.createGroup(parent, "Assignment Information");
		assignmentInfoGroup.setLayout(new FormLayout());
		assignmentInfoPageBook = AssignmentInformationPageBook.create(widgetFactory, assignmentInfoGroup);
		assignmentInfoGroup.setLayoutData(createAssignmentInfoGroupLayoutData());
		assignmentInfoPageBook.setVisible(false);
		swimlaneInfoComposite = widgetFactory.createComposite(assignmentInfoGroup);
		swimlaneInfoComposite.setLayoutData(createSwimlaneInfoCompositeLayoutData());
		swimlaneInfoComposite.setLayout(new FormLayout());
		swimlaneNameLabel = widgetFactory.createLabel(swimlaneInfoComposite, "Swimlane Name");
		swimlaneNameText = widgetFactory.createText(swimlaneInfoComposite, "");
		swimlaneNameLabel.setLayoutData(createSwimlaneNameLabelLayoutData());
		swimlaneNameText.setLayoutData(createSwimlaneNameTextLayoutData());
		swimlaneInfoComposite.setVisible(false);
	}
	
	private FormData createAssignmentInfoGroupLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(taskInfoGroup, 5);
		data.right = new FormAttachment(100, -5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createSwimlaneInfoCompositeLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.top = new FormAttachment(0, 0);
		data.bottom = new FormAttachment(100, 0);
		return data;
	}
	
	private FormData createSwimlaneNameLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createSwimlaneNameTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(swimlaneNameLabel, 5);
		data.top = new FormAttachment(0, 5);
		data.right = new FormAttachment(100, -5);
		return data;
	}
	
	private void createTaskInfoArea() {
		taskInfoGroup = widgetFactory.createGroup(parent, "Task Information");	
		taskInfoGroup.setLayout(new FormLayout());
		taskNameLabel = widgetFactory.createLabel(taskInfoGroup, "Task Name");
		taskNameText = widgetFactory.createText(taskInfoGroup, "");
		taskPriorityLabel = widgetFactory.createLabel(taskInfoGroup, "Priority");
		taskPriorityCombo = widgetFactory.createCCombo(taskInfoGroup, SWT.READ_ONLY | SWT.BORDER);
		taskBlockingButton = widgetFactory.createButton(taskInfoGroup, "Blocking", SWT.CHECK);
		taskSignallingButton = widgetFactory.createButton(taskInfoGroup, "Signalling", SWT.CHECK);
		taskNotifyButton = widgetFactory.createButton(taskInfoGroup, "Notify", SWT.CHECK);
		taskDescriptionLabel = widgetFactory.createLabel(taskInfoGroup, "Description");
		taskDescriptionText = widgetFactory.createText(taskInfoGroup, "");
		assignmentTypeLabel = widgetFactory.createLabel(taskInfoGroup, "Assignment Type");
		assignmentTypeCombo = widgetFactory.createCCombo(taskInfoGroup, SWT.READ_ONLY | SWT.BORDER);
		assignmentTypeCombo.setItems(getAssignmentTypes());
		assignmentTypeCombo.select(0);
		taskInfoGroup.setLayoutData(createTaskInfoGroupLayoutData());
		taskNameLabel.setLayoutData(createTaskNameLabelLayoutData());
		taskNameText.setLayoutData(createTaskNameTextLayoutData());
		taskPriorityLabel.setLayoutData(createTaskPriorityLabelLayoutData());
		taskPriorityCombo.setLayoutData(createTaskPriorityComboLayoutData());
		taskPriorityCombo.setItems(getPriorityTypes());
		taskPriorityCombo.setText("Normal");
		taskBlockingButton.setLayoutData(createTaskBlockingButtonLayoutData());
		taskSignallingButton.setLayoutData(createTaskSignallingButtonLayoutData());
		taskNotifyButton.setLayoutData(createTaskNotifyButtonLayoutData());
		taskDescriptionLabel.setLayoutData(createTaskDescriptionLabelLayoutData());
		taskDescriptionText.setLayoutData(createTaskDescriptionTextLayoutData());
		assignmentTypeLabel.setLayoutData(createAssignmentTypeLabelLayoutData());
		assignmentTypeCombo.setLayoutData(createAssignmentTypeComboLayoutData());
	}
	
	private FormData createTaskDescriptionTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskDescriptionLabel, 5);
		data.top = new FormAttachment(taskPriorityCombo, 5);
		data.right = new FormAttachment(100, -5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createTaskDescriptionLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(taskPriorityCombo, 5);
		return data;
	}
	
	private FormData createTaskNotifyButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskSignallingButton, 5);
		data.top = new FormAttachment(0, 7);
		return data;
	}
	
	private FormData createTaskSignallingButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskBlockingButton, 5);
		data.top = new FormAttachment(0, 7);
		return data;
	}
	
	private FormData createTaskBlockingButtonLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskPriorityCombo, 5);
		data.top = new FormAttachment(0, 7);
		return data;
	}
	
	private FormData createTaskPriorityComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskPriorityLabel, 5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createTaskPriorityLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(assignmentTypeCombo, 5);
		data.top = new FormAttachment(0, 7);
		return data;
	}
	
	private FormData createTaskInfoGroupLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(0, 5);
		data.right = new FormAttachment(100, -5);
		return data;
	}
	
	private FormData createTaskNameLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createTaskNameTextLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskDescriptionLabel, 5);
		data.top = new FormAttachment(0, 5);
		data.right = new FormAttachment(35, 0);
		return data;
	}
	
	private FormData createAssignmentTypeLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(taskNameText, 5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private FormData createAssignmentTypeComboLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(assignmentTypeLabel, 5);
		data.top = new FormAttachment(0, 5);
		return data;
	}
	
	private String[] getAssignmentTypes() {
		return new String[] { "", "Swimlane", "Expression", "Actor", "Pooled Actors", "Handler" };
	}

	private String[] getPriorityTypes() {
		return new String[] { "Highest", "High", "Normal", "Low", "Lowest" };
	}

	private void handleAssignmentComboSelected() {
		handleAssignmentComboDefaultSelected();
	}
	
	private void handleAssignmentComboDefaultSelected() {
		String text = assignmentTypeCombo.getText();
		if (assignmentTypeComboSelection.equals(text)) return;
		assignmentTypeComboSelection = text;
		if ("".equals(text)) {
			task.setAssignment(null);
			task.setSwimlane(null);
			assignmentInfoPageBook.setVisible(false);
			swimlaneInfoComposite.setVisible(false);
		} else if ("Swimlane".equals(text)) {
			task.setAssignment(null);
			task.setSwimlane(swimlaneNameText.getText());
			assignmentInfoPageBook.setVisible(false);
			swimlaneInfoComposite.setVisible(true);
		} else {
			task.setSwimlane(null);
			if (task.getAssignment() == null) {
				Assignment assignment = (Assignment)JpdlElementFactory.INSTANCE.createById("org.jbpm.gd.jpdl.core.assignment");
				task.setAssignment(assignment);
				assignmentInfoPageBook.setAssignment(assignment);
		    }
			assignmentInfoPageBook.updateAssignmentInfo(text);
			assignmentInfoPageBook.setVisible(true);
			swimlaneInfoComposite.setVisible(false);
		}
	}

	private void updateTaskName() {
		task.setName(taskNameText.getText());
	}
	
	private void updateTaskDescription() {
		task.setDescription(taskDescriptionText.getText());
	}
	
	private void updateTaskSwimlane() {
		task.setSwimlane(swimlaneNameText.getText());
	}

	private void updateTaskPriority() {
		task.setPriority(taskPriorityCombo.getText().toLowerCase());
	}

	private void updateTaskBlocking() {
		task.setBlocking(taskBlockingButton.getSelection() ? "true" : "false");
	}

	private void updateTaskSignalling() {
		task.setSignalling(taskSignallingButton.getSelection() ? "true" : "false");
	}

	private void updateTaskNotify() {
		task.setNotify(taskNotifyButton.getSelection() ? "true" : "false");
	}

	public void widgetDefaultSelected(SelectionEvent e) {
		if (e.widget == taskNameText) {
			updateTaskName();
		} else if (e.widget == taskDescriptionText) {
			updateTaskDescription();
		} else if (e.widget == swimlaneNameText) {
			updateTaskSwimlane();
		}
	}
	
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == assignmentTypeCombo) {
			handleAssignmentComboSelected();
		} else if (e.widget == taskPriorityCombo) {
			updateTaskPriority();
		} else if (e.widget == taskBlockingButton) {
			updateTaskBlocking();
		} else if (e.widget == taskSignallingButton) {
			updateTaskSignalling();
		} else if (e.widget == taskNotifyButton) {
			updateTaskNotify();
		}
	}
	
	public void focusGained(FocusEvent e) {
	}

	public void focusLost(FocusEvent e) {
		if (e.widget == taskNameText) {
			updateTaskName();
		} else if (e.widget == taskDescriptionText) {
			updateTaskDescription();
		} else if (e.widget == swimlaneNameText) {
			updateTaskSwimlane();
		}
	}	
	
}
