/******************************************************************************* 
 * Copyright (c) 2007 Red Hat, Inc. 
 * Distributed under license by Red Hat, Inc. All rights reserved. 
 * This program is made available under the terms of the 
 * Eclipse Public License v1.0 which accompanies this distribution, 
 * and is available at http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Red Hat, Inc. - initial API and implementation 
 ******************************************************************************/ 
package org.jboss.tools.cdi.core;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IProjectNature;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaModelException;
import org.jboss.tools.cdi.core.extension.CDIExtensionManager;
import org.jboss.tools.cdi.internal.core.impl.definition.AnnotationDefinition;
import org.jboss.tools.cdi.internal.core.impl.definition.BeansXMLDefinition;
import org.jboss.tools.cdi.internal.core.impl.definition.DefinitionContext;
import org.jboss.tools.cdi.internal.core.impl.definition.TypeDefinition;
import org.jboss.tools.cdi.internal.core.scanner.lib.ClassPathMonitor;
import org.jboss.tools.common.java.ParametedTypeFactory;
import org.jboss.tools.common.model.XJob;
import org.jboss.tools.common.model.XJob.XRunnable;
import org.jboss.tools.common.model.util.EclipseResourceUtil;
import org.jboss.tools.common.util.EclipseJavaUtil;
import org.jboss.tools.common.util.FileUtil;
import org.jboss.tools.common.validation.internal.ProjectValidationContext;
import org.jboss.tools.jst.web.kb.KbProjectFactory;
import org.jboss.tools.jst.web.kb.internal.KbProject;

public class CDICoreNature implements IProjectNature {
	public static String NATURE_ID = "org.jboss.tools.cdi.core.cdinature";

	IProject project = null;
	ICDIProject cdiProjectDelegate;

	ParametedTypeFactory typeFactory = new ParametedTypeFactory();

	ClassPathMonitor classPath = new ClassPathMonitor(this);
	DefinitionContext definitions = new DefinitionContext();

	ProjectValidationContext validationContext = null;

	boolean isBuilt = false;

//	Map<IPath, Object> sourcePaths2 = new HashMap<IPath, Object>(); //TODO

	private boolean isStorageResolved = false;

	Set<CDICoreNature> dependsOn = new HashSet<CDICoreNature>();
	
	Set<CDICoreNature> usedBy = new HashSet<CDICoreNature>();

	private CDIExtensionManager extensions = new CDIExtensionManager();
	
	public CDICoreNature() {
		extensions.setProject(this);
		definitions.setProject(this);
	}

	public void configure() throws CoreException {
		addToBuildSpec(CDICoreBuilder.BUILDER_ID);
	}

	public void deconfigure() throws CoreException {
		removeFromBuildSpec(CDICoreBuilder.BUILDER_ID);
		dispose();
	}

	public IProject getProject() {
		return project;
	}

	/**
	 * Convenience method.
	 * 
	 * @param qualifiedName
	 * @return
	 */
	public IType getType(String qualifiedName) {
		IJavaProject jp = EclipseResourceUtil.getJavaProject(getProject());
		if(jp == null) return null;
		try {
			return EclipseJavaUtil.findType(jp, qualifiedName);
		} catch (JavaModelException e) {
			CDICorePlugin.getDefault().logError(e);
		}
		return null;
	}

	public void setProject(IProject project) {
		this.project = project;
		classPath.init();
	}

	public void setCDIProject(ICDIProject cdiProject) {
		this.cdiProjectDelegate = cdiProject;
		cdiProject.setNature(this);
	}

	public Set<CDICoreNature> getCDIProjects() {
		return getCDIProjects(false);
	}

	public CDIExtensionManager getExtensionManager() {
		return extensions;
	}

	/**
	 * Returns all the project that are included into classpath of this project.
	 * @param hierarchy If false then return the projects explicitly included into the project classpath.
	 * If true then all the project from the entire hierarchy will be returned.
	 * @return
	 */
	public Set<CDICoreNature> getCDIProjects(boolean hierarchy) {
		if(hierarchy) {
			if(dependsOn.isEmpty()) return dependsOn;
			Set<CDICoreNature> result = new HashSet<CDICoreNature>();
			getAllCDIProjects(result);
			return result;
		} else {
			return dependsOn;
		}
	}

	void getAllCDIProjects(Set<CDICoreNature> result) {
		for (CDICoreNature n:dependsOn) {
			if(result.contains(n)) continue;
			result.add(n);
			n.getAllCDIProjects(result);
		}
	}

	public List<TypeDefinition> getAllTypeDefinitions() {
		Set<CDICoreNature> ps = getCDIProjects(true);
		if(ps == null || ps.isEmpty()) {
			return getDefinitions().getTypeDefinitions();
		}
		List<TypeDefinition> ds = getDefinitions().getTypeDefinitions();
		List<TypeDefinition> result = new ArrayList<TypeDefinition>();
		result.addAll(ds);
		Set<String> keys = new HashSet<String>();
		for (TypeDefinition d: ds) {
			keys.add(d.getKey());
		}
		for (CDICoreNature p: ps) {
			List<TypeDefinition> ds2 = p.getDefinitions().getTypeDefinitions();
			for (TypeDefinition d: ds2) {
				String key = d.getKey();
				if(!keys.contains(key)) {
					keys.add(key);
					result.add(d);
				}
			}
		}
		return result;
	}

	public List<AnnotationDefinition> getAllAnnotations() {
		Set<CDICoreNature> ps = getCDIProjects(false);
		if(ps == null || ps.isEmpty() || getCDIProjects(true).contains(this)) {
			return getDefinitions().getAllAnnotations();
		}
		List<AnnotationDefinition> result = new ArrayList<AnnotationDefinition>();
		Set<IType> types = new HashSet<IType>();
		for (CDICoreNature p: ps) {
			List<AnnotationDefinition> ds2 = p.getAllAnnotations();
			for (AnnotationDefinition d: ds2) {
				IType t = d.getType();
				if(t != null && !types.contains(t)) {
					types.add(t);
					result.add(d);
				}
			}
		}

		List<AnnotationDefinition> ds = getDefinitions().getAllAnnotations();
		for (AnnotationDefinition d: ds) {
			IType t = d.getType();
			if(t != null && !types.contains(t)) {
				types.add(t);
				result.add(d);
			}
		}

		return result;
	}

	/**
	 * Returns set of types that were to be marked as vetoed by CDI extensions, but 
	 * for which it was impossible to set isVetoed=true on the type definition object,
	 * because type is declared in another project where it is not vetoed.
	 * 
	 * @return
	 */
	public Set<String> getAllVetoedTypes() {
		Set<String> result = new HashSet<String>();
		result.addAll(definitions.getVetoedTypes());
		Set<CDICoreNature> ps = getCDIProjects(true);
		for (CDICoreNature n: ps) {
			result.addAll(n.getDefinitions().getVetoedTypes());
		}		
		return result;
	}

	public Set<BeansXMLDefinition> getAllBeanXMLDefinitions() {
		Set<CDICoreNature> ps = getCDIProjects(true);
		if(ps == null || ps.isEmpty()) {
			return getDefinitions().getBeansXMLDefinitions();
		}
		Set<BeansXMLDefinition> ds = getDefinitions().getBeansXMLDefinitions();
		Set<BeansXMLDefinition> result = new HashSet<BeansXMLDefinition>();
		result.addAll(ds);
		Set<IPath> paths = new HashSet<IPath>();
		for (BeansXMLDefinition d: ds) {
			IPath t = d.getPath();
			if(t != null) paths.add(t);
		}
		for (CDICoreNature p: ps) {
			Set<BeansXMLDefinition> ds2 = p.getDefinitions().getBeansXMLDefinitions();
			for (BeansXMLDefinition d: ds2) {
				IPath t = d.getPath();
				if(t != null && !paths.contains(t)) {
					paths.add(t);
					result.add(d);
				}
			}
		}
		return result;
	}

	/**
	 * Returns all the CDI projects that include this project into their class path.
	 * @return
	 */
	public Set<CDICoreNature> getDependentProjects() {
		return usedBy;
	}

	public CDICoreNature[] getAllDependentProjects(boolean resolve) {
		Map<CDICoreNature, Integer> set = new HashMap<CDICoreNature, Integer>();
		getAllDependentProjects(set, 0);
		if(resolve) {
			for (CDICoreNature n: set.keySet()) {
				n.resolve();
			}
		}
		CDICoreNature[] result = set.keySet().toArray(new CDICoreNature[set.size()]);
		Arrays.sort(result, new D(set));
		return result;
	}

	public CDICoreNature[] getAllDependentProjects() {
		return getAllDependentProjects(false);
	}

	private void getAllDependentProjects(Map<CDICoreNature, Integer> result, int level) {
		if(level > 10) return;
		for (CDICoreNature n:usedBy) {
			if(!result.containsKey(n) || result.get(n).intValue() < level) {
				result.put(n, level);
				n.getAllDependentProjects(result, level + 1);
			}
		}
	}
	private static class D implements Comparator<CDICoreNature> {
		Map<CDICoreNature, Integer> set;
		D(Map<CDICoreNature, Integer> set) {
			this.set = set;
		}
		@Override
		public int compare(CDICoreNature o1, CDICoreNature o2) {
			return set.get(o1).intValue() - set.get(o2).intValue();
		}
		
	}

	public void addCDIProject(final CDICoreNature p) {
		if(dependsOn.contains(p)) return;
		addUsedCDIProject(p);
		p.addDependentCDIProject(this);
		//TODO
		if(!p.isStorageResolved()) {
			XJob.addRunnableWithPriority(new XRunnable() {
				public void run() {
					p.resolve();
					if(p.getDelegate() != null) {
						p.getDelegate().update(true);
					}
				}
				
				public String getId() {
					return "Build CDI Project " + p.getProject().getName();
				}
			});
		}
	}

	public void removeCDIProject(CDICoreNature p) {
		if(!dependsOn.contains(p)) return;
		p.usedBy.remove(this);
		synchronized (dependsOn) {
			dependsOn.remove(p);
		}
		//TODO
	}

	void addUsedCDIProject(CDICoreNature p) {
		synchronized (dependsOn) {
			dependsOn.add(p);
		}
	}

	public void addDependentCDIProject(CDICoreNature p) {
		usedBy.add(p);
	}

	public DefinitionContext getDefinitions() {
		return definitions;
	}

	public ICDIProject getDelegate() {
		return cdiProjectDelegate;
	}

	public ParametedTypeFactory getTypeFactory() {
		return typeFactory;
	}

	public ClassPathMonitor getClassPath() {
		return classPath;
	}

	public boolean isStorageResolved() {
		return isStorageResolved;
	}
	/**
	 * 
	 * @param load
	 */
	public void resolveStorage(boolean load) {
		if(isStorageResolved) return;
		if(load) {
			load();
		} else {
			loadProjectDependenciesFromKBProject();
			synchronized(this) {
				isStorageResolved = true;
			}
		}
	}

	/**
	 * 
	 */
	public void resolve() {
		resolveStorage(true);
	}

	/**
	 * Loads results of last build, which are considered 
	 * actual until next build.
	 */	
	public void load() {
		if(isStorageResolved) return;
		synchronized(this) {
			if(isStorageResolved) return;
			isStorageResolved = true;
		}
		try {
			new CDICoreBuilder(this);
		} catch (CoreException e) {
			CDICorePlugin.getDefault().logError(e);
		}
		
		postponeFiring();
		
		try {		
//			boolean b = getClassPath().update();
//			if(b) {
//				getClassPath().validateProjectDependencies();
//			}
//			File file = getStorageFile();

			//Use kb storage for dependent projects since cdi is not stored.
			loadProjectDependenciesFromKBProject();
			//TODO

//			if(b) {
//				getClassPath().process();
//			}

		} finally {
			fireChanges();
		}
	}

	public void clean() {
		File file = getStorageFile();
		if(file != null && file.isFile()) {
			file.delete();
		}
		isBuilt = false;
		classPath.clean();
		postponeFiring();

		definitions.clean();
		if(cdiProjectDelegate != null) {
			cdiProjectDelegate.update(true);
		}
//		IPath[] ps = sourcePaths2.keySet().toArray(new IPath[0]);
//		for (int i = 0; i < ps.length; i++) {
//			pathRemoved(ps[i]);
//		}
		fireChanges();
	}

	public void cleanTypeFactory() {
		typeFactory.clean();
		CDICoreNature[] ps = getAllDependentProjects();
		for (CDICoreNature n: ps) {
			n.cleanTypeFactory();
		}
	}
	
	/**
	 * Stores results of last build, so that on exit/enter Eclipse
	 * load them without rebuilding project
	 * @throws IOException 
	 */
	public void store() throws IOException {
		isBuilt = true;
		File file = getStorageFile();
//TODO
//		file.getParentFile().mkdirs();
	}
	/**
	 * 
	 * @param builderID
	 * @throws CoreException
	 */
	protected void addToBuildSpec(String builderID) throws CoreException {
		IProjectDescription description = getProject().getDescription();
		ICommand command = null;
		ICommand commands[] = description.getBuildSpec();
		for (int i = 0; i < commands.length && command == null; ++i) {
			if (commands[i].getBuilderName().equals(builderID)) 
				command = commands[i];
		}
		if (command == null) {
			command = description.newCommand();
			command.setBuilderName(builderID);
			ICommand[] oldCommands = description.getBuildSpec();
			ICommand[] newCommands = new ICommand[oldCommands.length + 1];
			System.arraycopy(oldCommands, 0, newCommands, 0, oldCommands.length);
			newCommands[oldCommands.length] = command;
			description.setBuildSpec(newCommands);
			getProject().setDescription(description, null);
		}
	}

	/**
	 * 
	 */
	static String EXTERNAL_TOOL_BUILDER = "org.eclipse.ui.externaltools.ExternalToolBuilder"; //$NON-NLS-1$
	
	/**
	 * 
	 */
	static final String LAUNCH_CONFIG_HANDLE = "LaunchConfigHandle"; //$NON-NLS-1$

	/**
	 * 
	 * @param builderID
	 * @throws CoreException
	 */
	protected void removeFromBuildSpec(String builderID) throws CoreException {
		IProjectDescription description = getProject().getDescription();
		ICommand[] commands = description.getBuildSpec();
		for (int i = 0; i < commands.length; ++i) {
			String builderName = commands[i].getBuilderName();
			if (!builderName.equals(builderID)) {
				if(!builderName.equals(EXTERNAL_TOOL_BUILDER)) continue;
				Object handle = commands[i].getArguments().get(LAUNCH_CONFIG_HANDLE);
				if(handle == null || handle.toString().indexOf(builderID) < 0) continue;
			}
			ICommand[] newCommands = new ICommand[commands.length - 1];
			System.arraycopy(commands, 0, newCommands, 0, i);
			System.arraycopy(commands, i + 1, newCommands, i, commands.length - i - 1);
			description.setBuildSpec(newCommands);
			getProject().setDescription(description, null);
			return;
		}
	}

	/*
	 * 
	 */
	private File getStorageFile() {
		IPath path = CDICorePlugin.getDefault().getStateLocation();
		File file = new File(path.toFile(), "projects/" + project.getName()); //$NON-NLS-1$
		return file;
	}
	
	public void clearStorage() {
		File f = getStorageFile();
		if(f == null || !f.exists()) return;
		FileUtil.clear(f);
		f.delete();
	}

	public boolean hasNoStorage() {
			if(isBuilt) return false;
		File f = getStorageFile();
		return f == null || !f.exists();
	}

	public void postponeFiring() {
		//TODO
	}

	public void fireChanges() {
		//TODO
	}

	public long fullBuildTime;
	public List<Long> statistics;

	public void pathRemoved(IPath source) {
//		sourcePaths2.remove(source);
		definitions.getWorkingCopy().clean(source);
		//TODO
	}

	public ProjectValidationContext getValidationContext() {
		if(validationContext==null) {
			validationContext = new ProjectValidationContext();
		}
		return validationContext;
	}

	/**
	 * Test method.
	 */
	public void reloadProjectDependencies() {
		dependsOn.clear();
		usedBy.clear();
		synchronized (this) {
			projectDependenciesLoaded = false;
		}
		loadProjectDependenciesFromKBProject();
	}

	boolean projectDependenciesLoaded = false;

	public void loadProjectDependencies() {
		loadProjectDependenciesFromKBProject();
	}

	private void loadProjectDependenciesFromKBProject() {
		if(projectDependenciesLoaded) return;
		synchronized(this) {
			if(projectDependenciesLoaded) return;
			projectDependenciesLoaded = true;
		}
		_loadProjectDependencies();
	}
	
	private void _loadProjectDependencies() {
		KbProject kb = (KbProject)KbProjectFactory.getKbProject(project, true, true);

		if(kb == null) {
			return;
		}
		
		Set<KbProject> ps = kb.getKbProjects();
		
		for (KbProject kb1: ps) {
			IProject project = kb1.getProject();
			if(project == null || !project.isAccessible()) continue;
			KbProjectFactory.getKbProject(project, true, true);
			CDICoreNature sp = CDICorePlugin.getCDI(project, false);
			if(sp != null) {
				addUsedCDIProject(sp);
				sp.addDependentCDIProject(this);
			}
		}
		
		KbProject[] ps2 = kb.getDependentKbProjects();

		for (KbProject kb2: ps2) {
			IProject project = kb2.getProject();
			if(project == null || !project.isAccessible()) continue;
			KbProjectFactory.getKbProject(project, true, true);
			CDICoreNature sp = CDICorePlugin.getCDI(project, false);
			if(sp != null) {
				addDependentCDIProject(sp);
			}
		}
	}

	public void dispose() {
		CDICoreNature[] ds = dependsOn.toArray(new CDICoreNature[dependsOn.size()]);
		for (CDICoreNature d: ds) {
			removeCDIProject(d);
		}
		CDICoreNature[] us = usedBy.toArray(new CDICoreNature[usedBy.size()]);
		for (CDICoreNature u: us) {
			u.removeCDIProject(this);
		}
	}

}