package org.jboss.eap.util.xp.patch.stream.manager;

import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.Manifest;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ManagerManifestConfig {
    private static final String PATCH_STREAM_KEYS = "patch-stream-keys";
    private static final String CURRENT_PATCH_STREAM_KEY = "current-patch-stream-key";

    private static final String XP_MANAGER_VERSION = "eap-xp-manager-version";


    private static final String VERSION_SUFFIX = "version";
    private static final String PATCH_STREAMS_SUFFIX = "patch-streams";
    private static final String SERVER_TARGET_LAYERS_SUFFIX = "layers";
    private static final String MINIMUM_BASE_SERVER_VERSION_SUFFIX = "minimum-base-server-version";
    private static final String PATCH_NAME_PATTERNS_SUFFIX = "patch-name-patterns";

    static final ManagerManifestConfig INSTANCE;

    static {
        try {
            INSTANCE = ManagerManifestConfig.create();
        } catch (Exception e) {
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new RuntimeException(e);
        }
    }

    private final String managerVersion;
    private final List<String> patchStreamKeys;
    private final String currentPatchStreamKey;
    private final Map<String, XPConfig> xpConfigs;

    private ManagerManifestConfig(String managerVersion, List<String> patchStreamKeys, String currentPatchStreamKey, Map<String, XPConfig> xpConfigs) {
        this.managerVersion = managerVersion;
        this.patchStreamKeys = patchStreamKeys;
        this.currentPatchStreamKey = currentPatchStreamKey;
        this.xpConfigs = xpConfigs;
    }

    private static ManagerManifestConfig create() throws Exception {
        String managerVersion = readValue(XP_MANAGER_VERSION);
        List<String> patchStreamKeys = readList(PATCH_STREAM_KEYS);
        String currentPatchStreamKey = readValue(CURRENT_PATCH_STREAM_KEY);

        Map<String, XPConfig> xpConfigs = new LinkedHashMap<>();
        for (String key : patchStreamKeys) {
            String xpName = readValue(key + "-" + VERSION_SUFFIX);
            ServerVersion minimumBaseServerVersion = ServerVersion.parse(readValue(key + "-" + MINIMUM_BASE_SERVER_VERSION_SUFFIX));
            List<String> streamNames = readList(key + "-" + PATCH_STREAMS_SUFFIX);
            List<String> moduleLayerNames = readList(key + "-" + SERVER_TARGET_LAYERS_SUFFIX);
            List<String> patchNamePatterns = readList(key + "-" + PATCH_NAME_PATTERNS_SUFFIX);
            XPConfig xpConfig = new XPConfig(key, xpName, minimumBaseServerVersion, streamNames, moduleLayerNames, patchNamePatterns);
            xpConfigs.put(key, xpConfig);
        }
        return new ManagerManifestConfig(managerVersion, patchStreamKeys, currentPatchStreamKey, xpConfigs);
    }

    private static List<String> readList(String name) throws Exception {
        String input = readValue(name);
        if (input == null) {
            return Collections.emptyList();
        }
        List<String> list = new ArrayList<>();
        String[] tokens = input.split(",");
        for (String layer : tokens) {
            layer = layer.trim();
            if (layer.length() > 0) {
                list.add(layer);
            }
        }
        return list;
    }

    private static String readValue(String name) throws Exception {
        // Doing a simple ManifestUtils.class.getClassLoader().getResource("META-INF/MANIFEST.MF") doesn't always work
        // since it sometimes first tries to load from jar:file:/System/Library/Java/Extensions/MRJToolkit.jar!/META-INF/MANIFEST.MF
        for (Enumeration<URL> e = ManagerManifestConfig.class.getClassLoader().getResources("META-INF/MANIFEST.MF"); e.hasMoreElements(); ) {
            URL url = e.nextElement();
            try (InputStream stream = url.openStream()) {
                Manifest manifest = null;
                if (stream != null) {
                    manifest = new Manifest(stream);
                    String value = manifest.getMainAttributes().getValue(name);
                    if (value != null) {
                        return value;
                    }
                }
            }
        }

        throw ManagerLogger.LOGGER.manifestEntryNotFound(name);
    }

    public String getManagerVersion() {
        return managerVersion;
    }

    public List<String> getPatchStreamKeys() {
        return patchStreamKeys;
    }

    public String getCurrentPatchStreamKey() {
        return currentPatchStreamKey;
    }

    public XPConfig getXpConfig(String key) {
        return xpConfigs.get(key);
    }

    public XPConfig getCurrentXpConfig() {
        return getXpConfig(currentPatchStreamKey);
    }

    static class XPConfig {
        private final String key;
        private final String version;
        private final ServerVersion minimumBaseServerVersion;
        private final List<String> patchStreamNames;
        private final List<String> moduleLayerNames;
        private final List<String> patchNamePatterns;

        private XPConfig(
                String key,
                String version,
                ServerVersion minimumBaseServerVersion,
                List<String> patchStreamNames,
                List<String> moduleLayerNames,
                List<String> patchNamePatterns) {
            this.key = key;
            this.version = version;
            this.minimumBaseServerVersion = minimumBaseServerVersion;
            this.patchStreamNames = Collections.unmodifiableList(patchStreamNames);
            this.moduleLayerNames = Collections.unmodifiableList(moduleLayerNames);
            this.patchNamePatterns = Collections.unmodifiableList(patchNamePatterns);
        }

        String getKey() {
            return key;
        }

        String getVersion() {
            return version;
        }

        List<String> getPatchStreamNames() {
            return patchStreamNames;
        }

        List<String> getModuleLayerNames() {
            return moduleLayerNames;
        }

        public List<String> getPatchNamePatterns() {
            return patchNamePatterns;
        }

        public ServerVersion getMinimumBaseServerVersion() {
            return minimumBaseServerVersion;
        }
    }
}
