package org.jboss.eap.util.xp.patch.stream.manager.server.wrapper;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.net.Socket;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;

import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.client.Operation;
import org.jboss.dmr.ModelNode;
import org.jboss.eap.util.xp.patch.stream.manager.ManagerLogger;
import org.jboss.eap.util.xp.patch.stream.manager.OperationBuilder;
import org.wildfly.core.embedded.Configuration;
import org.wildfly.core.embedded.EmbeddedProcessFactory;
import org.wildfly.core.embedded.EmbeddedProcessStartException;
import org.wildfly.core.embedded.StandaloneServer;

/**
 * @author <a href="mailto:kabir.khan@jboss.com">Kabir Khan</a>
 */
public class ServerWrapperRemote implements Runnable {
    private final Configuration configuration;
    private StandaloneServer server;
    private ModelControllerClient client;
    private Path jbossHome;
    private int port;
    private volatile boolean closed;

    private ServerWrapperRemote(Path jbossHome, Path modulesDir, int port) {
        this.jbossHome = jbossHome;
        this.port = port;
        configuration = Configuration.Builder.of(jbossHome.toFile())
                .setModulePath(modulesDir.toString())
                .addSystemPackages("org.jboss.logging", "org.jboss.logmanager")
                .setCommandArguments("--admin-only")
                .build();
    }

    public static void main(String[] args) {
        System.out.println(ManagerLogger.LOGGER.startingExternalProcessContainingEmbeddedServer());
        // This will be called internally so we don't need validation + i18n
        if (args.length != 3) {
            throw new IllegalStateException(Arrays.toString(args));
        }
        Path jbossHome = Paths.get(args[0]);
        Path modulesDir = Paths.get(args[1]);
        int port = Integer.valueOf(args[2]);
        ServerWrapperRemote main = new ServerWrapperRemote(jbossHome, modulesDir, port);
        Thread t = new Thread(main);
        t.start();
    }

    @Override
    public void run() {
        System.out.println(ManagerLogger.LOGGER.connectingToParentProcessOnPort(port));
        try {
            try (Socket socket = new Socket((String) null, port)) {
                System.out.println(ManagerLogger.LOGGER.connectedToParentProcess());

                try (DataOutputStream out = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()))) {
                    try (DataInputStream in = new DataInputStream(new BufferedInputStream(socket.getInputStream()))) {
                        String input = in.readUTF();
                        while (input != null) {
                            handleCommand(out, input);
                            input = in.readUTF();
                        }
                    } catch (EOFException e) {
                        if (!closed) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void handleCommand(DataOutputStream out, String input) {
        try {
            if (input.equals(Protocol.START_CMD)) {
                start();
                sendResponse(out, Protocol.OK_RESPONSE, null);
                return;
            } else if (input.equals(Protocol.STOP_CMD)) {
                stop();
                sendResponse(out, Protocol.OK_RESPONSE, null);
                return;
            } else if (input.startsWith(Protocol.EXECUTE_OPERATION_CMD)) {
                String payload = Protocol.extractPayload(input);
                ModelNode op = ModelNode.fromJSONString(payload);
                ModelNode response = execute(op);
                sendResponse(out, Protocol.OK_RESPONSE, response.toJSONString(true));
                return;
            } else if (input.startsWith(Protocol.APPLY_PATCH_CMD)) {
                String payload = Protocol.extractPayload(input);
                Path patchPath = Paths.get(payload);
                applyPatch(patchPath);
                sendResponse(out, Protocol.OK_RESPONSE, null);
                return;
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        sendResponse(out, Protocol.ERROR_RESPONSE, null);
    }


    private void start() throws EmbeddedProcessStartException {
        System.out.println(ManagerLogger.LOGGER.startingEmbeddedServer(jbossHome));
        if (server != null) {
            throw new IllegalStateException();
        }
        server = EmbeddedProcessFactory.createStandaloneServer(configuration);
        server.start();
        client = server.getModelControllerClient();
    }

    private ModelNode execute(ModelNode operation) throws IOException {
        ModelNode response = client.execute(operation);
        return response;
    }

    private ModelNode execute(Operation operation) throws IOException {
        ModelNode response = client.execute(operation);
        return response;
    }


    private void applyPatch(Path patch) throws Exception {
        ModelNode operation = new OperationBuilder("patch")
                .addr("core-service", "patching")
                .param("override-modules", "false")
                .param("override-all", "false")
                .param("input-stream-index", "0")
                .build();

        final org.jboss.as.controller.client.OperationBuilder operationBuilder = org.jboss.as.controller.client.OperationBuilder.create(operation);
        operationBuilder.addFileAsAttachment(patch.toFile());
        execute(operationBuilder.build());
        System.out.println(ManagerLogger.LOGGER.patchAppliedSuccessFully(patch.getFileName().toString()));
    }


    void stop() throws Exception {
        System.out.println(ManagerLogger.LOGGER.stoppingExternalServerInExternalProcess());
        closed = true;
        try {
            if (client != null) {
                client.close();
            }
        } finally {
            client = null;
            StandaloneServer server = this.server;
            this.server = null;
            if (server != null) {
                server.stop();
            }
        }
    }

    private void sendResponse(DataOutputStream out, String response, String payload) {
        try {
            out.writeUTF(Protocol.formatWithPayload(response, payload));
            out.flush();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
