/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.processor;

import java.util.Map;

import org.apache.camel.CamelContext;
import org.apache.camel.Component;
import org.apache.camel.Consumer;
import org.apache.camel.ContextTestSupport;
import org.apache.camel.Endpoint;
import org.apache.camel.Exchange;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.support.DefaultComponent;
import org.apache.camel.support.DefaultConsumer;
import org.apache.camel.support.DefaultEndpoint;
import org.junit.jupiter.api.Test;

public class ConsumerRouteIdAwareTest extends ContextTestSupport {

    @Override
    protected RouteBuilder createRouteBuilder() throws Exception {
        return new RouteBuilder() {
            @Override
            public void configure() throws Exception {
                context.addComponent("my", new MyComponent(context));

                from("my:foo").routeId("foo").to("mock:result");
            }
        };
    }

    @Test
    public void testRouteIdAware() throws Exception {
        getMockEndpoint("mock:result").expectedBodiesReceived("Hello from consumer route foo");

        assertMockEndpointsSatisfied();
    }

    private class MyComponent extends DefaultComponent {

        public MyComponent(CamelContext context) {
            super(context);
        }

        @Override
        protected Endpoint createEndpoint(String uri, String remaining, Map<String, Object> parameters) throws Exception {
            return new MyEndpoint(uri, this);
        }
    }

    private class MyEndpoint extends DefaultEndpoint {

        public MyEndpoint(String endpointUri, Component component) {
            super(endpointUri, component);
        }

        @Override
        public Producer createProducer() throws Exception {
            throw new UnsupportedOperationException("Not supported");
        }

        @Override
        public Consumer createConsumer(Processor processor) throws Exception {
            return new MyConsumer(this, processor);
        }
    }

    private class MyConsumer extends DefaultConsumer {

        public MyConsumer(Endpoint endpoint, Processor processor) {
            super(endpoint, processor);

            Runnable run = () -> {
                Exchange exchange = endpoint.createExchange();
                exchange.getMessage().setBody("Hello from consumer route " + getRouteId());
                try {
                    Thread.sleep(100);
                    processor.process(exchange);
                } catch (Exception e) {
                    // ignore
                }
            };
            Thread t = new Thread(run);
            t.start();
        }

    }

}
