/*
 * Copyright 2020 Red Hat
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.apicurio.registry.connector.util;

import com.google.protobuf.Message;
import com.google.protobuf.Parser;
import io.apicurio.registry.utils.kafka.ProtoSerde;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaAndValue;
import org.apache.kafka.connect.storage.Converter;

import java.util.Map;

/**
 * @author Ales Justin
 */
public class ProtoConverter<T extends Message> implements Converter {

    private final ProtoSerde<T> serde;

    public ProtoConverter(Parser<T> parser) {
        this.serde = ProtoSerde.parsedWith(parser);
    }

    @Override
    public void configure(Map<String, ?> configs, boolean isKey) {
        serde.configure(configs, isKey);
    }

    @Override
    public byte[] fromConnectData(String topic, Schema schema, Object value) {
        //noinspection unchecked
        return serde.serialize(topic, (T) value);
    }

    @Override
    public SchemaAndValue toConnectData(String topic, byte[] bytes) {
        T value = serde.deserialize(topic, bytes);
        return new SchemaAndValue(null, value);
    }
}
