package org.jboss.brmsbpmsuite.patching.client;

import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.attribute.FileTime;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.core.Is.is;

public class ConfigurationPatcherTest {

    private static final File TEMP_DIR = new File("./target/tmp-tests-dir");
    private static final File PATCHES_DIR = createFileFromResource("/config-test/updates");
    private static final File SERVER_CONFIG_DIR = createFileFromResource("/config-test/eap-servers/eap-version-1/standalone/configuration");

    private ConfigurationPatcher testedConfigPatcher;

    @Before
    public void setUp() throws ParserConfigurationException, TransformerConfigurationException, IOException {
        TEMP_DIR.mkdirs();
        FileUtils.copyDirectory(SERVER_CONFIG_DIR, TEMP_DIR);
        testedConfigPatcher = new ConfigurationPatcher(PATCHES_DIR, TEMP_DIR);
    }

    @After
    public void tearDown() throws IOException {
        FileUtils.deleteDirectory(TEMP_DIR);
    }

    @Test
    public void shouldNotApplyPatchesToFilesNotInEAPInstall() throws IOException {
        testedConfigPatcher.apply();

        File file = new File(TEMP_DIR, "standalone-4.xml");
        assertThat(file.exists(), is(false));
    }

    @Test
    public void shouldApplyPatchesToFilesInEAPInstall() throws IOException {
        File file = new File(TEMP_DIR, "standalone-2.xml");
        Long sizeBeforePatch = Files.size(file.toPath());
        FileTime modifiedTimeBeforePatch = Files.getLastModifiedTime(file.toPath());

        testedConfigPatcher.apply();

        assertThat(Files.size(file.toPath()), is(greaterThan(sizeBeforePatch)));
        assertThat(Files.getLastModifiedTime(file.toPath()), is(greaterThan(modifiedTimeBeforePatch)));
    }

    @Test
    public void shouldNotBackupNotPatchedFiles() throws IOException {
        File backupDir = new File(TEMP_DIR, "backup");
        backupDir.mkdirs();

        testedConfigPatcher.backup(backupDir);

        File file = new File(backupDir, "standalone.xml");
        assertThat(file.exists(), is(false));
    }

    @Test
    public void shouldBackupPatchedFiles() throws IOException {
        File backupDir = new File(TEMP_DIR, "backup");
        backupDir.mkdirs();

        testedConfigPatcher.backup(backupDir);

        File file = new File(backupDir, "configuration/standalone-3.xml");
        assertThat(file.exists(), is(true));
    }

    private static File createFileFromResource(String filePath) {
        return new File(ConfigurationDiffsGeneratorTest.class.getResource(filePath).getPath());
    }

}