package com.redhat.installer.asconfiguration.processpanel.postinstallation;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.redhat.installer.installation.processpanel.ProcessPanelHelper;
import org.apache.commons.io.IOUtils;

import java.io.*;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Jsf extends PostInstallation {

    @Override
    protected Class getClassName() {
        return Jsf.class;
    }

    @Override
    protected boolean performOperation() {
        return installJsf(arguments);
    }

    /**
     * Copies a list of files into the appropriate module directory. Creates the directory if it doesn't exist
     *
     * @param jarFile  the jar implementation to copy
     * @param moduleDir the location to copy to
     * @return
     */
    private static boolean createJsfFile(String jarFile, String moduleDir, String filename) {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        try {
            File moduleFile = createDirectory(idata.getInstallPath(), moduleDir);
            File jar = new File(jarFile);
            boolean fromUrl = jarFile.startsWith("http://") || jarFile.startsWith("ftp://") || jarFile.startsWith("https://")
                    || jarFile.startsWith("file://");

            try (InputStream in = fromUrl ? new BufferedInputStream(new URL(jarFile).openStream()) : new FileInputStream(jar)) {
                moveFileToDir(in, filename, moduleFile.getPath());
            }

        } catch (Throwable e) {
            ProcessPanelHelper.printToPanel(mHandler,
                    String.format(idata.langpack.getString("postinstall.processpanel.jsfCopy.failure"), jarFile, moduleDir), true);
            ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
            return false;
        }
        return true;
    }

    /**
     * Creates a module xml in the given location.
     *
     * @param moduleDirectory the location to create the 'module.xml'
     * @param toSet           of the form {decorator -> valueToSet}
     * @return
     */
     private static boolean createJsfModuleXml(String moduleDirectory, HashMap<String, String> toSet, String templateName ) {
         AutomatedInstallData idata = AutomatedInstallData.getInstance();
         String jbossHome = idata.getInstallPath();
         String moduleXmlPath = jbossHome + File.separator + moduleDirectory;
         File xmlModule = new File(moduleXmlPath + File.separator + "module.xml");
         if (!xmlModule.exists()) {
             try (InputStream xmlTemplate = Jsf.class.getClassLoader().getResourceAsStream("XMLTemplates/"+templateName);
                  FileOutputStream fileOutputStream = new FileOutputStream(xmlModule.getPath())) {
                 IOUtils.copy(xmlTemplate, fileOutputStream);

                 //find and replace decorators in template
                 Path path = Paths.get(xmlModule.getPath());
                 Charset charset = StandardCharsets.UTF_8;
                 String content;

                 for (Map.Entry<String, String> setDecorators : toSet.entrySet()) {
                     String decorator = setDecorators.getKey();
                     String value = setDecorators.getValue();
                     content = new String(Files.readAllBytes(path), charset);
                     content = content.replaceAll(decorator, value);
                     Files.write(path, content.getBytes(charset));
                 }

                 ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("postinstall.processpanel.jsfCopyXml.success"), xmlModule.getPath()), false);
                 return true;

             } catch (IOException e) {
                 ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("postinstall.processpanel.jsfCopyXml.failure"), xmlModule.getPath()), true);
                 ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
                 return false;
             }
         }
         return true;
     }

    private static boolean installJsfImp(){
        AutomatedInstallData idata = AutomatedInstallData.getInstance();

        String tmpPath = idata.getVariable("jsfimpl.input.tmp");
        String path = tmpPath == null ? idata.getVariable("jsfimpl.input") : tmpPath;
        String jsfProjectName = idata.getVariable("jsf.project.name");
        String jsfVersion = idata.getVariable("jsf.version");
        String jsfDirStruct= "modules"+File.separator+"com"+File.separator+"sun"+File.separator+"jsf-impl"+File.separator+""+jsfProjectName+'-'+jsfVersion;
        Boolean isDefault = idata.getVariable("jsfDefault") != null && Boolean.parseBoolean(idata.getVariable("jsfDefault"));

        if (!createJsfFile(path, jsfDirStruct, jsfProjectName+"-impl-"+jsfVersion+".jar")) {
            return false;
        }

        String template;
        HashMap <String, String> toSet = new HashMap<>();

        toSet.put("\\$\\{jsf-impl-name\\}", jsfProjectName);
        toSet.put("\\$\\{jsf-version\\}", jsfVersion);

        template = (jsfProjectName.equals("mojarra"))? "mojarra-impl-module.xml" : "myfaces-impl-module.xml";

        if (!createJsfModuleXml(jsfDirStruct, toSet, template)) {
            return false;
        }

        if(isDefault) {
            boolean result;
            result = serverCommands.installJsfDefault(jsfProjectName, jsfVersion);
            if (!result) {
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("postinstall.processpanel.jsf.error"), jsfProjectName+'-'+jsfVersion), true);
            } else {
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("postinstall.processpanel.jsf.success"), jsfProjectName+'-'+jsfVersion), false);
            }
            return result;
        }
        return true;
    }

    private static boolean installJsfApi(){

        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        String tmpPath = idata.getVariable("jsfapi.input.tmp");
        String path = tmpPath == null ? idata.getVariable("jsfapi.input"): tmpPath;
        String jsfProjectName = idata.getVariable("jsf.project.name");
        String jsfVersion = idata.getVariable("jsf.version");
        String jsfDirStruct = "modules"+File.separator+"javax"+File.separator+"faces"+File.separator+"api"+File.separator+"" + jsfProjectName + "-" + jsfVersion;

        if (!createJsfFile(path, jsfDirStruct, jsfProjectName+"-api-"+jsfVersion+".jar")) {
            return false;
        }
        String template;
        HashMap <String, String> toSet = new HashMap<>();

        toSet.put("\\$\\{jsf-impl-name\\}", jsfProjectName);
        toSet.put("\\$\\{jsf-version\\}", jsfVersion);

        template = (jsfProjectName.equals("mojarra"))? "mojarra-api-module.xml" : "myfaces-api-module.xml";

        if (!createJsfModuleXml(jsfDirStruct, toSet, template)) {
            return false;
        }
        return true;
    }

    private static String findFirstFilename(File dir, String startWith) {
        String[] files = dir.list((file, s) -> s.startsWith(startWith));

        if (files == null || files.length == 0) {
            return "";
        } else {
            return files[0];
        }
    }

    private static String extractSubstring(String regex, String str) {
        Pattern pattern = Pattern.compile(regex);
        Matcher matcher = pattern.matcher(str);
        String sub = "";
        if (matcher.find()) {
            sub = matcher.group(1);
        }
        return sub;
    }

    private static boolean installJsfInjection() {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        String jsfProjectName = idata.getVariable("jsf.project.name");
        String jsfVersion = idata.getVariable("jsf.version");

        String jbossHome = idata.getInstallPath();
        String jsfDirStruct = "modules"+File.separator+"org"+File.separator+"jboss"+File.separator+"as"+File.separator+"jsf-injection" + File.separator + jsfProjectName + "-" + jsfVersion;
        String wildflyModules = jbossHome + File.separator + "modules"+File.separator+"system"+File.separator+"layers"+File.separator+"base"+File.separator+"org"+File.separator+"jboss"+File.separator+"as"+File.separator+"jsf-injection"+File.separator+"main"+File.separator;

        File wildflyModulesDir = new File(wildflyModules);

        String weldCore = findFirstFilename(wildflyModulesDir, "weld-core-jsf");
        String wildfly = findFirstFilename(wildflyModulesDir, "wildfly-jsf-injection");

        if (weldCore.equals("") || wildfly.equals("")) {
            ProcessPanelHelper.printToPanel(mHandler,
                        String.format(idata.langpack.getString("postinstall.processpanel.jsf.wildfly.modules.not.found")), true);
            return false;
        }

        String versionWeldCore = extractSubstring("^weld-core-jsf-(.*?)\\.jar$", weldCore);
        String versionJbossAs = extractSubstring("^wildfly-jsf-injection-(.*?)\\.jar$", wildfly);

        if (versionJbossAs.equals("") || versionWeldCore.equals("")) {
            ProcessPanelHelper.printToPanel(mHandler,
                        String.format(idata.langpack.getString("postinstall.processpanel.jsf.wildfly.versions.not.found")), true);
            return false;
        }

        if (!createJsfFile(wildflyModulesDir + File.separator + weldCore, jsfDirStruct, weldCore) ||
                !createJsfFile(wildflyModulesDir + File.separator + wildfly, jsfDirStruct, wildfly)) {
            return false;
        }

        String template;
        HashMap <String, String> toSet = new HashMap<>();

        toSet.put("\\$\\{jsf-impl-name\\}", jsfProjectName);
        toSet.put("\\$\\{jsf-version\\}", jsfVersion);
        toSet.put("\\$\\{version.jboss.as\\}", versionJbossAs);
        toSet.put("\\$\\{version.weld.core\\}", versionWeldCore);

        template = (jsfProjectName.equals("mojarra"))? "mojarra-injection-module.xml" : "myfaces-injection-module.xml";

        return createJsfModuleXml(jsfDirStruct, toSet, template);

    }

    private static boolean installJsfDigester() {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        String jsfProjectName = idata.getVariable("jsf.project.name");

        if (jsfProjectName.contains("myfaces")) {
            String jsfDirStruct = "modules"+File.separator+"org"+File.separator+"apache"+File.separator+"commons"+File.separator+"digester"+File.separator+"main";
            String commonsDigester = "commons-digester-1.8.jar";

            try (InputStream in = Jsf.class.getClassLoader().getResourceAsStream("jars/" + commonsDigester)){
                File moduleFile = createDirectory(idata.getInstallPath(), jsfDirStruct);
                moveFileToDir(in, commonsDigester, moduleFile.getPath());

            } catch (Throwable e) {
                ProcessPanelHelper.printToPanel(mHandler,
                        String.format(idata.langpack.getString("postinstall.processpanel.jsfCopy.failure"), commonsDigester, jsfDirStruct), true);
                ProcessPanelHelper.printToPanel(mHandler, e.getMessage(), true);
                return false;
            }

            if (!createJsfModuleXml(jsfDirStruct, new HashMap<>(), "myfaces-digester-module.xml")) {
                return false;
            }
        }

        return true;
    }

    private static boolean installJsf(String[] args) {

        String flag = "";

        for (String arg : args) {
            if (arg.startsWith("--stage")) {
                flag = arg.split("=")[1];
            }
        }

        switch (flag){

            case "impl":
                return installJsfImp();
            case "api":
                return installJsfApi();
            case "injection":
                return installJsfInjection();
            case "digest":
                return installJsfDigester();
            default:
                return false;

        }

    }

     private static File createDirectory(String basePath, String modulePath) throws Throwable {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();
        File location = new File(basePath, modulePath);
        if (!location.exists()) {
            if (!location.mkdirs()){
                ProcessPanelHelper.printToPanel(mHandler, String.format(idata.langpack.getString("directory.creation.failed"), location.toString()), true);
            }
        }
        return location;
    }

    private static File moveFileToDir(InputStream in, String filename, String destDir) throws Throwable {
        AutomatedInstallData idata = AutomatedInstallData.getInstance();

        File destination = new File(destDir);
        File fileLoc = new File(destination.getPath() + File.separator + filename);

        if (!fileLoc.exists()) {

            try (OutputStream out = new FileOutputStream(fileLoc)) {
                byte[] buffer = new byte[1024];
                int l;
                while ((l = in.read(buffer, 0, 1024)) > 0) {
                    out.write(buffer, 0, l);
                }
            }
            ProcessPanelHelper.printToPanel(mHandler,
                    String.format(idata.langpack.getString("postinstall.processpanel.jsfCopy.success"), filename, destination.getPath()), false);
        }

        return fileLoc;
    }
}

