/*
 * IzPack - Copyright 2001-2008 Julien Ponge, All Rights Reserved.
 * 
 * http://izpack.org/
 * http://izpack.codehaus.org/
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 *     
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.izforge.izpack.panels;

import com.izforge.izpack.gui.*;
import com.izforge.izpack.installer.InstallData;
import com.izforge.izpack.installer.InstallerFrame;
import com.izforge.izpack.installer.IzPanel;
import com.izforge.izpack.installer.ResourceManager;

import javax.accessibility.AccessibleContext;
import javax.swing.*;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.net.URL;
import java.util.Enumeration;

/**
 * The IzPack HTML license panel.
 *
 * @author Julien Ponge
 */
public class HTMLLicencePanel extends IzPanel implements HyperlinkListener, ActionListener, FocusListener {

    /**
     *
     */
    private static final long serialVersionUID = 3256728385458746416L;

    /**
     * The text area.
     */
    private JEditorPane textArea;

    /**
     * The radio buttons.
     */
    private JRadioButton yesRadio;
    private JRadioButton noRadio;

    final ButtonGroup group = new ButtonGroup();

    /**
     * The constructor.
     *
     * @param idata  The installation data.
     * @param parent Description of the Parameter
     */
    public HTMLLicencePanel(InstallerFrame parent, InstallData idata)
    {
        super(parent, idata, new GridBagLayout());
        buildUI();
    }

    private void setFocus() {
        for (Enumeration<AbstractButton> buttons = group.getElements(); buttons.hasMoreElements(); ) {
            AbstractButton button = buttons.nextElement();
            if (button.isSelected()) {
                setInitialFocus(button);
                return;
            }
        }
        setInitialFocus(noRadio);
    }

    private void buildUI() {
        add(getTitle(), GridBagConstraintsFactory.getTitleConstraints());
        addScroller();
        addButtons();
    }

    private void addButtons() {
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 2;
        add(getRadioButtons(),gbc);
    }

    private void addScroller() {
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        gbc.gridy = 1;
        gbc.insets.right = 10;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1;
        gbc.weighty = 1;
        add(getLicenceScroller(),gbc);
    }

    private JPanel getRadioButtons() {
        JPanel buttonPanel = PanelFactory.createRadioButtonPanel(!parent.hasBackground, group);
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(0, 16, 0, 0));
        Box verticalBox = Box.createVerticalBox();
        GridBagConstraints gbc = GridBagConstraintsFactory.getBasicConstraints();
        gbc.insets.top = 0;
        gbc.insets.bottom = 0;

        yesRadio = new JRadioButton(parent.langpack.getString("LicencePanel.agree"), false);
        yesRadio.setOpaque(!parent.hasBackground);
        yesRadio.setFont(FontResources.getOpenSansRegular());
        group.add(yesRadio);
        verticalBox.add(yesRadio, gbc);
        gbc.gridy++;
        yesRadio.addFocusListener(this);

        noRadio = new JRadioButton(parent.langpack.getString("LicencePanel.notagree"), true);
        noRadio.setOpaque(!parent.hasBackground);
        noRadio.setFont(FontResources.getOpenSansRegular());
        group.add(noRadio);
        verticalBox.add(noRadio, gbc);
        noRadio.addFocusListener(this);

        buttonPanel.add(verticalBox);

        /*
         * AccessibleContext and ActionCommand setting for marathon automated testing
         */
        AccessibleContext ac = yesRadio.getAccessibleContext();
        ac.setAccessibleDescription("JRadioButton to accept the licence agreement");
        yesRadio.setActionCommand("Selected =  Accept the licence agreement");
        ac = yesRadio.getAccessibleContext();
        ac.setAccessibleDescription("JRadioButton to reject the licence agreement");
        noRadio.setActionCommand("Selected = Reject the licence agreement");
        return buttonPanel;
    }

    private Component getLicenceScroller() {
        JScrollPane scroller = null;
        try
        {
            textArea = new JEditorPane();
            textArea.setFont(FontResources.getOpenSansRegular());
            textArea.setEditable(false);
            textArea.addHyperlinkListener(this);
            scroller = ScrollPaneFactory.createScroller(textArea);
            scroller.setFocusable(true);
            textArea.setPage(loadLicence());
            textArea.setFocusable(false);
        }
        catch (Exception err)
        {
            err.printStackTrace();
        }
        return scroller;
    }

    private JLabel getTitle() {
        JLabel title = LabelFactory.createTitleLabel(idata.langpack.getString("HTMLLicencePanel.headline"),!parent.hasBackground);
        return title;
    }

    /**
     * Loads the license text.
     *
     * @return The license text URL.
     */
    private URL loadLicence()
    {
        String resNamePrifix = "HTMLLicencePanel.licence";
        try
        {
            return ResourceManager.getInstance().getURL(resNamePrifix);
        }
        catch (Exception ex)
        {
            ex.printStackTrace();
        }
        return null;
    }

    /**
     * Actions-handling method (here it launches the installation).
     *
     * @param e The event.
     */
    public void actionPerformed(ActionEvent e)
    {
        if (yesRadio.isSelected())
        {
            parent.unlockNextButton();
        }
        else
        {
            parent.lockNextButton();
        }
    }

    /**
     * Indicates wether the panel has been validated or not.
     *
     * @return true if the user agrees with the license, false otherwise.
     */
    public boolean isValidated()
    {
        if (noRadio.isSelected())
        {
            parent.exit();
            return false;
        }
        return (yesRadio.isSelected());
    }

    /**
     * Hyperlink events handler.
     *
     * @param e The event.
     */
    public void hyperlinkUpdate(HyperlinkEvent e)
    {
        try
        {
            if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED)
            {
                try {
                    Desktop.getDesktop().browse(e.getURL().toURI());
                } catch (Exception f) {
                    textArea.setPage(e.getURL());
                }
            }
        }
        catch (Exception err)
        {
            // TODO: Extend exception handling.
        }
    }

    /**
     * Called when the panel becomes active.
     */
    public void panelActivate()
    {
        if (!yesRadio.isSelected())
        {
            parent.lockNextButton();
        }
        setFocus();
    }

    @Override
    public void focusGained(FocusEvent focusEvent) {
        JRadioButton b = (JRadioButton)focusEvent.getComponent();
        b.doClick();
        if(focusEvent.getComponent() == yesRadio) parent.unlockNextButton(false);
        if(focusEvent.getComponent() == noRadio) parent.lockNextButton();
    }

    @Override
    public void focusLost(FocusEvent focusEvent) {
    }
}
